package org.apache.lucene.store;

/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import java.io.IOException;

/**
 * A memory-resident {@link IndexOutput} implementation.
 *
 * @version $Id: RAMOutputStream.java 488330 2006-12-18 16:45:29Z mikemccand $
 */

public class RAMOutputStream extends BufferedIndexOutput {
  private RAMFile file;
  private long pointer = 0;

  /** Construct an empty output buffer. */
  public RAMOutputStream() {
    this(new RAMFile());
  }

  RAMOutputStream(RAMFile f) {
    file = f;
  }

  /** Copy the current contents of this buffer to the named output. */
  public void writeTo(IndexOutput out) throws IOException {
    flush();
    final long end = file.length;
    long pos = 0;
    int buffer = 0;
    while (pos < end) {
      int length = BUFFER_SIZE;
      long nextPos = pos + length;
      if (nextPos > end) {                        // at the last buffer
        length = (int)(end - pos);
      }
      out.writeBytes((byte[])file.buffers.get(buffer++), length);
      pos = nextPos;
    }
  }

  /** Resets this to an empty buffer. */
  public void reset() {
    try {
      seek(0);
    } catch (IOException e) {                     // should never happen
      throw new RuntimeException(e.toString());
    }

    file.setLength(0);
  }

  public void flushBuffer(byte[] src, int len) throws IOException {
    byte[] buffer;
    int bufferPos = 0;
    while (bufferPos != len) {
      int bufferNumber = (int)(pointer/BUFFER_SIZE);
      int bufferOffset = (int)(pointer%BUFFER_SIZE);
      int bytesInBuffer = BUFFER_SIZE - bufferOffset;
      int remainInSrcBuffer = len - bufferPos;
      int bytesToCopy = bytesInBuffer >= remainInSrcBuffer ? remainInSrcBuffer : bytesInBuffer;

      if (bufferNumber == file.buffers.size())
        buffer = file.addBuffer(BUFFER_SIZE);
      else
        buffer = (byte[]) file.buffers.get(bufferNumber);

      System.arraycopy(src, bufferPos, buffer, bufferOffset, bytesToCopy);
      bufferPos += bytesToCopy;
      pointer += bytesToCopy;
    }

    if (pointer > file.length)
      file.setLength(pointer);

    file.setLastModified(System.currentTimeMillis());
  }

  public void close() throws IOException {
    super.close();
  }

  public void seek(long pos) throws IOException {
    super.seek(pos);
    pointer = pos;
  }
  public long length() {
    return file.length;
  }
}
