/*
 * JBoss, Home of Professional Open Source
 * Copyright 2006, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */

package org.jboss.test.remoting.transport.multiplex.config;

import java.io.ByteArrayOutputStream;
import java.lang.reflect.Field;
import java.nio.ByteBuffer;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;

import org.jboss.remoting.transport.Connector;
import org.jboss.remoting.transport.multiplex.InputMultiplexor;
import org.jboss.remoting.transport.multiplex.Multiplex;
import org.jboss.remoting.transport.multiplex.MultiplexServerInvoker;
import org.jboss.remoting.transport.multiplex.MultiplexingManager;
import org.jboss.remoting.transport.multiplex.OutputMultiplexor;
import org.jboss.remoting.transport.multiplex.VirtualServerSocket;


/**
 * 
 * @author <a href="mailto:ron.sigal@jboss.com">Ron Sigal</a>
 * @version $Revision$
 * <p>
 * Copyright (c) May 19, 2006
 * </p>
 */
public class ConfigurationByInvokerSettersTestClient extends ConfigurationTestClient
{

   /**
    *  In this test a Connector is configured with a configuration Map
    *  and then the parameters are set by way of MultiplexServerInvoker setter
    *  methods.  The Connector is started and the parameters of the MultiplexingManager,
    *  InputMultiplexor, and OutputMultiplexor are tested to verify that
    *  the values set in the MultiplexServerInvoker override the values
    *  in the configuration Map.
    *  
    *  Note that the setter methods are called after Connector.create() but before
    *  Connector.start().  Other than maxAcceptErrors, all of these parameters are
    *  passed to MultiplexingManager, and, in the case of a master MultiplexServerInvoker,
    *  the first MultiplexingManager created by the MultiplexServerInvoker is created
    *  when the MasterServerSocket processes the priming socket, which can happen any
    *  time after Connector.start().  
    *  
    *  In the case of a virtual MultiplexServerInvoker, the first  MultiplexingManager
    *  is also created for the priming socket, but the priming socket is created during
    *  Connector.create().
    *  
    *  In general, then, setting parameters by way of setter methods is guaranteed to
    *  have an effect only up until Connector.create() is called.
    */
   public void testConfigurationByInvokerSetters()
   {
      log.info("entering " + getName());
      
      try
      {
         // Create HashMap with parameters.
         HashMap conf = new HashMap();
         
         // MultiplexingManager parameters
         conf.put(Multiplex.STATIC_THREADS_MONITOR_PERIOD,
                  new Integer(2 + Multiplex.STATIC_THREADS_MONITOR_PERIOD_DEFAULT));
         conf.put(Multiplex.SHUTDOWN_REQUEST_TIMEOUT,
                  new Integer(3 + Multiplex.SHUTDOWN_REQUEST_TIMEOUT_DEFAULT));
         conf.put(Multiplex.SHUTDOWN_REFUSALS_MAXIMUM,
                  new Integer(4 + Multiplex.SHUTDOWN_REFUSALS_MAXIMUM_DEFAULT));
         conf.put(Multiplex.SHUTDOWN_MONITOR_PERIOD,
                  new Integer(5 + Multiplex.SHUTDOWN_MONITOR_PERIOD_DEFAULT));
         
         // InputMultiplexor parameters
         conf.put(Multiplex.INPUT_BUFFER_SIZE,
                  new Integer(2 + Multiplex.INPUT_BUFFER_SIZE_DEFAULT));
         conf.put(Multiplex.INPUT_MAX_ERRORS,
                  new Integer(3 + Multiplex.INPUT_MAX_ERRORS_DEFAULT));
         
         // OutputMultiplexor parameters
         conf.put(Multiplex.OUTPUT_MESSAGE_POOL_SIZE,
                  new Integer(2 + Multiplex.OUTPUT_MESSAGE_POOL_SIZE_DEFAULT));
         conf.put(Multiplex.OUTPUT_MESSAGE_SIZE,
                  new Integer(3 + Multiplex.OUTPUT_MESSAGE_SIZE_DEFAULT));
         conf.put(Multiplex.OUTPUT_MAX_CHUNK_SIZE,
                  new Integer(4 + Multiplex.OUTPUT_MAX_CHUNK_SIZE_DEFAULT));
         conf.put(Multiplex.OUTPUT_MAX_TIME_SLICE,
                  new Integer(5 + Multiplex.OUTPUT_MAX_TIME_SLICE_DEFAULT));
         conf.put(Multiplex.OUTPUT_MAX_DATA_SLICE,
                  new Integer(6 + Multiplex.OUTPUT_MAX_DATA_SLICE_DEFAULT));
         
         // MultiplexServerInvoker parameter
         conf.put(Multiplex.MAX_ACCEPT_ERRORS,
                  new Integer(2 + Multiplex.MAX_ACCEPT_ERRORS_DEFAULT));
         
         // Create Connector.
         Connector connector = new Connector(ConfigurationTestServer.connectorURI5, conf);
         connector.create();

         // Set parameter values through master MultiplexServerInvoker.
         MultiplexServerInvoker masterServerInvoker = (MultiplexServerInvoker) connector.getServerInvoker();
         
         // MultiplexingManager parameters:
         masterServerInvoker.setStaticThreadsMonitorPeriod(4 + Multiplex.STATIC_THREADS_MONITOR_PERIOD_DEFAULT);
         masterServerInvoker.setShutdownRequestTimeout(5 + Multiplex.SHUTDOWN_REQUEST_TIMEOUT_DEFAULT);
         masterServerInvoker.setShutdownRefusalsMaximum(6 + Multiplex.SHUTDOWN_REFUSALS_MAXIMUM_DEFAULT);
         masterServerInvoker.setShutdownMonitorPeriod(7 + Multiplex.SHUTDOWN_MONITOR_PERIOD_DEFAULT);
         
         // InputMultiplexor parameters:
         masterServerInvoker.setInputBufferSize(4 + Multiplex.INPUT_BUFFER_SIZE_DEFAULT);
         masterServerInvoker.setInputMaxErrors(5 + Multiplex.INPUT_MAX_ERRORS_DEFAULT);
        
         // OutputMultiplexor parameters:
         masterServerInvoker.setOutputMessagePoolSize(4 + Multiplex.OUTPUT_MESSAGE_POOL_SIZE_DEFAULT);
         masterServerInvoker.setOutputMessageSize(5 + Multiplex.OUTPUT_MESSAGE_SIZE_DEFAULT);
         masterServerInvoker.setOutputMaxChunkSize(6 + Multiplex.OUTPUT_MAX_CHUNK_SIZE_DEFAULT);
         masterServerInvoker.setOutputMaxTimeSlice(7 + Multiplex.OUTPUT_MAX_TIME_SLICE_DEFAULT);
         masterServerInvoker.setOutputMaxDataSlice(8 + Multiplex.OUTPUT_MAX_DATA_SLICE_DEFAULT);
         
         // MultiplexServerInvoker parameter:
         masterServerInvoker.setMaxAcceptErrors(4 + Multiplex.MAX_ACCEPT_ERRORS_DEFAULT);
         
         connector.start();
         log.info("started Connector on: " + connector.getInvokerLocator());

         // Tell test server to connect.
         os.write(29);
         
         /////////////////////////////////////////////////////////////////////////////////////
         // Get MultiplexingManager.
         while (masterServerInvoker.getServerInvokers().isEmpty())
         {
            try
            {
               Thread.sleep(1000);
            }
            catch (Exception e) {}
         }
         
         Collection virtualServerInvokers = masterServerInvoker.getServerInvokers();
         MultiplexServerInvoker virtualServerInvoker = (MultiplexServerInvoker) virtualServerInvokers.iterator().next();
         VirtualServerSocket vss = (VirtualServerSocket) virtualServerInvoker.getServerSocket();
         MultiplexingManager manager = vss.getMultiplexingManager();

         // Test MultiplexingManager parameters.
         Field field = MultiplexingManager.class.getDeclaredField("staticThreadsMonitorPeriod");
         field.setAccessible(true);
         assertEquals(field.getInt(manager), 4 + Multiplex.STATIC_THREADS_MONITOR_PERIOD_DEFAULT);
         
         field = MultiplexingManager.class.getDeclaredField("shutdownRequestTimeout");
         field.setAccessible(true);
         assertEquals(field.getInt(manager), 5 + Multiplex.SHUTDOWN_REQUEST_TIMEOUT_DEFAULT);
         
         field = MultiplexingManager.class.getDeclaredField("shutdownRefusalsMaximum");
         field.setAccessible(true);
         assertEquals(field.getInt(manager), 6 + Multiplex.SHUTDOWN_REFUSALS_MAXIMUM_DEFAULT);
         
         field = MultiplexingManager.class.getDeclaredField("shutdownMonitorPeriod");
         field.setAccessible(true);
         assertEquals(field.getInt(manager), 7 + Multiplex.SHUTDOWN_MONITOR_PERIOD_DEFAULT);
         
         /////////////////////////////////////////////////////////////////////////////////////
         // Get InputMultiplexor.
         field =  MultiplexingManager.class.getDeclaredField("inputMultiplexor");
         field.setAccessible(true);
         InputMultiplexor inputMultiplexor = (InputMultiplexor) field.get(manager);
         
         // Test InputMultiplexor parameters.
         field = InputMultiplexor.class.getDeclaredField("bufferSize");
         field.setAccessible(true);
         assertEquals(field.getInt(inputMultiplexor), 4 + Multiplex.INPUT_BUFFER_SIZE_DEFAULT);
         
         field = InputMultiplexor.class.getDeclaredField("maxErrors");
         field.setAccessible(true);
         assertEquals(field.getInt(inputMultiplexor), 5 + Multiplex.INPUT_MAX_ERRORS_DEFAULT);
         
         /////////////////////////////////////////////////////////////////////////////////////
         // Get OutputMultiplexor.
         field =  MultiplexingManager.class.getDeclaredField("outputMultiplexor");
         field.setAccessible(true);
         OutputMultiplexor outputMultiplexor = (OutputMultiplexor) field.get(manager);
         
         // Test OutputMultiplexor parameters.
         field = OutputMultiplexor.class.getDeclaredField("messagePoolSize");
         field.setAccessible(true);
         assertEquals(field.getInt(outputMultiplexor), 4 + Multiplex.OUTPUT_MESSAGE_POOL_SIZE_DEFAULT);
         
         field = OutputMultiplexor.class.getDeclaredField("messagePool");
         field.setAccessible(true);
         List messagePool = (List) field.get(outputMultiplexor);
         assertTrue(messagePool.size() <=  4 + Multiplex.OUTPUT_MESSAGE_POOL_SIZE_DEFAULT);
          
         field = OutputMultiplexor.class.getDeclaredField("messageSize");
         field.setAccessible(true);
         assertEquals(field.getInt(outputMultiplexor), 5 + Multiplex.OUTPUT_MESSAGE_SIZE_DEFAULT);
         
         Object message = messagePool.get(0);
         field = message.getClass().getDeclaredField("baos");
         field.setAccessible(true);
         ByteArrayOutputStream baos = (ByteArrayOutputStream) field.get(message);
         field = ByteArrayOutputStream.class.getDeclaredField("buf");
         field.setAccessible(true);
         byte[] buf = (byte[]) field.get(baos);
         assertEquals(buf.length, 5 + Multiplex.OUTPUT_MESSAGE_SIZE_DEFAULT);
        
         field = OutputMultiplexor.class.getDeclaredField("maxChunkSize");
         field.setAccessible(true);
         assertEquals(field.getInt(outputMultiplexor), 6 + Multiplex.OUTPUT_MAX_CHUNK_SIZE_DEFAULT);
         
         field = OutputMultiplexor.class.getDeclaredField("buffer");
         field.setAccessible(true);
         ByteBuffer bb = (ByteBuffer) field.get(outputMultiplexor);
         field = OutputMultiplexor.class.getDeclaredField("HEADER_SIZE");
         field.setAccessible(true);
         int headerSize = ((Integer) field.get(outputMultiplexor)).intValue();
         assertEquals(bb.capacity(), 6 + Multiplex.OUTPUT_MAX_CHUNK_SIZE_DEFAULT + headerSize);
         
         field = OutputMultiplexor.class.getDeclaredField("maxTimeSlice");
         field.setAccessible(true);
         assertEquals(field.getInt(outputMultiplexor), 7 + Multiplex.OUTPUT_MAX_TIME_SLICE_DEFAULT);
         
         field = OutputMultiplexor.class.getDeclaredField("maxDataSlice");
         field.setAccessible(true);
         assertEquals(field.getInt(outputMultiplexor), 8 + Multiplex.OUTPUT_MAX_DATA_SLICE_DEFAULT);
         
         /////////////////////////////////////////////////////////////////////////////////////
         // Test MultiplexServerInvoker parameter.
         field =  MultiplexServerInvoker.class.getDeclaredField("maxAcceptErrors");
         field.setAccessible(true);
         assertEquals(field.getInt(masterServerInvoker), 4 + Multiplex.MAX_ACCEPT_ERRORS_DEFAULT);
         assertEquals(field.getInt(virtualServerInvoker), 4 + Multiplex.MAX_ACCEPT_ERRORS_DEFAULT);
         
         connector.stop();
         log.info(getName() + " PASSES");
      }
      catch (Exception e)
      {
         log.error(e);
         e.printStackTrace();
         fail();
         log.info(getName() + " FAILS");
      }
   }
}
