package jdepend.framework;

import java.io.*;
import java.util.*;

/**
 * The <code>FileManager</code> class is responsible for
 * extracting Java class files (<code>.class</code> files) 
 * from a collection of registered directories.
 *
 * @author <b>Mike Clark</b> (mike@clarkware.com)
 * @author Clarkware Consulting, Inc.
 */

public class FileManager {

	private ArrayList directories;
    private boolean acceptInnerClasses;
	
	/**
	 * Constructs an empty <code>FileManager</code> 
	 * instance.
	 */
	public FileManager() {
		directories = new ArrayList();
        acceptInnerClasses = true;
	}
    
    /**
     * Determines whether inner classes should be collected.
     * 
     * @param b <code>true</code> to collect inner classes;
     *          <code>false</code> otherwise.
     */
    public void acceptInnerClasses(boolean b) {
        acceptInnerClasses = b;
    }
	
	/**
	 * Adds the specified directory to the collection 
	 * of directories to be managed.
	 *
	 * @param name Directory name.
	 * @throws IOException If the directory is invalid.
	 */
	public void addDirectory(String name) throws IOException {
		
		File directory = new File(name);
		
		if (directory.isDirectory()) {
			directories.add(directory);
		} else { 
			throw new IOException("Invalid directory: " + name);
		}
	}

    /**
     * Indicates whether the specified file is
     * a valid file.
     *
     * @param file Candidate file.
     * @return <code>true</code> if the file is valid;
     *         <code>false</code> otherwise.
     */
    public boolean acceptFile(File file) {
        return acceptClassFile(file) || acceptJarFile(file);
    }
    
	/**
	 * Indicates whether the specified file is
	 * a valid Java class file.
	 *
	 * @param file Candidate file.
	 * @return <code>true</code> if the file is valid;
	 *         <code>false</code> otherwise.
	 */
	public boolean acceptClassFile(File file) {
		
		if (!file.isFile()) {
			return false;
		}
        
        return acceptClassFileName(file.getName());
	}
    
    /**
     * Indicates whether the specified file name 
     * is a valid Java class file.
     *
     * @param name Candidate file name.
     * @return <code>true</code> if the name is valid;
     *         <code>false</code> otherwise.
     */
    public boolean acceptClassFileName(String name) {
      
        if (!acceptInnerClasses) {
            if (name.toLowerCase().indexOf("$") > 0) {
                return false;
            }
        }
        
        if (!name.toLowerCase().endsWith(".class")) {
            return false;
        }
        
        return true;
    }

    /**
     * Indicates whether the specified file is
     * a valid jar file file.
     *
     * @param file Candidate file.
     * @return <code>true</code> if the file is valid;
     *         <code>false</code> otherwise.
     */
    public boolean acceptJarFile(File file) {
        
        if (!file.isFile()) {
            return false;
        }
        
        if (!file.getName().toLowerCase().endsWith(".jar") &&
            !file.getName().toLowerCase().endsWith(".zip") &&
            !file.getName().toLowerCase().endsWith(".war")) {
            return false;
        }
        
        return true;
    }
    	
	/**
	 * Returns the Java class files 
	 * contained in the managed directories.
	 *
	 * @return Collection of source and class files.
	 */
	public Collection extractFiles() {
		
		Collection files = new ArrayList();
			
		Iterator dirIter = directories.iterator();
		while (dirIter.hasNext()) {
			File directory = (File)dirIter.next();
            collectFiles(directory, files);
		}
		
		return files;
	}
	
    private void collectFiles(File directory, Collection files) {
		
		String[] directoryFiles = directory.list();
		
		for (int i = 0; i < directoryFiles.length; i++) {
			
			File file = new File(directory, directoryFiles[i]);
            if (acceptFile(file)) {
				addFile(file, files);
			} else if (file.isDirectory()) {
                collectFiles(file, files);
			}
		}
	}
	
    private void addFile(File f, Collection files) {
		if (!files.contains(f)) {
            files.add(f);
		}
	}
}
