/*
 * Copyright (c) 2005-2007 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.lafwidget.scroll;

import java.awt.event.HierarchyEvent;
import java.awt.event.HierarchyListener;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.JComponent;
import javax.swing.JScrollPane;
import javax.swing.plaf.basic.ComboPopup;

import org.jvnet.lafwidget.*;
import org.jvnet.lafwidget.preview.PreviewPainter;

/**
 * Widget that decorates scroll panes with selector.
 * 
 * @author Kirill Grouchnikov
 */
public class ScrollPaneSelectorWidget extends LafWidgetAdapter {
	/**
	 * The associated scroll pane.
	 */
	protected JScrollPane scrollPane;

	/**
	 * The scroll pane selector for the associated scroll pane.
	 */
	protected ScrollPaneSelector scrollPaneSelector;

	/**
	 * Hierarchy listener - remove the selector in the scroll pane of a combo
	 * popup.
	 */
	protected HierarchyListener hierarchyListener;

	/**
	 * Property change listener - listens on the changes to
	 * {@link LafWidget#COMPONENT_PREVIEW_PAINTER} property.
	 */
	protected PropertyChangeListener propertyChangeListener;

	@Override
	public void setComponent(JComponent jcomp) {
		super.setComponent(jcomp);
		this.scrollPane = (JScrollPane) jcomp;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidget#requiresCustomLafSupport()
	 */
	public boolean requiresCustomLafSupport() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#installUI()
	 */
	@Override
	public void installUI() {
		if (LafWidgetRepository.getRepository().getLafSupport()
				.toInstallExtraElements(this.scrollPane)) {

			PreviewPainter pPainter = LafWidgetUtilities2
					.getComponentPreviewPainter(this.scrollPane);
			if (pPainter == null)
				return;
			this.scrollPaneSelector = new ScrollPaneSelector();
			this.scrollPaneSelector.installOnScrollPane(this.scrollPane);
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#uninstallUI()
	 */
	@Override
	public void uninstallUI() {
		if (this.scrollPaneSelector != null) {
			this.scrollPaneSelector.uninstallFromScrollPane();
			this.scrollPaneSelector = null;
		}
	}

	@Override
	public void installListeners() {
		this.hierarchyListener = new HierarchyListener() {
			/*
			 * (non-Javadoc)
			 * 
			 * @see java.awt.event.HierarchyListener#hierarchyChanged(java.awt.event.HierarchyEvent)
			 */
			public void hierarchyChanged(HierarchyEvent e) {
				if (scrollPane.getParent() instanceof ComboPopup) {
					if (scrollPaneSelector != null) {
						scrollPaneSelector.uninstallFromScrollPane();
						scrollPaneSelector = null;
					}
				}
			}
		};
		this.scrollPane.addHierarchyListener(this.hierarchyListener);

		this.propertyChangeListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if (LafWidget.COMPONENT_PREVIEW_PAINTER.equals(evt
						.getPropertyName())) {
					PreviewPainter pPainter = LafWidgetUtilities2
							.getComponentPreviewPainter(scrollPane);
					// Uninstall old scroll pane selector
					if (scrollPaneSelector != null) {
						scrollPaneSelector.uninstallFromScrollPane();
						scrollPaneSelector = null;
					}
					// Install new scroll pane selector
					if (pPainter != null) {
						scrollPaneSelector = new ScrollPaneSelector();
						scrollPaneSelector.installOnScrollPane(scrollPane);
					}
				}
			}
		};
		this.scrollPane.addPropertyChangeListener(this.propertyChangeListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.lafwidget.LafWidgetAdapter#uninstallListeners()
	 */
	@Override
	public void uninstallListeners() {
		this.scrollPane.removeHierarchyListener(this.hierarchyListener);
		this.hierarchyListener = null;

		this.scrollPane
				.removePropertyChangeListener(this.propertyChangeListener);
		this.propertyChangeListener = null;
	}
}
