/*
 * Copyright (c) 2005-2008 Laf-Widget Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Laf-Widget Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.lafwidget.animation;

import java.awt.Component;

/**
 * Global callback for the fade tracker. Is used when the application wishes to
 * execute some code on the fade.
 * 
 * @author Kirill Grouchnikov
 */
public interface GlobalFadeTrackerCallback {
	/**
	 * Indicates that a single cycle of fade has been performed.
	 * 
	 * @param comp
	 *            Component for the fade event.
	 * @param subID
	 *            Optional ID of the subcomponent of the fade event. May be
	 *            <code>null</code>. Is non-<code>null</code> when fade events
	 *            can happen on different visual areas of the same top-level
	 *            component. For example, each tab in a tabbed pane has its own
	 *            fade sequence for rollovers.
	 * @param fadeKind
	 *            Fade animation kind.
	 * @param fadeCycle
	 *            The current fade cycle. Is guaranteed to be in 0.0-1.0 range.
	 */
	public void fadePerformed(Component comp, Comparable<?> subID,
			FadeKind fadeKind, float fadeCycle);

	/**
	 * Indicates that the fade sequence has started.
	 * 
	 * @param comp
	 *            Component for the fade event.
	 * @param subID
	 *            Optional ID of the subcomponent of the fade event. May be
	 *            <code>null</code>. Is non-<code>null</code> when fade events
	 *            can happen on different visual areas of the same top-level
	 *            component. For example, each tab in a tabbed pane has its own
	 *            fade sequence for rollovers.
	 * @param fadeKind
	 *            Fade animation kind.
	 * @param initialFadeCycle
	 *            The initial value for the fade cycle. Will be in 0.0-1.0 range.
	 */
	public void fadeStarted(Component comp, Comparable<?> subID,
			FadeKind fadeKind, float initialFadeCycle);

	/**
	 * Indicates that the fade sequence has ended.
	 * 
	 * @param comp
	 *            Component for the fade event.
	 * @param subID
	 *            Optional ID of the subcomponent of the fade event. May be
	 *            <code>null</code>. Is non-<code>null</code> when fade events
	 *            can happen on different visual areas of the same top-level
	 *            component. For example, each tab in a tabbed pane has its own
	 *            fade sequence for rollovers.
	 * @param fadeKind
	 *            Fade animation kind.
	 */
	public void fadeEnded(Component comp, Comparable<?> subID, FadeKind fadeKind);

	/**
	 * Indicates that the fade sequence has been reversed. This can happen, for
	 * example, when the user moves the mouse over a button (rollover fade) and
	 * then quickly moves the mouse out. In this case, the fade sequence is
	 * reverted halfway.
	 * 
	 * 
	 * @param comp
	 *            Component for the fade event.
	 * @param subID
	 *            Optional ID of the subcomponent of the fade event. May be
	 *            <code>null</code>. Is non-<code>null</code> when fade events
	 *            can happen on different visual areas of the same top-level
	 *            component. For example, each tab in a tabbed pane has its own
	 *            fade sequence for rollovers.
	 * @param fadeKind
	 *            Fade animation kind.
	 * @param isFadingIn
	 *            Indicates the <b>new</b> direction of fade.
	 * @param fadeCycle
	 *            Indicates the <b>new</b> value of fade cycle. Will be in 0.0-1.0
	 *            range.
	 */
	public void fadeReversed(Component comp, Comparable<?> subID,
			FadeKind fadeKind, boolean isFadingIn, float fadeCycle);
}