/*
 *   mwmv80l.c -- Mwave Modem AT Command Parser
 *
 *  Written By: Paul Schroeder IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#define   PFT_ENTRIES 18
#include <mwmspcfc.h>
#include <mwmcntnd.h>
#include <psv8itcb.h>
#include <stddef.h>

static char szThisFile[] = "mwmv80l.c";

VOID mwmParseLockedCodeSegmentForInterrupt(VOID)
{
}

/*****************************************************************************/
/* This routine loads modules and makes the proper connections for V8 and  $F*/
/* V21 (V22 for now).                                                       */
/* MTS 7757 - 02/12/97 DR - Add flag to determine if GPC connection should   */
/* be made to TAIO or the echo canceller.                                    */
/*****************************************************************************/
ULONG mwmV80LV8V21AndConnect(PMWM_DSPINFO pmwmDspInfo, BOOL bConnectToEchoCanceller)  /*$1*/
{
  ULONG  ulRC = 0;
  USHORT usTempValue = 0;
  ULONG  ulAddress = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LV8V21AndConnect entry\n");  }


  DPF("mwmV80LV8V21AndConnect\n");

  ulRC = mwmCntndLoadModule(pmwmDspInfo,"V8","V8",&pmwmDspInfo->hmodV8,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  ulRC = mwmCntndLoadModule(pmwmDspInfo,"V22FILE","V22",&pmwmDspInfo->hmodV22,szThisFile,__LINE__);
  if (ulRC)
    return ulRC;

  /***************************************************************************/
  /* Get v8 Task handle                                                      */
  /***************************************************************************/
  ulRC = dspNameToTaskHandle(pmwmDspInfo->hmodV8, "V8", &pmwmDspInfo->hmtV8);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    return(ulRC);
  }

  /***************************************************************************/
  /* Get v22 Task handle                                                     */
  /***************************************************************************/
  ulRC = dspNameToTaskHandle(pmwmDspInfo->hmodV22, "V22", &pmwmDspInfo->hv22Task);
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    return(ulRC);
  }

  /***************************************************************************/
  /* Make V.8 connections...                                                 */
  /***************************************************************************/
  ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtV8,"XMITDATA",
                          pmwmDspInfo->hv22Task,"XMITV8",0,NULL,
                          pmwmDspInfo, szThisFile,__LINE__ );
  if (ulRC)
    return(ulRC);

  ulRC=mwmCntndConnectGPC(pmwmDspInfo->hv22Task,"RCVDATA",
                          pmwmDspInfo->hmtV8,"RCVDATA",0,NULL,
                          pmwmDspInfo, szThisFile,__LINE__ );
  if (ulRC)
    return(ulRC);

  /* MTS 7757 - 02/12/97 DR - Conditional added */
  if (bConnectToEchoCanceller) {
    ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtCANCEL,"ECHO_CANCEL_OUTPUT",  /*$4*/
                            pmwmDspInfo->hv22Task,"SIGIN",0,NULL,
                            pmwmDspInfo, szThisFile,__LINE__ );
    if (ulRC)
      return(ulRC);
  } else {
    ulRC=mwmCntndConnectGPC(NULL, "TAIO_OUTPUT",
                            pmwmDspInfo->hv22Task,"SIGIN",0,NULL,
                            pmwmDspInfo, szThisFile,__LINE__ );
    if (ulRC)
      return(ulRC);
  } /* endif */

  /***************************************************************************/
  /* Make V.8 connections...                                                 */
  /***************************************************************************/

  ulRC=dspConnectITCB(pmwmDspInfo->hv22Task,"DPV8ITCB",
                      pmwmDspInfo->hmtV8   ,"DPV8ITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hv22Task,"DPITCB",
                    pmwmDspInfo->hmtV8   ,"DPITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hmtProtSupr,"PSV8ITCB",
                      pmwmDspInfo->hmtV8      ,"PSV8ITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hmctlTask,"OREGITCB",                      /*$2*/
                      pmwmDspInfo->hmtV8    ,"OREGITCB");
  if (ulRC != DSP_NOERROR)
  {
    DPF("WARNING:  OREGITCB.MCTL.MODEM to OREGITCB.V8.V8 itcb connection failed.\n");
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hv22Task,"DPV8ITCB",                       /*$2*/
                      pmwmDspInfo->hmctlTask,"DPV8ITCB");
  if (ulRC != DSP_NOERROR)
  {
    DPF("WARNING:  DPV8ITCB.V22.V22 to DPV8ITCB.MCTL.MODEM itcb connection failed.\n");
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hv22Task,"MCDPITCB",                       /*$2*/
                      pmwmDspInfo->hmtV8   ,"MCDPITCB");
  if (ulRC != DSP_NOERROR)
  {
    DPF("WARNING:  MCDPITCB.V22.V22 to MCDPITCB.V8.V8 itcb connection failed.\n");
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hmtProtSupr,"VDETECT",                     /*$2*/
                      pmwmDspInfo->hv22Task,"VDETECT");
  if (ulRC != DSP_NOERROR)
  {
    DPF("WARNING:  VDETECT.PROTSUPR.MODEM to VDETECT.V22.V22 itcb connection failed.\n");
  }

  /***************************************************************************/
  /*  BIOS.TAIO000..WTITCB      CONNECT  V8..WTITCB                          */
  /***************************************************************************/
  ulRC=dspConnectITCB(NULL,"WTITCB",pmwmDspInfo->hmtV8,"WTITCB");              /*$4*/
  if (ulRC != DSP_NOERROR)
  {
    DPF("WARNING:  WTITCB.TAIO.MWAVEOS to WTITCB.V8.V8 itcb connection failed.\n");
  }

  /***************************************************************************/
  /*  BIOS.TAIO000..WTITCB      CONNECT  V22..WTITCB                         */
  /***************************************************************************/
  ulRC=dspConnectITCB(NULL,"WTITCB",pmwmDspInfo->hv22Task,"WTITCB");          /*$4*/
  if (ulRC != DSP_NOERROR)
  {
    DPF("WARNING:  WTITCB.TAIO.MWAVEOS to WTITCB.V22.V22 itcb connection failed.\n");
  }

  if (pmwmDspInfo->ulFeaturesToLoad & FEATURE_V80)     /*$1*/
  {
    /***************************************************************************/
    /* Set CIDETECT for v.8                                                    */
    /***************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtV8, "CIDETECT", &ulAddress);
    if (ulRC!=DSP_NOERROR)
    {
        mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                       MWM_DSP_ERROR, ulRC);
        return(ulRC);
    }
    else
    {
      usTempValue = 1;
      ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                            &usTempValue, 1,
                            DSP_MEMXFER_DATA_WRITE);
      if (ulRC!=DSP_NOERROR)
      {
          mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                         MWM_DSP_ERROR, ulRC);
          return(ulRC);
      }
    } /* endif */
  }


  /*$5 activate V22 before V8*/
  ulRC = dspChangeModuleState(pmwmDspInfo->hmodV22, DSP_ACTIVATE_ACTIVE );
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulRC = dspChangeModuleState(pmwmDspInfo->hmodV8, DSP_ACTIVATE_ACTIVE );
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LV8V21AndConnect exit\n");  }


  return 0;
}



/*****************************************************************************/
/* This routine unloads V8 and V21 (V22 for now).                            */
/*****************************************************************************/
ULONG mwmV80LV8V21Unload(PMWM_DSPINFO pmwmDspInfo)  /*$1*/
{
  ULONG ulRC;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LV8V21Unload entry\n");  }


  DPF("mwmV80LV8V21Unload\n");

  if (pmwmDspInfo->hmodV22)
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV22);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC );
      return ulRC;
    }
    pmwmDspInfo->hmodV22 = 0;
    pmwmDspInfo->hv22Task  = 0;
  }

  if (pmwmDspInfo->hmodV8)
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV8);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC );
      return ulRC;
    }
    pmwmDspInfo->hmodV8 = 0;
    pmwmDspInfo->hmtV8  = 0;
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LV8V21Unload exit\n");  }

  return 0;
}


/*****************************************************************************/
/* This routine transitions from the dial state to the V80 Pumps loaded      */
/* state.                                                                    */
/*****************************************************************************/
ULONG mwmV80LDialToV80nPumps(PMWM_DSPINFO pmwmDspInfo) /*$1*/
{
  ULONG  ulRC;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LDialToV80nPumps entry\n");  }


  DPF("mwmV80LDialToV80nPumps\n");

  ulRC = mwmLoadDialToModem(pmwmDspInfo);

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LDialToV80nPumps exit\n");  }


  return ulRC;
}



/*****************************************************************************/
/* This routine transitions from the Discrim to Dial state.                  */
/*****************************************************************************/
ULONG mwmV80LAnywhereV80ToDial(PMWM_DSPINFO pmwmDspInfo)  /*$1*/
{
  ULONG  ulRC = 0;
  ULONG  ulAddress;
  USHORT usTempValue;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LAnywhereV80ToDial entry\n");  }


  DPF("mwmV80LAnywhereV80ToDial\n");

  /*************************************************************************/
  /* Get the Address of the v.80 online Segment Pointer.                 $3*/
  /*************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmtCLASSx, "ONLIN", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    DPF("  WARNING:  dspLabelToAddress failed for V80 OLINE segment pointer.\n");
    ulRC = DSP_NOERROR;
  }
  else
  {
    /* set the online segment pointer to 0 */
    ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                          &usTempValue, 1,
                          DSP_MEMXFER_DATA_WRITE);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }
  }

  /***************************************************************************/
  /* unload the V80_ON segment                                             $3*/
  /***************************************************************************/

  if (pmwmDspInfo->hsegCLASSxV80_ON != 0)
  {
    ulRC = dspFreeSegment(pmwmDspInfo->hsegCLASSxV80_ON);
    if (ulRC)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC);
      return(ulRC);
    }
    pmwmDspInfo->hsegCLASSxV80_ON = 0;
  }

  /***************************************************************************/
  /* Set V80 cycles for "OFFLINE Mode"                                     $4*/
  /***************************************************************************/
  if (pmwmDspInfo->hmodCLASSx && pmwmDspInfo->hmtCLASSx)
  {
    ulRC = mwmSpcfcSetV80Cycles(pmwmDspInfo, 0, MWM_MUSTHAVE);
    if (ulRC)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC );
      return ulRC;
    }
  }

  /***************************************************************************/
  /* unload V80MAC                                                         $3*/
  /***************************************************************************/

  if (pmwmDspInfo->hmodV80MAC != 0)
  {
    ulRC = dspFreeModule(pmwmDspInfo->hmodV80MAC); /*$5*/
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC );
      return ulRC;
    }
    pmwmDspInfo->hmodV80MAC = 0;
    pmwmDspInfo->hmtV80MAC = 0;
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LAnywhereV80ToDial exit\n");  }


  return ulRC;
}


/*****************************************************************************/
/* This routine transitions from the Discrim to Dial state.                  */
/*****************************************************************************/
ULONG mwmV80LV80DiscrimToDial(PMWM_DSPINFO pmwmDspInfo)    /*$3*/
{
  ULONG ulRC = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LV80DiscrimToDial entry\n");  }


  DPF("mwmV80LV80DiscrimToDial\n");
                                                           /*$C deleted function call*/
  ulRC = mwmLoadDiscrimToDial(pmwmDspInfo);

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LV80DiscrimToDial exit\n");  }


  return ulRC;
}

/*****************************************************************************/
/* This routine loads modules and makes the proper connections for V80     $F*/
/* online mode.                                                              */
/*****************************************************************************/
ULONG mwmV80LCyclesAdjustToLineSpeed(PMWM_DSPINFO pmwmDspInfo, /*$H*/
                                     USHORT usProtocol,
                                     USHORT OnLine)
{

  ULONG   ulAddress = 0;
  ULONG   ulRC;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LCyclesAdjustToLineSpeed entry\n");  }


  DPF("mwmV80LCyclesAdjustToLineSpeed\n");

  if (OnLine) {
    /***************************************************************************/
    /* Adjust UART and V80 cycles                                            $D*/
    /***************************************************************************/
    ulRC = mwmSpcfcRaiseUARTV80Cycles(pmwmDspInfo, usProtocol);
    if (ulRC)
      return ulRC;
  } /* endif */



  if ( (pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) ||               /* if SLIMBASE or*/
       ( !(pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) && !OnLine) ) /* if not SLIMBASE*/
  {                                                                        /* and not online*/
    DPF("  Loading V80MAC task\n");
    /***************************************************************************/
    /* Load the V80MAC task.                                                   */
    /***************************************************************************/
    ulRC = mwmCntndLoadModule(pmwmDspInfo,"V80MAC","V80MAC",&pmwmDspInfo->hmodV80MAC,szThisFile,__LINE__);
    if (ulRC)
      return(ulRC);

    /***************************************************************************/
    /* Get the V80MAC Task Handle                                              */
    /***************************************************************************/
    ulRC = dspNameToTaskHandle(pmwmDspInfo->hmodV80MAC, "V80MAC", &pmwmDspInfo->hmtV80MAC);
    if (ulRC)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC );
      return (ulRC);
    }

    DPF("  Connecting V80MAC task GPCs\n");
    /***************************************************************************/
    /* Connect the V80MAC GPCs.                                              $9*/
    /*                                                                         */
    /* MACXDATA - v80 owner                                                    */
    /* XV80GPC - v80mac user                                                   */
    /*                                                                         */
    /* RV80GPC  - v80mac owner                                                 */
    /* MACRDATA - v80 user                                                     */
    /*                                                                         */
    /* XDPGPC - v80mac owner                                                   */
    /* XMITDATA - v34 user                                                     */
    /*                                                                         */
    /* RCVDATA - v34 owner                                                     */
    /* RDPGPC - v80mac user                                                    */
    /***************************************************************************/
    ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtCLASSx,"MACXDATA",
                            pmwmDspInfo->hmtV80MAC,"XV80GPC",0,NULL,
                            pmwmDspInfo, szThisFile,__LINE__ );
    if (ulRC)
      return(ulRC);

    ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtV80MAC,"RV80GPC",
                            pmwmDspInfo->hmtCLASSx,"MACRDATA",0,NULL,
                            pmwmDspInfo, szThisFile,__LINE__ );
    if (ulRC)
      return(ulRC);

    ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtV80MAC,"XDPGPC",
                            pmwmDspInfo->hmtV34,"XMITDATA",0,NULL,
                            pmwmDspInfo, szThisFile,__LINE__ );
    if (ulRC)
      return(ulRC);


    ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtV34,"RCVDATA",
                            pmwmDspInfo->hmtV80MAC,"RDPGPC",0,NULL,
                            pmwmDspInfo, szThisFile,__LINE__ );
    if (ulRC)
      return(ulRC);

    DPF("  Connecting V80MAC task ITCBs\n");
    /***************************************************************************/
    /* Connect the V80MAC ITCBs.                                             $9*/
    /*                                                                         */
    /* V80MITCB - v80 primary                                                  */
    /* V80MITCB - v80mac secondary                                             */
    /*                                                                         */
    /* OREGITCB - mctl primary                                                 */
    /* OREGITCB - v80mac secondary                                             */
    /***************************************************************************/
    ulRC=dspConnectITCB(pmwmDspInfo->hmtCLASSx,"V80MITCB",
                        pmwmDspInfo->hmtV80MAC,"V80MITCB");
    if (ulRC != DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC );
      return (ulRC);
    }

    ulRC=dspConnectITCB(pmwmDspInfo->hmctlTask,"OREGITCB",
                        pmwmDspInfo->hmtV80MAC,"OREGITCB");
    if (ulRC != DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR,
                     ulRC );
      return (ulRC);
    }

  } /* endif */

  if (OnLine) {
    /*************************************************************************/
    /* Get the Address of the v.80 online Segment Pointer.                   */
    /*************************************************************************/
    ulRC = dspLabelToAddress(pmwmDspInfo->hmtCLASSx, "ONLIN", &ulAddress);
    if (ulRC!=DSP_NOERROR)
    {
      mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                     MWM_DSP_ERROR, ulRC);
      return(ulRC);
    }

    DPF("  Loading V80_ON online segment\n");
    /*************************************************************************/
    /* Load the V80_ON Segment                                               */
    /*************************************************************************/
    ulRC = mwmCntndLoadSegment(pmwmDspInfo,
                               "V80",
                               "V80_ON",
                               "V80_ON",
                               pmwmDspInfo->hmtCLASSx,
                               &pmwmDspInfo->hsegCLASSxV80_ON,
                               ulAddress,
                               szThisFile , __LINE__, MWM_MUSTHAVE );
    if (ulRC)
      return (ulRC);
  } /* endif */


  if ( (pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) ||               /* if SLIMBASE or*/
       ( !(pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) && !OnLine) ) /* if not SLIMBASE*/
  {                                                                        /* and not already online*/
    DPF("  Activating V80MAC task\n");
    /***************************************************************************/
    /* Activate the V80MAC task.                                               */
    /***************************************************************************/
    ulRC = dspChangeModuleState(pmwmDspInfo->hmodV80MAC, DSP_ACTIVATE_ACTIVE );
    if (ulRC != DSP_NOERROR)
    {
       mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                      MWM_DSP_ERROR, ulRC);
       return(ulRC);
    }
  } /* endif */


	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LCyclesAdjustToLineSpeed exit\n");  }


  return(ulRC);
}


ULONG mwmV80LReceivedSubclassV80(PMWM_DSPINFO pmwmDspInfo) /*$1*/
{
  ULONG ulRC = 0l;
  /*ULONG ulAddress = 0;*/
  MWM_FCLASS_SUBCLASS  ClassStruct;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LReceivedSubclassV80 entry\n");  }


  DPF("mwmV80LReceivedSubclassV80\n");

  ulRC = mwmFaxGetFCLASS(pmwmDspInfo, &ClassStruct.usFCLASS, &ClassStruct.usSubclass);
  if (ulRC!=DSP_NOERROR)
  {
    return(ulRC);
  }


  /***************************************************************************/
  /* Figure out where we are, then move to the discrim state                 */
  /***************************************************************************/
  if (ClassStruct.usFCLASS != 0 || !ClassStruct.usSubclass)
  {
    ulRC = mwmFaxClassXToDiscrim(pmwmDspInfo, ClassStruct.usFCLASS);
    if (ulRC != DSP_NOERROR)
    {
      return ulRC;
    }

    /*************************************************************************/
    /* Now the modem is either in data discrim, or data dial state.          */
    /*************************************************************************/
    /* Call DiscrimToV80Discrim                                              */
    /* if !slimbase or we have received ATD....;(ulDiscrimState==0)          */
    /*   Call V80DiscrimToDial                                               */
    /*   Call DialToV80nPumps                                                */
    /*************************************************************************/
    ulRC = mwmV80LDiscrimToV80Discrim(pmwmDspInfo, 0);
    if (ulRC != DSP_NOERROR)
    {
      return ulRC;
    }

    if ( (!(pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE)) ||
       (!pmwmDspInfo->ulDiscrimState))
    {
      if (pmwmDspInfo->ulDiscrimState)
      {
        ulRC = mwmV80LV80DiscrimToDial(pmwmDspInfo);
        if (ulRC != DSP_NOERROR)
        {
          return ulRC;
        }
      }

      ulRC = mwmLoadSetCurrentSpeed(pmwmDspInfo);
      if (ulRC)
        return ulRC;

      ulRC = mwmV80LDialToV80nPumps(pmwmDspInfo);
      if (ulRC != DSP_NOERROR)
      {
        return ulRC;
      }

      if ( !(pmwmDspInfo->ulFeaturesToLoad & FEATURE_SLIMBASE) )  /*$C*/
      {
        ulRC = mwmV80LCyclesAdjustToLineSpeed(pmwmDspInfo, 0, 0); /*$H*/
        if (ulRC != DSP_NOERROR)
          return ulRC;
      }
    }

    /*************************************************************************/
    /* Tell the parser and modem control that we are now in class 0 subclass */
    /* v.80 mode.                                                            */
    /*************************************************************************/
    mwmParseSetParserMode(MWM_MODE_REGULAR_MODEM);

  } /* if Not already in Class 0 subclass V.80 mode */

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LReceivedSubclassV80 exit\n");  }


  return ulRC;
}


ULONG mwmV80LDiscrimToV80Discrim(PMWM_DSPINFO pmwmDspInfo, USHORT usClass)  /*$3*/
{
  ULONG ulRC = DSP_NOERROR;
  ULONG ulAddress;
  char  *pszCLASSx;
  USHORT usTempValue = 0;                                                  /*$8*/

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LDiscrimToV80Discrim entry\n");  }


  DPF("mwmV80LDiscrimToV80Discrim:  usClass=%hd\n", usClass);

  if (usClass == 0)
     pszCLASSx = "V80";
  else
  {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DEBUG_MESSAGE, (ULONG)"FCLASS=?");
     return 1;
  }

  /***************************************************************************/
  /* Put the UART Task in standby                                            */
  /***************************************************************************/
  ulRC = dspChangeTaskState(pmwmDspInfo->hmtUART, DSP_ACTIVATE_STANDBY);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return (ulRC);
  }


  /***************************************************************************/
  /* Put Async in Standby...                                                 */
  /***************************************************************************/
  ulRC = dspChangeModuleState(pmwmDspInfo->hmodAsync, DSP_ACTIVATE_STANDBY);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* Swap to the dummy Async ITCB                                            */
  /***************************************************************************/
  ulRC = mwmLoadSwapToDummyAsync(pmwmDspInfo);
  if (ulRC)
    return ulRC;


  ulRC = dspFreeModule(pmwmDspInfo->hmodAsync);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return ulRC;
  }
  pmwmDspInfo->hmodAsync = 0;
  pmwmDspInfo->hasyTask = 0;


  /***************************************************************************/
  /* Load the V80 (CLASSx) Async Replacement.                                */
  /***************************************************************************/
  ulRC = mwmCntndLoadModule(pmwmDspInfo,pszCLASSx,pszCLASSx,&pmwmDspInfo->hmodCLASSx,szThisFile,__LINE__);
  if (ulRC)
    return(ulRC);

  /***************************************************************************/
  /* Get the CLASSx Task Handle                                              */
  /***************************************************************************/
  ulRC = dspNameToTaskHandle(pmwmDspInfo->hmodCLASSx, pszCLASSx, &pmwmDspInfo->hmtCLASSx);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* If we're not offline (on hook) unload the V80_ON segment              $5*/
  /***************************************************************************/
  ulRC = dspNameToSegmentHandle(pmwmDspInfo->hmtCLASSx,
                                "V80_ON",
                                &pmwmDspInfo->hsegCLASSxV80_ON);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    return(ulRC);
  }

  ulRC = dspFreeSegment(pmwmDspInfo->hsegCLASSxV80_ON);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    return(ulRC);
  }

  /* Get the Address of the v.80 online Segment Pointer.                 $8*/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmtCLASSx, "ONLIN", &ulAddress);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  /* set the online segment pointer to 0 */
  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddress,
                        &usTempValue, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  pmwmDspInfo->hsegCLASSxV80_ON = 0;

  /***************************************************************************/
  /*    CLASSx..ATCMD             CONNECT   MCTL..ATCMD                      */
  /***************************************************************************/
  ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtCLASSx,"ATCMD",pmwmDspInfo->hmctlTask,"ATCMD",0,NULL,
                          pmwmDspInfo, szThisFile,__LINE__ );
  if (ulRC)
    return(ulRC);


  /***************************************************************************/
  /*    MCTL..ATRESP              CONNECT   CLASSx..ATRESP                   */
  /***************************************************************************/
  ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmctlTask,"ATRESP",pmwmDspInfo->hmtCLASSx,"ATRESP",0,NULL,
                          pmwmDspInfo, szThisFile,__LINE__ );
  if (ulRC)
    return(ulRC);


  /***************************************************************************/
  /*  CLASSx..ASYNITCB          CONNECT  MCTL..ASYNITCB                      */
  /***************************************************************************/
  ulRC=dspConnectITCB(pmwmDspInfo->hmtCLASSx,"ASYNITCB",pmwmDspInfo->hmctlTask,"ASYNITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  /***************************************************************************/
  /*  BIOS.UART000..UART_OUTPUT CONNECT   CLASSx..UART_OUTPUT                */
  /***************************************************************************/
/*  ulRC=mwmCntndConnectGPC(NULL,"UART_OUTPUT",pmwmDspInfo->hmtCLASSx,"UART_OUTPUT",0,NULL,*/
/*                          pmwmDspInfo, szThisFile,__LINE__ );*/
/*  if (ulRC)*/
/*    return(ulRC);*/


  /***************************************************************************/
  /*  CLASSx..UART_INPUT        CONNECT   BIOS.UART000..UART_INPUT           */
  /***************************************************************************/
/*  ulRC=mwmCntndConnectGPC(pmwmDspInfo->hmtCLASSx,"UART_INPUT",NULL,"UART_INPUT",0,NULL,*/
/*                          pmwmDspInfo, szThisFile,__LINE__ );*/
/*  if (ulRC)*/
/*    return(ulRC);*/

  ulRC=dspConnectITCB(pmwmDspInfo->hmtCLASSx,"V80ITCB",pmwmDspInfo->hmctlTask,"V80ITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC);
    return(ulRC);
  }

  ulRC=dspConnectITCB(pmwmDspInfo->hmctlTask,"V34ITCB",pmwmDspInfo->hmtCLASSx,"V34ITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }
  /***************************************************************************/
  /*  CLASSx..OEITCB            CONNECT  UART..OEITCB                        */
  /***************************************************************************/
  ulRC=dspConnectITCB(pmwmDspInfo->hmtCLASSx,"OEITCB",NULL,"OEITCB");
  if (ulRC != DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  /***************************************************************************/
  /* Activate the ASYNC Replacement.                                         */
  /***************************************************************************/
  ulRC = dspChangeModuleState(pmwmDspInfo->hmodCLASSx, DSP_ACTIVATE_ACTIVE );
  if (ulRC != DSP_NOERROR)
  {
     mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                    MWM_DSP_ERROR, ulRC);
     return(ulRC);
  }

  /***************************************************************************/
  /* Swap from the Dummy Async to the Class x ITCB                           */
  /***************************************************************************/
  ulRC = mwmLoadSwapToRealAsync(pmwmDspInfo);
  if (ulRC)
    return ulRC;


  /***************************************************************************/
  /* Activate the UART Task                                                  */
  /***************************************************************************/
  ulRC = dspChangeTaskState(pmwmDspInfo->hmtUART, DSP_ACTIVATE_ACTIVE);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR,
                   ulRC );
    return (ulRC);
  }

  /***************************************************************************/
  /* Set CISEND for v.8 to tell it to send CI (and to init v.21)           $A*/
  /***************************************************************************/
  ulRC = mwmV80LSetCISEND(pmwmDspInfo, 1);
  if (ulRC!=DSP_NOERROR)
  {
    return(ulRC);
  }

  /***************************************************************************/
  /* Set the V80 mode indicator (SUBCLASS = 1)                               */
  /***************************************************************************/
  ulRC = mwmV80LSetSUBCLASS(pmwmDspInfo, 1);  /*$6 we're in v.80 mode now*/
  if (ulRC!=DSP_NOERROR)
  {
    return(ulRC);
  }

  ulRC = mwmFaxSetFCLASS(pmwmDspInfo, 0);    /*$6 keep FCLASS in sync with SUBCLASS*/
  if (ulRC!=DSP_NOERROR)
  {
    return(ulRC);
  }

  pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_V42; /*$G modem control needs v.42*/
  pmwmDspInfo->ulFeaturesToLoad &= ~FEATURE_MNP; /*   and MNP feature bits*/
   { MW_SYSLOG_2(TRACE_MWMLW32,"mwmv80l::mwmV80LDiscrimToV80Discrim turned off V42 %lx\n",pmwmDspInfo->ulFeaturesToLoad);  }
  ulRC = mwmSpcfcSetFeatures(pmwmDspInfo, 1);    /*   turned off while we're*/
  if (ulRC)                                      /*   in v.80 mode*/
  {
    return (ulRC);
  }

   { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LDiscrimToV80Discrim exit\n");  }


  return 0;

}


/*******************************************************************************/
/* Set CISEND for V8 task                                                    $A*/
/*  0 - V8 will not send CI and will not init V21 (V22 for now) task           */
/*  1 - V8 will send CI and will init V21 (V22 for now) task                   */
/*******************************************************************************/
ULONG mwmV80LSetCISEND(PMWM_DSPINFO pmwmDspInfo, USHORT usValue)
{
  ULONG   ulRC;
  ULONG   ulAddressPSV8ITCB = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LSetCISEND entry\n");  }


  DPF("mwmV80LSetCISEND:  setting CISEND to %d\n",usValue);

  /***************************************************************************/
  /*          Get Address of ITCBPSV8                                        */
  /***************************************************************************/
  ulRC = dspLabelToAddress(pmwmDspInfo->hmtProtSupr, "ITCBPSV8", &ulAddressPSV8ITCB);
  if (ulRC)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return (ulRC);
  }

  /***************************************************************************/
  /*          Set CISEND                                                     */
  /***************************************************************************/
  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, (ulAddressPSV8ITCB + offsetof(PSV8ITCB, CISEND)),
                        &usValue, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return ulRC;
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LSetCISEND exit\n");  }


  return ulRC;
}



ULONG mwmV80LGetSUBCLASS(PMWM_DSPINFO pmwmDspInfo, USHORT *pusSUBCLASS)  /*$3*/
{
  ULONG   ulRC;
  ULONG   ulAddressSUBCLASS = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LGetSUBCLASS entry\n");  }


/*  DPF("mwmV80LGetSUBCLASS\n");*/

  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "SUBCLASS", &ulAddressSUBCLASS);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }


  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddressSUBCLASS,
                        pusSUBCLASS, 1,
                        DSP_MEMXFER_DATA_READ);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

/*  DPF("  SUBCLASS = %d\n", *pusSUBCLASS);*/

	 { MW_SYSLOG_2(TRACE_MWMLW32,"mwmv80l::mwmV80LGetSUBCLASS exit usSUBCLASS %lx\n",*pusSUBCLASS);  }


  return 0;
}


ULONG mwmV80LSetSUBCLASS(PMWM_DSPINFO pmwmDspInfo, USHORT usSUBCLASS)  /*$3*/
{
  ULONG   ulRC;
  ULONG   ulAddressSUBCLASS = 0;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LSetSUBCLASS entry\n");  }


  DPF("mwmV80LSetSUBCLASS:  setting SUBCLASS to %d\n",usSUBCLASS);

  ulRC = dspLabelToAddress(pmwmDspInfo->hmctlTask, "SUBCLASS", &ulAddressSUBCLASS);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

  ulRC = dspMemTransfer(pmwmDspInfo->hDSP, ulAddressSUBCLASS,
                        &usSUBCLASS, 1,
                        DSP_MEMXFER_DATA_WRITE);
  if (ulRC!=DSP_NOERROR)
  {
    mwmHandleError(pInstanceInfo,szThisFile,__LINE__,
                   MWM_DSP_ERROR, ulRC);
    return(ulRC);
  }

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LSetSUBCLASS exit\n");  }


  return 0;
}

/*****************************************************************************/
/* This routine determines if V8 and V21 should be loaded based on the       */
/* A8E settings and call the routine to load them.                           */
/*****************************************************************************/
ULONG mwmV80LLoadV8AndV21(PMWM_DSPINFO pmwmDspInfo, BOOL bConnectToEC)
{

  ULONG     ulRC = 0;
  REGISTERS Registers;

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LLoadV8AndV21 entry\n");  }


  ulRC = mwmParseQueryModemRegisters(&Registers);
  if (ulRC)
    return (ulRC);

  if ((Registers.PlusA8E[0] == 6) || (Registers.PlusA8E[1] == 5)) {
    if ((!pmwmDspInfo->hmodV22) && (!pmwmDspInfo->hmodV8)) {
      ulRC = mwmV80LV8V21AndConnect(pmwmDspInfo, bConnectToEC);
      if (ulRC) {
        return ulRC;
      } /* endif */
    } /* endif */
  } /* endif */

	 { MW_SYSLOG_1(TRACE_MWMLW32,"mwmv80l::mwmV80LLoadV8AndV21 exit\n");  }


  return(ulRC);

}




