/*
 *   meio_api.c - This modules provides a single point of reference for
 *                all function calls to MEIO
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
#include <port_types.h> 
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h>
#include <meio_api.h>          /* MEIO API Definititions                  */
#include <meioddr2.h>          /* MEIO Device Driver definitions          */

ULONG APIENTRY MEIOAPIENTRY(UINT, LPARAM, LPARAM);
BOOL meio_loaded=FALSE;

/* function to issue pseudo ioctl to meiodvp.c method */
ULONG meiowinIOCTL(UINT wMessage,LPARAM lParam1) {
  ULONG ulRC=0;

  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::meiowinIOCTL entry wMessage %x\n",wMessage);

  ulRC=MEIOAPIENTRY(wMessage,lParam1,0L); 
  
  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::meiowinIOCTL exit ulRC %x\n",(ulRC ? 0:1));
  return (ulRC ? 0:1);
}

/*------------------------------------------------------------------------*/
/* function to close Device Driver Interface                               */
/*------------------------------------------------------------------------*/
APIRET MEIO_OpenDD()
{
  MW_SYSLOG_1(TRACE_MEIO_API,"meio_api::MEIO_OpenDD entry\n");

  if (meio_loaded == FALSE) {
    meiowinIOCTL(1,(LPARAM) 0);
    meio_loaded=TRUE;
  }
 
  MW_SYSLOG_1(TRACE_MEIO_API,"meio_api::MEIO_OpenDD exit rc=0\n");
  return (0);
}


/*------------------------------------------------------------------------*/
/* function to close Device Driver Interface                               */
/*------------------------------------------------------------------------*/
APIRET MEIO_CloseDD()
{
  MW_SYSLOG_1(TRACE_MEIO_API,"meio_api::MEIO_CloseDD entry\n");

  if (meio_loaded == TRUE) {
    meiowinIOCTL(5,(LPARAM) 0); 
    meio_loaded=FALSE;
  } else {
    MW_SYSLOG_1(TRACE_MEIO_API,"meio_api::MEIO_CloseDD error meio not loaded\n");
  }

  MW_SYSLOG_1(TRACE_MEIO_API,"meio_api::MEIO_CloseDD exit rc=0\n");
  return (0);
}

 
/*************************************************************************/
/*                                                                       */
/*  dspMeioOpen() - open the MEIO interface, get an owner handle.        */
/*                  On first call the connection to the MEIO Device      */
/*                  Driver must be made.                                 */
/*                                                                       */
/*************************************************************************/
ULONG APIENTRY dspMeioOpen(HDSP       hDsp,    /* Handle of DSP device from DSPMGR  */
			   HMEIO FAR *phMeio,  /* Handle of MEIO Instance returned  */
			   ULONG      Options) /* reserved - must be 0              */
{
  MEIO_IOCTL_OPEN_PARMS RParm;
  ULONG                 ulRC;

  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioOpen entry hDsp %p Options %lx\n",hDsp,Options);

  MEIO_OpenDD();
  /*---------------------------------------------------------------------*/
  /* call the driver...                                                  */
  /*---------------------------------------------------------------------*/
  RParm.hDsp          = hDsp;        /* put API Parms in Request Block */
  RParm.phMeio        = phMeio;
  RParm.ulOptions     = Options;
  /* and call the MEIO Device Driver */
  ulRC=meiowinIOCTL(MEIO_IOCtl_Open,
		    (LPARAM)&RParm);

  if (!ulRC) {
    ulRC = RParm.ulRC;        /* yup, then return code from function */
  } else {
    ulRC = MEIO_FAIL_DRIVER_CALL;
  }   

  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioOpen exit hMeio %lx ulRC %lx\n",*phMeio,ulRC);
  return(ulRC);
}

/*************************************************************************/
/*                                                                       */
/*  dspMeioClose()- close a MEIO handle, possibly disconnect connections */
/*                  owned by this handle.                                */
/*                                                                       */
/*************************************************************************/
ULONG APIENTRY dspMeioClose(HMEIO      hMeio,   /* Handle of MEIO Instance returned  */
			    ULONG      Options) /* reserved - must be 0              */
{
  MEIO_IOCTL_CLOSE_PARMS RParm;
  ULONG ulRC;

  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioClose entry hMeio %lx Options %lx\n",hMeio,Options);

  RParm.hMeio         = hMeio;       /* put API Parms in Request Block */
  RParm.ulOptions     = Options;
                                     /* and call the MEIO Device Driver */
  ulRC = meiowinIOCTL(MEIO_IOCtl_Close,
                      (LPARAM)&RParm);

  if (0 == ulRC) {
    ulRC = RParm.ulRC;        /* yup, then return code from function */
  } else {
    ulRC = MEIO_FAIL_DRIVER_CALL;
  } 
   
  MEIO_CloseDD();
  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::dspMeioClose exit rc=%lx\n",ulRC);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO QUERY API                                                         */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioQuery(HMEIO           hMeio,        /* MEIO Handle                     */
			    MEIO_QUERYTYPE  Request,      /* type of Query                   */
			    ULONG    FAR   *fpBufferSize, /* size of buffer for returned data*/
			                                  /* required size is returned if    */
                                                          /* insufficient for type of query  */
			    VOID     FAR   *fpBuffer)     /* buffer for returned data        */
{
  MEIO_IOCTL_QUERYAB_PARMS  RParm;          /* request parameter block */
  ULONG        ulRC;
  
  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioQuery entry hMeio %lx Request %x \n",hMeio,Request);
  
  if (meio_loaded == TRUE) {
    RParm.hMeio         = hMeio     ;  /* put API Parms in Request Block */
    RParm.sRequest      = (short)Request;
    RParm.fp16Buffer    = fpBuffer    ;
    RParm.fp16Buffersize= fpBufferSize;
    /* and call the MEIO Device Driver */
    ulRC=meiowinIOCTL(MEIO_IOCtl_Query_Abilities,
		      (LPARAM)&RParm);
    
    if (0 == ulRC) {
      ulRC = RParm.ulRC;        /* yup, then return code from function */
    } else {
      ulRC = MEIO_FAIL_DRIVER_CALL;
    } 
  } else {
    ulRC=MEIO_FAIL_DRIVER_OPEN;
  }
  
  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::dspMeioQuery exit ulRC %lx\n",ulRC);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO CONNECT API                                                       */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioConnect(HMEIO            hMeio,          /* MEIO Handle               */
			      HCONNECTION FAR *fpConnection,   /* THE Connection is returned*/
			      ULONG            OwnerPort,      /* Connection Owner          */
			      ULONG            UserPort,       /* Connection User           */
			      ULONG            SharingOptions, /* willingness to share      */
			      ULONG            ulState,        /* set switch to this stateCH02*/
			      ULONG            Reserved)       /* reserved - must be 0      */
{
  MEIO_IOCTL_CONNECT_PARMS RParm;  /* request parameter block */
  ULONG                    ulRC;
  
  MW_SYSLOG_5(TRACE_MEIO_API,"meio_api::dspMeioConnect entry hMeio %lx OwnerPort %lx UserPort %lx ulState %lx\n",
	 hMeio,OwnerPort,UserPort,ulState);
  
  if (meio_loaded == TRUE) {
    RParm.hMeio         = hMeio;
    RParm.fp16Connection= fpConnection;
    RParm.OwnerPort     = OwnerPort;
    RParm.UserPort      = UserPort;
    RParm.SharingOptions= SharingOptions;
    RParm.ulState       = ulState;                               
    RParm.Reserved      = Reserved;
    
    ulRC=meiowinIOCTL(MEIO_IOCtl_Connect,
		      (LPARAM)&RParm);
    
    if (0 == ulRC) {
      ulRC = RParm.ulRC;        /* yup, then return code from function */
    } else {
      ulRC = MEIO_FAIL_DRIVER_CALL;
    } 
  } else {
    ulRC=MEIO_FAIL_DRIVER_OPEN;
  }
  
  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioConnect exit fpConnection %lx ulRC %lx\n",
	 *fpConnection,ulRC);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO DISCONNECT API                                                    */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioDisconnect(HCONNECTION hConnection,         /* handle returned by     */
				 /* dspMeioConnect()       */
				 ULONG       ulReserved)          /* must be 0          CH02*/
{
  MEIO_IOCTL_DISCONNECT_PARMS RParm;  /* request parameter block */
  ULONG                       ulRC;
  
  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::dspMeioDisconnect entry hConnection %lx\n",
	 hConnection);
  
  if (meio_loaded == TRUE) {
    RParm.hConnection = hConnection;
    RParm.Reserved    = ulReserved;                                  
    
    ulRC=meiowinIOCTL(MEIO_IOCtl_Disconnect,
		      (LPARAM)&RParm);
    
    if (0 == ulRC) {
      ulRC = RParm.ulRC;        /* yup, then return code from function */
    } else {
      ulRC = MEIO_FAIL_DRIVER_CALL;
    }
  } else {
    ulRC=MEIO_FAIL_DRIVER_OPEN;
  }
  
  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::dspMeioDisconnect exit ulRC %lx\n",
	 ulRC);
  return(ulRC);
}


/*------------------------------------------------------------------------*/
/* MEIO Update Connection Attributes                                      */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioUpdateConnection(HCONNECTION  hConnection,  /* handle returned by             */
				       /* dspMeioConnect()               */
				       MEIO_CONNECTATTRIBUTE Attribute,    /* define attribute to update.    */
				       LONG                  lValue)       /* value of attribute.            */
{
  MEIO_IOCTL_UPDATECON_PARMS  RParm;  /* request parameter block */
  ULONG                       ulRC;

  MW_SYSLOG_4(TRACE_MEIO_API,"meio_api::dspMeioUpdateConnection entry hConnection %lx Attribute %x lValue %lx\n",
	 hConnection,Attribute,lValue);

  if (meio_loaded == TRUE) {
    RParm.hConnection = hConnection;
    RParm.sAttribute  = (short)Attribute;
    RParm.lValue      = lValue;
    
    ulRC=meiowinIOCTL(MEIO_IOCtl_UpdateConnection,
		      (LPARAM)&RParm);
    
    if (0 == ulRC) {
      ulRC = RParm.ulRC;        /* yup, then return code from function */
    } else {
      ulRC = MEIO_FAIL_DRIVER_CALL;
    } 
  } else {
    ulRC=MEIO_FAIL_DRIVER_OPEN;
  }

  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::dspMeioUpdateConnection exit ulRC %lx\n",
	 ulRC);
  return(ulRC);
}

/*------------------------------------------------------------------------*/
/* MEIO Query Connection Attributes                                       */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioQueryConnection(HCONNECTION           hConnection,  /* handle returned by             */
                                                                          /* dspMeioConnect()               */
				      MEIO_CONNECTATTRIBUTE Attribute,    /* define attribute to query.     */
				      LONG             FAR *fpValue)      /* value of attribute.            */
{
  MEIO_IOCTL_QUERYCON_PARMS   RParm;  /* request parameter block */
  ULONG                       ulRC;

  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioQueryConnection entry hConnection %lx Attribute %x\n",
	 hConnection,Attribute);

  if (meio_loaded == TRUE) {  
    RParm.hConnection = hConnection;
    RParm.sAttribute  = (short)Attribute;
    RParm.fp16lValue  = fpValue;
    
    ulRC=meiowinIOCTL(MEIO_IOCtl_QueryConnection,
		      (LPARAM)&RParm);
    
    if (0 == ulRC) {
      ulRC = RParm.ulRC;        /* yup, then return code from function */
    } else {
      ulRC = MEIO_FAIL_DRIVER_CALL;
    } 
  } else {
    ulRC=MEIO_FAIL_DRIVER_OPEN;
  }  
  
  MW_SYSLOG_3(TRACE_MEIO_API,"meio_api::dspMeioQueryConnection exit fpValue %lx ulRC %lx\n",
	 *fpValue,ulRC);
  return(ulRC);
}

/*------------------------------------------------------------------------*/
/* MEIO RESYNC API - reload MEIO Dsp task, etc.                      CH01 */
/*------------------------------------------------------------------------*/
ULONG APIENTRY dspMeioResync( MEIO_OBJ_TYPE ObjectType,    /* type of handle being passed*/
			      ULONG         Object,        /* some Mwave handle          */
			      /* dspMeioConnect()           */
			      ULONG         ulOptions)     /* resync options, must beCH01*/
{
  MEIO_IOCTL_RESYNC_PARMS     RParm;  /* request parameter block */
  ULONG                       ulRC;

  MW_SYSLOG_4(TRACE_MEIO_API,"meio_api::dspMeioResync entry ObjectType %x Object %lx ulOptions %lx\n",
	 ObjectType,Object,ulOptions);

  if (meio_loaded == TRUE) {
    RParm.sObjectType = (short)ObjectType;
    RParm.Object      = Object;
    RParm.ulOptions   = ulOptions;
    
    ulRC=meiowinIOCTL(MEIO_IOCtl_Resync,
		      (LPARAM)&RParm);
    
    if (0 == ulRC) {
      ulRC = RParm.ulRC;        /* yup, then return code from function */
    } else {
      ulRC = MEIO_FAIL_DRIVER_CALL;
    } 
  } else {
    ulRC=MEIO_FAIL_DRIVER_OPEN; 
  }
  
  MW_SYSLOG_2(TRACE_MEIO_API,"meio_api::dspMeioResync exit ulRC %lx\n",
	 ulRC);
  return(ulRC);
}
