/* ============================================================
 * File  : streamtuned.cpp
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2004-02-12
 * Description : streamtuned gui
 *
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// development snapshot, many todo's

#include <iostream>


#include <qapplication.h>
#include <unistd.h>
#include <qpainter.h>
#include <math.h>

#include "fft.h"

#ifndef M_PI
#define M_PI 3.14159265358979323846264338327950288
#endif

// #butif M_PI 3.14 would do too. #wouldntif ?


using namespace std;

static QString emptyStr = "";


SampleObject::SampleObject() : QObject()
{
    timer = new QTimer( this );

    connect(
             timer,
             SIGNAL(timeout()),
             this,
             SLOT(checkSamples())
           );

    // todo: get mapped mem filename and audio buffer size from mplayer output
    // audio buffer size is now fixed to 4x mapped memory size to compensate
    // for output vs. fft delay.
    QString username = QString(getenv("USER"));
    QString file = "/tmp/mplayer-af_export_" + username;

    if ((fd = open(file,O_RDWR)) < 0)
    {
       cerr << "cannot open " << file << endl;
       return;
    }

    if (fstat(fd, &fileStat) < 0)
    {
       cerr << "cannot stat " << file << endl;
       return;
    }

    //cout << "export file size: " << fileStat.st_size << endl;

    fileBase = (commArea*)mmap(0,fileStat.st_size,PROT_READ|PROT_WRITE,MAP_SHARED,fd,0);

    if ( (long)fileBase == -1)
    {
        cerr << "mmap error" << endl;
        return;
    }

    prevCounter = 1; //fileBase->counter;

    timer->start( 100, false);
}

SampleObject::~SampleObject()
{
    if ( (long)fileBase != -1 )
      munmap(fileBase, fileStat.st_size);

    close(fd);
}

void SampleObject::checkSamples()
{
    if ( prevCounter != fileBase->counter )
    {
      prevCounter = fileBase->counter;
      emit sampleReady(fileBase);
    }
}

// ----------------------------------------------------------------------

FFTStarter::FFTStarter() : QObject(), QThread() {};

void FFTStarter::run()
{
  emit threadedTrigger();
}

// ----------------------------------------------------------------------

FFTConverter::FFTConverter(int numpoints, int numsamples) : QObject()
{
    initialized = false;
    linkedSampler = 0;
    fftNumPoints = numpoints;
    sampleWindow = numsamples;

    in = 0;
    out = 0;

    if (fftNumPoints > 100)
    {
      cerr << "FFTConverter error: spectrum points > 100" << endl;
      exit(-1);
    }

    if (sampleWindow % 2 != 0)
    {
      cerr << "FFTConverter error: only even sample window size allowed" << endl;
      exit(-1);
    }

    FFTStarter *starter;
    starter = new FFTStarter();

    connect(
             starter,
             SIGNAL(threadedTrigger()),
             this,
             SLOT(initTrigger())
           );

    starter->start();
}

FFTConverter::~FFTConverter()
{
  unloadSampler();
  if (plan)
    fftw_destroy_plan(plan);
  free(in);
  free(out);
}

void FFTConverter::loadSampler(SampleObject *sampler)
{
    unloadSampler();
    linkedSampler = sampler;

    processIndex = 0;
    displayIndex = -4;

    connect(
             sampler,
             SIGNAL(sampleReady(commArea*)),
             this,
             SLOT(processSamples(commArea*))
           );

}

void FFTConverter::unloadSampler()
{
   if (linkedSampler)
     linkedSampler->disconnect(this);
   linkedSampler = 0;

   // clear fft values
   for (int i=0; i < fftNumPoints; i++)
     for (int k=0; k<10; k++)
       fftValues.entry[k].values[i] = 0;
}

void FFTConverter::processSamples(commArea *fileBase)
{
    if (!initialized)
    {
        if ( fileBase && fileBase->nch != 0 &&
             sampleWindow > fileBase->size / 2 / fileBase->nch )
        {
          cerr << "FFTConverter: sampleWindow size=" << sampleWindow << "("
               << fileBase->nch << "ch) exeeds player shared memory ("
               << fileBase->size << "KByte)" << endl;
          unloadSampler();
          return;
        }

        return;
        init();
    }

    // cout << fileBase->counter << endl;

    double tot, re, im, drad;
    int loop;

    signed short int *lbuf, *rbuf;
    lbuf = (signed short int *)(fileBase + 1);
    rbuf = lbuf + sampleWindow; // should be buf for every nch

    drad = M_PI / sampleWindow;

    for (int i = 0; i < sampleWindow; i++)
    {
        // samples are windowed
        c_re( in[i] ) = (int) ( lbuf[i] * sin(drad*i) );
        //c_re( in[i] ) = 32000 - (i%2)*64000 ;
        c_im( in[i] ) = 0;
    }

    // should use real fft...
    fftw(plan, 1, in, 1, 0, out, 1, 0);

    // remove DC offset
    c_re(out[0]) = 0;
    c_im(out[0]) = 0;

    // display x-scale: start linear, handover to log function
    int fftPoint = 0;       // display x-axis
    double linCoeff = 0.28; // linear coefficient (linCoeff*freq = display pos)
    double logPos, linPos;  // holds log and lin display pos for frequency
    double linWeight;       // holds relative weight of linPos scale
    int curPos;             // holds weighted display pos for frequency

    int nfreq = sampleWindow/2; // fft freq points
    int handoverDiv = 4;        // nfreq/handoverDiv is frequency with linWeight=0

    int handoverFreq = nfreq / handoverDiv; // frequency with linWeight = 0
    // log function disabled for freq<10,
    // factor 10 used to reduce high freq step size.
    double div = log10((double)nfreq/10);
    tot  = 0;
    loop = 0;

    for (int i = 0; i < nfreq; i++)
    {
        re = c_re(out[i+1]) / sampleWindow;
        im = c_im(out[i+1]) / sampleWindow;

        // lin and log display pos:
        linPos = linCoeff*i;
        if (i>=10)
          logPos = log10((double)i/10)*fftNumPoints/div;
        else
          logPos = 0;

        linWeight = (double)(handoverFreq - i)/handoverFreq;
        if (linWeight < 0) linWeight = 0;
        curPos = (int)(linWeight*linPos + (1-linWeight)*logPos);

        //cout << "i l c: " << i << " " << logPos << " " << curPos << endl;
        if ( curPos <= fftPoint)
        {
          tot += re*re + im*im;
          loop++;
        }
          else
        {
          tot = tot / loop; // average
          tot = 80 * log10(tot)/log10(1E9);  // log power (0..80)
          //cerr << "store at " << fftPoint << ": " << tot << endl;
          fftValues.entry[processIndex].values[fftPoint] = (int)tot;

          fftPoint++;
          tot  = re*re + im*im;
          loop = 1;
        }
    }

    if (loop > 0) // will be if fft out not empty
    {
      tot = tot / loop; // normalize
      tot = 80 * log10(tot)/log10(1E9);  // log power
      //cerr << "store at " << fftPoint << ": " << tot << endl;
      fftValues.entry[processIndex].values[fftPoint] = (int)tot;
    }


    if (displayIndex >= 0)
      emit fftReady(&fftValues.entry[displayIndex]);

    if (++processIndex > 9) processIndex = 0;
    if (++displayIndex > 9) displayIndex = 0;

}

void FFTConverter::initTrigger()
{
   init();
}

void FFTConverter::init()
{
    in = (FFTW_COMPLEX*)malloc(sampleWindow * sizeof(FFTW_COMPLEX));
    out = (FFTW_COMPLEX*)malloc(sampleWindow * sizeof(FFTW_COMPLEX));

    plan = fftw_create_plan(sampleWindow, FFTW_FORWARD, FFTW_MEASURE);
    if (plan == NULL)
    {
       cerr << "FFTConverter: Error creating fft plan" << endl;
       unloadSampler();
       return;
    }

    initialized = true;
}



