/* ============================================================
 * File  : MythStream.cpp
 * Author: Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date  : 2004-02-12
 * Description : MythStream gui
 *
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <iostream>

#include <qapplication.h>
#include <unistd.h>
#include <qpushbutton.h>
#include <qfont.h>
#include <qpainter.h>
#include <mythtv/mythdialogs.h>
#include <mythtv/mythcontext.h>
#include <mythtv/mythdbcon.h>

#include "mythstream.h"
#include "../libs/streambrowser.h"

using namespace std;

// todo: add streamCustomInfo's to audio_panel/video_panel (Artist, Genre, ...)
//       replace fft by OpenGL
//       check painting order / update events

static QString emptyStr = "";
QColor fftLow   = QColor(0x07, 0x99, 0x00); //#079900
QColor fftMid   = QColor(0xd0, 0xd3, 0x00); //#d0d300
QColor fftHigh  = QColor(0xef, 0xed, 0xed); //#efeded

// ----------------------------------------------------------------------

FFTBox::FFTBox(QWidget *parent, int barcnt, unsigned int x, unsigned int y, unsigned int w, unsigned int h)
     : QWidget( parent, "", Qt::WRepaintNoErase)
{
   go = false;
   percentage = 0;

   // we want 20 bars, space width = 2
   barCount   = barcnt;
   if (barCount > 100)
   {
     cerr << "gauge supports 100 bars max" << endl;
     exit(-1);
   }

   barWidth = int ( ( w - 2*(barCount - 1) ) /barCount );

   // bars must have width > 0
   if ( barWidth < 1 )
     barWidth = 1;

   gaugeWidth  = barCount * barWidth + 2*(barCount - 1) + 1;// - 1;
   div =  float(h) / 100;

   setBackgroundOrigin(QWidget::WindowOrigin);
   setGeometry( x, y, gaugeWidth, h );

   hide();

   // hack: wait for screen buildup
   grabTimer = new QTimer( this );
   connect( grabTimer, SIGNAL(timeout()), this, SLOT(buildFFTBackground()) );
   grabTimer->start( 1000, TRUE );
   //buildFFTBackground();
}

FFTBox::~FFTBox()
{
}

void setPainterCol(QPainter *p, QColor col)
{
   p->setPen( col );
   p->setBrush( QBrush(col, QBrush::SolidPattern) );
}

void FFTBox::buildFFTBackground()
{
    //fftPict.fill( QColor(0x07, 0x99, 0) );
    //return;
    if (go) return;
    // grabbing window, not widget
    QWidget *parent = parentWidget();
    bckPict = QPixmap::grabWindow( parent->winId(), x(), y(), gaugeWidth, height());

    fftPict = QPixmap(bckPict);
    QColor col = QColor(0x07, 0x99, 0);

    QPainter *painter = new QPainter( &fftPict );
    painter->setPen( col );
    painter->setBrush( QBrush(col, SolidPattern) );

    int barHeight = height();
    int divSpace = 2;
    int barDiv = (barHeight - divSpace * 19) / 20;
    if (barDiv <= 0)
    {
      cerr << "MythStream: fft window not large enough" << endl;
      go = false;
      return;
    }

    for ( int i = 0; i < barCount; i++)
      for ( int j = 0; j < 20; j++)
      {
        if (j == 0)  setPainterCol(painter, fftLow);
        if (j == 5)  setPainterCol(painter, fftMid);
        if (j == 10) setPainterCol(painter, fftHigh);
        painter->drawRect( i*(barWidth + 2) + 1, barHeight - (j + 1) * (barDiv + divSpace) + divSpace, barWidth, barDiv);
      }

    delete(painter);

    fftMask = QBitmap(fftPict.size(), true);
    drawPict = QPixmap(fftPict.size());
    show();
    go = true;
}

void FFTBox::paintEvent( QPaintEvent * )
{
    if (!go) return;

    QColor col = Qt::color1;

    fftMask.fill(Qt::color0);
    int barHeight;

    QPainter *painter = new QPainter( &fftMask );
    painter->setPen( col );
    painter->setBrush( QBrush(col, SolidPattern) );

    if (percentage)
    {
        for ( int i = 0; i < barCount; i++)
        {
          barHeight = int ( float( percentage->values[i] ) * div );
          if ( barHeight > height() ) barHeight = height();
          if ( barHeight < 0 ) barHeight = 0;
          painter->drawRect( i*(barWidth + 2) + 1, height() - barHeight, barWidth, height() );
        }
    }

    painter->setPen( Qt::color1 );
    painter->setBrush( QBrush(Qt::color1, NoBrush) );
    painter->drawRect( 0, height()-1, gaugeWidth, height() );
    delete(painter);

    fftPict.setMask(fftMask);

    bitBlt(&drawPict, QPoint(0,0), &bckPict, rect(), Qt::CopyROP);
    bitBlt(&drawPict, QPoint(0,0), &fftPict, rect(), Qt::CopyROP);
    bitBlt(this, QPoint(0,0), &drawPict, rect(), Qt::CopyROP);
}

void FFTBox::setPercentage(Spectrum* percent)
{
   percentage = percent;
   update();
}

void FFTBox::resetDisplay()
{
   if (percentage)
     for (int i=0; i<100; i++)
       percentage->values[i] = 0;
   update();
}

// ----------------------------------------------------------------------


PlayerState::PlayerState()
{
   streamStatus = "idle";
   playingVideo = false;
   displayStatus = browse;
   displayResetDelay = -1;
}

void PlayerState::streamStatusChanged(QString status)
{
   streamStatus = status;
   displayResetDelay = 1;
   pollMe();
}

void PlayerState::browserActivityDetected()
{
   displayStatus = browse;
   displayResetDelay = 3;
}

/*
void PlayerState::toggleInformationRequested()
{
   if ( displayStatus == information )
      displayStatus = browse;
   else
      displayStatus = information;
      
   displayResetDelay = 3;
}
*/

void PlayerState::videoDetected(bool video)
{
   playingVideo = video;
}

bool PlayerState::pollMe()
{
   if (displayResetDelay != -1 && --displayResetDelay == 0)
   {
     if (streamStatus == "playing" || streamStatus == "buffering" )
       if (playingVideo)
         displayStatus = video;
       else
         displayStatus = audio;

     if (streamStatus == "idle")
       displayStatus = browse;

     return true;
   }

   return false;
}



MythStream::MythStream(MythMainWindow *parent,
                       const char *name )
          : MythDialog(parent, name)
{
    setFocusPolicy( StrongFocus );
    
    harvesting = false;
    storageBin = new ReposStorage();
    streamStorage = new StreamStorage("streams", "mythstream" );
    
    m_ItemDisplaySize = 7;
    m_FolderDisplaySize = 4;

    // Load theme and ui file - hacked to introduct UIIconTextType, should be defined in uitypes...
    m_Theme = new XMLParse();
    m_Theme->SetWMult(wmult);
    m_Theme->SetHMult(hmult);
    QDomElement xmldata;
    m_Theme->LoadTheme(xmldata, "stream", "stream-");
    
    LoadWindow(xmldata);

    // scale images, use one of the loaded images
    LayerSet* layerSet = m_Theme->GetSet("myicons");
    UIImageType *itype;
    if (layerSet)
    {
      itype = (UIImageType *)layerSet->GetType("streampict");
      if (itype) streamPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("downloadpict");
      if (itype) downloadPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("urlpict");
      if (itype) urlPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("unknownpict");
      if (itype) unknownPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("infopict");
      if (itype) infoPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("htmlpict");
      if (itype) htmlPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("scheduledpict");
      if (itype) scheduledPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("recordingpict");
      if (itype) recordingPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("recordedpict");
      if (itype) recordedPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("generalpict");
      if (itype) generalPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("headerpict");
      if (itype) headerPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("markedpict");
      if (itype) markedPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("emptypict");
      if (itype) emptyPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("leftpict");
      if (itype) leftPict = itype->GetImage();
      itype = (UIImageType *)layerSet->GetType("rightpict");
      if (itype) rightPict = itype->GetImage();
    }

    
    clearVideo = false; // was video screen drawn?

    setNoErase();
    updateBackground();

    streamBrowser = new StreamBrowser(this, this, streamStorage);
    streamBrowser->setVideoRect(m_VideoRect);
    streamBrowser->setVideoMaxRect(m_VideoMaxRect);
    streamBrowser->setViewerRect(m_ViewerRect);
    streamBrowser->setViewerMaxRect(m_ViewerMaxRect);
    //streamBrowser->setItemDisplayFolded(m_DisplayOffset);
    //streamBrowser->setFolderDisplayFolded(m_DisplayOffset);
    streamBrowser->setItemDisplayFolded(-1, false);
    streamBrowser->setFolderDisplayFolded(-1, false);

    connect (
              streamBrowser,
              SIGNAL(eventItemTreeSwitchedTo(int)),
              this,
              SLOT(slotItemTreeSwitchedTo(int))
            );

    connect (
              streamBrowser,
              SIGNAL(eventHarvesterBusy(bool, QString)),
              this,
              SLOT(slotHarvesterBusy(bool, QString))
            );

    connect (
              streamBrowser,
              SIGNAL(eventVideoActive(bool)),
              this,
              SLOT(slotVideoActive(bool))
            );

    connect (
              streamBrowser,
              SIGNAL(eventUserMessage()),
              this,
              SLOT(slotUserMessage())
            );

    connect (
              streamBrowser,
              SIGNAL( eventValuesUpdated(int) ),
              this,
              SLOT( slotValuesUpdated(int) )
            );

    itemTreeLoaded = StreamBrowser::streams;
    
    layerSet = m_Theme->GetSet("dyn_panel");

    if (layerSet)
    {
      UIBlackHoleType *bhtype = (UIBlackHoleType *)layerSet->GetType("fft");
      if (bhtype)
        m_fftRect = bhtype->getScreenArea();
    }

    fft = new FFTBox( this, 20, m_fftRect.left(), m_fftRect.top(), m_fftRect.width(), m_fftRect.height());
    connect (
              streamBrowser,
              SIGNAL( eventFFTReady(Spectrum*) ),
              fft,
              SLOT( setPercentage(Spectrum*) )
            );
    streamBrowser->enableFFT(20);

//  all signals connected, now get streams

    connect (
               streamStorage,
               SIGNAL( storageEvent(int, int, bool ) ),
               this,
               SLOT  ( slotStorageEvent(int, int, bool ) )
             );

    storageBin->checkParsers(); // (stream list parsers) oops, should be somewhere else...
    selectStorages(storageBin, streamStorage, 100);

    QString error;
    if ( !streamStorage->loadList(100, error) )
       cerr << error << endl;

// let streamBrowser initialize displays

    streamBrowser->initReady();

    if (storageBin->oldParserMoved)
      reportEvent("Renamed old parser versions as *.pre-v"VERSION" (see MythStream README)");
}

MythStream::~MythStream()
{
  delete ( streamBrowser );
  if (streamStorage)
    delete ( streamStorage );
  if (storageBin)
    delete ( storageBin );
  delete fft;
}

void MythStream::parseContainer(QDomElement& e, QString& name, QRect& area)
{
    int context;

    m_Theme->parseContainer(e, name, context, area);

    if (name.lower() == "status_panel")
        m_TopRect = area;
    if (name.lower() == "dyn_panel")
        m_MidRect = area;
    if (name.lower() == "browse_panel")
        m_BotRect = area;
    if (name.lower() == "video")
        m_VideoRect = area;
    if (name.lower() == "maxvideo")
        m_VideoMaxRect = area;
    if (name.lower() == "viewer")
        m_ViewerRect = area;
    if (name.lower() == "maxviewer")
        m_ViewerMaxRect = area;

}

void MythStream::LoadWindow(QDomElement &element)
{
    QRect area;
    QString name;

    for (QDomNode child = element.firstChild(); !child.isNull();
         child = child.nextSibling())
    {
        QDomElement e = child.toElement();
        if (!e.isNull())
        {
            if (e.tagName() == "font")
            {
                m_Theme->parseFont(e);
            }
            else if (e.tagName() == "container")
            {
                parseContainer(e, name, area);
            }
            else
            {
                cerr << "Unknown element: " << e.tagName()
                     << endl;
                exit(0);
            }
        }
    }

    QString text;

    //text = "format: ";
    //loadField("selector", "formatlabel", text);

    //text = "codec: ";
    //loadField("selector", "codeclabel", text);

    //text = "FPS: ";
    //loadField("selector", "fpslabel", text);

    //text = "bandwidth: ";
    //loadField("selector", "bandwidthlabel", text);

    //text = "length: ";
    //loadField("selector", "lengthlabel", text);

    text = "status";
    loadField("status_panel", "statuslabel", text);

    text = "time";
    loadField("status_panel", "timelabel", text);

    text = "cache";
    loadField("status_panel", "cachelabel", text);

    text = "stability";
    loadField("status_panel", "stabilitylabel", text);

    text = "player";
    loadField("status_panel", "playlabel", text);

    text = "harvester";
    loadField("status_panel", "harvestlabel", text);

    loadBar("status_panel", "playled", 0);
    loadBar("status_panel", "harvestled", 0);
    loadBar("status_panel", "cachebar", 0);
    loadBar("status_panel", "stabilitybar", 0);

    loadBar("dyn_panel", "volumebar", 0);
    displayedVolume = 0;

}

/*
void MythStream::displayDetails()
{
    QString title, field1, field2, temp = "";
    
    bool ok = streamBrowser->getCurrentObjectDetails(title, field1, field2, false);
    
    if (ok)
    {
      title = "Showing details of: " + title;
      loadField("information_panel", "title", title);
      loadField("information_panel", "field1", field1);
      
      if (true)
      {    
        loadField("information_panel", "field2", field2);
        loadField("information_panel", "area1", temp);
      } 
        else
      {    
        loadField("information_panel", "field2", temp);
        loadField("information_panel", "area1", field2);
      }         
    }
    
}
*/

void MythStream::getStreamStatusValues()
{

    QString value, area, temp, containerName;
    UITextType *ttype;
    QString m_audioCh, m_audioSps, m_audioBps;

  // update status_panel fields
    value = streamBrowser->getStreamParameter(StreamBrowser::streamStatusStr);
    loadField("status_panel", "status", value);

    // todo: need status engine in StreamBrowser
    if ( currentStreamStatus != value )
    {
      temp = "";
      loadField("status_panel", "message", temp);

      playerState.streamStatusChanged(value);

      if ( value == "buffering" )
      {
        loadBar("status_panel", "playled", 50);
      }
      else if ( value == "playing" )
      {
        loadBar("status_panel", "playled", 100);
      }
      else if ( value == "idle" or value == "starting" )
      {
        fft->resetDisplay(); // labels are reset from StreamBrowser
        loadField("status_panel", "custominfo", emptyStr);	
;
		loadField("audio_panel", "title", emptyStr);
		loadField("audio_panel", "description", emptyStr);
		loadField("audio_panel", "audformat", emptyStr);
		loadField("audio_panel", "quality", emptyStr);
		loadField("audio_panel", "custom1", emptyStr);
		loadField("audio_panel", "custom2", emptyStr);
		loadField("audio_panel", "custom3", emptyStr);
		loadField("audio_panel", "custom4", emptyStr);
		loadField("audio_panel", "custom5", emptyStr);

        if (clearVideo)
        {
          updateBotView(true); // oops, gui is mess... (clear video window)
          clearVideo = false;
        }

      }
      else
        loadBar("status_panel", "playled", 0);

      currentStreamStatus = value;

      if ( playerState.getDisplayStatus() == PlayerState::video )
        fft->hide();
      else
        fft->show();
    }

    switch (playerState.getDisplayStatus())
    {
      case PlayerState::audio:
         containerName = "audio_panel";
         break;
      case PlayerState::video:
         containerName = "video_panel";
         break;
/*
      case PlayerState::information:
         containerName = "information_panel";
         break;
*/
      default:
         return;
         break;
    }

    value = temp = streamBrowser->getStreamParameter(StreamBrowser::streamName);
    loadField(containerName, "title", value);
    
    value = streamBrowser->getStreamParameter(StreamBrowser::streamDescr);
    if ( value == "" ) value = temp;
    loadField(containerName, "description", value);
    value = streamBrowser->getStreamParameter(StreamBrowser::streamUrl);

    value = streamBrowser->getStreamParameter(StreamBrowser::streamAudioFormat);
    loadField(containerName, "audformat", value);

    value = streamBrowser->getStreamParameter(StreamBrowser::streamAudioBps);
    m_audioBps = value + " Kbps ";
    value = streamBrowser->getStreamParameter(StreamBrowser::streamAudioSps);
    m_audioSps = value + " KHz ";
    value = streamBrowser->getStreamParameter(StreamBrowser::streamAudioChannels);
    if ( value == "1" ) m_audioCh = "mono";
    else
      if ( value == "2" ) m_audioCh = "stereo";
         else m_audioCh = value + " ch";

    if ( playerState.isVideoActive() )
    {
      loadField("video_panel", "videocodec", value);
      value = streamBrowser->getStreamParameter(StreamBrowser::streamVideoCodec);
      loadField("video_panel", "videocodec", value);
    }
    else
      value = "";

    if ( playerState.isVideoActive() )
    {
      value  = streamBrowser->getStreamParameter(StreamBrowser::streamVideoBps) + " Kbps ";
      if (value == " Kbps ") value = "-" + value;
      value += streamBrowser->getStreamParameter(StreamBrowser::streamVideoFps) + " Fps";
      loadField("video_panel", "videoquality", value);
    }
      else value = "";

    //value = streamBrowser->getStreamParameter(StreamBrowser::streamVideoFormat);
    //loadField("selector", "format", value);
    //value = streamBrowser->getStreamParameter(StreamBrowser::streamVideoLength);
    //loadField("selector", "length", value);

    value = m_audioBps + m_audioSps + m_audioCh;

    if (m_audioBps == " Kbps ")
      value = ""; // no stream

    loadField(containerName, "quality", value);

    
//    value = streamBrowser->getStreamCustomParameter(1, area);
//    if (ttype = findTtype(area)) ttype->SetText(value);
}

void MythStream::getStreamPlayValues()
{
    QString value, area, title;
    UITextType *ttype;
    bool converted;
    int i;

    // reset audio or video display after browse activity
    if ( playerState.pollMe() )
    {
      if ( playerState.getDisplayStatus() == PlayerState::video )
        streamBrowser->showVideo();
      update(m_BotRect);
    }

    value = streamBrowser->getStreamParameter(StreamBrowser::streamTime);
    loadField("status_panel", "time", value);

    value = streamBrowser->getStreamParameter(StreamBrowser::streamCache);
    i = value.toInt(&converted);
    if (!converted) i = 0;
    loadBar("status_panel", "cachebar", i);

    value = streamBrowser->getStreamParameter(StreamBrowser::streamStability);
    i = value.toInt(&converted);
    if (!converted) i = 0;
    loadBar("status_panel", "stabilitybar", i);
    
    value = streamBrowser->getStreamParameter(StreamBrowser::streamVolume);
    i = value.toInt(&converted);
    //cout << " volume: " << value << " = int: " << i << endl;
    if (!converted) i = 0;

    if ( i != displayedVolume )
    {
      displayedVolume = i;
      loadBar("dyn_panel", "volumebar", i);
    }

	for ( i = 0; i < streamBrowser->getStreamCustomParameterCount(); i++)
      if ( (value = streamBrowser->getStreamCustomParameter(i, area, title)) && value != "" )
      {
		if (title && title != "")
			value = title + ": " + value;
		if (ttype = findTtype(area))
		  ttype->SetText(value);
	  }
    
}

void MythStream::slotStorageEvent(int ident, int eventType, bool error)
{
   if ( ident != 100 ) return;

   if (error)
   {
     reportEvent( streamStorage->getLastError() );
     return;
   }

   QString err;

   switch ( eventType )
   {
     case StreamStorage::selected:
        if ( streamStorage->getAccessType() == StreamStorage::web )
          if ( !streamStorage->loadList(100, err) )
            reportEvent ( err );
     break;

     case StreamStorage::loaded:
     break;

     default:
     break;
   }
}

void MythStream::updateBackground()
{
    QPixmap pix(size());
    pix.fill(this, 0, 0);

    QPainter p(&pix);
     LayerSet *container = m_Theme->GetSet("background");
    if (container)
    {
        container->Draw(&p, 0, 0);
    }
    p.end();

    setPaletteBackgroundPixmap(pix);
}


// VideoContainer events
bool MythStream::eventFilter( QObject *obj, QEvent *ev)
{
  // events for VideoContainer, already handled by mplayer
  if (obj && ev->type() == QEvent::KeyPress )
  {
    // pass key if not handled by mplayer (and how would I know it isn't?)
    QKeyEvent *kev = static_cast<QKeyEvent*>(ev);
    if ( kev->key() != Key_P )
      qApp->sendEvent(this, ev);
    return true;
  }

/*
  if (e->type() == QEvent::Paint)
  {
     update();
     return true;
  }
*/

  return false;

}


bool MythStream::processAction( QString action )
{
    Command key_command = nada;
    bool isnumber;   
    
    int actNum = action.toInt(&isnumber);
    if (!isnumber) actNum = -1;
    
    if (action == "PREVITEM")    
      key_command = previtem;
    else if (action == "NEXTITEM")
      key_command = nextitem;
    else if (action == "PREVFOLDER")
      key_command = prevfolder;
    else if (action == "NEXTFOLDER")
      key_command = nextfolder;
    else if (action == "SELECT")
    {
      if ( playerState.getDisplayStatus() == PlayerState::video )
        key_command = fullscreen;
      else
        key_command = selectitem;
    }
    else if (action == "ESCAPE")
      key_command = escape;
    else if (action == "END")
      key_command = end;
    else if (action == "DUMP")
      key_command = dump;
    else if (action == "FULLSCREEN")
      key_command = fullscreen;
    else if (action == "PAUSE")
      key_command = pauseitem;
    else if (action == "RECORD")
      key_command = record;
    else if (action == "STOPRECORD")
      key_command = stoprecord;
    else if (action == "STOPALLRECORD")
      key_command = stopallrecord;
    else if (action == "VOLDN")
      key_command = voldn;
    else if (action == "MUTE")
      key_command = mute;
    else if (action == "VOLUP")
      key_command = volup;
    else if (action == "AVDEC")
      key_command = avdec;
    else if (action == "AVINC")
      key_command = avinc;
    else if (action == "MARK")
      key_command = mark;
    else if (action == "STOREMARKED")
      key_command = storemarked;
    else if (action == "INSPECT")
      key_command = inspect;
    else if (action == "FORWARD")
      key_command = forwardstream;
    else if (action == "REWIND")
      key_command = rewindstream;
    else if ( actNum > -1 && actNum < 10 )
      selectStorageByIndex(storageBin, streamStorage, actNum);

    return streamBrowser->handlePressedKey(key_command);
}

void MythStream::keyPressEvent(QKeyEvent *e)
{
    if (!e) return;


    bool handled = false;
    QStringList actions;

    gContext->GetMainWindow()->TranslateKeyPress("Stream", e, actions);

    for (unsigned int i = 0; i < actions.size() && !handled; i++)
    {
        QString action = actions[i];
        QString browserAction = "";
                
        if ( action == "LEFT"  ) browserAction = "PREVFOLDER";
        if ( action == "RIGHT" ) browserAction = "NEXTFOLDER";
        if ( action == "UP"    ) browserAction = "PREVITEM";
        if ( action == "DOWN"  ) browserAction = "NEXTITEM";

        if (browserAction != "")
        {
          playerState.browserActivityDetected();
          streamBrowser->hideVideo();
          action = browserAction;
        }

        handled = processAction(action);        
    }

    if (!handled) MythDialog::keyPressEvent(e);
}


void MythStream::slotItemTreeSwitchedTo(int itemTree)
{
    itemTreeLoaded = StreamBrowser::ItemTreeLoaded(itemTree);
    
    harvesting = ( itemTreeLoaded == StreamBrowser::harvester );
    
    if ( harvesting )
      loadBar("status_panel", "harvestled", 50);
    else
      loadBar("status_panel", "harvestled", 0);

}

void MythStream::slotHarvesterBusy(bool busy, QString message)
{
    QString value = "";

    if ( busy )
    {
      value = "harvesting";
      loadBar("status_panel", "harvestled", 100);
    }
      else
    {
      // fore browse mode
      playerState.streamStatusChanged("idle");
      update( m_BotRect);
      slotItemTreeSwitchedTo(int(itemTreeLoaded));
    }

    loadField("status_panel", "message", message);
    loadField("status_panel", "status", value);
    update( m_TopRect);
}

void MythStream::slotVideoActive(bool active)
{
   playerState.videoDetected(active);

   update( m_BotRect );
}

void MythStream::slotUserMessage()
{
  QString value = streamBrowser->getUserMessage();
  loadField("status_panel", "message", value);
  update( m_TopRect );
}

void MythStream::reportEvent(QString msg)
{
  loadField("status_panel", "message", msg);
  update( m_TopRect );
/*
   userMessageBox->setText( msg );
   userMessageBox->show();
*/
}


void MythStream::getFolderList()
{
    QStringList list;
    bool onTop, atBottom;
    int cursor = streamBrowser->getDisplayFolderList(m_FolderDisplaySize, list, onTop, atBottom);
    loadListFields("browse_panel", "folder", list, cursor);
    
    if (onTop) 
      loadIconSource("browse_panel", "folder_left", emptyPict);
    else
      loadIconSource("browse_panel", "folder_left", leftPict);
      
    if (atBottom) 
      loadIconSource("browse_panel", "folder_right", emptyPict);
    else
      loadIconSource("browse_panel", "folder_right", rightPict);
}


void MythStream::getItemList()
{
    QStringList list;
    QString msg;
    QString tmp = "";
    
    bool onTop, atBottom;
    
    int cursor = streamBrowser->getDisplayItemList(m_ItemDisplaySize, list, onTop, atBottom);
    
    msg = streamBrowser->getCurrentFolderCaption();
  
    if ( harvesting )
    {
        loadField("browse_panel", "browse_title", tmp );
        loadField("browse_panel", "harvest_title", msg );
    }
    else
    {
        loadField("browse_panel", "browse_title", msg );
        loadField("browse_panel", "harvest_title", tmp );
    }
    
    loadListFields("browse_panel", "item", list, cursor);
}


void MythStream::slotValuesUpdated(int updatedType)
{
   switch( updatedType )
   {
     case StreamBrowser::streamstatus:
       getStreamStatusValues();
       update(m_BotRect);  // stream info fields
       update(m_TopRect);  // status field
     break;
     case StreamBrowser::streamplay:
       getStreamPlayValues();
       update(m_BotRect);  // stream play status
       update(m_TopRect);  // stream play status
     break;
     case StreamBrowser::folderlist:
       getFolderList();
       getItemList();
       update(m_BotRect);
     break;
     case StreamBrowser::itemlist:
       getItemList();
       update(m_BotRect);
     break;
    /*
     case StreamBrowser::information:
       playerState.toggleInformationRequested();
       displayDetails();
       update(m_BotRect); // stream info fields
       //update(m_BotRect);
     break;
    */
   }
}


// orig to bottom

void MythStream::paintEvent(QPaintEvent *e)
{
    QRect r = e->rect();

    if (r.intersects(m_TopRect))
        updateTopView();
    if (r.intersects(m_MidRect))
    {
        updateMidView();
    }
    if (r.intersects(m_BotRect))
    {
        updateBotView(false);
    }
}

UITextType * MythStream::findTtype(const QString field)
{
  // find field in status_panel or audio_panel containers
  QString container = "status_panel";
  UITextType *ttype;
  LayerSet* layerSet;

  layerSet = m_Theme->GetSet(container);

  if (layerSet)
  {
    ttype = (UITextType *)layerSet->GetType(field);
    if (!ttype)
    {
      container = "audio_panel";
      layerSet = m_Theme->GetSet(container);
      ttype = (UITextType *)layerSet->GetType(field);
    }
  }

  return ttype;
}

void MythStream::loadField(const QString& container, const QString& field, QString& value)
{
  LayerSet* layerSet = m_Theme->GetSet(container);

  if (layerSet)
  {
    UITextType *ttype = (UITextType *)layerSet->GetType(field);
    if (ttype)
      ttype->SetText(value);
    else cerr << "MythStream: UITextType " << field << " not found" << endl;
  }
    else cerr << "MythStream: container " << container << " not found" << endl;
}


void MythStream::loadUIImageType(UIImageType *icon, QChar prefix)
{
   if (prefix == QChar() ) prefix = ' ';
   
   switch (prefix)
   {
     case '~':
       icon->SetImage(streamPict);
       //icon->LoadImage();
     break;
     case 'D':
       icon->SetImage(downloadPict);
       //icon->LoadImage();
     break;
     case '-':
       icon->SetImage(unknownPict);
       //icon->LoadImage();
     break;
     case 'I':
       icon->SetImage(infoPict);
       //icon->LoadImage();
     break;
     case 'H':
       icon->SetImage(htmlPict);
       //icon->LoadImage();
     break;
     case ' ':
       icon->SetImage(emptyPict);
       //icon->LoadImage();
     break;
     case '_':
       icon->SetImage(urlPict);
      // icon->LoadImage();
     break;
     case 'R':
       icon->SetImage(recordingPict);
      // icon->LoadImage();
     break;
     case 'S':
       icon->SetImage(scheduledPict);
      // icon->LoadImage();
     break;
     case '#':
       icon->SetImage(recordedPict);
      // icon->LoadImage();
     break;
     case '>': // header (keeptopline=1)
       icon->SetImage(headerPict);
      // icon->LoadImage();
     break;
     case 'E': // empty line
       icon->SetImage(emptyPict);
     //  icon->LoadImage();
     break;
     case '.': // empty line
       icon->SetImage(generalPict);
     //  icon->LoadImage();
     break;
     case '!': // marked item
       icon->SetImage(markedPict);  // myth
       //icon->LoadImage();
     break;
     default:
       icon->SetImage(generalPict);
       //icon->LoadImage();
     break;
   }
}

void MythStream::loadIconSource(QString containerName, QString name, QPixmap &source)
{
   LayerSet* layerSet = m_Theme->GetSet(containerName);
   if (layerSet)
   {
     UIImageType *itype = (UIImageType *)layerSet->GetType(name);
     if (itype)
       itype->SetImage(source);
   }
}

void MythStream::loadListFields(const QString& container, const QString& fieldbasename, QStringList& list, int cursor)
{
  LayerSet* layerSet = m_Theme->GetSet(container);
  int i = 0;
  QString fieldName, caption;
  QChar prefix;
  
  if (layerSet)
  {
      for ( QStringList::Iterator iter = list.begin();
            iter != list.end();
            ++iter )
      {
         i++;
         caption = *iter;
         prefix  = ' ';
         
         // set icon if any
         fieldName = fieldbasename + "_image" + QString::number(i);
         UIImageType *itype = (UIImageType *)layerSet->GetType(fieldName);
         if (itype)
         {
            prefix = caption.at(0);
            caption = caption.remove(0, 1);
            loadUIImageType( itype, prefix );
         }
         
         fieldName = fieldbasename + QString::number(i);
         UITextType *ttype = (UITextType *)layerSet->GetType(fieldName);
         if (ttype)
           ttype->SetText(caption);
         else cerr << "MythStream: UITextType " << fieldName << " not found" << endl;

         UIImageType *imtype = (UIImageType *)layerSet->GetType(fieldbasename + "_cursor");
         if (imtype)
         {
           QPoint pt = imtype->DisplayPos();
           if ( fieldbasename == "folder" ) 
             pt.setX( int(cursor * 188 * wmult) );
           else 
             pt.setY( int(42 * hmult + cursor * 28 * hmult) );
           imtype->SetPosition(pt);
         }
         else cerr << "MythStream: UIImageType " << fieldbasename << "_cursor not found" << endl;
         
      }
  }
    else cerr << "MythStream: container " << container << " not found" << endl;
    
  int m_DisplaySize = 0;
  if ( fieldbasename == "folder" ) m_DisplaySize = m_FolderDisplaySize; else m_DisplaySize = m_ItemDisplaySize;
  
  while (i < m_DisplaySize)
  {
     i++;
     fieldName = fieldbasename + QString::number(i);
     UITextType *ttype = (UITextType *)layerSet->GetType(fieldName);
     if (ttype)
       ttype->SetText("");
         
     fieldName = fieldbasename + "_image" + QString::number(i);
     UIImageType *itype = (UIImageType *)layerSet->GetType(fieldName);
     if (itype)
         loadUIImageType(itype, ' ' );
   }
}
      

void MythStream::loadBar(const QString& container, const QString& field, int value)
{
  LayerSet* layerSet = m_Theme->GetSet(container);

  if (layerSet)
  {
    UIStatusBarType *ttype = (UIStatusBarType *)layerSet->GetType(field);
    if (ttype)
    {
      ttype->SetTotal(100);
      ttype->SetUsed(value);
      ttype->refresh();
    }
      else
       cerr << "MythStream: UIStatusBarType " << field << " not found" << endl;
  }
}

void MythStream::updateTopView()
{
   // unconditional update
    QPixmap pix(m_TopRect.size());
    pix.fill(this, m_TopRect.topLeft());
    QPainter p(&pix);

    LayerSet* container = m_Theme->GetSet("status_panel");

    if (container)
    {
      container->Draw(&p, 0, 0);
      container->Draw(&p, 1, 0);
      container->Draw(&p, 2, 0);
      container->Draw(&p, 3, 0);
      container->Draw(&p, 4, 0);
      container->Draw(&p, 5, 0);
      container->Draw(&p, 6, 0);
      container->Draw(&p, 7, 0);
      container->Draw(&p, 8, 0);
    }

    p.end();

    bitBlt(this, m_TopRect.left(), m_TopRect.top(),
           &pix, 0, 0, -1, -1, Qt::CopyROP);
}


void MythStream::updateMidView()
{
   // update if not display videomode
    if ( playerState.getDisplayStatus() == PlayerState::video ) return;

    LayerSet* container = m_Theme->GetSet("dyn_panel");
    QRect rect = container->GetAreaRect();

    QPixmap pix( rect.size() );
    pix.fill(this, rect.topLeft() );
    QPainter p(&pix);

    
    if (container)
    {
      container->Draw(&p, 0, 0);
      container->Draw(&p, 1, 0);
    }

    p.end();

    bitBlt(this, rect.left(), rect.top(),
           &pix, 0, 0, -1, -1, Qt::CopyROP);
}


void MythStream::updateBotView(bool clearOnly)
{
    LayerSet* container = 0;
    
    switch ( playerState.getDisplayStatus() )
    {
      case PlayerState::browse:
        container = m_Theme->GetSet("browse_panel");
        break;
      case PlayerState::audio:
        container = m_Theme->GetSet("audio_panel");
        break;
   /*
      case PlayerState::information:
        container = m_Theme->GetSet("information_panel");
        break;
   */
      case PlayerState::video:
        clearVideo = true;  // indicate video screen is drawn
        container = m_Theme->GetSet("video_panel");
        break;
      default:
        container = m_Theme->GetSet("browse_panel");
        break;
    }

   if (clearOnly) // todo
     container = m_Theme->GetSet("video_panel");

    QRect rect = container->GetAreaRect();

    QPixmap pix( rect.size() );
    pix.fill(this, rect.topLeft());
    QPainter p(&pix);

    if (container)
    {
      container->Draw(&p, 0, 0);

      if (!clearOnly) // a todo...
      {
        container->Draw(&p, 1, 0);
        container->Draw(&p, 2, 0);
        container->Draw(&p, 3, 0);
        container->Draw(&p, 4, 0);
        container->Draw(&p, 5, 0);
        container->Draw(&p, 6, 0);
        container->Draw(&p, 7, 0);
        container->Draw(&p, 8, 0);
      }
    }

    p.end();

    bitBlt(this, rect.left(), rect.top(),
           &pix, 0, 0, -1, -1, Qt::CopyROP);
}

void MythStream::slotWebStorageMaybeReady()
{
   QString error;
   
   if ( !streamStorage->loadList(100, error) )
        cerr << error << endl;

}
void MythStream::selectStorageByIndex(ReposStorage *storageBin, StreamStorage *streamStorage, int index)
{
    QString error;
    ValueList values;
    QString home = getenv("HOME");
    int i = 0;
    bool found = false;
    
    storageBin->resetRecordList();
    
    while ( i != index && i < 9 && (found = storageBin->getNextRecord(values)) )
    {
      cout << values[3] << endl;
      i++; 
    }

    switch (index)
    {
      case 0: 
          streamStorage->selectDefaultDb(0);
          if ( !streamStorage->loadList(100, error) )
            cerr << error << endl;
      break;
    
      case 9:
          streamStorage->selectFileStorage(0, " from v"VERSION" tarball", PREFIX"/share/"SUBPATH"/streams.res");
          if ( !streamStorage->loadList(100, error) )
            cerr << error << endl;
      break;
      
      default:
          if (found)
          {
            reportEvent("");
            storageBin->openStorage(streamStorage, ident_storagegroup, values, error);
            QTimer::singleShot( 600, this, SLOT(slotWebStorageMaybeReady()) );
          }
            else reportEvent("Stream storage index " + QString::number(index) + " not configured");
      break;
    }
}

void selectStorages(ReposStorage *storageBin, StreamStorage *streamStorage, int ident)
{
  // selects, not loads, proper storages
    
    QString error;
    ValueList values;
    bool loaded = storageBin->getDefaultRepository(values);

    if (loaded)
    {
      if (!streamStorage->selectStorage(ident, values))
      {
        cerr << "MythStream: cannot open default stream repository" << endl;
        return;
      }
    }
      else
    {
        if (!streamStorage->selectDefaultDb(0))
        {
          cout << "MythStream: adding table streams to mythtv db" << endl;
           // init
          QString queryString( "CREATE TABLE IF NOT EXISTS streams("
                               "folder varchar(100) NOT NULL, "
                               "name varchar(100) NOT NULL, "
                               "url varchar(255) NOT NULL, "
                               "description varchar(255), "
                               "handler varchar(50) default '');" );
          MSqlQuery query(MSqlQuery::InitCon());
          if (!query.exec(queryString))
          {
             cerr << "MythStream: cannot create table streams in mythtv db" << endl;
             return;
          }

          QString home = getenv("HOME");

          cout << "MythStream: opening default stream repository ./.mythtv/.../stream.res" << endl;
          if (streamStorage->selectFileStorage(ident, "default", home + "/."SUBPATH"/streams.res"))
          {
             if (!streamStorage->loadList(0, error))
               cerr << error << endl;
             streamStorage->selectDefaultDb(ident);
             cout << "MythStream: saving stream.res contents to db" << endl;
             if (!streamStorage->storeList(0, error))
               cerr << error << endl;
          }
            else
               cerr << "MythStream: cannot load stream.res" << endl;
        }
    }
}



