/* ============================================================
 * File        : storage.cpp
 * Author      : Eric Giesselbach <ericgies@kabelfoon.nl>
 * Date        : 2004-01-21
 * Description : repository access db, file or web
 *
 * Copyright 2003 by Eric Giesselbach

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published bythe Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include <iostream>

#include <mythtv/mythcontext.h>
#include <mythtv/mythdbcon.h>

#include <qnetwork.h>
#include <qdatetime.h>
#include <qdir.h>
#include <qstringlist.h>
#include <qdict.h>

#include <qapplication.h>
#include <qurl.h>
#include <qptrlist.h>

#include "storagehandlers.h"

using namespace std;

/*
  labels in file and web storage:
  [item]          record header
  [emptystring]   represents empty string
  [rmvd]          removed item header
*/

Record::Record(int off, int len)
{
    offset = off;
    length = len;
}

void ChangedRecord::resetState()
{
   error = false;

   for (unsigned int i=0; i<values.count(); i++)
   {
     values[i]    = "";
     oldValues[i] = "";
   }
}

// todo: implement abstract virtual functions instead of setting keys
int RecordList::compareItems( QPtrCollection::Item i1, QPtrCollection::Item i2 )
{
   int comp;
   Record *r1, *r2;

   if (!i1 || !i2)
   {
      cerr << TARGET" error: empty record" << endl;
      return 0;
   }

   r1 = static_cast<Record*>(i1);
   r2 = static_cast<Record*>(i2);

   if ( urlCompare )
      comp = r1->values[key_pri].compare( r2->values[key_pri] );
   else
   {
      comp = r1->values[key_mul1].compare( r2->values[key_mul1] );
      if ( comp == 0 )
        comp = r1->values[key_mul2].compare( r2->values[key_mul2] );
   }

   return comp;
}

bool RecordList::validateItem( Record *record )
{
  if (!record)
  {
    return false;
    cerr << TARGET" storage error: null record" << endl;
  }

  if (record->values.count() > 2)  //duh
     return true;
  else
  {
     cerr << TARGET" storage warning: missing properties (has " << record->values.count()
          << " out of 3/4 properties). Record rejected." << endl;
     return false;
  }
}

//----------------------- GenStorage base class ---------------------------

GenStorage::GenStorage(QString name, int accessType, int key_pri, int key_mul1, int key_mul2)
{
    recordList.setAutoDelete( true );
    recordList.key_pri  = key_pri;
    recordList.key_mul1 = key_mul1;
    recordList.key_mul2 = key_mul2;

    listreset    = true;

    changedRecord = new ChangedRecord();
    changedRecord->resetState();

    resetState();
    this->accessType = accessType;
    accessName = name;
}

GenStorage::~GenStorage()
{
  recordList.clear();
  delete( changedRecord );
}

bool GenStorage::getNextRecord(RecordList& list, ValueList& values)
{
    Record *record;

    if ( listreset )
    {
      record = list.first();
      listreset = false;
    }
      else
        record = list.next();

    if (record)
    {
      if ( values.count() < record->values.count() )
          values.resize( record->values.count(), "" );
      values = record->values;
      return true;
    }
      else
        return false;
}

int GenStorage::findItemKeyIndex(ValueList& values)
{
   recordList.urlCompare = false;
   Record *record = new Record(0, 0);
   //record->values.resize(values.count());
   record->values = values;

   int index = recordList.find( record );
   delete(record);

   return index;
}

// returns -1 if not found
int GenStorage::findItemResourceIndex(ValueList& values)
{
   recordList.urlCompare = true;

   Record *record = new Record(0, 0);
   //record->values.resize(values.count());
   record->values = values;

   int index = recordList.find( record );
   delete(record);

   return index;
}

ValueList GenStorage::getItemValues(int index)
{
   if ( index < 0 || index >= (int)recordList.count() )
     cerr << TARGET" storage says: aaaaaaarrcchhhh...." << flush << endl;
   Record *record = recordList.at(index);
   return record->values;
}

void GenStorage::resetState()
{
    accessResource      = "";
    accessInSync        = false;
    accessNeedsCompact  = false;
    accessReadOnly      = false;
    accessRequestType   = idle;
    accessLastError     = "";
    accessSourceIdent   = 0;
}

// called from derived classes

bool GenStorage::loadList(int, QString& error)
{
    if ( accessRequestType != idle )
    {
      error = "storage " + accessName + " is busy or unavailable";
      return false;
    }

    return true;
}

bool GenStorage::storeList(int, RecordList&, QString& error)
{
    if ( accessRequestType != idle )
    {
      error = "storage is busy";
      return false;
    }

    if ( accessReadOnly )
    {
      error = "storage is readonly";
      return false;
    }

    return true;
}


bool GenStorage::insertRecord(int ident, ValueList& values, QString& error)
{    
    if ( accessRequestType != idle )
    {
      error = "storage is busy";
      return false;
    }
      else
        accessRequestType = insert;

    if ( accessReadOnly )
    {
      error = "storage is readonly";
      accessRequestType = idle;
      return false;
    }

    if ( !accessInSync )
    {
        error = "storage is not synchronized";
        accessRequestType = idle;
        return false;
    }

      int index = findItemResourceIndex(values);

      if ( index > -1 )  // item found
      {
        values = getItemValues(index);
        error = "resource exists";
        accessRequestType = idle;
        return false;
      }

      index = findItemKeyIndex(values);

      if ( index > -1 )  // item found
      {
        values = getItemValues(index);
        error = "item exists";
        accessRequestType = idle;
        return false;
      }

      changedRecord->ident     = ident;
      changedRecord->oldValues = values;
      changedRecord->values    = values;

      return true;
}

bool GenStorage::updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error)
{
    if ( accessRequestType != idle )
    {
      error = "storage is busy";
      return false;
    }
      else
        accessRequestType = update;

    if ( accessReadOnly )
    {
      error = "storage is readonly";
      accessRequestType = idle;
      return false;
    }

   if ( !accessInSync )
   {
      error = "storage not synchronized";
      accessRequestType = idle;
      return false;
   }

   int resIndex = findItemResourceIndex(values);
   int keyIndex = findItemKeyIndex(oldValues);

   if ( keyIndex == -1 )
   {
      error = "cannot find item";
      accessRequestType = idle;
      return false;
   }

   // resource exists and is not owned by the updated item itself
    if ( resIndex > -1 && resIndex != keyIndex )
    {
      values = getItemValues(resIndex);
      error = "resource exists";
      accessRequestType = idle;
      return false;
    }

    int foundIndex = findItemKeyIndex(values);

    // KeyIndex for new values exist and this key is not owned by updated item itself
    if ( foundIndex > -1 && foundIndex != keyIndex )
    {
      values = getItemValues(foundIndex);
      error = "item exists";
      accessRequestType = idle;
      return false;
    }

    changedRecord->ident     = ident;
    changedRecord->oldValues = oldValues;
    changedRecord->values    = values;

    return true;
}

bool GenStorage::removeRecord(int ident, ValueList values, QString& error)
{
    if ( accessRequestType != idle )
    {
      error = "storage is busy";
      return false;
    }
      else
        accessRequestType = remove;

    if ( accessReadOnly )
    {
       error = "storage is readonly";
       accessRequestType = idle;
       return false;
    }

    if ( !accessInSync )
    {
       error = "storage not synchronized";
       accessRequestType = idle;
       return false;
    }

    int index = findItemKeyIndex( values );

    if (index == -1)
    {
        error = "item not found";
        accessRequestType = idle;
        return false;
    }

    changedRecord->ident     = ident;
    changedRecord->oldValues = values;
    changedRecord->values    = values;

    return true;
}

//----------------------- File Storage ---------------------------

FileStorage::FileStorage(QString name, int accessType, int key_pri, int key_mul1, int key_mul2)
           : GenStorage(name, accessType, key_pri, key_mul1, key_mul2) {}

bool FileStorage::openFileStorage(int ident, QString fileName)
{
    closeStorage();

    myFile.setName( fileName );

    if ( !myFile.open(IO_ReadWrite) && !myFile.open(IO_ReadOnly) )
    {
      accessLastError = "cannot open file for read";
      emit storageEvent(ident, selected, true);
      return false;
    }
      else
    {
      accessResource     = fileName;
      accessInSync       = false;
      accessNeedsCompact = false;
      accessReadOnly     = !myFile.isWritable();

      emit storageEvent(ident, selected, false);
      return true;
    }
}

FileStorage::~FileStorage()
{
   closeStorage();
}

//----------------------- list read functions ---------------------------

QString FileStorage::getStorageDescription()
{
   return "file: " + accessResource;
}

//----------------------- access selection ---------------------------

void FileStorage::closeStorage()
{
    if (myFile.isOpen())
    {
      if ( accessNeedsCompact )
        saveListToFile(recordList);
      myFile.close();
    }

    changedRecord->resetState();
    resetState();
}


//----------------------- public list manipulation functions ---------------------------
// check if command can be executed, set changedRecord and call dispatching functions

bool FileStorage::loadList(int ident, QString& error)
{
    if ( !GenStorage::loadList(ident, error) )
      return false;

    accessRequestType = getlist;

    if ( loadListFromFile() )
    {
      accessInSync       = true;
      accessNeedsCompact = false;
      accessRequestType = idle;
      emit storageEvent( ident, loaded, false );
    }
      else
    {
      accessLastError = "cannot read from storage";
      accessRequestType = idle;
      emit storageEvent( ident, loaded, true );
    }

    return true;
}


// - copy from resource1 to resource2
// - update file resource after removal
bool FileStorage::storeList(int ident, RecordList& list, QString& error)
{
    if ( !GenStorage::storeList(ident, list, error) )
      return false;

    accessRequestType = savelist;

    if ( saveListToFile(list) )
    {
      accessInSync = true;
      accessRequestType = idle;
      emit storageEvent( ident, saved, false );
    }
      else
    {
      accessLastError = "cannot write to storage";
      accessRequestType = idle;
      emit storageEvent( ident, saved, true );
    }

    return true;
}

// todo: item exists fail must lead to change of name
// todo: exists error means updated values
bool FileStorage::insertRecord(int ident, ValueList& values, QString& error)
{
    if ( !GenStorage::insertRecord(ident, values, error) )
      return false;

    changedRecord->error = !appendFileRecord();
    if ( changedRecord->error )
      accessLastError = "cannot write to storage";
    accessRequestType = idle;
    emit recordInserted( changedRecord );

    return true;
}

bool FileStorage::updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error)
{
    if ( !GenStorage::updateRecord(ident, oldValues, values, error) )
      return false;

    bool result = blankFileRecord();
    if (result)
      result = appendFileRecord();
    changedRecord->error = !result;

    if ( changedRecord->error )
      accessLastError = "cannot write to storage";

    accessRequestType = idle;
    emit recordUpdated( changedRecord );

    return true;
}

bool FileStorage::removeRecord(int ident, ValueList values, QString& error)
{
    if ( !GenStorage::removeRecord(ident, values, error) )
      return false;

    changedRecord->error = !blankFileRecord();

    if ( changedRecord->error )
      accessLastError = "cannot write to storage";
    accessRequestType = idle;
    emit recordRemoved( changedRecord );

    return true;
}


bool FileStorage::loadListFromFile()
{
    myFile.at(0);
    QTextStream stream( &myFile );

    recordList.clear();

    int offset;
    int stage = 0;
    int hdrlength = 8; // length header + 2
    QString line;
    Record *record = 0;
    bool hold = false;

    while ( !stream.eof() )
    {
       offset = myFile.at();
       line = stream.readLine();

       if (stage > 0) stage++;

       if ( line == "[item]" )
       {
         stage = 1;
         hold = false;
       }
       if ( line == "[rmvd]" ) hold = true;

       if (stage == 1)
       {
          if ( record )
          {
            record->length = myFile.at() - hdrlength - record->offset;
            if ( recordList.validateItem(record) )
              recordList.append( record );
            else
              delete(record);
          }
          record = new Record (offset, 0);
       }
         // ignore empty lines, interpret [emptystring] as empty string.
         else if (stage > 0 && !hold && line != "")
       {
         if ( line == "[emptystring]" )
           line = "";

         record->values.append(line);
       }
    }

    if ( record )
    {
      record->length = myFile.at() - record->offset;
      if ( recordList.validateItem(record) )
        recordList.append( record );
      else
        delete(record);
    }

    recordList.sort();

    accessInSync       = true;
    accessNeedsCompact = false;

    return true;
}

bool FileStorage::saveListToFile(RecordList& list)
{
    if ( accessReadOnly )
      return false;

    // write only: truncate
    myFile.close();
    if (!myFile.exists() || !myFile.open(IO_WriteOnly))
      return false;

    QString value;
    QTextStream stream( &myFile );

    resetRecordList();
    ValueList values;
    int cnt;

    stream << endl;

    while ( getNextRecord(list, values) )
    {
       cnt = values.size();
       stream << endl;
       stream << "[item]" << endl;
       for ( int i = 0; i < cnt; i++ )
       {
         value = values[i];
         if ( value == "" ) value = "[emptystring]";
         stream << value << endl;
       }
    }

    accessInSync       = true;
    accessNeedsCompact = false;

    openFileStorage(0, accessResource );

    // record positions changed: reload
    return loadListFromFile();
}

bool FileStorage::appendFileRecord()
{
    QString value;
    int offset = myFile.size();
    myFile.at( offset );

    QTextStream stream( &myFile );

    stream << endl;
    offset++;
    stream << "[item]" << endl;

    Record *record = new Record(offset, 0);

    int cnt = changedRecord->values.size();
    for ( int i = 0; i < cnt; i++ )
    {
      value = changedRecord->values[i];
      if ( value == "" ) value = "[emptystring]";
      stream << value << endl;
      record->values.append( changedRecord->values[i] );
    }

    myFile.flush();
    record->length = myFile.size() - offset;

    if ( recordList.validateItem(record) )
      recordList.inSort( record );
    else
      delete record;

    return true;
}


bool FileStorage::blankFileRecord()
{
   int index = findItemKeyIndex( changedRecord->oldValues );

   if (index>-1)
   {
      Record *record = recordList.current();
      //cerr << "blank " << record->values[1] << endl;
      //dumpRecord( record );

      myFile.at( record->offset );

      QTextStream stream( &myFile );
      QString buffer;
      buffer.fill( '.', record->length - 7 );
      stream << "[rmvd]" << endl;
      stream << buffer;

      myFile.flush();
      accessNeedsCompact = true;
      recordList.remove();

      return true;
   }
     else
       return false;
}


//----------------------- Database Storage ---------------------------

DatabaseStorage::DatabaseStorage(QString table, QString name,
                                 int accessType, int key_pri, int key_mul1, int key_mul2)
           : GenStorage(name, accessType, key_pri, key_mul1, key_mul2)
{
    defaultTable = table;
    defaultName  = name;
}

DatabaseStorage::~DatabaseStorage()
{
   closeStorage();
}

//----------------------- list read functions ---------------------------

QString DatabaseStorage::getStorageDescription()
{
  return "database table: " + accessResource;
}


//----------------------- access type selection ---------------------------

void DatabaseStorage::closeStorage()
{
    changedRecord->resetState();
    resetState();
}


bool DatabaseStorage::openDatabaseStorage(int ident, QString /*dbDriver*/, QString /*hostName*/,
                                          unsigned int /*port*/, QString /*dbName*/, QString table,
                                          QString /*login*/, QString /*password*/)
{
    closeStorage();

    accessResource     = table;
    accessInSync       = false;
    accessNeedsCompact = false;
    accessReadOnly     = false;

    bool result = loadDbFields();
    emit storageEvent( ident, selected, !result );
    return result;
}

bool DatabaseStorage::openDefaultDb(int ident)
{
    closeStorage();

    accessResource     = defaultTable;
    accessName         = defaultName; // overwrites name passed in constructor
    accessInSync       = false;
    accessNeedsCompact = false;
    accessReadOnly     = false;

    bool result = loadDbFields();
    emit storageEvent( ident, selected, !result );
    return result;
}

//----------------------- public list manipulation functions ---------------------------
// check if command can be executed, set changedRecord and call dispatching functions

bool DatabaseStorage::loadList(int ident, QString& error)
{
    if ( !GenStorage::loadList(ident, error) )
      return false;

    accessRequestType = getlist;

    if ( loadListFromDb() )
    {
      accessInSync       = true;
      accessRequestType = idle;
      emit storageEvent( ident, loaded, false );
    }
      else
    {
      accessLastError = "cannot read from storage";
      accessRequestType = idle;
      emit storageEvent( ident, loaded, true );
    }

    return true;
}

// - copy from resource1 to resource2
// - update file resource after removal
bool DatabaseStorage::storeList(int ident, RecordList& list, QString& error)
{
    if ( !GenStorage::storeList(ident, list, error) )
      return false;

    accessRequestType = savelist;

    if ( saveListToDb(list) )
    {
      accessInSync = true;
      accessRequestType = idle;
      emit storageEvent( ident, saved, false );
    }
      else
    {
      accessLastError = "cannot write to storage";
      accessRequestType = idle;
      emit storageEvent( ident, saved, true );
    }

    return true;
}

// todo: item exists fail must lead to change of name
// todo: exists error means updated values
bool DatabaseStorage::insertRecord(int ident, ValueList& values, QString& error)
{
    if ( !GenStorage::insertRecord(ident, values, error) )
      return false;
cout << "db storage insert: " << values[1] << endl;
    changedRecord->error = !insertDbRecord();
    if ( changedRecord->error )
      accessLastError = "cannot write to storage";
    accessRequestType = idle;
    emit recordInserted( changedRecord );

    return true;
}


bool DatabaseStorage::updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error)
{
    if ( !GenStorage::updateRecord(ident, oldValues, values, error) )
      return false;

    changedRecord->error = !updateDbRecord();
    if ( changedRecord->error )
      accessLastError = "cannot write to storage";
    accessRequestType = idle;
    emit recordUpdated( changedRecord );

    return true;
}

bool DatabaseStorage::removeRecord(int ident, ValueList values, QString& error)
{
    if ( !GenStorage::removeRecord(ident, values, error) )
      return false;

    changedRecord->error = !removeDbRecord();
    if ( changedRecord->error )
      accessLastError = "cannot write to storage";
    accessRequestType = idle;
    emit recordRemoved( changedRecord );

    return true;
}


//----------------------- private database handling functions ---------------------------

bool DatabaseStorage::loadDbFields()
{
    MSqlQuery query(MSqlQuery::InitCon());
    query.prepare("desc " + accessResource);
    if (!query.exec())
    {
       accessLastError = QString(TARGET": Couldn't load definition for table " + accessResource + " from database");
       return false;
    }

    accessDbFields.clear();

    while ( query.next() )
        accessDbFields.append( query.value(0).toString() );

    return true;
}

bool DatabaseStorage::loadListFromDb()
{
    MSqlQuery query(MSqlQuery::InitCon());

    // check fieldnames as additional security measure
    QString queryString = "select * from " + accessResource + ";";

    if ( !query.exec(queryString) )
    {
        cerr << TARGET" storage: Couldn't load table " + accessResource + " from database" << endl;
        return false;
    }

    recordList.clear();

    int cnt = accessDbFields.size();

    while ( query.next() )
    {
        Record *record = new Record (0, 0);

        for ( int i = 0; i < cnt; i++ )
          record->values.append( query.value(i).toString() );

        if ( recordList.validateItem(record) )
          recordList.append( record );
        else
          delete(record);
    }

    recordList.sort();
    return true;
}


bool DatabaseStorage::saveListToDb(RecordList& list)
{
    bool result = true;

    QString queryString;
    MSqlQuery query(MSqlQuery::InitCon());

    // check fieldnames as additional security measure
    queryString = "delete from " + accessResource + ";";

    if ( !query.exec(queryString) )
      return false;

    int cnt = accessDbFields.size();
    resetRecordList();

    ValueList values;
    while ( getNextRecord(list, values) )
    {
       queryString = "insert into " + accessResource + "(";

       for ( int i = 0; i < cnt; i++ )
       {
         if (i > 0) queryString += ",";
         queryString += accessDbFields[i];
       }

       queryString += ") values(";

       for ( int i = 0; i < cnt; i++ )
       {
         if (i > 0) queryString += ",";
         queryString += "'" + values[i] + "'";
       }

       queryString += ");";

       if ( !query.exec(queryString) )
         result = false;
       
    }

    return result;
}

// Proper escaping please...
QString escapeValue(QString val)
{
   val = val.replace("'","''");
   return val;
}

bool DatabaseStorage::insertDbRecord()
{
    QString queryString;
    int cnt = accessDbFields.size();

    queryString = "insert into " + accessResource + "(";

    for ( int i = 0; i < cnt; i++ )
    {
      if (i > 0) queryString += ",";
      queryString += accessDbFields[i];
    }

    Record *record = new Record (0, 0);
    queryString += ") values(";

    for ( int i = 0; i < cnt; i++ )
    {
      if (i > 0) queryString += ", ";
      queryString += "'" + escapeValue( changedRecord->values[i] ) + "'";
      record->values.append( changedRecord->values[i] );
    }

    queryString += ");";
    MSqlQuery query(MSqlQuery::InitCon());

    if ( recordList.validateItem(record) )
      recordList.inSort( record );
    else
      delete record;

    return query.exec(queryString);
}

bool DatabaseStorage::updateDbRecord()
{
    QString queryString, temp1, temp2;
    temp1 = ""; temp2 = "";
    int cnt = accessDbFields.size();

    int index = findItemKeyIndex(changedRecord->oldValues);
    Record *rec = 0;
    if ( index > -1 )
       rec = recordList.current();

    queryString = "update " + accessResource + " set ";

    for ( int i = 0; i < cnt; i++ )
    {
      if (i>0) { temp1 += ", "; temp2 += " and "; }
      temp1 += accessDbFields[i] + "='" + escapeValue( changedRecord->values[i] ) + "'";
      temp2 += accessDbFields[i] + "='" + escapeValue( changedRecord->oldValues[i] ) + "'";
      if (rec) rec->values[i] = changedRecord->values[i];
    }

    queryString += temp1 + " where " + temp2;
    MSqlQuery query(MSqlQuery::InitCon());
    
    return query.exec(queryString);
}


bool DatabaseStorage::removeDbRecord()
{
    QString queryString;
    int cnt = accessDbFields.size();
    queryString = "delete from " + accessResource + " where ";

    for ( int i = 0; i < cnt; i++ )
    {
      if (i>0) queryString += " and ";
      queryString += accessDbFields[i] + "='" + escapeValue( changedRecord->values[i] ) + "'";
    }

    MSqlQuery query(MSqlQuery::InitCon());

    int index = findItemKeyIndex(changedRecord->values);
    if ( index > -1 )
      recordList.remove();

    return query.exec(queryString);
}


//----------------------- Web Storage ---------------------------

WebStorage::WebStorage(QString name, int accessType, int key_pri, int key_mul1, int key_mul2)
           : GenStorage(name, accessType, key_pri, key_mul1, key_mul2)
{
  http = 0;
}

WebStorage::~WebStorage()
{
   closeStorage();
}

//----------------------- list read functions ---------------------------

QString WebStorage::getStorageDescription()
{
  return "url: " + accessResource;
}

//----------------------- access type selection ---------------------------

void WebStorage::closeStorage()
{
    if (http)
      delete( http );

    changedRecord->resetState();
    resetState();
}


void WebStorage::openWebStorage(int ident, QString url, QString login, QString password)
{
    closeStorage();

    QUrl *qurl = new QUrl( url );
    host = qurl->host();
    delete(qurl);

    accessResource    = url;
    accessRequestType = sethost;
    accessSourceIdent = ident;
    accessLogin       = login;
    accessPassword    = password;

    QUrl::encode(password);
    QUrl::encode(login);

    authQuery = "login=" + login + "&pass=" + password + "&";

    http = new QHttp();
    connect ( http, SIGNAL(requestFinished(int, bool)), this, SLOT(slotRequestFinished(int, bool)) );
    http->setHost(host);
}

//----------------------- public list manipulation functions ---------------------------
// check if command can be executed, set changedRecord and call dispatching functions

bool WebStorage::loadList(int ident, QString& error)
{
    if ( !GenStorage::loadList(ident, error) )
      return false;

    accessRequestType = getlist;
    accessSourceIdent = ident;

    postToWeb("command=list" , false);
    // wait for callback (slotRequestFinished)
    return true;
}


// - copy from resource1 to resource2
// - update file resource after removal
bool WebStorage::storeList(int ident, RecordList& list, QString& error)
{
    if ( !GenStorage::storeList(ident, list, error) )
      return false;

    accessRequestType = savelist;

    accessSourceIdent = ident;
    saveListToWeb(list);
    // wait for callback (slotRequestFinished)

    return true;
}

// todo: item exists fail must lead to change of name
// todo: exists error means updated values
bool WebStorage::insertRecord(int ident, ValueList& values, QString& error)
{
    if ( !GenStorage::insertRecord(ident, values, error) )
      return false;

    changedRecord->ident     = ident;
    changedRecord->oldValues = values;
    changedRecord->values    = values;

    insertWebRecord();
    // wait for callback  (slotRequestFinished)

    return true;
}


bool WebStorage::updateRecord(int ident, ValueList& oldValues, ValueList& values, QString& error)
{
    if ( !GenStorage::updateRecord(ident, oldValues, values, error) )
      return false;

    updateWebRecord();

    return true;
}

bool WebStorage::removeRecord(int ident, ValueList values, QString& error)
{
    if ( !GenStorage::removeRecord(ident, values, error) )
      return false;

    removeWebRecord();

    return true;
}


//----------------------- storage command dispatching ---------------------------
// dispatch command, handle access errors and emit ready signal for non-web commands


//----------------------- web access callback ---------------------------
// handle http ready:
// - manage recordList
// - send access ready signals

void WebStorage::slotRequestFinished(int, bool error)
{
   QString data;
   QByteArray ar;
   int index;
   Record *record;

   if ( http->error() != QHttp::NoError )
   {
     error = true; // isn't it already?
     accessLastError = http->errorString();
   }

   // todo: handle 404 etc.

    switch ( accessRequestType )
    {
      case sethost:
          if (error)
          {
            emit storageEvent( accessSourceIdent, selected, error );
            return;
          }
          accessRequestType = handshake;
          postToWeb( "command=hello" , false );
          break;

      case handshake:
            accessLastError = "handshake with web storage failed";
            if ( !error && getWebResponse() )
            {
              accessInSync       = false;
              accessNeedsCompact = false;
              accessRequestType  = idle;
              accessLastError    = "";
              emit storageEvent( accessSourceIdent, selected, error );
            }
              else
            {
              error = true;
              emit storageEvent( accessSourceIdent, selected, error );
              return;
            }
          break;

      case getlist:
          if ( !error && http->bytesAvailable() )
          {
            data = QString( http->readAll() );
            parseWebList( data );
          }
          accessInSync       = true;
          accessRequestType  = idle;
          if ( error )
            accessLastError = "error reading items from web storage";
          emit storageEvent( accessSourceIdent, loaded, error );
          break;

      case savelist:
          changedRecord->error = false;
          processWebResponse(error);
          accessInSync      = true;
          accessRequestType = idle;

          if ( error )
            accessLastError = "error saving items to web storage";

          if ( changedRecord->error )
            error = true;

          emit storageEvent( accessSourceIdent, saved, error );
          break;

      case insert:
          processWebResponse(error);
          accessRequestType = idle;
          record = new Record(0, 0);
          record->values = changedRecord->values;
          if ( recordList.validateItem(record) )
            recordList.inSort( record );
          else
            delete record;

          emit recordInserted( changedRecord );
          break;

      case update:
          processWebResponse(error);
          accessRequestType = idle;
          index = findItemKeyIndex( changedRecord->oldValues );
          if ( index > -1 )
          {
            Record *rec = recordList.current();
            rec->values = changedRecord->values;
          }
          emit recordUpdated( changedRecord );
          break;

      case remove:
          processWebResponse(error);
          accessRequestType = idle;
          if ( findItemKeyIndex( changedRecord->oldValues ) > -1 )
            recordList.remove();
          emit recordRemoved( changedRecord );
          break;

      default:
          break;
    }

    changedRecord->resetState();
}

void WebStorage::processWebResponse(bool error)
{
    QString data;

    if ( !error )
    {
      if ( http->bytesAvailable() )
      {
        data = QString( http->readAll() );
        if ( data != "OK" )
        {
            accessLastError = data;
            changedRecord->error = true;
        }
      }
        else
      {
          accessLastError = "no confirmation received";
          changedRecord->error = true;
      }
    }
      else changedRecord->error = true;
}

bool WebStorage::getWebResponse()
{
    QStringList lines;
    QString data = "";
    int stage = 0;
    bool storageError = true;

    if ( http->bytesAvailable() )
      data = QString( http->readAll() );

    lines  = QStringList::split( "\n", data, true );

    for ( QStringList::Iterator i = lines.begin();
          i != lines.end();
          ++i
        )
    {
       if (stage > 0) stage++;

       if (*i && *i == "[storage]" ) stage = 1;

       switch (stage)
       {
          case 2:
            storageError = false;
            if (*i == "writable")
              accessReadOnly = false;
            else if (*i == "readonly")
              accessReadOnly = true;
            else
              storageError = true;
          break;
       }
    }

    if (stage < 2 && data != "") // error message from web script / login failed
      accessLastError = data;

    return !storageError;
}

// todo: check values length in here and in callback procs, can be anything
void WebStorage::parseWebList(QString& data)
{
    int         stage;
    Record      *record;
    QStringList lines;

    stage  = 0;
    record = 0;
    lines  = QStringList::split( "\n", data, true );

    recordList.clear();

    for ( QStringList::Iterator i = lines.begin();
          i != lines.end();
          ++i
        )
    {
       if (stage > 0) stage++;

       if (*i && *i == "[item]" ) stage = 1;

       if (stage == 1)
       {
          if ( record )
          {
            if ( recordList.validateItem(record) )
              recordList.append( record );
            else
              delete(record);
          }
          record = new Record (0, 0);
       }
          else if (stage > 1)
       {
           if (*i)
           {
					if (*i != "[emptystring]")
						record->values.append(*i);
					else
						record->values.append("");
           }
       }
    }

    if ( record && recordList.validateItem(record) )
      recordList.append( record );
    else
      delete(record);

    recordList.sort();
}

//----------------------- private web handling functions ---------------------------

void WebStorage::postToWeb(QString data, bool usePost)
{
    QString getResource = accessResource.utf8() + QCString("?") + authQuery.utf8() + data.utf8();
    
    QHttpRequestHeader header;
    if (usePost) 
      header.setRequest("POST", accessResource);
    else
      header.setRequest("GET", getResource );
    header.setValue("Host", host);
    header.setContentType("application/x-www-form-urlencoded");
    http->request(header, authQuery.utf8() + data.utf8());
}

bool WebStorage::removeWebRecord()
{
    QString value;

    QString postData = "command=remove";

    int cnt = changedRecord->values.size();
    for ( int i = 0; i < cnt; i++ )
    {
      value = changedRecord->values[i];
      QUrl::encode(value);
      postData += "&val" + QString::number(i) + "=" + value;
    }

    accessRequestType = remove;
    postToWeb( postData );

    return true;
}

bool WebStorage::updateWebRecord()
{
    QString value, oldValue;
    QString postData = "command=update";
    int cnt = changedRecord->values.size();
    for ( int i = 0; i < cnt; i++ )
    {
      value    = changedRecord->values[i];
      oldValue = changedRecord->oldValues[i];
      QUrl::encode(value);
      QUrl::encode(oldValue);
      postData += "&val" + QString::number(i) + "=" + value +
                  "&oldval" + QString::number(i) + "=" + oldValue ;
    }

    accessRequestType = update;
    postToWeb( postData );

    return true;
}

bool WebStorage::insertWebRecord()
{
    QString value;
    QString postData = "command=insert";
    int cnt = changedRecord->values.size();

    for ( int i = 0; i < cnt; i++ )
    {
      value    = changedRecord->values[i];
      QUrl::encode(value);
      postData += "&val" + QString::number(i) + "=" + value;
    }

    accessRequestType = insert;
    postToWeb( postData );

    return true;
}

bool WebStorage::saveListToWeb(RecordList& list)
{
    int index = 0;
    QString postData = "";
    QString value, item;

    postData = "command=savelist&count=" + QString::number( list.count() );
    resetRecordList();

    ValueList values;
    while ( getNextRecord(list, values) )
    {
        int cnt = values.size();

        for ( int i = 0; i < cnt; i++ )
        {
          value = values[i];
          QUrl::encode(value);
          item = "val" + QString::number(i) + "[" + QString::number( index ) + "]";
          QUrl::encode(item);
          postData += "&" + item + "=" + value;
        }

        index++;
    }

    accessRequestType = savelist;
    postToWeb( postData );

    return true;
}



