/* Nestra x11.c */
/* Public Domain */

#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include "mapper.h"
#include "io.h"
#include "globals.h"

/* X11 stuff: */
Display *display;
Visual *visual;
Window rootwindow,w;
int screen;
Colormap colormap;
XEvent ev;
unsigned char *keystate[32];
GC gc,blackgc;
GC addgc,maskgc;
GC backgroundgc,solidbggc;
GC color1gc,color2gc,color3gc,bgcolorgc;
GC spritesolidcolorgc,spritecolor1gc,spritecolor2gc,spritecolor3gc;
GC planegc,blackplanegc,planenorgc,planeandgc,planexorgc;
int black,white;
XSizeHints sizehints;
XGCValues GCValues;
XSetWindowAttributes windowattrib;
unsigned long colortable[25];
int indexedcolor=0;
XColor color;

#define tilecachedepth 25

Pixmap background[tilecachedepth];  /* The pre-rendered background that sprites are drawn on top of */
Pixmap bgplane1[tilecachedepth],bgplane2[tilecachedepth],bgplane3[tilecachedepth],bgmask[tilecachedepth];  /* Bitmaps for each color plane of the rendered background */
Pixmap tilecache1[tilecachedepth],tilecache2[tilecachedepth],tilecache3[tilecachedepth],tilebgmask[tilecachedepth];  /* Bitmaps for the background tiles in each of the four colors */
Pixmap layout; /* To assemble the final image to be displayed */
XImage *tilebitimage,*tilebitimage2;
unsigned char tilebitdata[16384],tilebitdata2[16384];
XImage *image;

/* Flags for differential redraw */
unsigned char needsredraw=1; /* Refresh screen display */
unsigned char redrawbackground=1; /* Redraw tile background */
unsigned char redrawall=1; /* Redraw all scanlines */
unsigned char bgmask_changed[tilecachedepth];
unsigned char current_bgmask=0;
unsigned char tilebgmask_changed[tilecachedepth];
unsigned char tilemask1_changed[tilecachedepth];
unsigned char tilemask2_changed[tilecachedepth];
unsigned char tilemask3_changed[tilecachedepth];
unsigned char scanline_diff[240];
unsigned short int tileline_begin[60];
unsigned short int tileline_end[60];

/* Precalculated/Cached values */
unsigned char spritecache[256];
unsigned char ntcache[tilecachedepth][3840];
unsigned char vramcache[tilecachedepth][4096];
unsigned char tiledirty[tilecachedepth][256];
unsigned char tilechanged[256];
unsigned int displaytilecache[tilecachedepth][3840];
unsigned int displaycolorcache[tilecachedepth][3840];
unsigned char displaycachevalid[tilecachedepth][3840];
unsigned char bitplanecachevalid[tilecachedepth][3840];
unsigned char pallette_cache[tilecachedepth][32];
unsigned int currentbgcolor,oldbgcolor;
unsigned int bgcolor[tilecachedepth];

/* Misc globals */
int hoffset,voffset;
int oldhoffset,oldvoffset;
unsigned short int oldhscroll[240],oldvscroll[240];
unsigned int currentline;
int debug_bgoff=0;
int debug_spritesoff=0;
int debug_switchtable=0;
int basetime;
int desync=1;
int halfspeed=0,doublespeed=0,pause_display=0;
int magstep=1;
int currentcache=0;
int nextcache=1%tilecachedepth;

#define screen_on (RAM[0x2001]&8)
#define sprites_on (RAM[0x2001]&16)
#define maxsize 1 /* Can't resize window yet */


/* Here is the color palette.  Change it if you don't like it. */
unsigned int palette_24[64]={
  0x3f3f3f,0x001f3f,0x00003f,0x1f003f,0x3f003f,0x3f0020,0x3f0000,
    0x3f2000,0x3f3f00,0x203f00,0x003f00,0x003f20,0x003f3f,0,0,0,
  0x7f7f7f,0x405f7f,0x40407f,0x5f407f,0x7f407f,0x7f4060,0x7f4040,
    0x7f6040,0x7f7f40,0x607f40,0x407f40,0x407f60,0x407f7f,0,0,0,
  0xbfbfbf,0x809fbf,0x8080bf,0x9f80bf,0xbf80bf,0xbf80a0,0xbf8080,
    0xbfa080,0xbfbf80,0xa0bf80,0x80bf80,0x80bfa0,0x80bfbf,0,0,0,
  0xffffff,0xc0dfff,0xc0c0ff,0xdfc0ff,0xffc0ff,0xffc0e0,0xffc0c0,
    0xffe0c0,0xffffc0,0xe0ffc0,0xc0ffc0,0xc0ffe0,0xc0ffff,0,0,0,
  };
unsigned short palette_16[64];
unsigned char palette_8[64];


InitDisplay()
{
  int x,y;
  struct timeval time;
  display=XOpenDisplay(NULL);
  if(display==NULL) return;
  screen=XDefaultScreen(display);
  visual=XDefaultVisual(display,screen);
  rootwindow=XRootWindow(display,screen);
  colormap=DefaultColormap(display,screen);
  white=WhitePixel(display,screen);
  black=BlackPixel(display,screen);
  depth=DefaultDepth(display,screen);
  if(depth==8&&!staticcolors)
  {
    indexedcolor=1; /* Note: Should check to make sure display really is indexed color */
    if(XAllocColorCells(display,colormap,0,0,0,colortable,25)==0)
    {
      printf("Can't allocate colors!\n");
      exit(1);
    }
    /* Pre-initialize the colormap to known values */
    color.red=((palette_24[0]&0xFF0000)>>8);
    color.green=(palette_24[0]&0xFF00);
    color.blue=((palette_24[0]&0xFF)<<8);
    color.flags=DoRed|DoGreen|DoBlue;
    for(x=0;x<=24;x++)
    {
      ((unsigned char *)palette)[x]=color.pixel=colortable[x];
      XStoreColor(display,colormap,&color);
    }
  }
  if(depth==8&&staticcolors)
  {
    /* Allocate 64 color entries */
    for(x=0;x<64;x++)
    {
      color.red=((palette_24[x]&0xFF0000)>>8);
      color.green=(palette_24[x]&0xFF00);
      color.blue=((palette_24[x]&0xFF)<<8);
      color.flags=DoRed|DoGreen|DoBlue;
      if(XAllocColor(display,colormap,&color)==0)
      {
        printf("Can't allocate colors!\n");
        exit(1);
      }
      palette_8[x]=color.pixel;
    }
  }

  if(depth==16)
  {
    /* Create 16-bit pallette */
    for(x=0;x<64;x++)
    {
      palette_16[x]=((palette_24[x]>>3)&0x1f)|((palette_24[x]>>5)&0x7c0)
                     |((palette_24[x]>>8)&0xf800);
    }
  }
  w=XCreateSimpleWindow(display,rootwindow,0,0,256,240,0,black,black);
  gc=XCreateGC(display,w,0,0);
  blackgc=XCreateGC(display,w,0,0);
  GCValues.function=GXor;
  addgc=XCreateGC(display,w,GCFunction,&GCValues);
  GCValues.function=GXand;
  maskgc=XCreateGC(display,w,GCFunction,&GCValues);
  XSetForeground(display,gc,~0);
  XSetBackground(display,gc,0);
  XSetForeground(display,blackgc,0);
  XSetBackground(display,blackgc,0);
  GCValues.function=GXor;
  color1gc=XCreateGC(display,w,GCFunction,&GCValues);
  color2gc=XCreateGC(display,w,GCFunction,&GCValues);
  color3gc=XCreateGC(display,w,GCFunction,&GCValues);
  bgcolorgc=XCreateGC(display,w,GCFunction,&GCValues);
  XSetBackground(display,bgcolorgc,0);
  GCValues.function=GXcopy;
  spritesolidcolorgc=XCreateGC(display,w,GCFunction,&GCValues);
  spritecolor1gc=XCreateGC(display,w,GCFunction,&GCValues);
  spritecolor2gc=XCreateGC(display,w,GCFunction,&GCValues);
  spritecolor3gc=XCreateGC(display,w,GCFunction,&GCValues);

  /* set window title */
  XStoreName(display,w,"Nestra");
  /* set aspect ratio */
  sizehints.flags=PMinSize|PMaxSize|PResizeInc|PAspect|PBaseSize;
  sizehints.min_width=256;
  sizehints.min_height=240;
  sizehints.max_width=256*maxsize;
  sizehints.max_height=240*maxsize;
  sizehints.width_inc=256;
  sizehints.height_inc=240;
  sizehints.min_aspect.x=256;
  sizehints.min_aspect.y=240;
  sizehints.max_aspect.x=256;
  sizehints.max_aspect.y=240;
  sizehints.base_width=0;
  sizehints.base_height=0;
  XSetWMNormalHints(display,w,&sizehints);

  for(x=0;x<tilecachedepth;x++){
    background[x]=XCreatePixmap(display,w,512,480,depth); /* should make bigger for expanding window (fix this) */
    bgplane1[x]=XCreatePixmap(display,w,512,480,1);
    bgplane2[x]=XCreatePixmap(display,w,512,480,1);
    bgplane3[x]=XCreatePixmap(display,w,512,480,1);
    bgmask[x]=XCreatePixmap(display,w,512,480,1);
    tilecache1[x]=XCreatePixmap(display,w,256,64,1);
    tilecache2[x]=XCreatePixmap(display,w,256,64,1);
    tilecache3[x]=XCreatePixmap(display,w,256,64,1);
    tilebgmask[x]=XCreatePixmap(display,w,256,64,1);
  }
  planegc=XCreateGC(display,tilebgmask[0],0,0);
  XSetForeground(display,planegc,1);
  XSetBackground(display,planegc,0);
  blackplanegc=XCreateGC(display,tilebgmask[0],0,0);
  XSetForeground(display,blackplanegc,0);
  XSetBackground(display,blackplanegc,0);
  GCValues.function=GXnor;
  planenorgc=XCreateGC(display,tilebgmask[0],GCFunction,&GCValues);
  GCValues.function=GXand;
  planeandgc=XCreateGC(display,tilebgmask[0],GCFunction,&GCValues);
  GCValues.function=GXxor;
  planexorgc=XCreateGC(display,tilebgmask[0],GCFunction,&GCValues);
  backgroundgc=XCreateGC(display,w,0,0);
  XSetBackground(display,backgroundgc,black);
  solidbggc=XCreateGC(display,w,0,0);
  XSetBackground(display,solidbggc,black);
  layout=XCreatePixmap(display,w,256*maxsize,240*maxsize,depth);
  tilebitimage=XCreateImage(display,visual,1,XYBitmap,0,tilebitdata,256,64,8,32);
  tilebitimage2=XCreateImage(display,visual,1,XYBitmap,0,tilebitdata2,256,64,8,32);
  XMapWindow(display,w);
  XSelectInput(display,w,KeyPressMask|KeyReleaseMask|ExposureMask|FocusChangeMask|KeymapStateMask);
  XFlush(display);
  for(y=0;y<tilecachedepth;y++) for(x=0;x<256;x++) tiledirty[y][x]=1;
  gettimeofday(&time,NULL);
  basetime=time.tv_sec;
  fbinit();
  if(depth==24)
    image=XCreateImage(display,visual,depth,ZPixmap,0,fb,256,240,32,0);
  else
    image=XCreateImage(display,visual,depth,ZPixmap,0,fb,256,240,depth,0);
}

void
HandleKeyboard(void)
{
  KeySym keysym = XKeycodeToKeysym (display, ((XKeyEvent *) & ev)->keycode, 0);

  if (ev.type == KeyPress && keysym == XK_Escape)
    quit ();                    /* ESC */

  switch (keysym)
    {
      /* controller 1 keyboard mapping */
    case XK_Return:
      if (ev.type == KeyPress)
        controller1 |= STARTBUTTON;
      if (ev.type == KeyRelease)
        controller1 &= ~STARTBUTTON;
      break;
    case XK_Tab:
      if (ev.type == KeyPress)
        controller1 |= SELECTBUTTON;
      if (ev.type == KeyRelease)
        controller1 &= ~SELECTBUTTON;
      break;
    case XK_Up:
      if (ev.type == KeyPress)
        controller1 |= UP;
      if (ev.type == KeyRelease)
        controller1 &= ~UP;
      break;
    case XK_Down:
      if (ev.type == KeyPress)
        controller1 |= DOWN;
      if (ev.type == KeyRelease)
        controller1 &= ~DOWN;
      break;
    case XK_Left:
      if (ev.type == KeyPress)
        controller1 |= LEFT;
      if (ev.type == KeyRelease)
        controller1 &= ~LEFT;
      break;
    case XK_Right:
      if (ev.type == KeyPress)
        controller1 |= RIGHT;
      if (ev.type == KeyRelease)
        controller1 &= ~RIGHT;
      break;
    case XK_Z:
    case XK_z:
    case XK_X:
    case XK_x:
    case XK_D:
    case XK_d:
    case XK_Shift_L:
      if (ev.type == KeyPress)
        controller1 |= BUTTONB;
      if (ev.type == KeyRelease)
        controller1 &= ~BUTTONB;
      break;
    case XK_A:
    case XK_a:
    case XK_C:
    case XK_c:
    case XK_space:
      if (ev.type == KeyPress)
        controller1 |= BUTTONA;
      if (ev.type == KeyRelease)
        controller1 &= ~BUTTONA;
      break;
    }

  /* emulator keys */
  if (ev.type == KeyPress)
    switch (keysym)
      {
      case XK_F1:
        debug_bgoff = 1;
        break;
      case XK_F2:
        debug_bgoff = 0;
        break;
      case XK_F3:
        debug_spritesoff = 1;
        break;
      case XK_F4:
        debug_spritesoff = 0;
        break;
      case XK_F5:
        debug_switchtable = 1;
        break;
      case XK_F6:
        debug_switchtable = 0;
        break;
      case XK_F9:
        memset (displaycachevalid[currentcache], 0, 3840);
        break;
      case XK_Pause:
        START ();
	break;
      case XK_grave:
        desync = 1;
        halfspeed = 1;
        doublespeed = 0;
        pause_display = 0;
        break;
      case XK_1:
        desync = 1;
        halfspeed = 0;
        doublespeed = 0;
        pause_display = 0;
        break;
      case XK_2:
        desync = 1;
        halfspeed = 0;
        doublespeed = 2;
        pause_display = 0;
        break;
      case XK_3:
        desync = 1;
        halfspeed = 0;
        doublespeed = 3;
        pause_display = 0;
        break;
      case XK_4:
        desync = 1;
        halfspeed = 0;
        doublespeed = 4;
        pause_display = 0;
        break;
      case XK_5:
        desync = 1;
        halfspeed = 0;
        doublespeed = 5;
        pause_display = 0;
        break;
      case XK_6:
        desync = 1;
        halfspeed = 0;
        doublespeed = 6;
        pause_display = 0;
        break;
      case XK_7:
        desync = 1;
        halfspeed = 0;
        doublespeed = 7;
        pause_display = 0;
        break;
      case XK_8:
        desync = 1;
        halfspeed = 0;
        doublespeed = 8;
        pause_display = 0;
        break;
      case XK_0:
        desync = 1;
        pause_display = 1;
        break;
      }
}

UpdateDisplay()
{
  Window root;int junk;
  static unsigned int width,height;
  static unsigned int oldwidth,oldheight;
  struct timeval time;
  unsigned static int frame;
  unsigned int timeframe;
  static int sleep=0; /* Initially we start with the emulation running.  If you want to wait until the window receives input focus, change this. */
  int x;
  
  /* Check the time.  If we're getting behind, skip a frame to stay in sync. */
  gettimeofday(&time,NULL);
  timeframe=(time.tv_sec-basetime)*50+time.tv_usec/20000; /* PAL */
  timeframe=(time.tv_sec-basetime)*60+time.tv_usec/16666; /* NTSC */
  frame++;
  if(halfspeed) timeframe>>=1;
  if(doublespeed==2) timeframe<<=1;
  else if(doublespeed>2) timeframe*=doublespeed;
  if(desync) frame=timeframe;desync=0;
  if(frame<timeframe-20&&frame%20==0) desync=1; /* If we're more than 20 frames behind, might as well stop counting. */

  drawimage(81840);
  if(!frameskip)
  {
    update_colors();
    XPutImage(display,layout,gc,image,0,0,0,0,256,240);
    redrawall=needsredraw=1;
  }

  /* Slow down if we're getting ahead */
  if(frame>timeframe+1&&frameskip==0) {
    usleep(16666*(frame-timeframe-1));
  }
  
  /* Handle X input */
  while(XPending(display)||ev.type==-1||sleep||pause_display) {
    XNextEvent(display,&ev);
    /*printf("event %d\n",ev.type);*/
    if(ev.type==DestroyNotify) quit();
    if(ev.type==FocusIn) sleep=0;
    if(ev.type==FocusOut) sleep=desync=1;
    if(ev.type==KeyPress || ev.type==KeyRelease) HandleKeyboard();
    if(ev.type==KeymapNotify) {memcpy(keystate,((XKeymapEvent *)&ev)->key_vector,32);controller1=controller2=0;}
    if(ev.type==Expose) needsredraw=1;
    if((sleep||pause_display)&&needsredraw) {XCopyArea(display,layout,w,gc,0,0,256*magstep,240*magstep,0,0);needsredraw=0;}
  }

  if (needsredraw) {
    XCopyArea(display,layout,w,gc,0,0,256*magstep,240*magstep,0,0);
    XFlush(display);
    /* 
       The purpose of this is to wait for the acknowledgement from the
       X server (a NoExpose event on the window background in response
       to the CopyArea) before proceeding, so we don't send commands
       faster than the X server can process them.  The -1 acts as a flag
       to indicate that the XEvent loop should block.
    */
    ev.type=-1;
  }
  
  /* See if the window size changed */
  XGetGeometry(display,w,&root,&junk,&junk,&width,&height,&junk,&junk);
  if(width!=oldwidth||height!=oldheight)
  {
    if((magstep=((width*240 > height*256) ? width/256 : height/240))<1) magstep=1;
    else if(magstep>maxsize) magstep=maxsize;
    oldwidth=width;oldheight=height;
    memset(displaycachevalid[currentcache],0,3840);
    memset(bitplanecachevalid[currentcache],0,3840);
    for(x=0;x<256;x++) tilechanged[x]=tiledirty[currentcache][x]=1;
    desync=1;
    XFillRectangle(display,background[currentcache],blackgc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgplane1[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgplane2[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgplane3[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgmask[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
  }

  XFlush(display);
  needsredraw=0;
  redrawbackground=0;
  redrawall=0;
  
  /* Check the time.  If we're getting behind, skip next frame to stay in sync. */
  gettimeofday(&time,NULL);
  timeframe=(time.tv_sec-basetime)*60+time.tv_usec/16666; /* NTSC */
  if(halfspeed) timeframe>>=1;
  if(doublespeed==2) timeframe<<=1;
  else if(doublespeed>2) timeframe*=doublespeed;
  if(frame>=timeframe||frame%20==0) frameskip=0;
  else frameskip=1;  
  
}

/*
  The screen draw works as follows:
  
  The full 4-screen background is laid out in Pixmap background.  The
  current window (h/v scrolling area) is copied into Pixmap layout.
  Sprites are then drawn on top of this, and finally the completed pixmap
  is copied onto the window.
  
  The final image is prepared in a pixmap before copying onto the screen
  because flicker would result if the sprites were drawn/erased directly
  onto the exposed window.  Therefore, backing store need not be
  maintained, as obscured areas can be recopied as necessary in response
  to expose events.
*/

UpdateDisplay_orig()
{
  Window root;int junk;
  static unsigned int width,height;
  static unsigned int oldwidth,oldheight;
  struct timeval time;
  unsigned static int frame;
  unsigned int timeframe;
  static int sleep=0; /* Initially we start with the emulation running.  If you want to wait until the window receives input focus, change this. */
  int x;
  
  /* Check the time.  If we're getting behind, skip a frame to stay in sync. */
  gettimeofday(&time,NULL);
  timeframe=(time.tv_sec-basetime)*50+time.tv_usec/20000; /* PAL */
  timeframe=(time.tv_sec-basetime)*60+time.tv_usec/16666; /* NTSC */
  frame++;
  if(halfspeed) timeframe>>=1;
  if(doublespeed==2) timeframe<<=1;
  else if(doublespeed>2) timeframe*=doublespeed;
  if(desync) frame=timeframe;desync=0;
  if(frame<timeframe-20&&frame%20==0) desync=1; /* If we're more than 20 frames behind, might as well stop counting. */

  if(frame>=timeframe||frame%20==0) {
    /* If mode settings are different, force a redraw. */
    diff_update();

    /* If the pallette changed, update the colors. */
    update_colors();

    /* Layout the background with h/v-scrolling */
    layoutbackground();

    /* Draw the sprites on top */
    if(screen_on&&sprites_on) drawsprites();
  }

  /* Slow down if we're getting ahead */
  if(frame>timeframe+1) {
    usleep(16666*(frame-timeframe-1));
  }
  
  /* Handle X input */
  while(XPending(display)||ev.type==-1||sleep||pause_display) {
    XNextEvent(display,&ev);
    /*printf("event %d\n",ev.type);*/
    if(ev.type==DestroyNotify) quit();
    if(ev.type==FocusIn) sleep=0;
    if(ev.type==FocusOut) sleep=desync=1;
    if(ev.type==KeyPress || ev.type==KeyRelease) HandleKeyboard();
    if(ev.type==KeymapNotify) {memcpy(keystate,((XKeymapEvent *)&ev)->key_vector,32);controller1=controller2=0;}
    if(ev.type==Expose) needsredraw=1;
    if((sleep||pause_display)&&needsredraw) {XCopyArea(display,layout,w,gc,0,0,256*magstep,240*magstep,0,0);needsredraw=0;}
  }

  if (needsredraw) {
    XCopyArea(display,layout,w,gc,0,0,256*magstep,240*magstep,0,0);
    XFlush(display);
    /* 
       The purpose of this is to wait for the acknowledgement from the
       X server (a NoExpose event on the window background in response
       to the CopyArea) before proceeding, so we don't send commands
       faster than the X server can process them.  The -1 acts as a flag
       to indicate that the XEvent loop should block.
    */
    ev.type=-1;
  }
  
  /* See if the window size changed */
  XGetGeometry(display,w,&root,&junk,&junk,&width,&height,&junk,&junk);
  if(width!=oldwidth||height!=oldheight)
  {
    if((magstep=((width*240 > height*256) ? width/256 : height/240))<1) magstep=1;
    else if(magstep>maxsize) magstep=maxsize;
    oldwidth=width;oldheight=height;
    memset(displaycachevalid[currentcache],0,3840);
    memset(bitplanecachevalid[currentcache],0,3840);
    for(x=0;x<256;x++) tilechanged[x]=tiledirty[currentcache][x]=1;
    desync=1;
    XFillRectangle(display,background[currentcache],blackgc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgplane1[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgplane2[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgplane3[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
    XFillRectangle(display,bgmask[currentcache],blackplanegc,0,0,256*magstep,240*magstep);
  }

  XFlush(display);
  needsredraw=0;
  redrawbackground=0;
  redrawall=0;
}

/* Update the background as necessary */
dobackground(){
  unsigned int x,y,z;
  unsigned int h,v;
  signed int c;
  unsigned char *vramgroupptr,*cachegroupptr;
  unsigned char *colorptr;
  unsigned int tilecolor;
  unsigned int color1,color2,color3;
  static count=0;
  Pixmap plane1,plane2,plane3,bgplane;
  int hfirst,hlast,vfirst,vlast;
  
  /*count++; if (count<100) return;  /* This can be used to skip frames for testing purposes */
  count=0;

  /* Some games use the mapper to switch the graphic tiles on each frame,
     so all are cached so that they can be switched quickly.  */
  plane1=bgplane1[currentcache];plane2=bgplane2[currentcache];plane3=bgplane3[currentcache];bgplane=bgmask[currentcache];
  
  /* Go thru all rows and render any exposed tiles */
  for(y=0;y<60;y+=2)
  {
    c=((tileline_end[y]-tileline_begin[y])>>4);
    if(c<0) c=0;
    else if(c<16) {printf("Error: Short scanline!\n");exit(1);} /* Sanity check */
    else if(c>=32) c=32;
    else if(tileline_end[y]&15) c++;
    for(x=h=(tileline_begin[y]>>3)&62;c--;x=(x+2)&62)
    {
      if (h>63) exit(1); /* shouldn't happen */
      v=y;
      
      /* Colors are assigned to a group of four tiles, so we use
         "groupptr" to point to this group. */
      if(nomirror)
      {
        vramgroupptr=VRAM+0x2000+((x&32)<<5)+((v&~1)%30)*32+(x&30)+(0x800*(v>=30));
        colorptr=VRAM+0x23C0+((x&32)<<5)+((v%30)&28)*2+((x&28)>>2)+(0x800*(v>=30));
      }else
      if(hvmirror==0){
        vramgroupptr=VRAM+0x2000+((x&32)<<5)+((v&~1)%30)*32+(x&30);
        colorptr=VRAM+0x23C0+((x&32)<<5)+((v%30)&28)*2+((x&28)>>2);
      }else{
        vramgroupptr=VRAM+((0x2000+((v&~1)%30)*32+(x&30))^(0x400*(v>=30)));
        colorptr=VRAM+( (0x23C0+((v%30)&28)*2+((x&28)>>2))^(0x400*(v>=30)) );
      }
      cachegroupptr=ntcache[currentcache]+(v&~1)*64+(x&~1);
      tilecolor=((*colorptr)>>((((v%30)&2)<<1)|(x&2)))&3;

      if(tilecolor!=displaycolorcache[currentcache][(v&~1)*32+(x>>1)])
      {
        displaycachevalid[currentcache][(v&~1)*64+(x&~1)]=displaycachevalid[currentcache][(v&~1)*64+(x|1)]=
        displaycachevalid[currentcache][(v|1)*64+(x&~1)]=displaycachevalid[currentcache][(v|1)*64+(x|1)]=0;
      }

      /* When a new tile is exposed, first check to see if it's already
         on the screen.  If it is, then just recopy it.  If not, then we
         must go thru the (relatively slow) process of building it from
         the bitplanes and colormaps.  Since a group of 4 tiles share the
         same colormap, we do all four at once to keep things simpler. */

      if((!displaycachevalid[currentcache][v*64+x])||
        cachegroupptr[0]!=vramgroupptr[0]||
        cachegroupptr[1]!=vramgroupptr[1]||
        cachegroupptr[64]!=vramgroupptr[32]||
        cachegroupptr[65]!=vramgroupptr[33])
      {
        /* This only checks tiles in the same row.  The efficiency could be
           improved by copying larger areas at once when possible. */
        for(z=0;z<64;z+=2)
          if(displaytilecache[currentcache][(v&~1)*32+(z>>1)]==((vramgroupptr[0]<<24)|(vramgroupptr[1]<<16)|(vramgroupptr[32]<<8)|(vramgroupptr[33]))
          &&displaycolorcache[currentcache][(v&~1)*32+(z>>1)]==tilecolor
          &&displaycachevalid[currentcache][(v&~1)*64+z]
          &&displaycachevalid[currentcache][(v&~1)*64+z+1]
          &&displaycachevalid[currentcache][(v|1)*64+z]
          &&displaycachevalid[currentcache][(v|1)*64+z+1]
          &&z!=(x&~1))
            break;

        if(z<64)
        {
          count++; /* For profiling/debugging */
          XCopyArea(display,background[currentcache],background[currentcache],gc,z*8*magstep,(v&~1)*8*magstep,16*magstep,16*magstep,(x&~1)*8*magstep,(v&~1)*8*magstep);
          /* If only the color attributes changed, then the bitmaps do not need to be redrawn */
          if(cachegroupptr[0]!=vramgroupptr[0]||
             cachegroupptr[1]!=vramgroupptr[1]||
             cachegroupptr[64]!=vramgroupptr[32]||
             cachegroupptr[65]!=vramgroupptr[33]||
             (!bitplanecachevalid[currentcache][(v&~1)*64+(x&~1)])||
             (!bitplanecachevalid[currentcache][(v|1)*64+(x|1)])||
             (!bitplanecachevalid[currentcache][(v&~1)*64+(x&~1)])||
             (!bitplanecachevalid[currentcache][(v|1)*64+(x|1)]))
          {
            XCopyArea(display,plane1,plane1,planegc,z*8*magstep,(v&~1)*8*magstep,16*magstep,16*magstep,(x&~1)*8*magstep,(v&~1)*8*magstep);
            XCopyArea(display,plane2,plane2,planegc,z*8*magstep,(v&~1)*8*magstep,16*magstep,16*magstep,(x&~1)*8*magstep,(v&~1)*8*magstep);
            XCopyArea(display,plane3,plane3,planegc,z*8*magstep,(v&~1)*8*magstep,16*magstep,16*magstep,(x&~1)*8*magstep,(v&~1)*8*magstep);
            XCopyArea(display,bgplane,bgplane,planegc,z*8*magstep,(v&~1)*8*magstep,16*magstep,16*magstep,(x&~1)*8*magstep,(v&~1)*8*magstep);
            bgmask_changed[currentcache]=1;
            bitplanecachevalid[currentcache][(v&~1)*64+(x&~1)]=
            bitplanecachevalid[currentcache][(v&~1)*64+(x|1)]=
            bitplanecachevalid[currentcache][(v|1)*64+(x&~1)]=
            bitplanecachevalid[currentcache][(v|1)*64+(x|1)]=1;
          }
          displaycachevalid[currentcache][(v&~1)*64+(x&~1)]=
          displaycachevalid[currentcache][(v&~1)*64+(x|1)]=
          displaycachevalid[currentcache][(v|1)*64+(x&~1)]=
          displaycachevalid[currentcache][(v|1)*64+(x|1)]=1;
        }
        else
        {
          /* Assemble a group of 4 tiles from the respective bitplanes */
          count++; /* For profiling/debugging */
          XCopyArea(display,tilecache1[currentcache],plane1 ,planegc,(vramgroupptr[0]&0xF8),((vramgroupptr[0]&7)<<3),8,8,x*8*magstep,v*8*magstep);
          XCopyArea(display,tilecache2[currentcache],plane2 ,planegc,(vramgroupptr[0]&0xF8),((vramgroupptr[0]&7)<<3),8,8,x*8*magstep,v*8*magstep);
          XCopyArea(display,tilecache3[currentcache],plane3 ,planegc,(vramgroupptr[0]&0xF8),((vramgroupptr[0]&7)<<3),8,8,x*8*magstep,v*8*magstep);
          XCopyArea(display,tilebgmask[currentcache],bgplane,planegc,(vramgroupptr[0]&0xF8),((vramgroupptr[0]&7)<<3),8,8,x*8*magstep,v*8*magstep);
          XCopyArea(display,tilecache1[currentcache],plane1 ,planegc,(vramgroupptr[1]&0xF8),((vramgroupptr[1]&7)<<3),8,8,(x+1)*8*magstep,v*8*magstep);
          XCopyArea(display,tilecache2[currentcache],plane2 ,planegc,(vramgroupptr[1]&0xF8),((vramgroupptr[1]&7)<<3),8,8,(x+1)*8*magstep,v*8*magstep);
          XCopyArea(display,tilecache3[currentcache],plane3 ,planegc,(vramgroupptr[1]&0xF8),((vramgroupptr[1]&7)<<3),8,8,(x+1)*8*magstep,v*8*magstep);
          XCopyArea(display,tilebgmask[currentcache],bgplane,planegc,(vramgroupptr[1]&0xF8),((vramgroupptr[1]&7)<<3),8,8,(x+1)*8*magstep,v*8*magstep);
          XCopyArea(display,tilecache1[currentcache],plane1 ,planegc,(vramgroupptr[32]&0xF8),((vramgroupptr[32]&7)<<3),8,8,x*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilecache2[currentcache],plane2 ,planegc,(vramgroupptr[32]&0xF8),((vramgroupptr[32]&7)<<3),8,8,x*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilecache3[currentcache],plane3 ,planegc,(vramgroupptr[32]&0xF8),((vramgroupptr[32]&7)<<3),8,8,x*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilebgmask[currentcache],bgplane,planegc,(vramgroupptr[32]&0xF8),((vramgroupptr[32]&7)<<3),8,8,x*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilecache1[currentcache],plane1 ,planegc,(vramgroupptr[33]&0xF8),((vramgroupptr[33]&7)<<3),8,8,(x+1)*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilecache2[currentcache],plane2 ,planegc,(vramgroupptr[33]&0xF8),((vramgroupptr[33]&7)<<3),8,8,(x+1)*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilecache3[currentcache],plane3 ,planegc,(vramgroupptr[33]&0xF8),((vramgroupptr[33]&7)<<3),8,8,(x+1)*8*magstep,(v+1)*8*magstep);
          XCopyArea(display,tilebgmask[currentcache],bgplane,planegc,(vramgroupptr[33]&0xF8),((vramgroupptr[33]&7)<<3),8,8,(x+1)*8*magstep,(v+1)*8*magstep);
          bgmask_changed[currentcache]=1;
          if(indexedcolor)
          {
            color1=colortable[tilecolor*3];
            color2=colortable[tilecolor*3+1];
            color3=colortable[tilecolor*3+2];            
          }
          else if(depth==16)
          {
            color1=palette_16[VRAM[0x3f01+(tilecolor<<2)]&63];
            color2=palette_16[VRAM[0x3f02+(tilecolor<<2)]&63];
            color3=palette_16[VRAM[0x3f03+(tilecolor<<2)]&63];
          }
          else if(depth==24)
          {
            color1=palette_24[VRAM[0x3f01+(tilecolor<<2)]&63];
            color2=palette_24[VRAM[0x3f02+(tilecolor<<2)]&63];
            color3=palette_24[VRAM[0x3f03+(tilecolor<<2)]&63];
          }
          XFillRectangle(display,background[currentcache],blackgc,x*8*magstep,v*8*magstep,16*magstep,16*magstep);

          if(color1!=0){
            XSetBackground(display,color1gc,0);
            XSetForeground(display,color1gc,color1);
            XCopyPlane(display,plane1,background[currentcache],color1gc,x*8*magstep,v*8*magstep,16*magstep,16*magstep,x*8*magstep,v*8*magstep,1);
          }
          if(color2!=0){
            XSetBackground(display,color2gc,0);
            XSetForeground(display,color2gc,color2);
            XCopyPlane(display,plane2,background[currentcache],color2gc,x*8*magstep,v*8*magstep,16*magstep,16*magstep,x*8*magstep,v*8*magstep,1);
          }
          if(color3!=0){
            XSetBackground(display,color3gc,0);
            XSetForeground(display,color3gc,color3);
            XCopyPlane(display,plane3,background[currentcache],color3gc,x*8*magstep,v*8*magstep,16*magstep,16*magstep,x*8*magstep,v*8*magstep,1);
          }
          XCopyPlane(display,bgplane,background[currentcache],bgcolorgc,x*8*magstep,v*8*magstep,16*magstep,16*magstep,x*8*magstep,v*8*magstep,1);
          bitplanecachevalid[currentcache][(v&~1)*64+(x&~1)]=
          bitplanecachevalid[currentcache][(v&~1)*64+(x|1)]=
          bitplanecachevalid[currentcache][(v|1)*64+(x&~1)]=
          bitplanecachevalid[currentcache][(v|1)*64+(x|1)]=1;
          displaycachevalid[currentcache][(v&~1)*64+(x&~1)]=
          displaycachevalid[currentcache][(v&~1)*64+(x|1)]=
          displaycachevalid[currentcache][(v|1)*64+(x&~1)]=
          displaycachevalid[currentcache][(v|1)*64+(x|1)]=1;
        }
        cachegroupptr[0]=vramgroupptr[0];
        cachegroupptr[1]=vramgroupptr[1];
        cachegroupptr[64]=vramgroupptr[32];
        cachegroupptr[65]=vramgroupptr[33];
        redrawbackground=1;
        redrawall=1;
        displaytilecache[currentcache][(v&~1)*32+(x>>1)]=((cachegroupptr[0]<<24)|(cachegroupptr[1]<<16)|(cachegroupptr[64]<<8)|(cachegroupptr[65]));
        displaycolorcache[currentcache][(v&~1)*32+(x>>1)]=tilecolor;
      }
    }
  }
  /*if (count)printf("bg tiles changed: %d\n",count); /* for debugging */
}

updatetiles()
{
  int x,y,l,n;
  int b1,b2,d1,d2;
  int loc,invloc,h,v;
  int vfirstchange=64;
  int hfirstchange=256;
  int vlastchange=0;
  int hlastchange=0;
  /*unsigned char transparent,opaque;*/
  unsigned char *vramptr;
  int baseaddr=((linereg[currentline]&0x10)<<8)^(debug_switchtable<<12); /* 0 or 0x1000 */

  static count=0;
  /*count++; if (count<100) return;  /* This can be used to skip frames for testing purposes */
  count=0;

  /* Some games change the bitmaps on every frame, eg to display rotating
     blocks.  In this case it's best to cache all the different bitmaps,
     hence the use of multiple caches.  The size of the cache is defined
     by tilecachedepth. */

  for(x=0;x<256;x++)
  {
    if((((long long *)(VRAM+baseaddr))[x*2]!=((long long *)(vramcache[currentcache]))[x*2]) ||
     (((long long *)(VRAM+8+baseaddr))[x*2]!=((long long *)(vramcache[currentcache]+8))[x*2]) )
      break;
  }
  if(x<256)
  {
    for(y=(currentcache+1)%tilecachedepth;y!=(currentcache+tilecachedepth-1)%tilecachedepth;y=(y+1)%tilecachedepth)
    {
      for(x=0;x<256&&(
         (((long long *)(VRAM+baseaddr))[x*2]==((long long *)(vramcache[y]))[x*2]) &&
         (((long long *)(VRAM+8+baseaddr))[x*2]==((long long *)(vramcache[y]+8))[x*2])
         );x++) {}
      if(x==256) {currentcache=y;break;}
    }
    if(x<256)
    {
      /* When the cache is full, a cache entry must be overwritten to
         make room for a new one.  Copy the current image to the new
         cache entry. */
      if(nextcache==currentcache) nextcache=(nextcache+1)%tilecachedepth;
      for(x=0;x<256;x++) tiledirty[nextcache][x]=tiledirty[currentcache][x]; /* FIX - only dirty a rectangle */
      bgcolor[nextcache]=bgcolor[currentcache];
      bgmask_changed[nextcache]=
      tilebgmask_changed[nextcache]=
      tilemask1_changed[nextcache]=
      tilemask2_changed[nextcache]=
      tilemask3_changed[nextcache]=1;
      memcpy(ntcache[nextcache],ntcache[currentcache],3840);
      memcpy(vramcache[nextcache],vramcache[currentcache],4096);
      memcpy(displaycachevalid[nextcache],displaycachevalid[currentcache],3840);
      memcpy(bitplanecachevalid[nextcache],bitplanecachevalid[currentcache],3840);
      memcpy(displaytilecache[nextcache],displaytilecache[currentcache],3840*4); /* should use sizeof(...) */
      memcpy(displaycolorcache[nextcache],displaycolorcache[currentcache],3840*4); /* ditto */
      /*memcpy(tiletransparent[nextcache],tiletransparent[currentcache],512);*/
      /*memcpy(tileopaque[nextcache],tileopaque[currentcache],512);*/
      XCopyArea(display,background[currentcache],background[nextcache],gc,0,0,512,480,0,0);
      XCopyArea(display,bgplane1[currentcache],bgplane1[nextcache],planegc,0,0,512,480,0,0);
      XCopyArea(display,bgplane2[currentcache],bgplane2[nextcache],planegc,0,0,512,480,0,0);
      XCopyArea(display,bgplane3[currentcache],bgplane3[nextcache],planegc,0,0,512,480,0,0);
      XCopyArea(display,bgmask[currentcache],bgmask[nextcache],planegc,0,0,512,480,0,0);
      XCopyArea(display,tilecache1[currentcache],tilecache1[nextcache],planegc,0,0,256,64,0,0);
      XCopyArea(display,tilecache2[currentcache],tilecache2[nextcache],planegc,0,0,256,64,0,0);
      XCopyArea(display,tilecache3[currentcache],tilecache3[nextcache],planegc,0,0,256,64,0,0);
      XCopyArea(display,tilebgmask[currentcache],tilebgmask[nextcache],planegc,0,0,256,64,0,0);
      /*printf("invalidate cache %d (current=%d)\n",nextcache,currentcache); /* for debugging */
      currentcache=nextcache;
      nextcache=(nextcache+1)%tilecachedepth;
    }
    /*printf("currentcache=%d %d\n",y,x); /* for debugging */
  }

  /* This mess just compares each 16 bytes of VRAM with what's in our cache.
     If a tile has changed, set the 'dirty' flag so we will redraw the
     graphics for that tile. */

  for(x=0;x<256;x++)
    if((((long long *)(VRAM+baseaddr))[x*2]!=((long long *)(vramcache[currentcache]))[x*2]) ||
       (((long long *)(VRAM+8+baseaddr))[x*2]!=((long long *)(vramcache[currentcache]+8))[x*2]) )
      {
        tiledirty[currentcache][x]=1;
        ((long long *)(vramcache[currentcache]))[x*2]=((long long *)(VRAM+baseaddr))[x*2];
        ((long long *)(vramcache[currentcache]+8))[x*2]=((long long *)(VRAM+8+baseaddr))[x*2];
      }
  
  for(x=0;x<256;x++)
  {
    if(tiledirty[currentcache][x])
    {
      v=((x&7)<<3);h=(x&0xf8);
      if(h<hfirstchange) hfirstchange=h;
      if(v<vfirstchange) vfirstchange=v;
      if(h>hlastchange) hlastchange=h;
      if(v>vlastchange) vlastchange=v;
      if(v>=64) exit(1); /* sanity check - this should not happen! */
      if(h>=256) exit(1); /* sanity check - this should not happen! */
      if(v<0) exit(1); /* sanity check - this should not happen! */
      if(h<0) exit(1); /* sanity check - this should not happen! */
    }
  }
  for(h=hfirstchange;h<=hlastchange;h+=8)
  {
    for(v=vfirstchange;v<=vlastchange;v+=8)
    {
      x=h|((v>>3)&7);
      loc=((x&0xf8)>>3)+((x&7)<<8);
      /*transparent=0;opaque=255;*/
      for(l=0;l<8;l++)
      {
        b1=VRAM[baseaddr+((x&256)<<4)+((x&255)*16)+l];
        b2=VRAM[baseaddr+((x&256)<<4)+((x&255)*16)+l+8];
        /*transparent|=(b1|b2);opaque&=(b1|b2);*/
        /* Nintendo uses opposite bit-ordering from X11,
           so we need to reverse them.  A lookup table might
           improve this, if anyone feels so inclined... */
        d1=d2=0;
        for(n=0;n<8;n++) {
          d1=(d1>>1)|((b1<<n)&0x80);
          d2=(d2>>1)|((b2<<n)&0x80);
        }
        tilebitdata[loc+(l<<5)]=d1;
        tilebitdata2[loc+(l<<5)]=d2;
      }
      /*tiletransparent[currentcache][x]=(transparent==0);*/
      /*tileopaque[currentcache][x]=(opaque==255);*/
      tiledirty[currentcache][x]=0;tilechanged[x]=redrawbackground=1;
    }
  }

  if(hfirstchange<=hlastchange)
  {
    tilebgmask_changed[currentcache]=
    tilemask1_changed[currentcache]=
    tilemask2_changed[currentcache]=
    tilemask3_changed[currentcache]=1;
    /*printf("updatetiles: h=%d-%d (%d) v=%d-%d (%d),\n",hfirstchange,hlastchange,hlastchange-hfirstchange+8,vfirstchange,vlastchange,vlastchange-vfirstchange+8); /* debug */
    XPutImage(display,tilecache1[currentcache],planegc,tilebitimage,hfirstchange,vfirstchange,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8);
    XPutImage(display,tilecache2[currentcache],planegc,tilebitimage2,hfirstchange,vfirstchange,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8);

    /* bitwise-nor to create the background mask */
    XCopyArea(display,tilecache1[currentcache],tilebgmask[currentcache],planegc,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8,hfirstchange,vfirstchange);
    XCopyArea(display,tilecache2[currentcache],tilebgmask[currentcache],planenorgc,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8,hfirstchange,vfirstchange);
    /* Seperate the colors */
    XCopyArea(display,tilecache1[currentcache],tilecache3[currentcache],planegc,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8,hfirstchange,vfirstchange);
    XCopyArea(display,tilecache2[currentcache],tilecache3[currentcache],planeandgc,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8,hfirstchange,vfirstchange);
    XCopyArea(display,tilecache3[currentcache],tilecache1[currentcache],planexorgc,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8,hfirstchange,vfirstchange);
    XCopyArea(display,tilecache3[currentcache],tilecache2[currentcache],planexorgc,hfirstchange,vfirstchange,hlastchange-hfirstchange+8,vlastchange-vfirstchange+8,hfirstchange,vfirstchange);
    /* Invalidate the cache */
    for(y=0;y<60;y++)
    {
      for(x=0;x<63;x++)
      {
        if(hvmirror==0){
          vramptr=VRAM+0x2000+((x&32)<<5)+(y%30)*32+0x400*(y>=30)+(x&31);
        }else{
          vramptr=VRAM+((0x2000+(y%30)*32+(x&31))^(0x400*(y>=30)));
        }
        if(tilechanged[*vramptr])
        {
          displaycachevalid[currentcache][y*64+x]=0;
          bitplanecachevalid[currentcache][y*64+x]=0;
          tilechanged[*vramptr]=0;
        }
      }
    }
    /*desync=1;*/
  }
}

layoutbackground()
{
  int y,z;
  int x;
  unsigned int last;
  static int linecache[240];

  if((!screen_on)||debug_bgoff)
  {
    if(redrawbackground==0) return;
    XFillRectangle(display,layout,solidbggc,0,0,256*magstep,240*magstep);
    needsredraw=1;
    return;
  }
  else
  {
    currentline=0;
    updatetiles();
    if(!indexedcolor) update_tile_colors();  
    dobackground();
    if(redrawbackground==0) return;
  }

  last=linereg[0]&0x10;
  for(y=0;y<240;y++)
  {
    z=y+1;
    while(hscroll[y]==hscroll[z]&&vscroll[y]==vscroll[z]&&z<240
          &&(linereg[y]&0x10)==(linereg[z]&0x10)
          &&(redrawall||scanline_diff[y]==scanline_diff[z])) z++;

    if(y!=0&&(linereg[y]&0x10)!=last)
    {
      currentline=y;
      last=linereg[y]&0x10;
      updatetiles();
      if(!indexedcolor) update_tile_colors();
      dobackground();
    }
    
    for(x=y;x<z;x++)
      if(linecache[y]!=currentcache) redrawall=1;
    
    if(scanline_diff[y]||redrawall)
    {
      for(x=y;x<z;x++)
        linecache[y]=currentcache;
 
      if(!osmirror)
      {
        if(vscroll[y]+z<=480||vscroll[y]+y>=480)
          if(hscroll[y]<=256)
          {
            XCopyArea(display,background[currentcache],layout,gc,hscroll[y]*magstep,((vscroll[y]+y)%480)*magstep,256*magstep,(z-y)*magstep,0,y*magstep);
          }
          else
          {
            XCopyArea(display,background[currentcache],layout,gc,hscroll[y]*magstep,((vscroll[y]+y)%480)*magstep,(512-hscroll[y])*magstep,(z-y)*magstep,0,y*magstep);
            XCopyArea(display,background[currentcache],layout,gc,0,((vscroll[y]+y)%480)*magstep,(hscroll[y]-256)*magstep,(z-y)*magstep,(512-hscroll[y])*magstep,y*magstep);
          }
        else
          if(hscroll[y]<=256)
          {
            XCopyArea(display,background[currentcache],layout,gc,hscroll[y]*magstep,(vscroll[y]+y)*magstep,256*magstep,(480-vscroll[y]-y)*magstep,0,y*magstep);
            XCopyArea(display,background[currentcache],layout,gc,hscroll[y]*magstep,0,256*magstep,(vscroll[y]+z-480)*magstep,0,(480-vscroll[y])*magstep);
          }
          else
          {
            XCopyArea(display,background[currentcache],layout,gc,hscroll[y]*magstep,(vscroll[y]+y)*magstep,(512-hscroll[y])*magstep,(480-vscroll[y]-y)*magstep,0,y*magstep);
            XCopyArea(display,background[currentcache],layout,gc,hscroll[y]*magstep,0,(512-hscroll[y])*magstep,(vscroll[y]+z-480)*magstep,0,(480-vscroll[y])*magstep);
            XCopyArea(display,background[currentcache],layout,gc,0,(vscroll[y]+y)*magstep,(hscroll[y]-256)*magstep,(480-vscroll[y]-y)*magstep,(512-hscroll[y])*magstep,y*magstep);
            XCopyArea(display,background[currentcache],layout,gc,0,0,(hscroll[y]-256)*magstep,(vscroll[y]+z-480)*magstep,(512-hscroll[y])*magstep,(480-vscroll[y])*magstep);
          }
      }        
      else /*if(osmirror)*/
      {
        if((vscroll[y]+z-1)%240>=(vscroll[y]+y)%240)
          XCopyArea(display,background[currentcache],layout,gc,(hscroll[y]&255)*magstep,((vscroll[y]+y)%240)*magstep,(256-(hscroll[y]&255))*magstep,(z-y)*magstep, 0,(y%240)*magstep);
        else
        {
          XCopyArea(display,background[currentcache],layout,gc,(hscroll[y]&255)*magstep,((vscroll[y]+y)%240)*magstep,(256-(hscroll[y]&255))*magstep,((480-vscroll[y]-y)%240)*magstep,0,(y%240)*magstep);
          XCopyArea(display,background[currentcache],layout,gc,(hscroll[y]&255)*magstep,0,                           (256-(hscroll[y]&255))*magstep,((vscroll[y]+z)%240)*magstep,    0,((480-vscroll[y])%240)*magstep);
        }

        if(hscroll[y]&255)
        {
          if((vscroll[y]+z-1)%240>=(vscroll[y]+y)%240)
            XCopyArea(display,background[currentcache],layout,gc,0,((vscroll[y]+y)%240)*magstep,(hscroll[y]&255)*magstep,(z-y)*magstep,((512-hscroll[y])&255)*magstep,(y%240)*magstep);
          else
          {
            XCopyArea(display,background[currentcache],layout,gc,0,((vscroll[y]+y)%240)*magstep,(hscroll[y]&255)*magstep,((480-vscroll[y]-y)%240)*magstep,((512-hscroll[y])&255)*magstep,(y%240)*magstep);
            XCopyArea(display,background[currentcache],layout,gc,0,0,                           (hscroll[y]&255)*magstep,((vscroll[y]+z)%240)*magstep,    ((512-hscroll[y])&255)*magstep,((480-vscroll[y])%240)*magstep);
          }
        }
      }
    }
    y=z-1;
  }
  needsredraw=1;
}

/* Update the colors on the screen if the pallette changed */
update_colors()
{
  int x,y,t;
  
  /* Set Background color */
  oldbgcolor=currentbgcolor;
  if(indexedcolor)
  {
    color.pixel=currentbgcolor=colortable[24];
    color.red=((palette_24[VRAM[0x3f00]&63]&0xFF0000)>>8);
    color.green=(palette_24[VRAM[0x3f00]&63]&0xFF00);
    color.blue=((palette_24[VRAM[0x3f00]&63]&0xFF)<<8);
    color.flags=DoRed|DoGreen|DoBlue;
    XStoreColor(display,colormap,&color);
    XSetForeground(display,solidbggc,currentbgcolor);
    XSetForeground(display,bgcolorgc,currentbgcolor);
    XSetForeground(display,backgroundgc,currentbgcolor);
  }
  else /* truecolor */
  {
    if(depth==8) ((unsigned char *)palette)[24]=currentbgcolor=palette_8[VRAM[0x3f00]&63];
    else if(depth==16) ((short *)palette)[24]=currentbgcolor=palette_16[VRAM[0x3f00]&63];
    else if(depth==24) ((int *)palette)[24]=currentbgcolor=palette_24[VRAM[0x3f00]&63];
    if(oldbgcolor!=currentbgcolor/*||currentbgcolor!=bgcolor[currentcache]*/)
    {
      XSetForeground(display,solidbggc,currentbgcolor);
      XSetForeground(display,bgcolorgc,currentbgcolor);
      XSetForeground(display,backgroundgc,currentbgcolor);
      redrawbackground=1;
      needsredraw=1;
    }
  }

  /* Tile colors */
  if(indexedcolor)
  {
    for(x=0;x<24;x++)
    {
      if(VRAM[0x3f01+x+(x/3)]!=pallette_cache[0][1+x+(x/3)])
      {
        color.pixel=colortable[x];
        color.red=((palette_24[VRAM[0x3f01+x+(x/3)]&63]&0xFF0000)>>8);
        color.green=(palette_24[VRAM[0x3f01+x+(x/3)]&63]&0xFF00);
        color.blue=((palette_24[VRAM[0x3f01+x+(x/3)]&63]&0xFF)<<8);
        color.flags=DoRed|DoGreen|DoBlue;
        XStoreColor(display,colormap,&color);
        /*printf("color %d (%d) = %6x\n",x,colortable[x],palette_24[VRAM[0x3f01+x+(x/3)]&63]);*/
      }
    }
    memcpy(pallette_cache[0],VRAM+0x3f00,32);
  }

  /* Set pallette tables */
  if(indexedcolor)
  {
    /* Already done in InitDisplay */
  }
  else /* truecolor */
  {
    if(depth==8)
    {
      for(x=0;x<24;x++)
      {
        ((unsigned char *)palette)[x]=palette_8[VRAM[0x3f01+x+(x/3)]&63];
      }
    }
    if(depth==16)
    {
      for(x=0;x<24;x++)
      {
        ((short *)palette)[x]=palette_16[VRAM[0x3f01+x+(x/3)]&63];
      }
    }
    else if(depth==24)
    {
      for(x=0;x<24;x++)
      {
        ((int *)palette)[x]=palette_24[VRAM[0x3f01+x+(x/3)]&63];
      }
    }
  }
}

/* Update the tile colors for the current cache if the pallette changed */
/* (direct color mode only) */
update_tile_colors()
{
  int x,y,t;
  
  /* Set Background color */

    if(currentbgcolor!=bgcolor[currentcache])
    {
      bgcolor[currentcache]=currentbgcolor;
      redrawbackground=1;
      needsredraw=1;
      if (bgmask_changed[currentcache]||currentcache!=current_bgmask)
      {
        XSetClipMask(display,backgroundgc,bgmask[currentcache]);
        bgmask_changed[current_bgmask=currentcache]=0;
      }
      XFillRectangle(display,background[currentcache],backgroundgc,0,0,512,480);
      redrawall=1;
    }

  /* Tile colors */

  /* For programs which change the pallettes affecting only a few tiles,
     in truecolor mode, it's best to just invalidate the cache and redraw
     those few tiles, as done below.  For programs which change the colors
     across the whole screen, it's best to paint the whole thing using a
     clipmask, as for the background above.  Will have to see what works
     best.  This only affects truecolor mode; for 8-bit mode, just change
     the pallettes. */

    for(y=0;y<30;y++)
    {
      for(x=0;x<32;x++)
      {
        t=displaycolorcache[currentcache][y*64+x];
        if(pallette_cache[currentcache][t*4+1]!=VRAM[0x3f00+t*4+1]||
           pallette_cache[currentcache][t*4+2]!=VRAM[0x3f00+t*4+2]||
           pallette_cache[currentcache][t*4+3]!=VRAM[0x3f00+t*4+3])
        {
          displaycachevalid[currentcache][y*128+x*2]=0;
        }
      }
    }
    memcpy(pallette_cache[currentcache],VRAM+0x3f00,32);

  /* Sprite colors for truecolor mode are handled by drawsprites() */
}


/* This looks up a pixel on the screen and returns 1 if it is in the
   foreground and 0 if it is in the background.  This is used for the
   sprite transparency. */
getpixel(unsigned int x,unsigned int y)
{
  unsigned int h,v;
  unsigned int tile;
  unsigned int baseaddr=((linereg[y]&0x10)<<8)^(debug_switchtable<<12); /* 0 or 0x1000 */
  unsigned char d1,d2;
  unsigned int page;
  
  x+=hscroll[y];
  y+=vscroll[y];
  h=((x&255)>>3);
  v=((y%240)>>3);
  if(nomirror) page=0x2000+((x&256)<<2)+(((y%480)>=240)<<10);
  else if(hvmirror) page=0x2000+(((y%480)>=240)<<10);
  else page=0x2000+((x&256)<<2);
  tile=VRAM[page+h+(v<<5)];
  d1=VRAM[baseaddr+(tile<<4)+(y&7)];
  d2=VRAM[baseaddr+(tile<<4)+(y&7)+8];
  return ((d1|d2)>>(~x&7))&1;
}

drawsprites()
{
  int x,y;
  int s;
  int spritetile,spritecolor,pixelcolor,pixelcolor2;
  int hflip,vflip,behind;
  int d1,d2;
  int v,n,b1,b2,b3,b4;
  int baseaddr=(RAM[0x2000]&0x08)<<9;/* 0x0 or 0x1000 */
  int spritesize=8<<((RAM[0x2000]&0x20)>>5); /* 8 or 16 */
  unsigned int color1,color2,color3;
  static unsigned int currentcolor1,currentcolor2,currentcolor3,currentsolidcolor;
  static int currentmask1=-1,currentmask2=-1,currentmask3=-1,currentmaskbg=-1;
  static unsigned int random;
  static unsigned long long spritepallettecache[2];
  signed char linebuffer[256];

  if(debug_spritesoff) return;

  /* If the sprite pallette changed, we must redraw the sprites in
     truecolor mode.  This isn't necessary for 8-bit indexed color mode */
  if(!indexedcolor&&
     (*((long long *)(VRAM+0x3f10))!=spritepallettecache[0] ||
      *((long long *)(VRAM+0x3f18))!=spritepallettecache[1]) )
  {
    memcpy(spritepallettecache,VRAM+0x3f10,16);
    redrawall=needsredraw=1;
  }

  /* If any sprite entries have changed since last time, redraw */
  for(s=0;s<64;s++)
  {
    if(spriteram[s*4]<240||spritecache[s*4]<240)
    {
      if(((int *)spriteram)[s]!=((int *)spritecache)[s])
      {
        redrawall=needsredraw=1;
        ((int *)spritecache)[s]=((int *)spriteram)[s];
      }
    }
  }

  if(redrawbackground||redrawall)
  {
    for(y=1;y<240;y++)
    {
      if(redrawall||scanline_diff[y])
      {
        memset(linebuffer,0,256); /* Clear buffer for this scanline */
        baseaddr=((linereg[y]&0x08)<<9); /* 0 or 0x1000 */
        for(s=63;s>=0;s--)
        {
          if(spriteram[s*4]<y&&spriteram[s*4]<240&&spriteram[s*4+3]<249)
          {
            if(spriteram[s*4]+spritesize>=y)
            {
              spritetile=spriteram[s*4+1];
              if(spritesize==16) baseaddr=(spritetile&1)<<12;
              behind=spriteram[s*4+2]&0x20;
              hflip=spriteram[s*4+2]&0x40;
              vflip=spriteram[s*4+2]&0x80;
  
              /*linebuffer[spriteram[s*4+3]]=1;*/
              /*XDrawPoint(display,layout,spritecolor3gc,spriteram[s*4+3],y);*/
              
              /* This finds the memory location of the tiles, taking into account
                 that vertically flipped sprites are in reverse order. */
              if(vflip)
              {
                if(spriteram[s<<2]>=y-8) /* 8x8 sprites and first half of 8x16 sprites */
                {
                  d1=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+spritesize*2-8-y+spriteram[s*4]];
                  d2=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+spritesize*2-y+spriteram[s*4]];
                }
                else /* Do second half of 8x16 sprites */
                {
                  d1=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+spritesize*2-16-y+spriteram[s*4]];
                  d2=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+spritesize*2-8-y+spriteram[s*4]];
                }
              }
              else
              {
                if(spriteram[s<<2]>=y-8) /* 8x8 sprites and first half of 8x16 sprites */
                {
                  d1=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+y-1-spriteram[s*4]];
                  d2=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+y+7-spriteram[s*4]];
                }
                else /* Do second half of 8x16 sprites */
                {
                  d1=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+y+7-spriteram[s*4]];
                  d2=VRAM[baseaddr+((spritetile&(~(spritesize>>4)))<<4)+y+15-spriteram[s*4]];
                }
              }
              for(x=7*(!hflip);x<8&&x>=0;x+=1-((!hflip)<<1))
              {
                if(d1&d2&1) linebuffer[spriteram[s*4+3]+x]=3+((spriteram[s*4+2]&3)<<2);
                else if(d1&1) linebuffer[spriteram[s*4+3]+x]=1+((spriteram[s*4+2]&3)<<2);
                else if(d2&1) linebuffer[spriteram[s*4+3]+x]=2+((spriteram[s*4+2]&3)<<2);
                if(behind&&(d1|d2))
                  if(getpixel(spriteram[s*4+3]+x,y)) linebuffer[spriteram[s*4+3]+x]=0; /* Sprite hidden behind background */
                d1>>=1;d2>>=1;
              }
            }
          }
        }
        
        for(x=0;x<256;x++)
        {
          if(linebuffer[x])
          {
            if(indexedcolor)
              color1=colortable[(linebuffer[x]>>2)*3+11+(linebuffer[x]&3)];
            else
            {
              if(depth==16)
                color1=palette_16[VRAM[0x3f10+(linebuffer[x]&15)]&63];
              else
                color1=palette_24[VRAM[0x3f10+(linebuffer[x]&15)]&63];
            }
            if(color1!=currentcolor1)
              XSetForeground(display,spritecolor1gc,currentcolor1=color1);
  
            XDrawPoint(display,layout,spritecolor1gc,x,y);
          }
        }
      }
    }
  }
}

diff_update()
{
  static int old_screen_on;
  static int old_sprites_on;
  static unsigned char oldlinereg[240];
  unsigned int x,s;
  int spritesize=8<<((RAM[0x2000]&0x20)>>5); /* 8 or 16 */
  
  if(old_screen_on!=screen_on) redrawall=redrawbackground=1;
  old_screen_on=screen_on;
  
  if(screen_on)
  {
    if(old_sprites_on!=sprites_on) redrawall=needsredraw=1;
    old_sprites_on=sprites_on;
  }

  /* Update scanline redraw info */
  for(x=0;x<60;x++) {tileline_begin[x]=512;tileline_end[x]=256;}
  for(x=0;x<240;x++)
  {
    scanline_diff[x]=0;
    if (oldhscroll[x]!=hscroll[x]) redrawbackground=scanline_diff[x]=1;
    oldhscroll[x]=hscroll[x];
    if (oldvscroll[x]!=vscroll[x]) redrawbackground=scanline_diff[x]=1;
    oldvscroll[x]=vscroll[x];
    if(osmirror)
    {
      tileline_begin[((vscroll[x]+x)%240)>>3]=0;
      tileline_end[((vscroll[x]+x)%240)>>3]=256;
    }
    else
    {
      if(tileline_begin[((vscroll[x]+x)%480)>>3]>hscroll[x]) tileline_begin[((vscroll[x]+x)%480)>>3]=hscroll[x];
      if(tileline_end[((vscroll[x]+x)%480)>>3]<hscroll[x]+256) tileline_end[((vscroll[x]+x)%480)>>3]=hscroll[x]+256;
      if(tileline_begin[(((vscroll[x]+x)%480)>>3)&62]>hscroll[x]) tileline_begin[(((vscroll[x]+x)%480)>>3)&62]=hscroll[x];
      if(tileline_end[(((vscroll[x]+x)%480)>>3)&62]<hscroll[x]+256) tileline_end[(((vscroll[x]+x)%480)>>3)&62]=hscroll[x]+256;
    }
    if (oldlinereg[x]!=linereg[x]) redrawbackground=scanline_diff[x]=1;
    oldlinereg[x]=linereg[x];
  }
  
  /* See if any sprite entries have changed since last time and mark */
  /* those scanlines for redraw. */
  if(debug_spritesoff||!sprites_on) return;
  for(s=0;s<64;s++)
  {
    if(spritecache[s*4]<240)
    {
      if(((int *)spriteram)[s]!=((int *)spritecache)[s])
      {
        redrawbackground=1;
        for(x=1;x<=spritesize&&x+spritecache[s*4]<240;x++)
          scanline_diff[x+spritecache[s*4]]=1;
      }
    }
  }
}
