/* OGMRip - A library for DVD ripping and encoding
 * Copyright (C) 2004-2007 Olivier Rolland <billl@users.sf.net>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "ogmrip-mplayer.h"
#include "ogmrip-version.h"
#include "ogmrip-plugin.h"
#include "ogmrip-enums.h"
#include "ogmrip-fs.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

// static const gchar *deinterlacer[] = { "lb", "li", "ci", "md", "fd", "l5", "kerndeint", "yadif=3,mcdeint=2:1:10" };
static const gchar *deinterlacer[] = { "lb", "li", "ci", "md", "fd", "l5", "kerndeint", "yadif=0" };

static gint
ogmrip_mplayer_map_audio_id (OGMDvdAudioStream *astream)
{
  gint aid;

  aid = ogmdvd_stream_get_nr (OGMDVD_STREAM (astream));

  switch (ogmdvd_audio_stream_get_format (astream))
  {
    case OGMDVD_AUDIO_FORMAT_MPEG1:
    case OGMDVD_AUDIO_FORMAT_MPEG2EXT:
      break;
    case OGMDVD_AUDIO_FORMAT_LPCM:
      aid += 160;
      break;
    case OGMDVD_AUDIO_FORMAT_DTS:
      aid += 136;
      break;
    default:
      aid += 128;
      break;
  }

  return aid;
}

static gchar *
ogmrip_mplayer_get_output_fps (OGMRipCodec *codec, OGMDvdTitle *title)
{
  guint rate_numerator, rate_denominator;
  guint output_rate_numerator, output_rate_denominator;
  gint step;

  if (OGMRIP_IS_VIDEO (codec) && ogmrip_video_get_pullup (OGMRIP_VIDEO (codec)))
  {
    output_rate_numerator = 24000;
    output_rate_denominator = 1001;
  }
  else
    ogmrip_codec_get_framerate (codec, &output_rate_numerator, &output_rate_denominator);

  ogmdvd_title_get_framerate (title, &rate_numerator, &rate_denominator);
  step = ogmrip_codec_get_framestep (codec); 

  if (rate_numerator != output_rate_numerator || rate_denominator != output_rate_denominator * step)
    return g_strdup_printf ("%d/%d", output_rate_numerator, output_rate_denominator * step);

  return NULL;
}

static gchar *
ogmrip_mplayer_get_chapters (OGMRipCodec *codec, OGMDvdTitle *title)
{
  guint start, end;
  gint n_chap;

  ogmrip_codec_get_chapters (codec, &start, &end);

  n_chap = ogmdvd_title_get_n_chapters (title);

  if (start != 0 || end != n_chap - 1)
  {
    gchar *str;

    if (end != n_chap - 1)
    {
      ogmdvd_title_get_n_chapters (title);
      str = g_strdup_printf ("%d-%d", start + 1, end + 1);
    }
    else
      str = g_strdup_printf ("%d", start + 1);

    return str;
  }

  return NULL;
}

static void
ogmrip_mplayer_append_edl (OGMRipCodec *codec, GPtrArray *argv, gboolean hr)
{
  OGMRipEdl *edl;

  edl = ogmrip_codec_get_edl (codec);
  if (edl)
  {
    const gchar *edl_file;

    edl_file = ogmrip_edl_get_filename (edl);
    if (edl_file)
    {
      ogmrip_edl_dump (edl);

      if (hr)
        g_ptr_array_add (argv, g_strdup ("-hr-edl-seek"));
      g_ptr_array_add (argv, g_strdup ("-edl"));
      g_ptr_array_add (argv, g_strdup (edl_file));
    }
  }
}

static gint
ogmrip_mplayer_audio_file_get_demuxer (OGMRipAudioFile *audio)
{
  gint demuxer = OGMRIP_AUDIO_DEMUXER_AUTO;

  switch (ogmrip_file_get_format (OGMRIP_FILE (audio)))
  {
    case OGMRIP_FORMAT_AC3:
      demuxer = OGMRIP_AUDIO_DEMUXER_AC3;
      break;
    case OGMRIP_FORMAT_DTS:
      demuxer = OGMRIP_AUDIO_DEMUXER_DTS;
      break;
  }

  return demuxer;
}

gdouble
ogmrip_mencoder_codec_watch (OGMJobExec *exec, const gchar *buffer, OGMRipCodec *codec)
{
  gint frames, progress;
  gdouble seconds;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
  {
    seconds = strtod (pos, NULL);
    return seconds / ogmrip_codec_get_length (codec, NULL);
  }

  return -1.0;
}

gdouble
ogmrip_mencoder_container_watch (OGMJobExec *exec, const gchar *buffer, OGMRipContainer *container)
{
  gint frames, progress;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
    return progress / 100.;

  return -1.0;
}

GPtrArray *
ogmrip_mplayer_wav_command (OGMRipAudio *audio, gboolean header, const gchar *input, const gchar *output)
{
  OGMDvdTitle *title;
  OGMDvdAudioStream *astream;

  GPtrArray *argv;
  GString *options;

  const gchar *device;
  gchar *chap, *ofps;
  gint vid, aid;

#if MPLAYER_CHECK_VERSION(1,0,0,6)
  gint srate;
#endif /* MPLAYER_CHECK_VERSION */

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (audio));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (audio));
  g_return_val_if_fail (title != NULL, NULL);

  astream = ogmrip_audio_get_dvd_audio_stream (audio);
  g_return_val_if_fail (astream != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noframedrop"));

  ofps = ogmrip_mplayer_get_output_fps (OGMRIP_CODEC (audio), title);
  if (ofps)
    g_free (ofps);
  else
  {
    g_ptr_array_add (argv, g_strdup ("-mc"));
    g_ptr_array_add (argv, g_strdup ("0"));
  }

  g_ptr_array_add (argv, g_strdup ("-vc"));
  g_ptr_array_add (argv, g_strdup ("null"));
  g_ptr_array_add (argv, g_strdup ("-vo"));
  g_ptr_array_add (argv, g_strdup ("null"));
  g_ptr_array_add (argv, g_strdup ("-ao"));

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  options = g_string_new ("pcm");

  if (ogmrip_audio_get_fast (audio))
    g_string_append (options, ":fast");

  if (header)
    g_string_append (options, ":waveheader");
  else
    g_string_append (options, ":nowaveheader");

  g_string_append_printf (options, ":file=%s", output);

  g_ptr_array_add (argv, g_string_free (options, FALSE));
#elif MPLAYER_CHECK_VERSION(1,0,0,7)
  if (header)
    g_ptr_array_add (argv, g_strdup_printf ("pcm:waveheader:file=%s", output));
  else
    g_ptr_array_add (argv, g_strdup_printf ("pcm:nowaveheader:file=%s", output));
#else /* MPLAYER_CHECK_VERSION(1,0,0,7) 7 */
  g_ptr_array_add (argv, g_strdup ("pcm"));
  if (header)
    g_ptr_array_add (argv, g_strdup ("-waveheader"));
  else
    g_ptr_array_add (argv, g_strdup ("-nowaveheader"));
  g_ptr_array_add (argv, g_strdup ("-aofile"));
  g_ptr_array_add (argv, g_strdup (output));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  options = g_string_new (NULL);

  if (ogmrip_audio_get_normalize (audio))
  {
#if MPLAYER_CHECK_VERSION(1,0,0,8)
    g_string_append (options, "volnorm=1");
#elif MPLAYER_CHECK_VERSION(1,0,0,6)
    g_string_append (options, "volnorm");
#else /* MPLAYER_CHECK_VERSION(1,0,0,6) */
    g_string_append (options, "list=volnorm");
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */
  }

#if MPLAYER_CHECK_VERSION(1,0,0,6)
  srate = ogmrip_audio_get_sample_rate (audio);
  if (srate != 48000)
  {
    g_ptr_array_add (argv, g_strdup ("-srate"));
    g_ptr_array_add (argv, g_strdup_printf ("%d", srate));

    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "lavcresample=%d", srate);
  }
#endif /* MPLAYER_CHECK_VERSION(1,0,0,6) */

  if (options->len == 0)
    g_string_free (options, TRUE);
  else
  {
#if MPLAYER_CHECK_VERSION(1,0,0,6)
    g_ptr_array_add (argv, g_strdup ("-af"));
#else /* MPLAYER_CHECK_VERSION(1,0,0,6) */
    g_ptr_array_add (argv, g_strdup ("-aop"));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,6) */
    g_ptr_array_add (argv, g_string_free (options, FALSE));
  }

  g_ptr_array_add (argv, g_strdup ("-channels"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_audio_get_channels (audio) + 1));

  chap = ogmrip_mplayer_get_chapters (OGMRIP_CODEC (audio), title);
  if (chap)
  {
    g_ptr_array_add (argv, g_strdup ("-chapter"));
    g_ptr_array_add (argv, chap);
  }

  ogmrip_mplayer_append_edl (OGMRIP_CODEC (audio), argv, FALSE);

  aid = ogmdvd_stream_get_nr (OGMDVD_STREAM (astream));
  g_ptr_array_add (argv, g_strdup ("-aid"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_mplayer_map_audio_id (astream)));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_CHECK_VERSION(1,0,0,1)
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_CHECK_VERSION(1,0,0,1) */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,1) */

  return argv;
}

gdouble
ogmrip_mplayer_wav_watch (OGMJobExec *exec, const gchar *buffer, OGMRipAudio *audio)
{
  gchar a[10], v[10], av[15], ct[15];
  gint frame, decoded;

  if (sscanf (buffer, "A:%s V:%s A-V:%s ct:%s %d/%d", a, v, av, ct, &decoded, &frame) == 6)
    return decoded / (gdouble) ogmrip_codec_get_frames (OGMRIP_CODEC (audio));

  return -1.0;
}

GPtrArray *
ogmrip_mencoder_audio_command (OGMRipAudio *audio, OGMDvdTitle *title, OGMDvdAudioStream *astream, const gchar *output)
{
  GPtrArray *argv;
  const gchar *device;
  gchar *ofps, *chap;
  gint vid, aid;

  g_return_val_if_fail (OGMRIP_IS_AUDIO (audio), NULL);

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));

  ofps = ogmrip_mplayer_get_output_fps (OGMRIP_CODEC (audio), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }
  else
  {
    g_ptr_array_add (argv, g_strdup ("-mc"));
    g_ptr_array_add (argv, g_strdup ("0"));
  }

  chap = ogmrip_mplayer_get_chapters (OGMRIP_CODEC (audio), title);
  if (chap)
  {
    g_ptr_array_add (argv, g_strdup ("-chapter"));
    g_ptr_array_add (argv, chap);
  }

  ogmrip_mplayer_append_edl (OGMRIP_CODEC (audio), argv, TRUE);

  aid = ogmdvd_stream_get_nr (OGMDVD_STREAM (astream));
  g_ptr_array_add (argv, g_strdup ("-aid"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_mplayer_map_audio_id (astream)));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

  return argv;
}

GPtrArray *
ogmrip_mencoder_video_command (OGMRipVideo *video, OGMDvdTitle *title, const gchar *output)
{
  GPtrArray *argv;
  OGMRipScalerType scaler;
  OGMRipDeintType deint;
  OGMDvdAudioStream *astream;
  GString *options, *pp;

  const gchar *device;
  guint crop_x, crop_y, crop_width, crop_height;
  guint scale_width, scale_height;
  gboolean crop, scale;
  gchar *ofps, *chap;

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noslices"));

  astream = ogmrip_video_get_ensure_sync (video);
  if (astream)
  {
    g_ptr_array_add (argv, g_strdup ("-oac"));
    g_ptr_array_add (argv, g_strdup ("pcm"));
    g_ptr_array_add (argv, g_strdup ("-srate"));
    g_ptr_array_add (argv, g_strdup ("8000"));
    g_ptr_array_add (argv, g_strdup ("-af"));
    g_ptr_array_add (argv, g_strdup ("channels=1,lavcresample=8000"));
    g_ptr_array_add (argv, g_strdup ("-aid"));
    g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_mplayer_map_audio_id (astream)));
/*
    g_ptr_array_add (argv, g_strdup ("-channels"));
    g_ptr_array_add (argv, g_strdup ("1"));
*/
  }
  else
    g_ptr_array_add (argv, g_strdup ("-nosound"));

  scaler = ogmrip_video_get_scaler (video);
  g_ptr_array_add (argv, g_strdup ("-sws"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", MAX (scaler, 0)));

  scale = ogmrip_video_get_scale_size (video, &scale_width, &scale_height);
  g_ptr_array_add (argv, g_strdup (scale ? "-zoom": "-nozoom"));

  chap = ogmrip_mplayer_get_chapters (OGMRIP_CODEC (video), title);
  if (chap)
  {
    g_ptr_array_add (argv, g_strdup ("-chapter"));
    g_ptr_array_add (argv, chap);
  }

  ogmrip_mplayer_append_edl (OGMRIP_CODEC (video), argv, TRUE);

  options = g_string_new (NULL);
  pp = g_string_new (NULL);

  if (ogmrip_video_get_deblock (video))
  {
    if (pp->len > 0)
      g_string_append_c (pp, '/');
    g_string_append (pp, "ha/va");
  }

  if (ogmrip_video_get_dering (video))
  {
    if (pp->len > 0)
      g_string_append_c (pp, '/');
    g_string_append (pp, "dr");
  }

  if (ogmrip_video_get_pullup (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "pullup,softskip");
  }

  crop = ogmrip_video_get_crop_size (video, &crop_x, &crop_y, &crop_width, &crop_height);
  if (crop)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "crop=%u:%u:%u:%u", crop_width, crop_height, crop_x, crop_y);
  }

  deint = ogmrip_video_get_deinterlacer (video);
  if (deint != OGMRIP_DEINT_NONE)
  {
    if (deint == OGMRIP_DEINT_KERNEL || deint == OGMRIP_DEINT_YADIF)
    {
      if (options->len > 0)
        g_string_append_c (options, ',');
      g_string_append (options, deinterlacer[deint - 1]);
    }
    else
    {
      if (pp->len > 0)
        g_string_append_c (pp, '/');
      g_string_append (pp, deinterlacer[deint - 1]);
    }
  }

  if (pp->len > 0)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "pp=%s", pp->str);
  }
  g_string_free (pp, TRUE);

  if (scale)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "scale=%u:%u", scale_width, scale_height);
  }

  if (ogmrip_video_get_denoise (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "hqdn3d=2:1:2");
  }

  if (options->len > 0)
    g_string_append_c (options, ',');
  g_string_append (options, "harddup");

  if (options->len == 0)
    g_string_free (options, TRUE);
  else
  {
    g_ptr_array_add (argv, g_strdup ("-vf"));
    g_ptr_array_add (argv, g_string_free (options, FALSE));
  }

  ofps = ogmrip_mplayer_get_output_fps (OGMRIP_CODEC (video), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }
  else
  {
    g_ptr_array_add (argv, g_strdup ("-mc"));
    g_ptr_array_add (argv, g_strdup ("0"));
  }

  g_ptr_array_add (argv, g_strdup ("-dvdangle"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_video_get_angle (video)));

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup (output));

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  return argv;
}

gdouble
ogmrip_mplayer_video_watch (OGMJobExec *exec, const gchar *buffer, OGMRipVideo *video)
{
  gchar v[10];
  gint frame, decoded;

  if (sscanf (buffer, "V:%s %d/%d", v, &frame, &decoded) == 3)
    return decoded / (gdouble) ogmrip_codec_get_frames (OGMRIP_CODEC (video));

  return -1.0;
}

GPtrArray *
ogmrip_mplayer_video_command (OGMRipVideo *video, OGMDvdTitle *title, const gchar *output)
{
  OGMRipScalerType scaler;
  OGMRipDeintType deint;
  GString *options, *pp;
  GPtrArray *argv;

  const gchar *device;
  gboolean crop, scale;
  guint crop_x, crop_y, crop_width, crop_height;
  guint scale_width, scale_height;
  gchar *chap, *ofps;

  g_return_val_if_fail (OGMRIP_IS_VIDEO (video), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (video));
  g_return_val_if_fail (output != NULL, NULL);

  argv = g_ptr_array_new ();

  g_ptr_array_add (argv, g_strdup ("mplayer"));
  g_ptr_array_add (argv, g_strdup ("-nolirc"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noframedrop"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));
  g_ptr_array_add (argv, g_strdup ("-noslices"));

  scaler = ogmrip_video_get_scaler (video);
  g_ptr_array_add (argv, g_strdup ("-sws"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", MAX (scaler, 0)));

  scale = ogmrip_video_get_scale_size (video, &scale_width, &scale_height);
  g_ptr_array_add (argv, g_strdup (scale ? "-zoom" : "-nozoom"));

  options = g_string_new (NULL);
  pp = g_string_new (NULL);

  if (ogmrip_video_get_deblock (video))
  {
    if (pp->len > 0)
      g_string_append_c (pp, '/');
    g_string_append (pp, "ha/va");
  }

  if (ogmrip_video_get_dering (video))
  {
    if (pp->len > 0)
      g_string_append_c (pp, '/');
    g_string_append (pp, "dr");
  }

  if (ogmrip_video_get_pullup (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "pullup,softskip");
  }

  crop = ogmrip_video_get_crop_size (video, &crop_x, &crop_y, &crop_width, &crop_height);
  if (crop)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "crop=%u:%u:%u:%u", crop_width, crop_height, crop_x, crop_y);
  }

  deint = ogmrip_video_get_deinterlacer (video);
  if (deint != OGMRIP_DEINT_NONE)
  {
    if (deint == OGMRIP_DEINT_KERNEL || deint == OGMRIP_DEINT_YADIF)
    {
      if (options->len > 0)
        g_string_append_c (options, ',');
      g_string_append (options, deinterlacer[deint - 1]);
    }
    else
    {
      if (pp->len > 0)
        g_string_append_c (pp, '/');
      g_string_append (pp, deinterlacer[deint - 1]);
    }
  }

  if (pp->len > 0)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "pp=%s", pp->str);
  }
  g_string_free (pp, TRUE);

  if (scale)
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append_printf (options, "scale=%u:%u", scale_width, scale_height);
  }

  if (ogmrip_video_get_denoise (video))
  {
    if (options->len > 0)
      g_string_append_c (options, ',');
    g_string_append (options, "hqdn3d=2:1:2");
  }

  if (options->len > 0)
    g_string_append_c (options, ',');
  g_string_append (options, "harddup");

  if (options->len == 0)
    g_string_free (options, TRUE);
  else
  {
    g_ptr_array_add (argv, g_strdup ("-vf"));
    g_ptr_array_add (argv, g_string_free (options, FALSE));
  }

  ofps = ogmrip_mplayer_get_output_fps (OGMRIP_CODEC (video), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }
  else
  {
    g_ptr_array_add (argv, g_strdup ("-mc"));
    g_ptr_array_add (argv, g_strdup ("0"));
  }

  g_ptr_array_add (argv, g_strdup ("-dvdangle"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", ogmrip_video_get_angle (video)));

  chap = ogmrip_mplayer_get_chapters (OGMRIP_CODEC (video), title);
  if (chap)
  {
    g_ptr_array_add (argv, g_strdup ("-chapter"));
    g_ptr_array_add (argv, chap);
  }

  ogmrip_mplayer_append_edl (OGMRIP_CODEC (video), argv, FALSE);

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  return argv;
}

gdouble
ogmrip_mencoder_vobsub_watch (OGMJobExec *exec, const gchar *buffer, OGMRipSubp *subp)
{
  gint frames, progress;
  gdouble seconds;
  gchar pos[10];

  if (sscanf (buffer, "Pos:%s %df (%d%%)", pos, &frames, &progress) == 3)
  {
    seconds = strtod (pos, NULL);
    return 0.98 * seconds / ogmrip_codec_get_length (OGMRIP_CODEC (subp), NULL);
  }

  return -1.0;
}

GPtrArray *
ogmrip_mencoder_vobsub_command (OGMRipSubp *subp, const gchar *input, const gchar *output)
{
  OGMDvdTitle *title;
  OGMDvdSubpStream *sstream;

  GPtrArray *argv;
  const gchar *device;
  gchar *ofps, *chap;
  gint vid, sid;

  g_return_val_if_fail (OGMRIP_IS_SUBP (subp), NULL);

  if (!output)
    output = ogmrip_codec_get_output (OGMRIP_CODEC (subp));
  g_return_val_if_fail (output != NULL, NULL);

  title = ogmrip_codec_get_input (OGMRIP_CODEC (subp));
  g_return_val_if_fail (title != NULL, NULL);

  sstream = ogmrip_subp_get_dvd_subp_stream (OGMRIP_SUBP (subp));
  g_return_val_if_fail (sstream != NULL, NULL);

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-nosound"));

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  g_ptr_array_add (argv, g_strdup ("-of"));
  g_ptr_array_add (argv, g_strdup ("rawaudio"));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("copy"));

  ofps = ogmrip_mplayer_get_output_fps (OGMRIP_CODEC (subp), title);
  if (ofps)
  {
    g_ptr_array_add (argv, g_strdup ("-ofps"));
    g_ptr_array_add (argv, ofps);
  }

  g_ptr_array_add (argv, g_strdup ("-o"));
  g_ptr_array_add (argv, g_strdup ("/dev/null"));

  sid = ogmdvd_stream_get_nr (OGMDVD_STREAM (sstream));
  g_ptr_array_add (argv, g_strdup ("-vobsubout"));
  g_ptr_array_add (argv, g_strdup (output));
  g_ptr_array_add (argv, g_strdup ("-vobsuboutindex"));
  g_ptr_array_add (argv, g_strdup ("0"));
  g_ptr_array_add (argv, g_strdup ("-sid"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", sid));
/*
  if (ogmrip_subp_get_forced_only (subp))
    g_ptr_array_add (argv, g_strdup ("-forcedsubsonly"));
*/
  chap = ogmrip_mplayer_get_chapters (OGMRIP_CODEC (subp), title);
  if (chap)
  {
    g_ptr_array_add (argv, g_strdup ("-chapter"));
    g_ptr_array_add (argv, chap);
  }

  ogmrip_mplayer_append_edl (OGMRIP_CODEC (subp), argv, TRUE);

  device = ogmdvd_disc_get_device (ogmdvd_title_get_disc (title));
  g_ptr_array_add (argv, g_strdup ("-dvd-device"));
  g_ptr_array_add (argv, g_strdup (device));

  vid = ogmdvd_title_get_nr (title);

#if MPLAYER_CHECK_VERSION(1,0,0,1)
  g_ptr_array_add (argv, g_strdup_printf ("dvd://%d", vid + 1));
#else /* MPLAYER_CHECK_VERSION(1,0,0,1) */
  g_ptr_array_add (argv, g_strdup ("-dvd"));
  g_ptr_array_add (argv, g_strdup_printf ("%d", vid + 1));
#endif /* MPLAYER_CHECK_VERSION(1,0,0,1) */

  return argv;
}

static void
ogmrip_mencoder_container_append_audio_file (OGMRipContainer *container, 
    const gchar *filename, guint demuxer, gint language, GPtrArray *argv)
{
#if MPLAYER_CHECK_VERSION(1,0,0,8)
  g_ptr_array_add (argv, g_strdup ("-audiofile"));
  g_ptr_array_add (argv, g_strdup (filename));

  if (demuxer != OGMRIP_AUDIO_DEMUXER_AUTO)
  {
    g_ptr_array_add (argv, g_strdup ("-audio-demuxer"));
    g_ptr_array_add (argv, g_strdup ("rawaudio"));
    g_ptr_array_add (argv, g_strdup ("-rawaudio"));
    g_ptr_array_add (argv, g_strdup_printf ("format=0x%x", demuxer));
  }
#else /* MPLAYER_CHECK_VERSION(1,0,0,8) */
  if (demuxer == OGMRIP_AUDIO_DEMUXER_AUTO)
  {
    g_ptr_array_add (argv, g_strdup ("-audiofile"));
    g_ptr_array_add (argv, g_strdup (filename));
  }
  else
  {
    gchar *dirname, *new_name;

    dirname = g_path_get_dirname (filename);
    new_name = g_build_filename (dirname, "frameno.avi", NULL);
    ogmrip_fs_rename (filename, new_name, NULL);
    strcpy ((gchar *) filename, new_name);
  }
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */
}

static void
ogmrip_mencoder_container_foreach_audio (OGMRipContainer *container, 
    OGMRipCodec *codec, guint demuxer, gint language, GPtrArray *argv)
{
  gchar *input;

  input = (gchar *) ogmrip_codec_get_output (codec);

  if (ogmrip_plugin_get_audio_codec_format (G_OBJECT_TYPE (codec)) == OGMRIP_FORMAT_AAC)
  {
    gchar *str;

    str = g_strconcat (input, ".aac", NULL);
    ogmrip_fs_rename (input, str, NULL);

    ogmrip_codec_set_output (codec, str);
    input = str;
  }

  ogmrip_mencoder_container_append_audio_file (container, input, demuxer, language, argv);

  if (ogmrip_plugin_get_audio_codec_format (G_OBJECT_TYPE (codec)) == OGMRIP_FORMAT_AAC)
    g_free (input);
}

static void
ogmrip_mencoder_container_foreach_file (OGMRipContainer *container, OGMRipFile *file, GPtrArray *argv)
{
  gchar *filename;

  filename = ogmrip_file_get_filename (file);
  if (filename)
  {
    if (ogmrip_file_get_type (file) == OGMRIP_FILE_TYPE_AUDIO)
    {
      gint demuxer, format;

      format = ogmrip_file_get_format (file);
      if (format == OGMRIP_FORMAT_AAC && !g_str_has_suffix (filename, ".aac"))
      {
        gchar *s1, *s2;

        s1 = g_path_get_basename (filename);
        s2 = g_build_filename (g_get_tmp_dir (), s1, NULL);
        g_free (s1);

        s1 = g_strconcat (s2, ".aac", NULL);
        g_free (s2);

        if (symlink (filename, s1) < 0)
          g_free (s1);
        else
        {
          g_free (filename);
          filename = s1;
        }
      }

      demuxer = ogmrip_mplayer_audio_file_get_demuxer (OGMRIP_AUDIO_FILE (file));
      ogmrip_mencoder_container_append_audio_file (container, filename, demuxer, -1, argv);
    }
  }
  g_free (filename);
}

GPtrArray *
ogmrip_mencoder_container_command (OGMRipContainer *container)
{
  GPtrArray *argv;
  const gchar *fourcc;

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  const gchar *label;
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  argv = g_ptr_array_new ();
  g_ptr_array_add (argv, g_strdup ("mencoder"));
  g_ptr_array_add (argv, g_strdup ("-nocache"));
  g_ptr_array_add (argv, g_strdup ("-noskip"));
  g_ptr_array_add (argv, g_strdup ("-mc"));
  g_ptr_array_add (argv, g_strdup ("0"));

  g_ptr_array_add (argv, g_strdup ("-ovc"));
  g_ptr_array_add (argv, g_strdup ("copy"));
  g_ptr_array_add (argv, g_strdup ("-oac"));
  g_ptr_array_add (argv, g_strdup ("copy"));

  fourcc = ogmrip_container_get_fourcc (container);
  if (fourcc)
  {
    g_ptr_array_add (argv, g_strdup ("-ffourcc"));
    g_ptr_array_add (argv, g_strdup (fourcc));
  }

#if MPLAYER_CHECK_VERSION(1,0,0,8)
  label = ogmrip_container_get_label (container);
  if (label)
  {
    g_ptr_array_add (argv, g_strdup ("-info"));
    g_ptr_array_add (argv, g_strdup_printf ("name=%s", label));
  }
#endif /* MPLAYER_CHECK_VERSION(1,0,0,8) */

  ogmrip_container_foreach_audio (container, 
      (OGMRipContainerCodecFunc) ogmrip_mencoder_container_foreach_audio, argv);
  ogmrip_container_foreach_file (container, 
      (OGMRipContainerFileFunc) ogmrip_mencoder_container_foreach_file, argv);

  return argv;
}

