/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 8; tab-width: 8 -*-
 *
 * nautilus-burn-drive.c: easy to use cd burner software
 *
 * Copyright (C) 2002-2004 Red Hat, Inc.
 * Copyright (C) 2005 William Jon McCann <mccann@jhu.edu>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 * 
 * Authors: Alexander Larsson <alexl@redhat.com>
 *          Bastien Nocera <hadess@hadess.net>
 *          William Jon McCann <mccann@jhu.edu>
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <unistd.h>
#include <sys/ioctl.h>

#include "nautilus-burn-drive.h"
#include "nautilus-burn-drive-common.h"

#ifdef __linux__
#include "nautilus-burn-drive-linux.h"
#endif /* __linux__ */

#ifdef __FreeBSD__
#include "nautilus-burn-drive-freebsd.h"
#endif /* __FreeBSD__ */

#if defined(USE_HAL4) || defined(USE_HAL5)
#include <libhal.h>
#include "nautilus-burn-drive-hal.h"
#endif /* USE_HAL */

enum 
{
  MEDIA_ADDED,
  MEDIA_REMOVED,
  LAST_SIGNAL
};

enum 
{
  PROP_0,
  PROP_MONITOR_ENABLED
};

static int signals [LAST_SIGNAL] = { 0 };

gboolean drive_door_is_open (int fd);

G_DEFINE_TYPE(NautilusBurnDrive, nautilus_burn_drive, G_TYPE_OBJECT);

static void
nautilus_burn_drive_finalize (GObject *object)
{
  NautilusBurnDrive *drive = NAUTILUS_BURN_DRIVE (object);

  g_return_if_fail (object != NULL);

#ifdef USE_HAL5
  nautilus_burn_drive_hal_shutdown (drive);
#endif

  if (drive->poll_id > 0)
    g_source_remove (drive->poll_id);

  g_free (drive->udi);
  g_free (drive->monitor_udi);
  g_free (drive->display_name);
  g_free (drive->cdrecord_id);
  g_free (drive->device);

  if (G_OBJECT_CLASS (nautilus_burn_drive_parent_class)->finalize != NULL)
    (* G_OBJECT_CLASS (nautilus_burn_drive_parent_class)->finalize) (object);
}

static void
nautilus_burn_drive_set_property (GObject *object, guint prop_id, const GValue *value, GParamSpec *pspec)
{
  switch (prop_id) 
  {
    case PROP_MONITOR_ENABLED:
      nautilus_burn_drive_set_monitor_enabled (NAUTILUS_BURN_DRIVE (object), g_value_get_boolean (value));
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
nautilus_burn_drive_get_property (GObject *object, guint prop_id, GValue *value, GParamSpec *pspec)
{
  switch (prop_id) 
  {
    case PROP_MONITOR_ENABLED:
      g_value_set_boolean (value, NAUTILUS_BURN_DRIVE (object)->monitor_enabled);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
      break;
  }
}

static void
nautilus_burn_drive_class_init (NautilusBurnDriveClass *klass)
{
  GObjectClass *object_class;

  object_class = (GObjectClass *) klass;

  object_class->finalize = nautilus_burn_drive_finalize;
  object_class->get_property = nautilus_burn_drive_get_property;
  object_class->set_property = nautilus_burn_drive_set_property;

  /* Properties */
  g_object_class_install_property (object_class, PROP_MONITOR_ENABLED, 
      g_param_spec_boolean ("enable-monitor", NULL, NULL, FALSE, G_PARAM_READWRITE));

  /* Signals */
  signals [MEDIA_ADDED] = 
    g_signal_new ("media-added", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST, 
        G_STRUCT_OFFSET (NautilusBurnDriveClass, media_added), NULL, NULL,
        g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);

  signals [MEDIA_REMOVED] = 
    g_signal_new ("media-removed", G_TYPE_FROM_CLASS (klass), G_SIGNAL_RUN_LAST, 
        G_STRUCT_OFFSET (NautilusBurnDriveClass, media_removed), NULL, NULL, 
        g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
}

static void
nautilus_burn_drive_init (NautilusBurnDrive *drive)
{
#ifdef USE_HAL5
  nautilus_burn_drive_hal_init (drive);
#endif
}

NautilusBurnDrive *
nautilus_burn_drive_new (void)
{
  return g_object_new (NAUTILUS_BURN_TYPE_DRIVE, NULL);
}

NautilusBurnDrive *
nautilus_burn_drive_new_from_path (const gchar *device)
{
  NautilusBurnDrive *drive = NULL;
  GList *list, *link;

  list = nautilus_burn_drive_get_list ();
  for (link = list; link; link = link->next) 
  {
    drive = link->data;
    if (g_str_equal (device, drive->device)) 
      break;
  }

  if (link)
    g_object_ref (drive);

  g_list_foreach (list, (GFunc) g_object_unref, NULL);
  g_list_free (list);

  return drive;
}

gboolean
nautilus_burn_drive_door_is_open (NautilusBurnDrive *drive)
{
  gpointer ioctl_handle;
  int      fd;
  gboolean ret;

  g_return_val_if_fail (NAUTILUS_BURN_IS_DRIVE (drive), FALSE);

  ioctl_handle = open_ioctl_handle (drive->device);
  if (ioctl_handle == INVALID_HANDLE)
    return FALSE;

  fd = get_ioctl_handle_fd (ioctl_handle);

  ret = drive_door_is_open (fd);

  close_ioctl_handle (ioctl_handle);

  return ret;
}

gboolean
nautilus_burn_drive_eject (NautilusBurnDrive *drive)
{
  gchar *cmd;
  gboolean res;

  g_return_val_if_fail (NAUTILUS_BURN_IS_DRIVE (drive), FALSE);

  if (drive->device == NULL)
    return FALSE;

#ifdef __FreeBSD_
  cmd = g_strdup_printf ("cdcontrol -f %s eject", drive->priv->device);
#else
  cmd = g_strdup_printf ("eject %s", drive->device);
#endif

  res = g_spawn_command_line_sync (cmd, NULL, NULL, NULL, NULL);
  g_free (cmd);

  /* delay a bit to make sure eject finishes */
  /* sleep (2); */

  return res;
}

NautilusBurnMediaType
nautilus_burn_drive_get_media_type (NautilusBurnDrive *drive)
{
  g_return_val_if_fail (NAUTILUS_BURN_IS_DRIVE (drive), NAUTILUS_BURN_MEDIA_TYPE_ERROR);

#ifdef USE_HAL5
  return nautilus_burn_drive_hal_get_media_type_full (drive);
#else
  return nautilus_burn_drive_get_media_type_from_path_full (drive->device);
#endif
}

void
nautilus_burn_drive_set_monitor_enabled (NautilusBurnDrive *drive, gboolean enabled)
{
  g_return_if_fail (NAUTILUS_BURN_IS_DRIVE (drive));

  if (drive->monitor_enabled != enabled) 
  {
    drive->monitor_enabled = enabled;

#ifdef USE_HAL5
    nautilus_burn_drive_hal_set_monitor (drive, enabled);
#else
    nautilus_burn_drive_set_monitor (drive, enabled);
#endif
  }
}

gboolean
nautilus_burn_drive_get_monitor_enabled (NautilusBurnDrive *drive)
{
  g_return_val_if_fail (NAUTILUS_BURN_IS_DRIVE (drive), FALSE);

  return drive->monitor_enabled;
}

GList *
nautilus_burn_drive_get_list (void)
{
  GList *drives = NULL;

#if defined(USE_HAL4) || defined(USE_HAL5)
  drives = nautilus_burn_drive_hal_scan ();
#endif

  if (drives == NULL)
  {
#if defined (__linux__)
    drives = linux_scan (FALSE);
#elif defined (__FreeBSD__)
    drives = freebsd_scan (FALSE);
#else
    drives = cdrecord_scan (FALSE);
#endif
  }

  return drives;
}

