/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2008 Olivier Rolland <billl@users.sf.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include <glib/gi18n.h>
#include <glade/glade.h>

#include "ogmrip-progress.h"
#include "ogmrip-helper.h"

#ifdef HAVE_LIBNOTIFY_SUPPORT
#include <libnotify/notify.h>
#endif /* HAVE_LIBNOTIFY_SUPPORT */

#define OGMRIP_PROGRESS_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_PROGRESS_DIALOG, OGMRipProgressDialogPriv))

#define OGMRIP_ICON_FILE  "pixmaps/ogmrip.png"
#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-progress.glade"
#define OGMRIP_GLADE_ROOT "root"

struct _OGMRipProgressDialogPriv
{
  OGMJobSpawn *spawn;

  GtkWidget *title_label;
  GtkWidget *stage_label;
  GtkWidget *eta_label;

  GtkWidget *stage_progress;

  GtkWidget *suspend_button;
  GtkWidget *resume_button;
  GtkWidget *cancel_button;
  GtkWidget *quit_check;
/*
  guint stage;
  guint completed;
*/
  gulong start_time;
  gulong suspend_time;

  gchar *title;
  gchar *label;

#ifdef HAVE_LIBNOTIFY_SUPPORT
  NotifyNotification *notification;
  GtkStatusIcon *status_icon;
  gboolean iconified;
#endif /* HAVE_LIBNOTIFY_SUPPORT */

};

enum
{
  GTK_RESPONSE_SUSPEND = 1,
  GTK_RESPONSE_RESUME
};

static void ogmrip_progress_dialog_dispose  (GObject   *gobject);
static void ogmrip_progress_dialog_response (GtkDialog *dialog,
                                             gint      response_id);

#ifdef HAVE_LIBNOTIFY_SUPPORT
static gboolean
ogmrip_progress_dialog_get_visibility (OGMRipProgressDialog *dialog)
{
  GdkWindowState state;

  if (!GTK_WIDGET_REALIZED (dialog))
    return FALSE;

  if (dialog->priv->iconified)
    return FALSE;

  state = gdk_window_get_state (GTK_WIDGET (dialog)->window);
  if (state & (GDK_WINDOW_STATE_WITHDRAWN | GDK_WINDOW_STATE_ICONIFIED))
    return FALSE;

  return TRUE;
}

static gboolean
ogmrip_progress_dialog_state_changed (OGMRipProgressDialog *dialog, GdkEventWindowState *event)
{
  dialog->priv->iconified = ((event->new_window_state & GDK_WINDOW_STATE_ICONIFIED) != 0);

  return FALSE;
}

static void
ogmrip_progress_dialog_iconify (OGMRipProgressDialog *dialog)
{
  gtk_window_iconify (gtk_window_get_transient_for (GTK_WINDOW (dialog)));
}

static void
ogmrip_progress_dialog_cancel_menu_activated (OGMRipProgressDialog *dialog)
{
  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_CANCEL);
}

static void
ogmrip_progress_dialog_status_icon_popup_menu (OGMRipProgressDialog *dialog, guint button, guint activate_time)
{
  GtkWidget *menu, *menuitem, *image;

  menu = gtk_menu_new ();

  menuitem = gtk_image_menu_item_new_with_label (_("Suspend"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
  if (GTK_WIDGET_VISIBLE (dialog->priv->suspend_button))
    gtk_widget_show (menuitem);

  g_signal_connect_swapped (menuitem, "activate",
      G_CALLBACK (gtk_button_clicked), dialog->priv->suspend_button);

  image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PAUSE, GTK_ICON_SIZE_MENU);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);
  gtk_widget_show (image);

  menuitem = gtk_image_menu_item_new_with_label (_("Resume"));
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
  if (GTK_WIDGET_VISIBLE (dialog->priv->resume_button))
    gtk_widget_show (menuitem);

  g_signal_connect_swapped (menuitem, "activate",
      G_CALLBACK (gtk_button_clicked), dialog->priv->resume_button);

  image = gtk_image_new_from_stock (GTK_STOCK_MEDIA_PLAY, GTK_ICON_SIZE_MENU);
  gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM (menuitem), image);
  gtk_widget_show (image);

  menuitem = gtk_separator_menu_item_new ();
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
  gtk_widget_show (menuitem);

  menuitem = gtk_image_menu_item_new_from_stock (GTK_STOCK_CANCEL, NULL);
  gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
  gtk_widget_show (menuitem);

  g_signal_connect_swapped (menuitem, "activate",
      G_CALLBACK (ogmrip_progress_dialog_cancel_menu_activated), dialog);

  gtk_menu_popup (GTK_MENU (menu), NULL, NULL, gtk_status_icon_position_menu,
      dialog->priv->status_icon, button, activate_time);
}

#endif /* HAVE_LIBNOTIFY_SUPPORT */

static void
ogmrip_progress_dialog_run (OGMRipProgressDialog *dialog)
{
  GTimeVal tv;

  g_get_current_time (&tv);
  dialog->priv->start_time = tv.tv_sec;
}

static void
ogmrip_progress_dialog_progress (OGMRipProgressDialog *dialog, gdouble fraction)
{
  if (fraction < 0.0)
    gtk_progress_bar_pulse (GTK_PROGRESS_BAR (dialog->priv->stage_progress));
  else
  {
    GTimeVal tv;
    gulong eta;
    gchar *str;

    if (!dialog->priv->start_time)
    {
      g_get_current_time (&tv);
      dialog->priv->start_time = tv.tv_sec;
    }

    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (dialog->priv->stage_progress), fraction);

#ifdef HAVE_LIBNOTIFY_SUPPORT
    if (dialog->priv->status_icon)
    {
      str = g_strdup_printf (_("%s: %02.0lf%% done"), dialog->priv->label, fraction * 100);
      gtk_status_icon_set_tooltip (dialog->priv->status_icon, str);
      g_free (str);
    }
#endif /* HAVE_LIBNOTIFY_SUPPORT */

    g_get_current_time (&tv);
    eta = (1.0 - fraction) * (tv.tv_sec - dialog->priv->start_time) / fraction;

    if (eta >= 3600)
      str = g_strdup_printf ("%ld hour(s) %ld minute(s)", eta / 3600, (eta % 3600) / 60);
    else
      str = g_strdup_printf ("%ld minute(s)", eta / 60);

    gtk_label_set_text (GTK_LABEL (dialog->priv->eta_label), str);
    g_free (str);

    if (GTK_WINDOW (dialog)->transient_parent)
    {
      gchar *title;

      title = g_strdup_printf ("OGMRip: %s: %.0f%%", dialog->priv->label, CLAMP (fraction, 0.0, 1.0) * 100);
      gtk_window_set_title (GTK_WINDOW (dialog)->transient_parent, title);
      g_free (title);
    }
  }
}

static void
ogmrip_progress_dialog_suspend (OGMRipProgressDialog *dialog)
{
  GTimeVal tv;

  g_get_current_time (&tv);
  dialog->priv->suspend_time = tv.tv_sec;
}

static void
ogmrip_progress_dialog_resume (OGMRipProgressDialog *dialog)
{
  GTimeVal tv;

  g_get_current_time (&tv);
  dialog->priv->start_time += tv.tv_sec - dialog->priv->suspend_time;
}

G_DEFINE_TYPE (OGMRipProgressDialog, ogmrip_progress_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_progress_dialog_class_init (OGMRipProgressDialogClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  GtkDialogClass *dialog_class = GTK_DIALOG_CLASS (klass);

  gobject_class->dispose = ogmrip_progress_dialog_dispose;
  dialog_class->response = ogmrip_progress_dialog_response;

  g_type_class_add_private (klass, sizeof (OGMRipProgressDialogPriv));
}

static void
ogmrip_progress_dialog_init (OGMRipProgressDialog *dialog)
{
  GtkWidget *root;
  GtkSizeGroup *group;
  GladeXML *xml;

#ifdef HAVE_LIBNOTIFY_SUPPORT
  GtkAccelGroup *accel_group;
  GClosure *closure;
#endif /* HAVE_LIBNOTIFY_SUPPORT */

  dialog->priv = OGMRIP_PROGRESS_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  group = gtk_size_group_new (GTK_SIZE_GROUP_HORIZONTAL);

  gtk_button_box_set_layout (GTK_BUTTON_BOX (GTK_DIALOG (dialog)->action_area), GTK_BUTTONBOX_EDGE);

  dialog->priv->resume_button = gtk_button_new_with_image (_("Resume"), GTK_STOCK_MEDIA_PLAY);
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), dialog->priv->resume_button, GTK_RESPONSE_RESUME);
  gtk_size_group_add_widget (group, dialog->priv->resume_button);

  dialog->priv->suspend_button = gtk_button_new_with_image (_("Suspend"), GTK_STOCK_MEDIA_PAUSE);
  gtk_dialog_add_action_widget (GTK_DIALOG (dialog), dialog->priv->suspend_button, GTK_RESPONSE_SUSPEND);
  gtk_size_group_add_widget (group, dialog->priv->suspend_button);
  gtk_widget_show (dialog->priv->suspend_button);

  dialog->priv->cancel_button = gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL);

  gtk_window_set_title (GTK_WINDOW (dialog), _("Progress"));
  gtk_window_set_default_size (GTK_WINDOW (dialog), 450, -1);
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_EXECUTE);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CANCEL);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

  root = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), root, TRUE, TRUE, 0);
  gtk_widget_show (root);
/*
  dialog->priv->completed = -1;
*/
#ifdef HAVE_LIBNOTIFY_SUPPORT
  dialog->priv->status_icon = gtk_status_icon_new_from_file (OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE);

  g_signal_connect_swapped (dialog->priv->status_icon, "popup_menu",
      G_CALLBACK (ogmrip_progress_dialog_status_icon_popup_menu), dialog);

  dialog->priv->notification = notify_notification_new_with_status_icon ("Dummy", "Dummy",
      OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE, dialog->priv->status_icon);
  g_signal_connect (dialog, "window-state-event",
      G_CALLBACK (ogmrip_progress_dialog_state_changed), NULL);

  accel_group = gtk_accel_group_new ();
  gtk_window_add_accel_group (GTK_WINDOW (dialog), accel_group);

  closure = g_cclosure_new_swap (G_CALLBACK (ogmrip_progress_dialog_iconify), dialog, NULL);
  gtk_accel_group_connect (accel_group, 'w', GDK_CONTROL_MASK, 0, closure);
  g_closure_unref (closure);
#endif /* HAVE_LIBNOTIFY_SUPPORT */

  dialog->priv->title_label = glade_xml_get_widget (xml, "title-label");
  dialog->priv->stage_label = glade_xml_get_widget (xml, "stage-label");
  dialog->priv->stage_progress = glade_xml_get_widget (xml, "stage-progress");
  dialog->priv->eta_label = glade_xml_get_widget (xml, "eta-label");
  dialog->priv->quit_check = glade_xml_get_widget (xml, "quit-check");

  g_object_unref (xml);
}

static void
ogmrip_progress_dialog_dispose (GObject *gobject)
{
  OGMRipProgressDialog *dialog = OGMRIP_PROGRESS_DIALOG (gobject);

  if (dialog->priv->spawn)
    g_object_unref (dialog->priv->spawn);
  dialog->priv->spawn = NULL;

  if (dialog->priv->title)
    g_free (dialog->priv->title);
  dialog->priv->title = NULL;

  if (dialog->priv->label)
    g_free (dialog->priv->label);
  dialog->priv->label = NULL;

#ifdef HAVE_LIBNOTIFY_SUPPORT
  if (dialog->priv->notification)
    g_object_unref (dialog->priv->notification);
  dialog->priv->notification = NULL;

  if (dialog->priv->status_icon)
    g_object_unref (dialog->priv->status_icon);
  dialog->priv->status_icon = NULL;
#endif /* HAVE_LIBNOTIFY_SUPPORT */

  G_OBJECT_CLASS (ogmrip_progress_dialog_parent_class)->dispose (gobject);
}

static void
ogmrip_progress_dialog_response (GtkDialog *dialog, gint response_id)
{
  OGMRipProgressDialog *progress_dialog;

  progress_dialog = OGMRIP_PROGRESS_DIALOG (dialog);

  if (progress_dialog->priv->spawn)
  {
    switch (response_id)
    {
      case GTK_RESPONSE_SUSPEND:
        gtk_widget_hide (progress_dialog->priv->suspend_button);
        gtk_widget_show (progress_dialog->priv->resume_button);
        ogmjob_spawn_suspend (progress_dialog->priv->spawn);
        break;
      case GTK_RESPONSE_RESUME:
        gtk_widget_hide (progress_dialog->priv->resume_button);
        gtk_widget_show (progress_dialog->priv->suspend_button);
        ogmjob_spawn_resume (progress_dialog->priv->spawn);
        break;
      case GTK_RESPONSE_NONE:
        break;
      default:
        response_id = ogmrip_message_dialog (GTK_WINDOW (dialog), GTK_MESSAGE_QUESTION,
            _("Are you sure you want to cancel the encoding process?"));
        if (response_id == GTK_RESPONSE_YES)
          ogmjob_spawn_cancel (progress_dialog->priv->spawn);
        break;
    }
  }
}

GtkWidget *
ogmrip_progress_dialog_new (const gchar *title, guint stage)
{
  OGMRipProgressDialog *dialog;
  gchar *str;

  dialog = g_object_new (OGMRIP_TYPE_PROGRESS_DIALOG, NULL);
  dialog->priv->title = g_strdup (title);
/*
  dialog->priv->stage = stage;
*/
  str = g_strdup_printf ("<big><b>%s</b></big>", title);
  gtk_label_set_markup (GTK_LABEL (dialog->priv->title_label), str);
  g_free (str);

#ifdef HAVE_LIBNOTIFY_SUPPORT
  notify_notification_update (dialog->priv->notification, title, "Dummy",
      OGMRIP_DATA_DIR "/" OGMRIP_ICON_FILE);
#endif /* HAVE_LIBNOTIFY_SUPPORT */

  return GTK_WIDGET (dialog);
}

void
ogmrip_progress_dialog_set_spawn (OGMRipProgressDialog *dialog, OGMJobSpawn *spawn, const gchar *label)
{
  if (dialog)
  {
    dialog->priv->start_time = 0;
/*
    dialog->priv->completed ++;
*/
    if (spawn != dialog->priv->spawn)
    {
      g_object_ref (spawn);
      if (dialog->priv->spawn)
        g_object_unref (dialog->priv->spawn);
      dialog->priv->spawn = spawn;

      g_signal_connect_swapped (spawn, "run", G_CALLBACK (ogmrip_progress_dialog_run), dialog);
      g_signal_connect_swapped (spawn, "progress", G_CALLBACK (ogmrip_progress_dialog_progress), dialog);
      g_signal_connect_swapped (spawn, "suspend", G_CALLBACK (ogmrip_progress_dialog_suspend), dialog);
      g_signal_connect_swapped (spawn, "resume", G_CALLBACK (ogmrip_progress_dialog_resume), dialog);
    }

    if (label)
    {
      gchar *str;

      if (dialog->priv->label)
        g_free (dialog->priv->label);
      dialog->priv->label = g_strdup (label);

      str = g_strdup_printf ("<i>%s</i>", label);
      gtk_label_set_markup (GTK_LABEL (dialog->priv->stage_label), str);
      g_free (str);

#ifdef HAVE_LIBNOTIFY_SUPPORT
      if (dialog->priv->notification && !ogmrip_progress_dialog_get_visibility (dialog))
      {
        g_object_set (dialog->priv->notification, "body", label, NULL);
        notify_notification_show (dialog->priv->notification, NULL);
      }
#endif
    }

    gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR (dialog->priv->stage_progress), 0.0);
  }
}

gboolean
ogmrip_progress_dialog_get_quit (OGMRipProgressDialog *dialog)
{
  return gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dialog->priv->quit_check));
}

