/* OGMRip - A DVD Encoder for GNOME
 * Copyright (C) 2004-2009 Olivier Rolland <billl@users.sourceforge.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "config.h"

#include "ogmdvd.h"

#include "ogmrip-fs.h"
#include "ogmrip-pref.h"
#include "ogmrip-profiles.h"
#include "ogmrip-gconf.h"
#include "ogmrip-helper.h"
#include "ogmrip-plugin.h"
#include "ogmrip-preferences.h"

#include <string.h>
#include <glib/gi18n.h>
#include <glade/glade.h>

#define OGMRIP_GLADE_FILE "ogmrip/ogmrip-pref.glade"
#define OGMRIP_GLADE_ROOT "root"

#define OGMRIP_PREF_DIALOG_GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE ((o), OGMRIP_TYPE_PREF_DIALOG, OGMRipPrefDialogPriv))

struct _OGMRipPrefDialogPriv
{
  gboolean dummy;
};

extern gchar *ogmdvd_languages[][3];
extern guint  ogmdvd_nlanguages;

/*
 * Toggle
 */

static void
ogmrip_pref_dialog_connect_toggle (GtkWidget *toggle, const gchar *key, gboolean def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_bool (key, def);

  ogmrip_preferences_connect_toggle (toggle, key);
}

/*
 * Combo
 */

static void
ogmrip_pref_dialog_connect_combo (GtkWidget *combo, const gchar *key, gint def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (key, def);

  ogmrip_preferences_connect_combo (combo, key);
}

/*
 * Lang
 */

static GConfValue *
ogmrip_pref_dialog_lang_get_value (GtkWidget *combo)
{
  GConfValue *value;
  const gchar *lang;
  gint i;

  value = gconf_value_new (GCONF_VALUE_INT);

  i = gtk_combo_box_get_active (GTK_COMBO_BOX (combo));
  if (i > 0 && i + 1 < ogmdvd_nlanguages)
  {
    lang = ogmdvd_languages[i + 1][OGMDVD_LANGUAGE_ISO639_1];
    gconf_value_set_int (value, (lang[0] << 8) | lang[1]);
  }

  return value;
}

static gboolean
ogmrip_pref_dialog_lang_set_value (GtkWidget *combo, GConfValue *value)
{
  gint code;

  if (!value)
    return FALSE;

  code = gconf_value_get_int (value);
  if (!code)
    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), 0);
  else
  {
    const gchar *lang;
    gint i;

    lang = ogmdvd_get_language_iso639_1 (code);

    for (i = 2; i < ogmdvd_nlanguages; i++)
      if (g_str_equal (ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1], lang))
        break;

    if (i == ogmdvd_nlanguages)
      return FALSE;

    gtk_combo_box_set_active (GTK_COMBO_BOX (combo), i - 1);
  }

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_lang (GtkWidget *widget, const gchar *key, gint def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_int (key, def);

  if (widget)
    ogmrip_preferences_connect_combo_custom (widget, key,
        ogmrip_pref_dialog_lang_get_value,
        ogmrip_pref_dialog_lang_set_value);
}

/*
 * Chooser
 */

static void
ogmrip_pref_dialog_connect_output_chooser (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_chooser (widget, key);
}

static GConfValue *
ogmrip_pref_dialog_tmp_chooser_get_value (GtkWidget *chooser)
{
  GConfValue *value;
  gchar *str;

  value = gconf_value_new (GCONF_VALUE_STRING);

  str = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (chooser));
  if (!str)
    return NULL;

  gconf_value_set_string (value, str);
  g_free (str);
                
  return value;
}

static gboolean
ogmrip_pref_dialog_tmp_chooser_set_value (GtkWidget *chooser, GConfValue *value)
{
  const gchar *new_path;
  gchar *old_path;

  if (!value)
    return FALSE;

  new_path = gconf_value_get_string (value);
  if (!g_file_test (new_path, G_FILE_TEST_IS_DIR))
    return FALSE;

  ogmrip_fs_set_tmp_dir (new_path);

  old_path = gtk_file_chooser_get_current_folder (GTK_FILE_CHOOSER (chooser));
  if (!old_path || !g_str_equal (new_path, old_path))
    gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (chooser), new_path);

  return TRUE;
}

static void
ogmrip_pref_dialog_connect_tmp_chooser (GtkWidget *widget, const gchar *key, const gchar *def)
{
  GConfValue *value;

  value = ogmrip_preferences_get (key);
  if (value)
    gconf_value_free (value);
  else
    ogmrip_preferences_set_string (key, def);

  if (widget)
    ogmrip_preferences_connect_chooser_custom (widget, key,
        ogmrip_pref_dialog_tmp_chooser_get_value,
        ogmrip_pref_dialog_tmp_chooser_set_value);
}

/*
 * Dialog
 */

G_DEFINE_TYPE (OGMRipPrefDialog, ogmrip_pref_dialog, GTK_TYPE_DIALOG);

static void
ogmrip_pref_dialog_add_languages (GtkWidget *widget)
{
  gchar *str;
  gint i;

  for (i = 2; i < ogmdvd_nlanguages; i++)
  {
    str = g_strdup_printf ("%s (%s)", ogmdvd_languages[i][OGMDVD_LANGUAGE_NAME], 
        ogmdvd_languages[i][OGMDVD_LANGUAGE_ISO639_1]);
    gtk_combo_box_append_text (GTK_COMBO_BOX (widget), str);
    g_free (str);
  }
}

static void
ogmrip_pref_dialog_class_init (OGMRipPrefDialogClass *klass)
{
  g_type_class_add_private (klass, sizeof (OGMRipPrefDialogPriv));
}

static void
ogmrip_pref_dialog_init (OGMRipPrefDialog *dialog)
{
  GtkWidget *widget;
  GladeXML *xml;

  dialog->priv = OGMRIP_PREF_DIALOG_GET_PRIVATE (dialog);

  xml = glade_xml_new (OGMRIP_DATA_DIR "/" OGMRIP_GLADE_FILE, OGMRIP_GLADE_ROOT, NULL);
  if (!xml)
  {
    g_warning ("Could not find " OGMRIP_GLADE_FILE);
    return;
  }

  gtk_dialog_add_buttons (GTK_DIALOG (dialog),
      GTK_STOCK_CLOSE, GTK_RESPONSE_CLOSE,
      NULL);
  gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);
  gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
  gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_CLOSE);
  gtk_window_set_title (GTK_WINDOW (dialog), _("Preferences"));
  gtk_window_set_icon_from_stock (GTK_WINDOW (dialog), GTK_STOCK_PREFERENCES);

  widget = glade_xml_get_widget (xml, OGMRIP_GLADE_ROOT);
  gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), widget, TRUE, TRUE, 0);
  gtk_widget_show (widget);

  /*
   * General
   */

  widget = glade_xml_get_widget (xml, "output-dir-chooser");
  ogmrip_pref_dialog_connect_output_chooser (widget,
      OGMRIP_GCONF_OUTPUT_DIR, OGMRIP_DEFAULT_OUTPUT_DIR);

  gtk_file_chooser_set_action (GTK_FILE_CHOOSER (widget), 
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  widget = glade_xml_get_widget (xml, "filename-combo");
  ogmrip_pref_dialog_connect_combo (widget,
      OGMRIP_GCONF_FILENAME, OGMRIP_DEFAULT_FILENAME);

  widget = glade_xml_get_widget (xml, "pref-audio-combo");
  ogmrip_pref_dialog_add_languages (widget);
  ogmrip_pref_dialog_connect_lang (widget,
      OGMRIP_GCONF_PREF_AUDIO, OGMRIP_DEFAULT_PREF_AUDIO);

  widget = glade_xml_get_widget (xml, "pref-subp-combo");
  ogmrip_pref_dialog_add_languages (widget);
  ogmrip_pref_dialog_connect_lang (widget,
      OGMRIP_GCONF_PREF_SUBP, OGMRIP_DEFAULT_PREF_SUBP);

  widget = glade_xml_get_widget (xml, "chapter-lang-combo");
  ogmrip_pref_dialog_add_languages (widget);
  ogmrip_pref_dialog_connect_lang (widget,
      OGMRIP_GCONF_CHAPTER_LANG, OGMRIP_DEFAULT_CHAPTER_LANG);

  /*
   * Advanced
   */

  widget = glade_xml_get_widget (xml, "copy-dvd-check");
  ogmrip_pref_dialog_connect_toggle (widget,
      OGMRIP_GCONF_COPY_DVD, OGMRIP_DEFAULT_COPY_DVD);

  widget = glade_xml_get_widget (xml, "after-enc-combo");
  ogmrip_pref_dialog_connect_combo (widget,
      OGMRIP_GCONF_AFTER_ENC, OGMRIP_DEFAULT_AFTER_ENC);

  widget = glade_xml_get_widget (xml, "keep-tmp-check");
  ogmrip_pref_dialog_connect_toggle (widget,
      OGMRIP_GCONF_KEEP_TMP, OGMRIP_DEFAULT_KEEP_TMP);

  widget = glade_xml_get_widget (xml, "log-check");
  ogmrip_pref_dialog_connect_toggle (widget,
      OGMRIP_GCONF_LOG_OUTPUT, OGMRIP_DEFAULT_LOG_OUTPUT);

  widget = glade_xml_get_widget (xml, "tmp-dir-chooser");
  ogmrip_pref_dialog_connect_tmp_chooser (widget,
      OGMRIP_GCONF_TMP_DIR, OGMRIP_DEFAULT_TMP_DIR);

  gtk_file_chooser_set_action (GTK_FILE_CHOOSER (widget), 
      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER);

  g_object_unref (xml);
}

GtkWidget *
ogmrip_pref_dialog_new (void)
{
  return g_object_new (OGMRIP_TYPE_PREF_DIALOG, NULL);
}

