/* bunzip2_mod.c -- BZip2 decompression module for Penguin Linux Bootloader
 *
 * Copyright (c) 1999 Tony Mantler <tonym@mac.linux-m68k.org>
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "bzlib.h"
#include "stream.h"

#define STATIC static

#define BZ2_BUFSIZ	(32*1024)

STATIC int		bunzip2_open		(const char	*name);
STATIC long		bunzip2_fillbuf		(char		*buf);
STATIC int		bunzip2_close		(void);
STATIC int		bunzip2_fill_inbuf	(void);
STATIC void		bunzip2_error		(char		*x);

MODULE bunzip2_mod = {
	"bunzip2",
	BZ2_BUFSIZ,
	bunzip2_open,
	bunzip2_fillbuf,
	NULL,				/* no skipping */
	bunzip2_close,
	MOD_REST_INIT
};

STATIC bz_stream bunzip2_stream;

STATIC char *bunzip2_inbuf;

/* *** bunzip2_open *** */
STATIC int bunzip2_open	(const char *name) {

//	bunzip2_error("bunzip2_open called");
	
	/* try opening downstream channel */
	if (sopen( name ) < 0) {
		bunzip2_error("Can't open downstream module");
		return -1;
	}
	
	/* allocate input buffer */
	bunzip2_inbuf = malloc(BZ2_BUFSIZ);
	if (bunzip2_inbuf == NULL) {
		bunzip2_error("Can't allocate input buffer");
		return -1;
	}
	bunzip2_stream.avail_in = 0; /* no data in there yet */
	
	/* fix up the rest of the bz_stream struct */
	bunzip2_stream.state = NULL;
	bunzip2_stream.bzalloc = NULL; /* use malloc() */
	bunzip2_stream.bzfree = NULL; /* use free() */
	bunzip2_stream.opaque = NULL;
	
	/* init the decompression stuff */
	bzDecompressInit(&bunzip2_stream, 0, 0);
	
	cprintf( "BUnzip2ing %s\n", name );
	
	return 0;
}

STATIC long bunzip2_fillbuf (char *buf) {
	int err = 0;
	int insize = 0;
	
//	bunzip2_error("bunzip2_fillbuf called");
	
	bunzip2_stream.next_out = buf;
	bunzip2_stream.avail_out = BZ2_BUFSIZ;
	
	do {
		/* keep the input buffer atleast half full */
		if (bunzip2_stream.avail_in < BZ2_BUFSIZ/2) {
			insize = bunzip2_fill_inbuf();
		}
		err = bzDecompress(&bunzip2_stream);
	} while (err == 0 && bunzip2_stream.avail_out != 0 && insize > 0);
	
	if (err < 0 && err != BZ_OUTBUFF_FULL) {
		/* Uh oh. Error. Bad error. Panic! */
		bunzip2_error("Decompression error. Oops.");
//		cprintf("bunzip2_mod: error %d\n", err);
		return -1;
	}
	
	if (insize < 0) {
		/* Uh oh, can't read data. */
		bunzip2_error("Data read error. That's bad.");
		return -1;
	}
	
	return BZ2_BUFSIZ - bunzip2_stream.avail_out;
}

STATIC int bunzip2_close (void) {
//	bunzip2_error("bunzip2_close called");
	
	/* clean up the bz2 stream */
	bzDecompressEnd(&bunzip2_stream);
	
	if (bunzip2_inbuf != NULL) free(bunzip2_inbuf);
	bunzip2_inbuf = NULL;
	
	return 0;
}

STATIC int bunzip2_fill_inbuf (void) {
	int err;
	
//	bunzip2_error("bunzip2_fill_inbuf called");
	
	/* leftover data in the buffer */
	if (bunzip2_stream.avail_in != 0) {
		void *foo;
		foo = malloc(bunzip2_stream.avail_in);
		if (foo == NULL) {
			bunzip2_error("Can't allocate memory");
			return -1;
		}
		/* copy it back to the start of the buffer */
		memcpy(foo, bunzip2_stream.next_in, bunzip2_stream.avail_in);
		memcpy(bunzip2_inbuf, foo, bunzip2_stream.avail_in);
		free(foo);
	}
	/* fill up the rest */
	err = sread(
		(char *)(bunzip2_inbuf + bunzip2_stream.avail_in),
		(BZ2_BUFSIZ - bunzip2_stream.avail_in));
	
	bunzip2_stream.next_in = bunzip2_inbuf;
	bunzip2_stream.avail_in = BZ2_BUFSIZ;
	
	cprintf(".");
	
	return err;
}

/* *** bunzip2_error *** */
STATIC void bunzip2_error (char *x)
{
	cprintf("bunzip2_mod: %s\n", x);
}
