/*
 * gunzip_mod.c -- Decompressing module for Atari bootstrap
 *
 * Copyright (c) 1997 by Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *
 * This file is subject to the terms and conditions of the GNU General Public
 * License.  See the file COPYING in the main directory of this archive
 * for more details.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
//#include <unistd.h>

#include "penguin_prototypes.h"
#include "stream.h"


/*
 * gzip declarations
 */

#define OF(args)		args

#define	GZIP_MAGIC		"\037\213"	/* Magic header for gzip files, 1F 8B */
#define	OLD_GZIP_MAGIC	"\037\236"	/* Magic header for gzip 0.5 = freeze 1.x */

#define memzero(s, n)	memset ((s), 0, (n))

typedef unsigned char	uch;
typedef unsigned short	ush;
typedef unsigned long	ulg;

#define INBUFSIZ		(32*1024)	/* input buffer size; for optimum performance
								 	 * the same or a multiple of other buffer
									 * sizes, and file_mod uses 32k */
#define WSIZE			(32*1024)	/* window size--must be a power of two, and */
									/* at least 32K for zip's deflate method */
#define CHANGING_WINDOW				/* this tell inflate.c that 'window' can
									 * change in flush_window(), and that the
									 * previous window (needed to copy data) is
									 * preserved in 'previous_window' */

/***************************** Prototypes *****************************/

/*static*/ int			gunzip_open( const char *name );
/*static*/ long			gunzip_fillbuf( char *buf );
/*static*/ int			gunzip_close( void );
/*asm*/ /*static*/ int		call_gunzip( void );
/*static*/ void			gzip_mark( void **ptr );
/*static*/ void			gzip_release( void **ptr );
/*static*/ int			fill_inbuf( void );
/*asm*/ /*static*/ void		flush_window( void );
/*static*/ void			error( char *x );
/*static*/ void			return_from_flush(void);
/*static*/ void			return_to_flush(void);

/************************* End of Prototypes **************************/



MODULE gunzip_mod = {
	"gunzip",					/* name */
	WSIZE,						/* max. 512 bytes per TFTP packet */
	gunzip_open,
	gunzip_fillbuf,
	NULL,						/* cannot skip */
	gunzip_close,
	MOD_REST_INIT
};

/* special stack for gunzip() function */
/*static*/ char		*gunzip_stack;

/* some globals for storing values between stack switches */
/*static*/ long		gunzip_sp, main_sp, main_fp;
/*static*/ long		save_StkLowPt;

/* size of stack for gunzip(); 4k should be more than enough. (I guess 2k
 * would suffice also, but I don't want to track down stack overflows...) */
#define GUNZIP_STACK_SIZE	(4*1024)

/*static*/ uch *inbuf;
/*static*/ uch *window;
/*static*/ uch *previous_window;

/*static*/ long insize = 0;  /* valid bytes in inbuf */
/*static*/ long inptr = 0;   /* index of next byte to be processed in inbuf */
/*static*/ long outcnt = 0;  /* bytes in output buffer */
/*static*/ int exit_code = 0;
/*static*/ long bytes_out = 0;

#define get_byte()  (inptr < insize ? inbuf[inptr++] : fill_inbuf())
		
/* Diagnostic functions (stubbed out) */
#define Assert(cond,msg)
#define Trace(x)
#define Tracev(x)
#define Tracevv(x)
#define Tracec(c,x)
#define Tracecv(c,x)

#define STATIC

/*static*/ int gunzip_open( const char *name )
{
	int				rv;
	unsigned char	buf[2];

	gunzip_stack = NULL;
	gunzip_sp = 0;
	main_sp = 0;
	main_fp = 0;

	inbuf = NULL;
	window = NULL;
	previous_window = NULL;

	insize = 0;
	inptr = 0;
	outcnt = 0;
	exit_code = 0;
	bytes_out = 0;

	/* try opening downstream channel */
	if (sopen( name ) < 0)
		return( -1 );
	
	/* check if data is gzipped */
	rv = sread( (char *)buf, 2 );
	sseek( 0, SEEK_SET );
	if (rv < 2) {
		/* definitely not gzip'd, remove from stream */
			//cprintf("Oh no, it's not GZipped\n");
		return( 1 );
	}
    if (buf[0] != 037 || (buf[1] != 0213 && buf[1] != 0236))
		/* not compressed, remove this module from the stream */
			//cprintf("Oh no, it's really not GZipped\n");
		return( 1 );

	if (!(gunzip_stack = malloc( GUNZIP_STACK_SIZE ))) {
		cprintf( "Out of memory for gunzip stack!\n" );
		return( -1 );
	}
	gunzip_sp = 0;

	if (!(inbuf = malloc( INBUFSIZ ))) {
		free(gunzip_stack);
		gunzip_stack = 0;
		cprintf( "Out of memory for gunzip input buffer!\n" );
		return( -1 );
	}

	cprintf( "GUnzipping %s\n", name );
	return( 0 );
}

/*static*/ long gunzip_fillbuf( char *buf )
{
	previous_window = window;
	window = (unsigned char *)buf;
	return( call_gunzip() );
}

/*static*/ int gunzip_close( void )
{
	if (gunzip_stack)
		free( gunzip_stack );
	gunzip_stack = 0;

	if (inbuf)
		free( inbuf );
	inbuf = 0;

	sclose();

	return( 0 );
}

#include "inflate.c"

/*static*/ void gzip_mark( void **ptr )
{
//#pragma unused(ptr)
}

/*static*/ void gzip_release( void **ptr )
{
//#pragma unused(ptr)
}

#if 0
/* MPW chokes on inline asm. see file gunzip_mod_asm.a */

/* This function does most of the magic for implementing a second stack for
 * gunzip(). This is necessary because lib/inflate.c only provides a callback
 * (flush_window()) that can store data away. But we need to actually return
 * from out fillbuf method to implement the streaming. (Otherwise, the whole
 * file would have to be uncompressed and buffered as a whole.)
 *
 * The solution to this problem is a second stack on which gunzip() runs. If
 * flush_window() is called, it saves state and then switches back to the main
 * stack, from which we can return to our caller. For resuming at the point
 * where it left off, we simply restore the gunzip stack again. A second
 * return value in d1 distinguishes returns from inside flush_window() and
 * "normal" returns from gunzip() itself. The latter either indicate EOF or
 * error.
 */
asm static int call_gunzip( void )
{
	FRALLOC

	MOVEM.L	D3-D7/A2-A4,-(SP)

	MOVEQ	#0,D3

	TST.L	gunzip_sp
	BNE.S	sp_inited

	/* gunzip() wasn't called before: set up its stack and
	 * call the main function
	 */

	BSR.L	makecrc

	MOVEM.L	D1-D7/A2-A6,-(SP)
	MOVE.L	SP,main_sp							/* save current sp */
	MOVE.L	A6,main_fp							/* and fp */
	MOVE.L	0x110,save_StkLowPt					/* save Mac stack low ptr */
	CLR.L	0x110								/* disable stack checking */
	MOVE.L	gunzip_stack,D0						/* move to new stack */
	ADDI.L	#GUNZIP_STACK_SIZE-4,D0
	MOVE.L	D0,SP
	BSR.L	gunzip

entry static return_from_flush					/* point of coming back */
	
	MOVE.L	main_sp,SP							/* restore main sp */
	MOVE.L	main_fp,A6							/* and fp */
	MOVE.L	save_StkLowPt,0x110					/* restore stack checking */
	MOVEM.L	(SP)+,D1-D7/A2-A6
	MOVE.L	D0,D3								/* result */
	BRA.S	done


sp_inited:

	/* gunzip() is already active,
	 * jump to where it left off
	 */

	MOVEM.L	D1-D7/A2-A6,-(SP)
	MOVE.L	SP,main_sp							/* save current sp */
	MOVE.L	A6,main_fp							/* and fp */
	MOVE.L	0x110,save_StkLowPt					/* save Mac stack low ptr */
	CLR.L	0x110								/* disable stack checking */
	JMP		return_to_flush						/* go again */

done:
	MOVE.L	D3,D0		/* 0 for EOF, -1 for error, > 0 if from flush_window */
	
	MOVEM.L	(SP)+,D3-D7/A2-A4
	FRFREE
	RTS
}
#endif

/*
 * Fill the input buffer. This is called only when the buffer is empty
 * and at least one byte is really needed.
 */
/*static*/ int fill_inbuf( void )
{
    if (exit_code)
		return( -1 );

    insize = sread( (char *)inbuf, INBUFSIZ );
    if (insize <= 0)
		return( -1 );
	
	cprintf(".");
	
    inptr = 1;
    return( inbuf[0] );
}

#if 0
/* MPW chokes on inline asm. see file gunzip_mod_asm.a */

/*
 * Write the output window window[0..outcnt-1] and update crc and bytes_out.
 * (Used for the decompressed data only.)
 */
asm static void flush_window( void )
{
	FRALLOC

	MOVE.L	outcnt,D1
	ADD.L	D1,bytes_out
	MOVE.L	crc,D2
	MOVEA.L	window,A0
	LEA		crc_32_tab,A1
	BRA.S	check_n
next_n:
	MOVEQ	#0,D0
	MOVE.B	(A0)+,D0
	EOR.L	D2,D0
	ANDI.L	#0x000000FF,D0
	LSL.L	#2,D0
	MOVE.L	0(A1,D0.L),D0
	LSR.L	#8,D2
	EOR.L	D0,D2
check_n:
	SUBQ.L	#1,D1
	BGE.S	next_n
	MOVE.L	D2,crc

	MOVEM.L	D1-D7/A2-A6,-(SP)					/* save call-saved regs */
	MOVE.L	SP,gunzip_sp						/* save current sp */
	MOVE.L	outcnt,D0							/* return value */
	JMP		return_from_flush

entry static return_to_flush					/* point of coming back */
	
	MOVE.L	gunzip_sp,SP						/* restore sp */
	MOVEM.L	(SP)+,D1-D7/A2-A6					/* restore registers */

	CLR.L	outcnt

	FRFREE
	RTS

}
#endif

/*static*/ void error( char *x )
{
    cprintf( "\n%s\n", x);
    exit_code = 1;
}

/* Local Variables: */
/* tab-width: 4     */
/* End:             */
