;;
;; This file is part of q-tools, a collection of performance tools
;; Copyright (c) 2003 Hewlett-Packard Development Company, L.P.
;; Contributed by David Mosberger-Tang <davidm@hpl.hp.com>
;;
;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2 of the License, or
;; (at your option) any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 59 Temple Place, Suite 330,
;; Boston, MA  02111-1307  USA
;;
(use-modules (ice-9 regex))

(load "q-lib.scm")

(define (qview:print-flat-profile sym-tab hist-info
				  weight-key child-weight-key)
  (define (make-flat-profile sym-tab weight-key)
    ;; build a vector of symbols, sorted in order of decreasing
    ;; histogram counts.
    (let* ((v (make-vector (q:hash-length sym-tab)))
	   (sort-keys (list weight-key 'q:call-count)))
      ;; initialize the vector with a copy of the symbol-table:
      (hash-fold (lambda (hash-key value i)
		   (vector-set! v i value) (1+ i))
		 0 sym-tab)
      ;; now sort the vector:
      (stable-sort! v (lambda (left right)
			(> (q:sym-compare left right sort-keys) 0)))
      v))
  (let ((prof (make-flat-profile sym-tab weight-key))
	(sym-name #f)
	(sym #f)
	(weight #f)
	(child-weight #f)
	(calls #f)
	(filename (assq-ref hist-info 'q:file))
	(event-name (assq-ref hist-info 'q:event-name))
	(y-unit (assq-ref hist-info 'q:y-unit-label))
	(y-granule (assq-ref hist-info 'q:y-granularity))
	(total 0)
	(cumulative 0)
	(done #f))
    ;; calculate the histogram total:
    (do ((i 0 (1+ i)))
	((>= i (vector-length prof)))
      (set! total
	    (+ total (or (q:sym-ref (vector-ref prof i) weight-key) 0))))
    (printf "Flat profile %sin %s:\n"
	    (if event-name (string-append "of " event-name " ") "") filename)
    (if (and y-granule y-unit)
	(printf " Each histogram sample counts as %k %s\n" y-granule y-unit))
    (printf
     "%% time      self     cumul     calls self/call  tot/call name\n")
    (do ((i 0 (1+ i)))
	((or done (>= i (vector-length prof))))
      (set! sym-name (q:sym-name (vector-ref prof i)))
      (set! sym (vector-ref prof i))
      (set! weight (or (q:sym-ref sym weight-key) 0))
      (set! child-weight (q:sym-ref sym child-weight-key))
      (set! calls (q:sym-ref sym 'q:call-count))
      (set! cumulative (+ cumulative weight))
      (if (and (= weight 0) (= (or calls 0) 0))
	  (set! done #t)
	  (begin
	    (printf "%6.2f %9.2f %9.2f %s %s %s %s\n"
		    (if (> total 0) (* 100 (/ weight total)) 0)
		    weight
		    cumulative
		    (if calls (sprintf #f "%9.3k" calls)
			"        -")
		    (if calls (sprintf #f "%9.3k"
					    (/ weight calls))
			"        -")
		    (if (and calls child-weight)
			(sprintf #f "%9.3k" (/ (+ weight child-weight) calls))
			"        -")
		    sym-name))))
    (printf "\n")))

(define (qview:print-call-graph graph weight-key child-weight-key)
  (define (cg-name sym)
    ;; Return the call-graph name for symbol SYM.  This is normally
    ;; the symbol name followed by the topological order number in
    ;; square brackets.
    (string-append (q:sym-name sym)
		   (sprintf #f " [%u]" (q:sym-ref sym 'q:topo-order))))
  (let ((total-weight 0)
	(sorted-graph
	 (stable-sort graph
		      (lambda (l r)
			(let ((lw (+ (or (q:sym-ref l weight-key) 0)
				     (or (q:sym-ref l child-weight-key) 0)))
			      (rw (+ (or (q:sym-ref r weight-key) 0)
				     (or (q:sym-ref r child-weight-key) 0))))
			  (> lw rw))))))
    (printf "Call-graph table:\n")
    (printf "index %%time      self  children         called     name\n")
    ;; calculate the sum of all weights in the graph:
    (for-each (lambda (node)
		(let ((weight (q:sym-ref node weight-key)))
		  (if weight (set! total-weight (+ total-weight weight)))))
	      sorted-graph)
    (for-each
     (lambda (node)
       (let ((order (q:sym-ref node 'q:topo-order))
	     (in-edges (q:sym-ref node 'q:callers))
	     (out-edges (q:sym-ref node 'q:callees))
	     (self-count (q:sym-ref node 'q:call-count))
	     (self-weight (q:sym-ref node weight-key))
	     (child-weight (q:sym-ref node child-weight-key)))
	 ;; print info about parents:
	 (if in-edges
	     (for-each
	      (lambda (edge)
		(let ((caller (car edge))
		      (count (cdr edge))
		      (frac 0))
		  (if (and count self-count)
		      (set! frac (/ count self-count))
		      (set! frac 1))
		  (printf
		   "            %s %s %9.3k              %s\n"
		   (if self-weight
		       (sprintf #f "%9.3k" (* frac self-weight))
		       "        -")
		   (if child-weight
		       (sprintf #f "%9.3k" (* frac child-weight))
		       "        -")
		   count (cg-name caller))))
	      in-edges)
	     (printf "%68s\n" "<spontaneous>"))
	 ;; print info about this node itself:
	 (printf "%-5s %5.1f %s %s %s          %s\n"
		 (sprintf #f "[%u]" order)
		 (if (> total-weight 0)
		     (* 100 (/ (+ (or self-weight 0) (or child-weight 0))
			       total-weight))
		     0)
		 (if self-weight
		     (sprintf #f "%9.3k" self-weight)
		     "        -")
		 (if child-weight
		     (sprintf #f "%9.3k" child-weight)
		     "        -")
		 (if self-count
		     (sprintf #f "%9.3k" self-count)
		     "        -")
		 (q:sym-name node))
	 ;; print info about children:
	 (if out-edges
	     (for-each (lambda (edge)
			 (let ((callee (car edge))
			       (count (cdr edge))
			       (callee-calls #f)
			       (callee-weight #f)
			       (callee-child-weight #f)
			       (frac 0))
			   (set! callee-calls
				 (q:sym-ref callee 'q:call-count))
			   (set! callee-weight (q:sym-ref callee weight-key))
			   (set! callee-child-weight
				 (q:sym-ref callee child-weight-key))
			   (if (and count callee-calls)
			       (set! frac (/ count callee-calls))
			       (set! frac 1))
			   (printf "            %s %s %9.3k%s    %s\n"
				   (if callee-weight
				       (sprintf #f "%9.3k"
						(* frac callee-weight))
				       "        -")
				   (if callee-child-weight
				       (sprintf #f "%9.3k"
						(* frac callee-child-weight))
				       "        -")
				   count
				   (if callee-calls
				       (sprintf #f "/%-9.3k" callee-calls)
				       "         ")
				   (cg-name callee))))
		       out-edges)))
       (printf "----------------------------------------------------\n"))
     sorted-graph)))

(define qview:call-graph-help-text "
 Legend:

  Numbers:
    m: milli (10e-3) u: micro (10e-6) n: nano (10e-6)
    k: kilo  (10e3)  M: mega  (10e6)  G: giga (10e9)

  Flat profile columns:

    %%time  The percentage of the total running time spent in this function.

    self   The time accounted for by this function alone.  The profile is
           sorted in decreasing order of this time.

    cumul  A running sum of the time accounted for by this function and
           those listed above it.

    calls  The number of times this function was invoked.  If unknown,
           a dash (-) is printed instead.

    self/call
           The average time spent in this function per call.  If the number
           of calls is unknown, a dash (-) is printed instead.

    total/call
           The average time spent in this function and its children
           per call.  If unknown, a dash (-) is printed instead.

    name
           The name of the function.

  Call-graph table:

   This table describes the call-tree of the program.  Each entry in this
   table consists of several lines.  The line with the index number at
   the left hand margin lists the current function.  The lines above it
   list the functions that called this function, and the lines below it
   list the functions this one called.

   For line for the current function lists:

    index  A unique number given to each element of the table.  The index
           numbers are assigned such that if A calls B, then the index
           of B is bigger than that of A (unless A and B are part of a
           cycle, due to recursion).

    %%time  The percentage of the total time that was spent in this function
           and its children.

    self   The total amount of time spent in this function.

    children
           The total amount of time propagated into this function by its
           children.

    called The number of times the function was called.

    name   The name of the function.  The index number is printed after it.

   For the function's parents, the columns have the following meanings:

    self   The amount of time that was propagated directly from the
           function into this parent.

    children
           The amount of time that was propagated from the function's
           children into this parent.

    called The number of times this parent called the function.

    name   The name of the parent.  The parent's index number is printed
           after it.

   If the parents of the function cannot be determined, the word
   `<spontaneous>' is printed in the `name' column and all other columns
   are blank.

   For the function's children, the columns have the following meaning:

    self   The amount of time that was propagated directly from the child
           into the function.

    children
           The amount of time that was propagated from the child's
           children into the function.

    called The number of times the function called this child `/' the total
           number of times the child was called.

    name   The name of the child.  The child's index number is printed
	   after it.
")

(define (qview:call-graph->dot port graph title weight-key child-weight-key
			       threshold grayscale)
  "Translate the call-graph GRAPH into a dot language (as used by the
graphviz toolkit).  The output is printed to PORT and the resulting
graph is named TITLE.  WEIGHT-KEY is the symbol under which we find
the weight associated with a node (symbol) in the graph.
CHILD-WEIGHT-KEY is the symbol under which we find the weight
associated with the children of a node (symbol).  THRESHOLD is the
minimum total weight (self + child weight) that the node must have for
it to be included in the graph.  If GRAYSCALE is #t, the output uses
only shades of gray instead of a false-color image to represent the
weight/heat of nodes and edges."
  (define max-weight 0)
  (define max-call-count 0)
  (define color (if grayscale
		    #("#d0d0d0" "#c0c0c0" "#b0b0b0" "#a0a0a0"
		      "#909090" "#808080" "#707070" "#606060"
		      "#505050" "#404040" "#303030" "#202020")
 		    #("navy" "blue1" "cornflowerblue" "turquoise1"
 		      "green" "greenyellow" "yellow1" "gold1"
 		      "orangered1" "red1" "red2" "red3")))
  (define txt-color (if grayscale
			#("black" "black" "black" "black"
			  "black" "black" "white" "white"
			  "white" "white" "white" "white")
			#("white" "white" "white" "black"
			  "black" "black" "black" "black"
			  "black" "black" "black" "black")))
  (define line-width #(1 1 1 1 1 1 1 1.5 2 2.5 3 3.5 4))
  (define (percent->index pcnt)
    (if (< pcnt 1)
	0 (inexact->exact
	   (truncate (* (log pcnt)
			(/ (1- (vector-length color)) (log 100)))))))
  (define (node-attributes node weight-percentage)
    (let ((i (percent->index weight-percentage)))
      (if (q:call-graph-root? node)
	  (if grayscale
	      (sprintf #f "fillcolor=white, fontcolor=black")
	      (sprintf #f "fillcolor=lightgray, fontcolor=black"))
	  (sprintf #f "fillcolor=\"%s\", fontcolor=\"%s\""
		   (vector-ref color i) (vector-ref txt-color i)))))
  (define (edge-attributes call-percentage tot-weight)
    (let ((i (percent->index call-percentage)))
      (sprintf #f
  "color=\"%s\", label=\"%.1f%%\" style=\"setlinewidth(%g)\" weight=\"%.1f\""
	       (vector-ref color i) call-percentage
	       (vector-ref line-width i) tot-weight)))
  (define (emit-node node weight)
    (let* ((name (q:sym-name node))
	   (pcnt (* 100 (/ weight max-weight))))
      (fprintf port "\t%s [ label=\"%s\\\\n%.3k\", %s ]\n"
	       name name weight (node-attributes node pcnt))))
  (define (emit-edge node tot-weight edge)
    (let* ((child (car edge))
	   (calls (cdr edge))
	   (parent-name (q:sym-name node))
	   (child-name (q:sym-name child))
	   (parent-order (q:sym-ref node 'q:topo-order))
	   (child-order (q:sym-ref child 'q:topo-order))
	   ;; even though dot in theory only allows for DAGs, it seems to be
	   ;; OK with self-recursion and in fact you get the wrong-looking
	   ;; output if self-recursion is treated as a backedge, hence, we
	   ;; check with < instead of <=
	   (backedge (< child-order parent-order)))
      (fprintf port "\t%s -> %s [%s%s]\n"
	       (if backedge child-name parent-name)
	       (if backedge parent-name child-name)
	       (if backedge "dir=back, " "")
	       (edge-attributes (* 100 (/ calls max-call-count)) tot-weight))))

  (for-each
   (lambda (node)
     (let ((weight (or (q:sym-ref node weight-key) 0))
	   (edges (or (q:sym-ref node 'q:callees) '())))
       (set! max-weight (max max-weight weight))
       (for-each (lambda (edge)
		   (set! max-call-count (max max-call-count (cdr edge))))
		 edges)))
   graph)
  (fprintf port "/* Format this graph, e.g., with:\n")
  (fprintf port "     dot -Tps -Gcenter,rotate=90,size=\"10,7.5\" */\n")
  (fprintf port "digraph \"%s\" {\n" title)
  (fprintf port "\tnode [ fontname=Helvetica shape=ellipse, style=filled ]\n")
  (fprintf port "\tedge [ fontname=Helvetica ]\n")
  (for-each
   (lambda (node)
     (let* ((edges (or (q:sym-ref node 'q:callees) '()))
            (self-weight (or (q:sym-ref node weight-key) 0))
            (child-weight (or (q:sym-ref node child-weight-key) 0))
            (tot-weight (+ self-weight child-weight)))
       ;; XXX: This simply drops below-threshold nodes and all edges
       ;; from that node.  We may want to do something smarter with
       ;; the dangling edges though...
       (if (>= tot-weight threshold)
	   (begin (emit-node node self-weight)
		  (for-each (lambda (edge) (emit-edge node tot-weight edge))
			    edges)))))
   graph)
  (fprintf port "}\n"))


;;
;; Callback routines for qview:process-file:
;;

(define (qview:print-profiles sym-tab addr-map call-graph)
  "Print all profiles.  If multiple histograms were collected, one
histogram will be printed per event-type collected.  Histograms for
the same event-type are combined into a single flat profile."
  (for-each
   (lambda (aux-info)
     (let ((argv (assq-ref aux-info 'q:cmdline))
	   (line ""))
       (if argv
	   (begin
	     (for-each (lambda (arg)
			 (set! line (string-append line " " arg)))
		       argv)
	     (printf "Command:%s\n" line)))))
   q:info-list)
  (let ((keys-done '()))
    (for-each
     (lambda (hist-info)
       (let ((key (q:hist->weight-key hist-info))
	     (child-key (q:hist->child-weight-key hist-info)))
	 (if (not (memq key keys-done))
	     (begin (set! keys-done (append keys-done (list key)))
		    (qview:print-flat-profile sym-tab hist-info key child-key)
		    (qview:print-call-graph call-graph key child-key)))))
     q:histogram-list))
  (begin (printf qview:call-graph-help-text)))

(define (qview:write-dot-files sym-tab addr-map call-graph port threshold
			       grayscale)
  (for-each
   (lambda (hist-info)
     (qview:call-graph->dot port call-graph (assq-ref hist-info 'q:file)
			    (q:hist->weight-key hist-info)
			    (q:hist->child-weight-key hist-info)
			    threshold grayscale))
   q:histogram-list))


;;
;; Process one profile (info) file:
;;

(define (qview:process-file filename proc . extra-proc-args)
  "Process file FILENAME by reading its contents and building a
symbol-table, address-map, flat profile, and a call-graph.  Once these
data-structures are built, call PROC with arguments (SYM-TAB ADDR-MAP
CALL-GRAPH).  SYM-TAB is a hash-table of symbols, ADDR-MAP a vector of
symbols sorted by address, and CALL-GRAPH a topologically sorted list
of call-graph nodes (symbols)."
  ;; load up/evaluate the data file:
  (let ((port (q:open-file filename "r"))
	(sym-tab (make-hash-table 100000))
	(unaccounted-sym (q:sym "<unaccounted>" '((q:value . 0))))
	(call-graph #f)
	(addr-map #f))
;;    (printf "Reading input...\n")
    (do ((expr (read port) (read port)))
	((eof-object? expr))
      (catch #t
	     (lambda ()
	       (primitive-eval expr))
	     (lambda (key . args)
	       (fprintf (current-error-port)
			"warning: ignoring unknown expression `")
	       (write expr (current-error-port))
	       (fprintf (current-error-port) "'\n"))))

    ;; build the symbol table:
;;    (printf "Building symbol table...\n")
    (for-each
     (lambda (map-file-info)
       (let ((filename (car map-file-info))
	     (postfix (cdr map-file-info)))
	 (if (not postfix)
	     (set! postfix (if (= (q:hash-length q:object-hash) 0)
			       "" "<kernel>")))
	 (q:set-kallsyms! sym-tab (q:open-file filename "r") postfix)))
     q:map-file-list)
    (hash-fold (lambda (key value result)
		 (q:set-object-syms! sym-tab value "FUNC")
		 #f)
	       #f
	       q:object-hash)

    ;; build an address-map from the completed symbol-table:
;;    (printf "Building address map...\n")
    (set! addr-map (q:addr-map sym-tab unaccounted-sym))

    ;; assign call-counts to the symbols:
;;    (printf "Assigning call-counts...\n")
    (for-each
     (lambda (call-info)
       (let ((filename (assq-ref call-info 'q:file)))
	 (q:assign-call-counts! sym-tab
				 (q:open-file filename "r") call-info
				 addr-map)))
     q:call-count-list)

    ;; build the call-graph:
;;    (printf "Building call-graph...\n")
    (set! call-graph (q:call-graph-dag sym-tab))

    ;; assign weight from the histograms and distribute it according to the
    ;; call-graph:
;;    (printf "Assigning histogram weights...\n")
    (for-each
     (lambda (hist-info)
       (let ((filename (assq-ref hist-info 'q:file)))
	 (q:hist-assign-weights! sym-tab (q:open-file filename "r")
				  hist-info addr-map)
	 (q:propagate-child-weights! call-graph
				      (q:hist->weight-key hist-info)
				      (q:hist->child-weight-key hist-info))))
     q:histogram-list)
;;    (printf "Doing output-processing...\n")
    (apply proc (append (list sym-tab addr-map call-graph) extra-proc-args))))
