/*
 * Copyright (c) 2002, 2003 Red Hat, Inc. All rights reserved.
 *
 * This software may be freely redistributed under the terms of the
 * GNU General Public License.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Liam Stewart
 * Component of: Visual Explain GUI tool for PostgreSQL - Red Hat Edition
 */

package com.redhat.rhdb.treedisplay;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.plaf.*;
import javax.swing.tree.*;
import javax.swing.event.*;

/**
 * The TreeDisplay class is a custom Swing component that provides a
 * nice graphical view of trees. It is similar in many ways to the
 * standard Swing tree component, JTree, and can be used as a drop in
 * replacement for JTree in most cases as TreeDisplay has many of the
 * same methods and uses a TreeModel for its model. However,
 * TreeDisplay does not implement all of JTree's methods. Noticeably
 * absent are the methods that deal with rows - TreeDisplay does not
 * deal with rows like JTree does.
 * <p>
 * The view that TreeDisplay objects provide is static. That is, the
 * tree is displayed but interaction with it is not possible beyond
 * simple selection and deselection of nodes. Nodes cannot be moved
 * around and trees cannot be expanded and collapsed as they can with
 * JTree. Multiple selection is also not possible.
 * <p>
 * TreeDisplay does allow some degree of customization. One can zoom
 * in and out, zoom to fit the current tree in the window, zoom to fit
 * the width of the current tree in the window, set the drawing to be
 * anti-aliased, set the anchoring of the tree, set the position of
 * the root node. One can also customize the way the tree is laid out
 * and how it is rendered in an easy manner.
 * <p>
 * TreeDisplay also provides a popup "overview" window that, when
 * open, displays a birds-eye view of the current tree.
 * <p>
 * The class makes use of four other classes:
 * <ul>
 * <li>{@link javax.swing.tree.TreeModel TreeModel} - the tree to display
 * <li>{@link javax.swing.tree.TreeSelectionModel TreeSelectionModel} - the model used for selections
 * <li>{@link TreeLayoutModel} - handles laying out the tree
 * <li>{@link TreeRenderer} - draws the nodes and edges
 * </ul>
 * <p>
 * TreeDisplay objects generate events when certain things occur. See
 * {@link TreeDisplayEvent}. Objects wishing to be notified when
 * events occurs should be of a class that implements the {@link
 * TreeDisplayListener} interface and should register themselves with
 * TreeDisplay objects using the {@link #addTreeDisplayListener}
 * method.
 * <p>
 * <h4>Current Known Limitations</h4>
 * <ul>
 * <li>Setting the root position is not implemented.
 * <li>Large icon support is not implemented.
 * <li>Modes (Select, Pan, Zoom) are not implemented.
 * <li>Drawing is very, very slow, especially for large trees.
 * </ul>
 *
 * @author <a href="mailto:liams@redhat.com">Liam Stewart</a>
 * @version 0.0
 * @see javax.swing.JTree
 */

public class TreeDisplay extends JComponent implements Scrollable {
	private static String uiclassid = "TreeDisplayUI";
	private static String uiclassname = "com.redhat.rhdb.treedisplay.BasicTreeDisplayUI";

	private Point zoompoint;

	private TreeModel model;
	private TreeModelListener tdtml;
	private TreeSelectionModel selmodel;
	private TreeSelectionListener tdtsl;
	private TreeRenderer renderer;
	private TreeLayoutModel tlmodel;
	private EventListenerList listeners;

	// properties
	private boolean antialias;
	private double zoom;
	private int pad_x, pad_y;
	private int anchor;
	private int mode;
	private int zoommode;

	// "helper" properties
	private int translate_x, translate_y;

	// for the Scrollable interface
	private int blockincr = 100;
	private int unitincr = 20;

	// properties

	/** Root position property */
	public static final String ORIENTATION_PROPERTY = "orientation";

	/** Anti-alias property */
	public static final String ANTI_ALIAS_PROPERTY = "antiAlias";

	/** Mode property */
	public static final String MODE_PROPERTY = "mode";

	/** Left/right padding property */
	public static final String PAD_X_PROPERTY = "padX";

	/** Top/bottom padding property */
	public static final String PAD_Y_PROPERTY = "padY";

	/** Anchor property */
	public static final String ANCHOR_PROPERTY = "anchor";

	/** Zoom level property */
	public static final String ZOOM_LEVEL_PROPERTY = "zoomLevel";

	/** Zoom mode property */
	public static final String ZOOM_MODE_PROPERTY = "zoomMode";

	/** Selection model property */
	public static final String SELECTION_MODEL_PROPERTY = "selectionModel";

	/** Tree model property */
	public static final String TREE_MODEL_PROPERTY = "treeModel";

	/** Tree layout model property */
	public static final String TREE_LAYOUT_MODEL_PROPERTY = "treeLayoutModel";

	/** Tree renderer property */
	public static final String TREE_RENDERER_PROPERTY = "treeRenderer";

	// orientations

	/** Position the root node at the top of the drawing area. */
	public static final int TOP = 0;

	/** Position the root node at the bottom of the drawing area. */
	public static final int BOTTOM = 1;

	/** Position the root node at the right of the drawing area. */
	public static final int RIGHT = 2;

	/** Position the root node at the left of the drawing area. */
	public static final int LEFT = 3;

	// achoring

	/** Anchor the tree at the top of the drawing area. */
	public static final int NORTH = 0;

	/** Anchor the tree at the bottom of the drawing area. */
	public static final int SOUTH = 1;

	/** Anchor the tree at the right (east) side of the drawing area. */
	public static final int EAST = 2;

	/** Anchor the tree at the left (west) side of the drawing area. */
	public static final int WEST = 3;

	/** Anchor the tree in the north-east corner of the drawing area. */
	public static final int NORTHEAST = 4;

	/** Anchor the tree in the north-west corner of the drawing area. */
	public static final int NORTHWEST = 5;

	/** Anchor the tree in the south-east corner of the drawing area. */
	public static final int SOUTHEAST = 6;

	/** Anchor the tree in the south-west corner of the drawing area. */
	public static final int SOUTHWEST = 7;

	/** Anchor the tree in the middle of the drawing area. */
	public static final int CENTER = 8;

	// modes

	/** Select mode. */
	public static final int SELECT = 9;

	/** Pan mode. */
	public static final int PAN = 10;

	/** Zoom in mode. */
	public static final int ZOOMIN = 11;

	/** Zoom out mode. */
	public static final int ZOOMOUT = 12;

	// zoom modes

	/** When set, zoom is set to a constant, user-defined level */
	public static final int NO_ZOOM_MODE = -1;

	/** Zoom set at 100% */
	public static final int ACTUAL_SIZE = 20;

	/** Zoom so that the width of the current tree fits in the viewable area. */
	public static final int FIT_WIDTH = 21;

	/** Zoom so that the entire tree fits in the viewable area. */
	public static final int FIT_WINDOW = 22;
	
	//
	// static initializer
	//
	
	static
	{
		UIManager.put(uiclassid, uiclassname);
	}

	//
	// constructors
	//
	
	/**
	 * Creates a new <code>TreeDisplay</code> instance.
	 *
	 */
	public TreeDisplay()
	{
		this(getDefaultTreeModel());
	}

	/**
	 * Creates a new <code>TreeDisplay</code> instance that views the
	 * given TreeModel.
	 *
	 * @param tm a <code>TreeModel</code> value
	 */
	public TreeDisplay(TreeModel tm)
	{
		renderer = new DefaultTreeRenderer();
		
		model = tm;
		tdtml = new TreeDisplayTreeModelListener();
		tlmodel = new DefaultTreeLayoutModel(model);

		model.addTreeModelListener(tlmodel);
		tlmodel.addTreeModelListener(tdtml);
		
		selmodel = new DefaultTreeSelectionModel();
		tdtsl = new TreeDisplayTreeSelectionListener();
		selmodel.addTreeSelectionListener(tdtsl);
		selmodel.setSelectionMode(TreeSelectionModel.SINGLE_TREE_SELECTION);

		listeners = new EventListenerList();
		
		// set various properties to nice defaults
		antialias = false;
		zoommode = ACTUAL_SIZE;
		zoom = 1.0;
		pad_x = 50;
		pad_y = 50;
		anchor = NORTH;
		mode = SELECT;

		try {
			tlmodel.setOrientation(TOP);
		} catch (TreeDisplayException ex) { }

		// we are opaque
		setOpaque(true);
		
		// turn on tool tips
		setToolTipText("");

		// misc things
		setBackground(Color.white);
		setForeground(Color.black);

		// draw
		updateUI();
	}

	//
	// utility methods
	//

	/**
	 * Is the given point (x, y) contained within the bounds of the
	 * tree?
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>boolean</code> value. true if the point is
	 * contained in the bounds, false otherwise.
	 */
	public boolean contains(int x, int y)
	{
		return contains(new Point(x, y));
	}

	/**
	 * Is the given Point p contained in the bounds of the tree?
	 *
	 * @param p a <code>Point</code> value
	 * @return a <code>boolean</code> value. true if the point is
	 * contained in the bounds, false otherwise.
	 */	 
	public boolean contains(Point p)
	{
		Point q = translatePoint(p);
		String text = null;
		
		if ((q.x < 0 || q.x > getRawWidth() - 2*getPadX()) ||
			(q.y < 0 || q.y > getRawHeight() - 2*getPadY()))
			return false;

		return true;
	}

	/**
	 * Get the tool tip text for a specific location.
	 *
	 * @param e a <code>MouseEvent</code> value
	 * @return a <code>String</code> value
	 */
	public String getToolTipText(MouseEvent e)
	{
		String text = null;
		TreeLayoutNode node = getNode(e.getPoint());

		if (node != null)
			text = node.getData().toString();
		return text;
	}

	/**
	 * Get node that contains given point.
	 *
	 * @param x X coordinate
	 * @param y Y coordinate
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(int x, int y)
	{
		return getNode(new Point(x, y));
	}

	/**
	 * Get node that contains given point.
	 *
	 * @param p a <code>Point</code> value
	 * @return a <code>TreeLayoutNode</code> value
	 */
	public TreeLayoutNode getNode(Point p)
	{
		return tlmodel.getNode(translatePoint(p));
	}

	/**
	 * Get the x translation.
	 *
	 * @return an <code>int</code> value
	 */
	public int getTranslateX()
	{
		return translate_x;
	}
	
	/**
	 * Get the y translation.
	 *
	 * @return an <code>int</code> value
	 */
	public int getTranslateY()
	{
		return translate_y;
	}

	/**
	 * Center view on selected node. If no selected node, center on
	 * middle of viewport.
	 */
	public void centerOnSelected()
	{
		if (! (getParent() instanceof JViewport))
			return;
		JViewport vp = (JViewport) getParent();
		Rectangle vpr = vp.getViewRect();

		Point q = new Point();
		int nw = 0, nh = 0;

		TreePath tp = getSelectionPath();
		if (tp != null)
		{
			centerOnNode(tp);
		}
		else if (zoompoint != null)
		{
			centerOnPoint(zoompoint);
			zoompoint = null;
		}
	}

	/**
	 * Center view on root of tree.
	 */
	public void centerOnRoot()
	{
		centerOnNode(tlmodel.getRoot());
	}

	/**
	 * Center view on node represented by given TreePath.
	 *
	 * @param tp a <code>TreePath</code> value
	 */
	public void centerOnNode(TreePath tp)
	{
		if (tp != null)
		{
			Object d = tp.getLastPathComponent();
			TreeLayoutNode n = tlmodel.getNode(d);
			centerOnNode(n);
		}
	}

	/**
	 * Center view on given point (in tree layout coordinates).
	 *
	 * @param p a <code>Point</code> value
	 */
	public void centerOnPoint(Point p)
	{
		if (p == null)
			return;

		centerOnRectangle(new Rectangle(p.x, p.y, 1, 1));
	}

	/**
	 * Center view on given point (in screen coordinates).
	 *
	 * @param p a <code>Point</code> value
	 */
	public void centerOnScreenPoint(Point p)
	{
		if (p == null)
			return;

		centerOnPoint(translatePoint(p));
	}

	/**
	 * Center view on given rectangle (in tree layout coordinates and
	 * sizes).
	 *
	 * @param p a <code>Rectangle</code> value
	 */
	public void centerOnRectangle(Rectangle r)
	{
		if (r == null)
			return;
		if (! (getParent() instanceof JViewport))
			return;

		JViewport vp = (JViewport) getParent();
		Rectangle vpr = vp.getViewRect();

		Point q = new Point();
		int nw = 0, nh = 0;

		q = translatePointToScreen(new Point(r.x, r.y));
		nw = (int) (r.width * getZoomLevel());
		nh = (int) (r.height * getZoomLevel());

		int vx = q.x - (int)((vpr.width - nw) / 2.0);
		if (vx < 0) vx = 0;
		if (vx + vpr.width > getWidth()) vx = getWidth() - vpr.width;
		
		int vy = q.y - (int)((vpr.height - nh) / 2.0);
		if (vy < 0) vy = 0;
		if (vy + vpr.height > getHeight()) vy = getHeight() - vpr.height;

		vp.setViewPosition(new Point(vx, vy));
	}

	/**
	 * Center view on given rectangle (in screen coordinates and
	 * sizes).
	 *
	 * @param p a <code>Rectangle</code> value
	 */
	public void centerOnScreenRectangle(Rectangle r)
	{
		if (r == null)
			return;

		Point q = translatePoint(new Point(r.x, r.y));
		Rectangle s = new Rectangle(q.x,
									q.y,
									(int)(r.width / getZoomLevel()),
									(int)(r.height / getZoomLevel()));

		centerOnRectangle(s);
	}

	//
	// property methods (get/set)
	//
	
	/**
	 * Get the mode that the TreeDisplay object is currently in.
	 *
	 * @return an <code>int</code> value
	 */
	public int getMode()
	{
		return mode;
	}

	/**
	 * Set the TreeDisplay's mode.
	 *
	 * @param i an <code>int</code> value
	 */
	public void setMode(int i)
	{
		int old = mode;

		if (i < 9 || i > 12)
			throw new IllegalArgumentException("Mode must be an integer between 9 and 12");

		mode = i;

		firePropertyChange(MODE_PROPERTY, old, mode);
	}
	
	/**
	 * Get the amount of padding that is used on the right and left
	 * sides of the tree.
	 *
	 * @return an <code>int</code> value
	 */
	public int getPadX()
	{
		return pad_x;
	}

	/**
	 * Set the amount of padding that will be used to pad the right
	 * and left sides of the tree.
	 *
	 * @param i an <code>int</code> value
	 */
	public void setPadX(int i)
	{
		int old = pad_x;
		if (i < 0)
			throw new IllegalArgumentException("X padding must be non-negative");
		pad_x = i;
		firePropertyChange(PAD_X_PROPERTY, old, pad_x);
		resizeAndRepaint();
	}

	/**
	 * Get the amount of padding that is used on the top and bottom
	 * sides of the tree.
	 *
	 * @return an <code>int</code> value
	 */
	public int getPadY()
	{
		return pad_y;
	}

	/**
	 * Set the amount of padding that will be used to pad the top
	 * and bottom of the tree.
	 *
	 * @param i an <code>int</code> value
	 */
	public void setPadY(int i)
	{
		int old = pad_y;
		if (i < 0)
			throw new IllegalArgumentException("Y padding must be non-negative");
		this.pad_y = i;
		firePropertyChange(PAD_Y_PROPERTY, old, pad_y);
		resizeAndRepaint();
	}

	/**
	 * Get the current anchoring.
	 *
	 * @return an <code>int</code> value
	 */
	public int getAnchor()
	{
		return anchor;
	}

	/**
	 * Set the anchoring of the tree.
	 *
	 * @param i an <code>int</code> value
	 */
	public void setAnchor(int i)
	{
		int old = anchor;
		if (i < 0 || i > 8)
			throw new IllegalArgumentException("Anchor value must be between 0 and 8");
		anchor = i;
		firePropertyChange(ANCHOR_PROPERTY, old, anchor);
		resizeAndRepaint();
	}
	
	/**
	 * Is anti-aliasing being done when drawing the tree?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isAntiAlias()
	{
		return antialias;
	}

	/**
	 * Turn on/off anti-aliasing.
	 *
	 * @param b a <code>boolean</code> value
	 */
	public void setAntiAlias(boolean b)
	{
		boolean old = antialias;
		antialias = b;
		firePropertyChange(ANTI_ALIAS_PROPERTY, old, antialias);
		resizeAndRepaint();
	}

	/**
	 * Gets the current zoom level.
	 *
	 * @return a <code>double</code> value
	 */
	public double getZoomLevel()
	{
		return zoom;
	}

	/**
	 * Sets the zoom level.
	 * <p>
	 * 1 is normal, 2 is double size, 0.5 is
	 * half-size etc. If the component is in a zoom mode other than
	 * NO_ZOOM_MODE, the zoom mode will be set to NO_ZOOM_MODE.
	 *
	 * @param l a <code>double</code> value
	 */
	public void setZoomLevel(double l)
	{
		double old = zoom;
		if (l <= 0)
			throw new IllegalArgumentException("Zoom level must be greater than 0.");
		createZoomPoint();
		zoom = l;
		setZoomMode(NO_ZOOM_MODE);
		firePropertyChange(ZOOM_LEVEL_PROPERTY, old, zoom);
		notifyTreeDisplayListeners(new TreeDisplayEvent(this, TreeDisplayEvent.ZOOM_PERFORMED));
	}

	/**
	 * Gets the current zoom mode.
	 *
	 * @return a <code>integer</code> value
	 */
	public int getZoomMode()
	{
		return zoommode;
	}

	/**
	 * Sets the current zoom mode. Currently, the following modes are
	 * supported:
	 *
	 *  ACTUAL_SIZE: Keep zoom at 100%
	 *  FIT_WIDTH: Zoom so that it fits width-wise in its window
	 *  FIT_WINDOW: Zoom so that the entire tree fits in its window
	 *  NO_ZOOM_MODE: Zoom is at an arbitrary value.
	 * @param i an <code>int</code> value
	 */
	public void setZoomMode(int i)
	{
		int old = zoommode;

		if (zoompoint == null)
			createZoomPoint();

		switch (i)
		{
			case NO_ZOOM_MODE:
			case ACTUAL_SIZE:
			case FIT_WIDTH:
			case FIT_WINDOW:
				zoommode = i;
				break;
			default:
				zoommode = NO_ZOOM_MODE;
		}

		firePropertyChange(ZOOM_MODE_PROPERTY, old, zoommode);
		private_layout();
		centerOnSelected();
		repaint();
	}

	/**
	 * Get the current root position.
	 *
	 * @return an <code>int</code> value
	 */
	public int getOrientation()
	{
		return tlmodel.getOrientation();
	}

	/**
	 * Set the root position.
	 *
	 * @param i an <code>int</code> value
	 */
	public void setOrientation(int i) throws TreeDisplayException
	{
		int old = tlmodel.getOrientation();

		if (i < 0 || i > 3)
			throw new IllegalArgumentException("Orientation must be between 0 and 3");

		if (tlmodel.supportsOrientation(i))
		{
			tlmodel.setOrientation(i);
			firePropertyChange(ORIENTATION_PROPERTY, old, i);
			private_layout();
			centerOnSelected();
			repaint();
		}
		else
			throw new TreeDisplayException("Unable to set orientation " + i);
	}

	/**
	 * Get the current TreeRenderer used for drawing nodes and edges.
	 *
	 * @return a <code>TreeRenderer</code> value
	 */
	public TreeRenderer getTreeRenderer()
	{
		return renderer;
	}

	/**
	 * Set a new TreeRenderer.
	 *
	 * @param r a <code>TreeRenderer</code> value
	 */
	public void setTreeRenderer(TreeRenderer r)
	{
		TreeRenderer old = renderer;

		if (r == null)
			renderer = new DefaultTreeRenderer();
		else
			renderer = r;

		firePropertyChange(TREE_RENDERER_PROPERTY, old, renderer);
		
		resizeAndRepaint();
	}
	
	/**
	 * Get the current TreeModel.
	 *
	 * @return a <code>TreeModel</code> value
	 */
	public TreeModel getModel()
	{
		return model;
	}

	/**
	 * Set a new TreeModel.
	 *
	 * @param tm a <code>TreeModel</code> value
	 */
	public void setModel(TreeModel tm)
	{
		TreeModel old = model;

		if (model != null && tlmodel != null)
			model.removeTreeModelListener(tlmodel);

		if (tm == null)
			model = getDefaultTreeModel();
		else
			model = tm;

		clearSelection();
		tlmodel.setModel(model);
		model.addTreeModelListener(tlmodel);

		firePropertyChange(TREE_MODEL_PROPERTY, old, model);

		private_layout();
		centerOnRoot();
		repaint();
	}

	/**
	 * Get the current TreeLayoutModel.
	 *
	 * @return a <code>TreeLayoutModel</code> value
	 */
	public TreeLayoutModel getTreeLayoutModel()
	{
		return tlmodel;
	}

	/**
	 * Set a new TreeLayoutModel.
	 *
	 * @param m a <code>TreeLayoutModel</code> value
	 */
	public void setTreeLayoutModel(TreeLayoutModel m)
	{
		TreeLayoutModel old = tlmodel;

		if (m == null)
			tlmodel = new DefaultTreeLayoutModel();
		else
			tlmodel = m;

		// FIXME?
		model.addTreeModelListener(tlmodel);
		tlmodel.addTreeModelListener(tdtml);
		tlmodel.setModel(model);

		firePropertyChange(TREE_LAYOUT_MODEL_PROPERTY, old, tlmodel);
	}
	
	/**
	 * Get the current TreeSelectionModel.
	 *
	 * @return a <code>TreeSelectionModel</code> value
	 */
	public TreeSelectionModel getSelectionModel()
	{
		return selmodel;
	}

	/**
	 * Set a new TreeSelectionModel.
	 *
	 * @param tsm a <code>TreeSelectionModel</code> value
	 */
	public void setSelectionModel(TreeSelectionModel tsm)
	{
		TreeSelectionModel old = tsm;

		if (tsm == null)
			selmodel = new DefaultTreeSelectionModel();
		else
			selmodel = tsm;

		addTreeSelectionListener(tdtsl);

		firePropertyChange(SELECTION_MODEL_PROPERTY, old, selmodel);
	}

	//
	// selection model convenience wrapper methods
	//

	/**
	 * Get the current selection mode.
	 *
	 * @return an <code>int</code> value
	 */
	public int getSelectionMode()
	{
		return selmodel.getSelectionMode();
	}

	/**
	 * The the selection mode.
	 *
	 * @param b an <code>int</code> value
	 */
	public void setSelectionMode(int b)
	{
		selmodel.setSelectionMode(b);
	}
	
	/**
	 * If there is a selection, return the last object (the selected
	 * one) in the last selection path.
	 *
	 * @return an <code>Object</code> value
	 */
	public Object getLastSelectedPathComponent()
	{
		if (selmodel.isSelectionEmpty())
			return null;
		
		TreePath paths[] = selmodel.getSelectionPaths();
		TreePath last = paths[paths.length-1];

		return last.getPathComponent(last.getPath().length - 1);
	}

	/**
	 * Return the first selection path.
	 *
	 * @return a <code>TreePath</code> value
	 */
	public TreePath getLeadSelectionPath()
	{
		if (selmodel.isSelectionEmpty())
			return null;

		return selmodel.getLeadSelectionPath();
	}

	/**
	 * Return the number of selected nodes.
	 *
	 * @return an <code>int</code> value
	 */
	public int getSelectionCount()
	{
		return selmodel.getSelectionCount();
	}

	/**
	 * Is there currently nothing selected?
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean isSelectionEmpty()
	{
		return selmodel.isSelectionEmpty();
	}

	/**
	 * Get the current selection path (or first, if there are many
	 * selected nodes).
	 *
	 * @return a <code>TreePath</code> value
	 */
	public TreePath getSelectionPath()
	{
		return selmodel.getSelectionPath();
	}

	/**
	 * Set the current selection path.
	 *
	 * @param tp a <code>TreePath</code> value
	 */
	public void setSelectionPath(TreePath tp)
	{
		selmodel.setSelectionPath(tp);
	}

	/**
	 * Get all selection paths.
	 *
	 * @return a <code>TreePath[]</code> value
	 */
	public TreePath[] getSelectionPaths()
	{
		return selmodel.getSelectionPaths();
	}

	/**
	 * Set multiple selections with multiple paths.
	 *
	 * @param tps a <code>TreePath[]</code> value
	 */
	public void setSelectionPaths(TreePath[] tps)
	{
		selmodel.setSelectionPaths(tps);
	}

	/**
	 * Add a new selection path.
	 *
	 * @param tp a <code>TreePath</code> value
	 */
	public void addSelectionPath(TreePath tp)
	{
		selmodel.addSelectionPath(tp);
	}

	/**
	 * Add multiple new selections.
	 *
	 * @param tps a <code>TreePath[]</code> value
	 */
	public void addSelectionPaths(TreePath[] tps)
	{
		selmodel.addSelectionPaths(tps);
	}

	/**
	 * Clear all selections.
	 */
	public void clearSelection()
	{
		selmodel.clearSelection();
	}

	/**
	 * Is the given path selected (ie: one of the selection model's
	 * current selection paths)?
	 *
	 * @param tp a <code>TreePath</code> value
	 * @return a <code>boolean</code> value
	 */
	public boolean isPathSelected(TreePath tp)
	{
		return selmodel.isPathSelected(tp);
	}

	/**
	 * Remove the given selection path.
	 *
	 * @param tp a <code>TreePath</code> value
	 */
	public void removeSelectionPath(TreePath tp)
	{
		selmodel.removeSelectionPath(tp);
	}

	/**
	 * Remove the given set of selection paths.
	 *
	 * @param tps a <code>TreePath[]</code> value
	 */
	public void removeSelectionPaths(TreePath[] tps)
	{
		selmodel.removeSelectionPaths(tps);
	}

	//
	// useful TreePath methods
	//

	/**
	 * Get the path for the node closets to the point (x, y)
	 *
	 * TODO: implement this method
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>TreePath</code> value
	 */
	public TreePath getClosestPathForLocation(int x, int y)
	{
		return null;
	}

	/**
	 * Get the path for the node (if any) that contains the point (x, y).
	 *
	 * @param x an <code>int</code> value
	 * @param y an <code>int</code> value
	 * @return a <code>TreePath</code> value
	 */
	public TreePath getPathForLocation(int x, int y)
	{
		TreeLayoutNode node = getNode(x, y);

		if (node == null)
			return null;
		else
			return tlmodel.getTreePath(node);
	}

	/**
	 * Get the rectangle that represents where the given node will be
	 * drawn.
	 *
	 * @param tp a <code>TreePath</code> value
	 * @return a <code>Rectangle</code> value
	 */
	public Rectangle getPathBounds(TreePath tp)
	{
		if (tp != null)
		{
			Object d = tp.getLastPathComponent();
			TreeLayoutNode n = tlmodel.getNode(d);
			if (n == null)
				return null;
			Point q = translatePointToScreen(new Point(n.getX(), n.getY()));

			Rectangle r = new Rectangle(q.x, q.y,
										(int) (n.getWidth() * getZoomLevel()),
										(int) (n.getHeight() * getZoomLevel()));

			return r;
		}

		return null;
	}

	/**
	 * Scroll so that the given node is displayed.
	 *
	 * @param tp a <code>TreePath</code> value
	 */
	public void scrollPathToVisible(TreePath tp)
	{
		if (tp != null)
		{
			Rectangle bounds = getPathBounds(tp);
			if (bounds != null)
				scrollRectToVisible(bounds);
		}
	}

	//
	// methods overriding JComponent
	//

	/**
	 * Get the class name of the UI delegate of this class.
	 *
	 * @return <description>
	 */
	public String getUIClassID()
	{
		return uiclassid;
	}

	/**
	 * Set the UI delegate.
	 *
	 * @param ui a <code>TreeDisplayUI</code> value
	 */
	public void setUI(TreeDisplayUI ui)
	{
		super.setUI(ui);
	}

	/**
	 * Get the UI delegate object.
	 *
	 * @return a <code>TreeDisplayUI</code> value
	 */
	public TreeDisplayUI getUI()
	{
		return (TreeDisplayUI) ui;
	}

	/**
	 * Reset the UI.
	 *
	 */
	public void updateUI()
	{
		setUI((TreeDisplayUI) UIManager.getUI(this));
		resizeAndRepaint();
	}

	/**
	 * Layout everything.
	 */
	public void doLayout()
	{
		private_layout();

		super.doLayout();
	}

	//
	// Scrollable
	//

	/**
	 * {@link javax.swing.Scrollable#getPreferredScrollableViewportSize}
	 *
	 * @return a <code>Dimension</code> value
	 */
	public Dimension getPreferredScrollableViewportSize()
	{
		return getPreferredSize();
	}

	/**
	 * {@link javax.swing.Scrollable#getScrollableUnitIncrement}
	 *
	 * @param visibleRec a <code>Rectangle</code> value
	 * @param orientation an <code>int</code> value
	 * @param direction an <code>int</code> value
	 * @return an <code>int</code> value
	 */
	public int getScrollableUnitIncrement(Rectangle visibleRec, int orientation, int direction)
	{
		return unitincr;
	}

	/**
	 * {@link javax.swing.Scrollable#getScrollableBlockIncrement}
	 *
	 * @param visibleRec a <code>Rectangle</code> value
	 * @param orientation an <code>int</code> value
	 * @param direction an <code>int</code> value
	 * @return an <code>int</code> value
	 */
	public int getScrollableBlockIncrement(Rectangle visibleRec, int orientation, int direction)
	{
		return blockincr;
	}

	/**
	 * {@link javax.swing.Scrollable#getScrollableTracksViewportWidth}
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean getScrollableTracksViewportWidth()
	{
		private_layout();

		if (getZoomMode() == FIT_WIDTH || getZoomMode() == FIT_WINDOW)
			return true;
		if (getZoomedWidth() <= getParent().getWidth())
			return true;

		return false;
	}

	/**
	 * {@link javax.swing.Scrollable#getScrollableTracksViewportHeight}
	 *
	 * @return a <code>boolean</code> value
	 */
	public boolean getScrollableTracksViewportHeight()
	{
		private_layout();

		if (getZoomMode() == FIT_WINDOW)
			return true;
		if (getZoomedHeight() <= getParent().getHeight())
			return true;

		return false;
	}

	//
	// protected methods
	//

	/**
	 * Do a layout, revalidate, and then repaint.
	 */
	protected void resizeAndRepaint()
	{
		private_layout();
		revalidate();
		repaint();
	}

	public void repaint()
	{
		revalidate();
		super.repaint();
	}

	/**
	 * Get the real (display) width of the tree.
	 *
	 * @return an <code>int</code> value
	 */
	protected int getZoomedWidth()
	{
		return (int) (getRawWidth() * getZoomLevel());
	}

	/**
	 * Get the real (display) height of the tree.
	 *
	 * @return an <code>int</code> value
	 */
	protected int getZoomedHeight()
	{
		return (int) (getRawHeight() * getZoomLevel());
	}

	/**
	 * Get the width of the tree before zooming is done.
	 *
	 * @return an <code>int</code> value
	 */
	protected int getRawWidth()
	{
		return getTreeLayoutModel().getWidth() + 2*getPadX();
	}

	/**
	 * Get the height of the tree before zooming is done.
	 *
	 * @return an <code>int</code> value
	 */
	protected int getRawHeight()
	{
		return getTreeLayoutModel().getHeight() + 2*getPadY();
	}

	/**
	 * Get a default TreeModel.
	 *
	 * @return a <code>TreeModel</code> value
	 */
	protected static TreeModel getDefaultTreeModel()
	{
		DefaultMutableTreeNode root = new DefaultMutableTreeNode("Root");
		DefaultMutableTreeNode parent;

		parent = new DefaultMutableTreeNode("Letters");
		root.add(parent);

		parent.add(new DefaultMutableTreeNode("A"));
		parent.add(new DefaultMutableTreeNode("B"));

		parent = new DefaultMutableTreeNode("Numbers");
		root.add(parent);

		parent.add(new DefaultMutableTreeNode("1"));
		parent.add(new DefaultMutableTreeNode("2"));

		return new DefaultTreeModel(parent);
	}

	/**
	 * Set the zoom level but don't invoke a repaint etc.
	 *
	 * @param l a <code>double</code> value
	 */
	protected void uiSetZoomLevel(double l)
	{
		this.zoom = l;

		notifyTreeDisplayListeners(new TreeDisplayEvent(this, TreeDisplayEvent.ZOOM_PERFORMED));
	}

	/**
	 * Set the x translation but don't invoke a repaint etc.
	 *
	 * @param i an <code>int</code> value
	 */
	protected void setTranslateX(int i)
	{
		this.translate_x = i;
	}

	/**
	 * Set the y translation but don't invoke a repaint etc.
	 *
	 * @param i an <code>int</code> value
	 */
	protected void setTranslateY(int i)
	{
		this.translate_y = i;
	}

	/**
	 * Center view on given TreeLayoutNode.
	 *
	 * @param n a <code>TreeLayoutNode</code> value
	 */
	protected void centerOnNode(TreeLayoutNode n)
	{
		if (n == null)
			return;

		centerOnRectangle(new Rectangle(n.getX(), n.getY(),
										n.getWidth(), n.getHeight()));
	}

	//
	// listener stuff
	//
	
	/**
	 * Add a TreeModelListener.
	 *
	 * @param tml a <code>TreeModelListener</code> value
	 */
	public void addTreeModelListener(TreeModelListener tml)
	{
		model.addTreeModelListener(tml);
	}

	/**
	 * Remove a TreeModelListener.
	 *
	 * @param tml a <code>TreeModelListener</code> value
	 */
	public void removeTreeModelListener(TreeModelListener tml)
	{
		model.removeTreeModelListener(tml);
	}

	/**
	 * Add a TreeSelectionListener.
	 *
	 * @param tsl a <code>TreeSelectionListener</code> value
	 */
	public void addTreeSelectionListener(TreeSelectionListener tsl)
	{
		selmodel.addTreeSelectionListener(tsl);
	}

	/**
	 * Remove a TreeSelectionListener.
	 *
	 * @param tsl a <code>TreeSelectionListener</code> value
	 */
	public void removeTreeSelectionListener(TreeSelectionListener tsl)
	{
		selmodel.removeTreeSelectionListener(tsl);
	}

	/**
	 * Add a TreeDisplayListener.
	 *
	 * @param tdl a <code>TreeDisplayListener</code> value
	 */
	public void addTreeDisplayListener(TreeDisplayListener tdl)
	{
		listeners.add(TreeDisplayListener.class, tdl);
	}

	/**
	 * Remove a TreeDisplayListener.
	 *
	 * @param tdl a <code>TreeDisplayListener</code> value
	 */
	public void removeTreeDisplayListener(TreeDisplayListener tdl)
	{
		listeners.remove(TreeDisplayListener.class, tdl);
	}

	/**
	 * Notify all TreeDisplayListeners that something of importance
	 * has happened.
	 *
	 * @param tde a <code>TreeDisplayEvent</code> value
	 */
	protected void notifyTreeDisplayListeners(TreeDisplayEvent tde)
	{
		Object[] list = listeners.getListenerList();

		for (int i = 1; i < list.length; i += 2)
		{
			TreeDisplayListener l = (TreeDisplayListener) list[i];
			switch (tde.getID())
			{
				case TreeDisplayEvent.ZOOM_PERFORMED:
					l.treeDisplayZoomed(tde);
					break;
				default:
					// ignore
			}
		}
	}

	//
	// private stuff
	//

	private Point translatePoint(Point p)
	{
		Point q = new Point();
		double zoom = getZoomLevel();
		
		q.x = (int) ((p.x - getTranslateX() - ((int) (getPadX() * zoom))) / zoom);
		q.y = (int) ((p.y - getTranslateY() - ((int) (getPadY() * zoom))) / zoom);

		return q;
	}

	private Point translatePointToScreen(Point p)
	{
		Point q = new Point();
		double zoom = getZoomLevel();

		q.x = (int) (zoom * (p.x + getPadX())) + getTranslateX();
		q.y = (int) (zoom * (p.y + getPadY())) + getTranslateY();

		return q;
	}

	private void createZoomPoint()
	{
		if (! (getParent() instanceof JViewport))
		{
			zoompoint = null;
			return;
		}

		JViewport vp = (JViewport) getParent();
		Rectangle vpr = vp.getViewRect();

		Point p = new Point(vpr.x + (int)((vpr.width) / 2.0),
							vpr.y + (int)((vpr.height) / 2.0));
		zoompoint = translatePoint(p);
	}

	private void private_layout()
	{
		Component parent = getParent();
		Component pp = parent;
		double zoom = 1, scale_x, scale_y;
		int trans_x = 0, trans_y = 0;

		if (parent == null)
			return;

		int new_width = getRawWidth();
		int new_height = getRawHeight();

		// zoom
		switch (getZoomMode())
		{
			case NO_ZOOM_MODE:
				break;
			case ACTUAL_SIZE:
				uiSetZoomLevel(1);
				break;
			case FIT_WIDTH:
				// find scale based on width
				zoom = (double) parent.getWidth() / (double) new_width;

				if (getZoomLevel() != zoom)
					uiSetZoomLevel(zoom);
				break;
			case FIT_WINDOW:
				// find scale based on width
				scale_x = (double) parent.getWidth() / (double) new_width;

				// find scale based on height
				scale_y = (double) parent.getHeight() / (double) new_height;

				// take the smaller one
				zoom = (scale_x < scale_y ? scale_x : scale_y);
			
				// set the zoom level in component without invoking a repaint
				if (getZoomLevel() != zoom)
					uiSetZoomLevel(zoom);
				break;
			default:
				// do nothing
		}

		zoom = getZoomLevel();
		new_width = (int) (new_width * zoom);
		new_height = (int) (new_height * zoom);

		// where is the tree anchored?
		int anchor = getAnchor();
		if (anchor == NORTH ||
			anchor == NORTHEAST ||
			anchor == NORTHWEST)
		{
			if (new_width < parent.getWidth())
				trans_x = (parent.getWidth() - new_width) / 2;
			trans_y = 0;
		}

		if (anchor == SOUTH ||
			anchor == SOUTHEAST ||
			anchor == SOUTHWEST)
		{
			if (new_width < parent.getWidth())
				trans_x = (parent.getWidth() - new_width) / 2;
			if (new_height < parent.getHeight())
				trans_y = parent.getHeight() - new_height;
		}
		
		if (anchor == EAST ||
			anchor == NORTHEAST ||
			anchor == SOUTHEAST)
		{
			if (new_width < parent.getWidth())
				trans_x = parent.getWidth() - new_width;
			if (new_height < parent.getHeight())
				trans_y = (parent.getHeight() - new_height) / 2;
		}

		if (anchor == WEST ||
			anchor == NORTHWEST ||
			anchor == SOUTHWEST)
		{
			trans_x = 0;
			if (new_height < parent.getHeight())
				trans_y = (parent.getHeight() - new_height) / 2;
		}

		if (anchor == CENTER)
		{
			if (new_width < parent.getWidth())
				trans_x = (parent.getWidth() - new_width) / 2;
			if (new_height < parent.getHeight())
				trans_y = (parent.getHeight() - new_height) / 2;
		}

		setTranslateX(trans_x);
		setTranslateY(trans_y);

		setSize(Math.max(getZoomedWidth(), parent.getWidth()), Math.max(getZoomedHeight(), parent.getHeight()));
	}

	//
	// implementation of javax.swing.event.TreeModelListener interface
	//

	/**
	 * Describe class <code>TreeDisplayTreeModelListener</code> here.
	 *
	 */
	protected class TreeDisplayTreeModelListener implements TreeModelListener {
		/**
		 * Describe <code>treeNodesChanged</code> method here.
		 *
		 * @param e a <code>TreeModelEvent</code> value
		 */
		public void treeNodesChanged(TreeModelEvent e)
		{
			resizeAndRepaint();
		}

		/**
		 * Describe <code>treeNodesInserted</code> method here.
		 *
		 * @param e a <code>TreeModelEvent</code> value
		 */
		public void treeNodesInserted(TreeModelEvent e)
		{
			resizeAndRepaint();
		}

		/**
		 * Describe <code>treeNodesRemoved</code> method here.
		 *
		 * @param e a <code>TreeModelEvent</code> value
		 */
		public void treeNodesRemoved(TreeModelEvent e)
		{
			resizeAndRepaint();
		}

		/**
		 * Describe <code>treeStructureChanged</code> method here.
		 *
		 * @param e a <code>TreeModelEvent</code> value
		 */
		public void treeStructureChanged(TreeModelEvent e)
		{
			resizeAndRepaint();
		}
	}

	//
	// implementation of javax.swing.event.TreeSelectionModelListener interface
	//

	/**
	 * Describe class <code>TreeDisplayTreeSelectionListener</code> here.
	 *
	 */
	protected class TreeDisplayTreeSelectionListener implements TreeSelectionListener {
		/**
		 * Describe <code>valueChanged</code> method here.
		 *
		 * @param e a <code>TreeSelectionEvent</code> value
		 */
		public void valueChanged(TreeSelectionEvent e)
		{
			resizeAndRepaint();
		}
	}
}// TreeDisplay
