///////////// main.h ///////////////////////////////////
/** \file
  * Implements main functions for all supported systems.
  * This include file should be included by, and only by, your application's main cpp file. 
  * Unlike every other libRT header file, <tt>main.h</tt> includes system include files.
  * That means that you should not use the <tt>using namespace lrt;</tt> directive in your
  * application's main cpp file, as you can never know what will be in the top namespace
  * for other systems.
  */


#include "rtsystem.h"
#include "rtstring.h"
#include "rtcollect.h"
#include "rtfile.h"

// fwd declare: user main function
/** The main function for libRT users. Every application using libRT should implement
  * this function, rather than the system-dependent one.
  * @param args The application's command line arguments. Note that they do <b>not</b> 
  *             include the executable name.
  * @return The application's return code.
  * @see lrt::File::getExecutableFile()
  */
int rtMain(const lrt::Array<lrt::String> &args);


#ifdef __SYMBIAN32__
#include <e32base.h>
#include <e32std.h>

#include "rtetools.h"

GLDEF_C TInt E32Main() // main function called by E32
{
	lrt::System::cleanup=CTrapCleanup::New(); // get clean-up stack

	RProcess myProcess;
	TFileName fn = myProcess.FileName();
#ifndef _UNICODE    // actually: #if Epoc Version 5, but that's impossible
	TCommand cmd = myProcess.CommandLine();
#else
	TBuf<0x100> cmd;
	myProcess.CommandLine(cmd);
#endif
	lrt::String cmdLine = MYSTRING(cmd);
	cmdLine = cmdLine.trim();
	lrt::String file = MYSTRING(fn);
	file = file.trim();
	if(!file.compareIgnoreCase(cmdLine)) // started (tapped) from System 
	{
		cmdLine = lrt::System::readLine("Arguments: ");
	}
	lrt::File::initialize(file);
	lrt::File::setCurrentFolder(lrt::File(file));
	lrt::Array<lrt::String> args = cmdLine.split(" ");

	int retval = 0;
	TRAPD(error,retval = rtMain(args));
	__ASSERT_ALWAYS(!error,User::Panic(_L("LibRT Error"),error));

	if(lrt::System::isInteractive()) {
		lrt::System::println("--- Press any key to exit ---");
		lrt::System::read();
	}
	lrt::System::exit(retval);

	return 0; // unreachable
}

#else

/** LibRT's implementation of the stdlib main function. 
  * This method basically just calls the user main function <tt>rtMain()</tt>,
  * which should be implemented instead by library users.
  */
int main(int argn, char **argv) // main function called by Windows et al.
{ 
	lrt::File::initialize(argv[0]);
	lrt::Array<lrt::String>* arr 
	    = new lrt::Array<lrt::String>(lrt::Math::max(argn-1,0));
	for(int i = 1; i < argn; i++)
		(*arr)[i-1] = lrt::String(argv[i]);

	int ret = rtMain(*arr);
	delete arr;
	lrt::System::exit(ret);
	return 0; // unreachable
}
#endif
