/////////// streams.cpp: implementing simple input & output streams
#include "rtstreams.h"
#include "rtcollect.h"
#include "rtstring.h"

#ifdef __SYMBIAN32__
  #include "rtstreams.epoc.cpp"
#else
  #include "rtstreams.stdlib.cpp"
#endif

namespace lrt {

///////////// InputStream /////////////
bool InputStream::markSupported()
{
	return false;
}

void InputStream::mark() {}
void InputStream::reset() {}

int InputStream::read(Array<char> &b, int off, int len)
{
	int numRead = 0;
	for(int i = off; i < off + len; i++)
	{
		int c = read();
		if(c != -1) {
			numRead++;
			b[i] = (char)c;
		}
	}
	if(numRead == 0) numRead = -1;

	return numRead;
}

int InputStream::read(Array<char> &b)
{
	return read(b, 0, b.length());
}

bool InputStream::fail() 
{
	return _fail;
}

void InputStream::close() {}

InputStream::~InputStream() 
{
	close();
}

/// protected ////
InputStream::InputStream() : _fail(false) {}


////////////// FilterInputStream //////////////

FilterInputStream::FilterInputStream(InputStream *in) : in(in), _detach(false)
{
}

bool FilterInputStream::markSupported()
{
	if(in) 
		return in->markSupported();
	else 
		return false;
}

void FilterInputStream::mark()
{
	if(in) in->mark();
}

void FilterInputStream::reset()
{
	if(in) in->reset();
}

int FilterInputStream::read()
{
	if(in) 
		return in->read();
	else 
		return -1;
}

int FilterInputStream::read(Array<char> &b, int off, int len)
{
	if(in)
		return in->read(b, off, len);
	else
		return 0;
}

bool FilterInputStream::fail()
{
	if(in)
		return in->fail();
	else
		return true; 
}

InputStream* FilterInputStream::detach()
{
	_detach = true; 
	return in;
}

bool FilterInputStream::eos()
{
	return in->eos();
}

void FilterInputStream::close()
{
	if(!_detach)
		in->close();
}

FilterInputStream::~FilterInputStream()
{
	close();
	if(!_detach)
		delete in;
}

//////////////////// ParseInputStream /////////////////////////

const String ParseInputStream::whitespace("\r\n\t ");

ParseInputStream::ParseInputStream(InputStream *in, const String &separators, 
	const char commentStart) : FilterInputStream(in), separators(separators), 
	commentStart(commentStart), lineNum(1), markLineNum(1), buf(-1), 
	ignoreComments(false), autoTrim(false), eol(false)
{
	this->separators += commentStart; // make sure that comment char is a separator
}

void ParseInputStream::setIgnoreComments(bool val)
{
	ignoreComments = val;
}

void ParseInputStream::setAutoTrim(bool val)
{
	autoTrim = val;
}

void ParseInputStream::mark()
{
	markLineNum = lineNum;
	FilterInputStream::mark();
}

void ParseInputStream::reset()
{
	FilterInputStream::reset();
	lineNum = markLineNum;
}

int ParseInputStream::read()
{
	if(eol) { lineNum++; eol = false; }
	int ret;
	if(buf >= 0) { ret = buf; buf = -1; }
	else ret = FilterInputStream::read();
	if(ret == '\n') eol = true;
	return ret;
}

//! Can't handle multiple successive unreads! Characters will get lost!
void ParseInputStream::unread(int c)
{
	buf = c;
	if(c == '\n') eol = false;
}

String ParseInputStream::getWord(const String& specialSeps)
{
	const String& separators = (specialSeps == "" ? this->separators : specialSeps);
	String ret;
	bool eof = false;
	bool first = true;
	int c;
	while(true) {
		c = read();
		if(c < 0) { eof = true; break; }
		
		if(first) { // for first char of word, things are a bit different...
			if(c == commentStart) {
				if(!ignoreComments) // return the whole comment as a word
				{ unread(c); return getLine(); }
				else // skip comment: return first word of next line
				{ skipRestOfLine(); return getWord(specialSeps); }
			}
			else {
				ret += (char)c; // separators are allowed as first char
				if(separators.indexOf((char)c) >= 0) break;
			}
			first = false;
		}
		else { // more chars to the word
			if(separators.indexOf((char)c) >= 0) 
			{ unread(c); break; } // break on separator, put it to next word
			ret += (char)c;
		}
	}

	if(autoTrim) {
		ret = ret.trim();
	}
	if(!eof && (ret.length() == 0)) // word is empty => fetch next word
		return getWord(specialSeps);
	else return ret; // return this word
}

String ParseInputStream::getLine(bool skipEmpty)
{
	String ret;
	bool eof = false;
	int c;
	while(true) {
		c = read();
		if(c < 0) { eof = true; break; }
		if(c == '\n') break; // never ever append the newline to the string
		ret += (char)c;
	}
	if(ignoreComments) {
		int commentIndex = ret.lastIndexOf(commentStart);
		if(commentIndex >= 0)
			ret = ret.substring(0, commentIndex);
	}
	if(autoTrim) {
		ret = ret.trim();
	}
	if(skipEmpty && !eof && (ret.length() == 0)) // line is empty => return next line
		return getLine(skipEmpty);
	else return ret;	// return this line
}

int ParseInputStream::getLineNumber()
{
	return lineNum;
}

bool ParseInputStream::skipRestOfLine()
{
	int c;
	while(true) {
		c = read();
		if(c < 0) break;
		if(c == '\n') break;
	}
	return (c >= 0);
}

bool ParseInputStream::eos()
{
	if(buf >= 0) return false;
	buf = read();
	return (buf < 0);
}


//////////////////// OutputStream /////////////////////////////

bool OutputStream::write(const Array<char> &b, int off, int len)
{
	bool ret = true;
	for(int i = off; i < off + len; i++)
		ret &= write(b[i]);
	return ret;
}

bool OutputStream::write(const Array<char> &b)
{
	return write(b, 0, b.length());
}

bool OutputStream::write(const char* c)
{
	return write(String(c));
}

void OutputStream::flush() {}

bool OutputStream::fail()
{
	return _fail;
}

void OutputStream::close() {}

OutputStream::~OutputStream()
{
	close();
}

//////// protected //////////
OutputStream::OutputStream()  : _fail(false) {}


///////////////// FileOutputStream ///////////////////////////

bool FileOutputStream::write(const Array<char> &b)
{
	return write(b, 0, b.length());
}

////////////// FilterOutputStream //////////////

FilterOutputStream::FilterOutputStream(OutputStream *out) : out(out)
{
}

bool FilterOutputStream::write(int b)
{
	return out->write(b);
}
	
bool FilterOutputStream::write(const Array<char> &b, int off, int len)
{
	return out->write(b, off, len);
}

bool FilterOutputStream::write(const Array<char> &b)
{
	return out->write(b, 0, b.length());
}

	
void FilterOutputStream::flush()
{
	out->flush();
}

void FilterOutputStream::close()
{
	out->close();
}

FilterOutputStream::~FilterOutputStream()
{
	close();
	delete out;
}


} // namespace

