/** \mainpage libRT - The Random Target Library
  * \section intro Introduction
  * LibRT is a platform abstraction library for C++ console applications.
  * Supported platforms are currently Windows, Unix and EPOC32.
  * It provides access to files and the file system, the console, and
  * includes some collection templates.
  */

/** \file rtsystem.h
  * Contains the <tt>System</tt> class, which is the core and glue of libRT
  * and should be included by any libRT application. <br>
  * While you're at it, include <tt>rtmain.h</tt> as well.
  */

/** \namespace lrt
  * All classes in libRT are defined inside the namespace <tt>lrt</tt>.
  * However, there are (almost) no global members in the namespace:
  * they are all implemented as static constants or functions of wrapper classes
  * (e.g. <tt>System</tt>, <tt>Char</tt> and <tt>Math</tt>).
  */

#ifndef __LRT_SYSTEM__
#define __LRT_SYSTEM__

// System forward declares
#ifdef __SYMBIAN32__
class CConsoleBase;
class CLineEdit;
class CTrapCleanup;
#endif

namespace lrt {

// Forward declares
class String;
template<class T> class Array;

/** A time value. The values should be interpreted towards the "Unix birthday" 
  * 1/1/1970 00:00:00, if they are absolute (e.g. the result of a <tt>getCurrentTime()</tt>
  * operation), but they may also be time differences (e.g. the result of a - operation).
  */
class Time
{
public:
	/** Get the current time. */
	static Time getCurrentTime();

	/** Creates a Time object set to 1/1/1970, 00:00:00. */
	Time() : sec(0), msec(0) {}

	/** Creates a Time object set to a custom time duration. */
	Time(int sec, int msec) : sec(sec), msec(msec) {}

	/** Creates a Time object by converting from the given String, which must be in 
	  * the standard Unix format (as returned by toString()). If the string was not
	  * in correct format, the values of <tt>sec</tt> and <tt>msec</tt> will still
	  * be 0 after the construction. */
	Time(const String&);

	/** Subtracts another time from this one and returns the difference. 
	  * If this time is earlier than the given one, the difference will be negative. */
	Time operator-(const Time& t);

	/** Gets the time as milliseconds. If the time value is too large, 
	  * the result is undefined. (This will always happen if this Time is an absolute
	  * time value.) */
	int toInt();

	/** Converts this time (milliseconds are ignored) to the standard Unix time string.
	  * The result looks like <tt>Mon Apr 08 15:38:42 2002</tt>.
	  */
	String toString();


	/** Seconds since 1/1/1970, 00:00:00. */
	int sec;
	/** Milliseconds in the current second. */
	int msec;

private:
	// an array of 12 month names
	static String months[];
	// convert 'Jan' etc. to a number
	int getMonthNum(const String& monthName);
};


/** Class System provides methods to access the system console and exit the application.
  */
class System
{
public:
	//// types
	/** Possible break types. */
	enum BreakType { CTRL_C,	///< as by CTRL+C in the command line
					 TERMINATE	///< as by kill command
	};
	/** A break handler function (see System::setOnBreak()) */
	typedef void (*BreakFunction)(BreakType);

	/** Sets the application name. It is used in message box titles, for example. */
	static void setAppName(const String& appName);
	/** Sets if we're in interactive mode. By default, this is true. 
	  * If libRT is in interactive mode, some actions may be presented to the user using 
	  * GUI elements (message boxes), e.g. the functions <tt>message()</tt> and 
	  * <tt>exit(int, String)</tt>. If not, the console will be used in every case. */
	static void setInteractive(bool);
	/** Returns the current interactive mode. */
	static bool isInteractive();

	/** Print a message to the user. If we're in interactive mode and it is supported 
	  * on the current system, a GUI element (message box) will be used to present the message.
	  * Otherwise, the console will be used. */
	static void message(const String& str);

	/** Exits the application. */
	static void exit();
	/** Exits the application with a specified return code. */
	static void exit(int);
	/** Exits the application abnormally, with a specified return code and error message. */
	static void exit(int, const String &);
	/** Exits the application abnormally, with a specified return code and error message. 
		Note: There are some places where you can't even use lrt::String. */
	static void exit(int, const char*);
	/** Sets a signal handler function, which will be called when the user presses CTRL+C 
	  * on the command line. As soon as you set such a signal handler, the program will no
	  * longer just exit (which it would do otherwise); instead your function will be called,
	  * whereafter the program continues normally. 
	  * @param func The function to call. May be 0, which restores the default behaviour. 
	  * @param catchKill (optional) If given, the 'normal' kill command (without -SIGKILL) 
	  *                  is caught by your function, too. (This only works on Unix. The Windows
	  *                  Task manager ignores this parameter.)
	  */
	static void setOnBreak(BreakFunction func, bool catchKill);

	/** Prints some characters to the system console. */
	static void print(const char *cStr);
	/** Prints a String to the system console. */
	static void print(const String &str);
	/** Prints a newline to the system console. */
	static void println();
	/** Prints a String to the console and finishes the line. */
	static void println(const String &str);
	/** Reads a single character from the system input. (This is usually the keyboard.) */
	static char read();
	/** Reads a whole line from the system input. (This is usually the keyboard.)
	  * The line ending character (usually '\n') is never included in the returned string.
	  * @param query The query string for the input. */
	static String readLine(const String &query);

	/** Sets the console position to these new coordinates. */
	static void setPos(int x, int y);

	/** Returns <tt>true</tt> if the underlying system is little-endian, <tt>false</tt>
	  * if it is big-endian. */
	static bool isLittleEndian();
	/** Fills a block of memory with the given byte value.
	  * @param ptr Location of the block
	  * @param length Number of bytes to fill
	  * @param data Value to fill the bytes with
	  */
	static void fillMemory(void* ptr, int length, char value);
	/** Copies a given block of memory.
	  * @param src Source pointer
	  * @param dest Destination pointer
	  * @param length Number of bytes to copy
	  */
	static void copyMemory(const void* src, void* dest, int length);

#ifdef __SYMBIAN32__
	static CTrapCleanup *cleanup; // need to set it from main()
#endif
public: // internal API
	static BreakFunction lastBreakFunction; // required for Windows, where it must be given on un-registration
private:
	static bool init;
	static bool interactive;
	static String* appName;
#ifdef __SYMBIAN32__
	static CConsoleBase *console;
	static CLineEdit *edit;
	static void printInternal(const char *cStr, bool isLong, bool addLine);
#endif
	static void initialize();
	static void finalize();

	System() {}
	~System() {}
};

/** A standard comparison function for default usage with <tt>Array::sort()</tt> and <tt>Map</tt>.
  * It uses the operators &lt; and == to compare any objects. */
template<class T> int stdCompare(const T& elem1, const T& elem2);
int stdCompare(char elem1, char elem2);

} // namespace

#include "rtsystem.templ.cpp"

#endif
