#ifndef __RT_ROBBASE__
#define __RT_ROBBASE__

#include <rtcollect.h>
#include <rtstring.h>
#include <rtlist.h>
#include <rtmap.h>

#include "robvars.h"
#include "robglob.h"

//using namespace lrt;

namespace rt {

// forward declares
class Program;
class ErrorHandler;
class Bot;
class Task;
class Bank;
class Instr;
class SimSupervisor;
class ExecSupervisor;
class ProgramLoader;

/** Implements one simulation. Use run() to run the simulation.  */
class Simulation {
public:
	const ErrorHandler* errHandler;
	const Globals& glob; ///< global constants (timings etc.)
	rint vars[simVarLength];	///< global vars (#PUB, $FIELDS, etc.)
	rint fieldWidth, fieldHeight;

	lrt::Array<Bot*> field;	///< for position-based access
	lrt::List<Bot*> bots;	///< for exec() and all walkthrough access
	unsigned long curCycle;	///< current game time

	Simulation(const Globals& glob, const ErrorHandler*);
	~Simulation();
	
	Program* run(); ///< returns the winner, or 0 if a tie occured
	rint getAllNumBots(); ///< returns sum of the bot counts for all programs
	void addProgram(Program* program); ///< adds a program to the current sim; only before starting it
	int getNumPrograms();			///< Returns number of programs in this simulation. 
	Program* getProgram(int index); ///< Returns a given program. 
	void addSupervisor(SimSupervisor* supervis); ///< adds a supervisor
	void addExecSupervisor(ExecSupervisor* es); ///< adds another
	
	/** Moves the given coordinates by one field to the given direction. */
	void getPosBefore(rint& x, rint& y, rint dir);
private:
	lrt::Vector<Program*> programs;
	lrt::List<SimSupervisor*> supervisors;
	lrt::List<ExecSupervisor*> execSupervisors;
	bool started;

	void processBots(); //!< just exec()s all the bots, and kills the <tt>fail</tt>ed ones

	friend class Task; // needs access to execSupervisors
};

/** One of the players on the field. However, a Program can play in multiple
  * simulations. The program is loaded using its <tt>ProgramLoader</tt>. */
class Program {
public:
	struct HeaderField {
		HeaderField();
		HeaderField(const lrt::String& name, const lrt::String& value, HeaderType type);
		lrt::String name;
		lrt::String value;
		HeaderType type;
	};

	static lrt::Vector<ProgramLoader*> exampleLoaders;
	static ProgramLoader* getLoader(const lrt::String& file); 
	static Program* create(const lrt::String& file, const Globals& glob, int globalNum);

	Simulation* sim;
	
	rint vars[progVarLength];
	/// loads banks if not done, and returns them
	const lrt::Vector<Bank*>& getBanks();
	lrt::StringMap<HeaderField> headers;
	int programNum; ///< index in Sim::programs
	int globalNum; ///< index in the TourResults
	Bank* emptyBank; ///< the empty bank belonging to this program

	lrt::String print();

	/** Initializes the program loader. Loads headers immediately, banks only on request. */
	Program(ProgramLoader* loader);
	/** Initializes sim and vars; MUST be called before each sim.
	  * @param num The program's number (index in Sim::programs) in the current sim. */
	void init(rint num, Simulation* sim); 
	/** Gets the loader for this program. */
	ProgramLoader* getLoader();
	/** deletes the loader. */
	~Program();

private:
	lrt::Vector<Bank*> banks;
	ProgramLoader* loader;
};

/** Handles all kinds of errors. */
class ErrorHandler {
public:
	/** This method is called when a robot is deleted due to a RoboCom fatal error. */
	virtual void handleBotError(const Bot* affectedBot, ExecReturnType type) const = 0;
	/** This method is called when a robot file cannot be loaded due to a syntax error. */
	virtual void handleLoadError(const lrt::String& affectedFile, const lrt::String& message) const = 0;
	/** This method is called when a system error was detected; the application should exit. */
	virtual void handleSystemError(int num, const lrt::String& message) const = 0;
	/** This method is called when the user should be warned about an unexpected, but 
	  * recoverable event. For example, if a specified robot file cannot be found (it can
	  * just be ignored). */
	virtual void handleWarning(const lrt::String& message) const {};
};

/** A robot on the field. */
class Bot {
public:
	Program* owner;

	rint vars[botVarLength];  ///< named vars (e.g. #ACTIVE)
	rint* stdVars;			///< unnamed vars (e.g. #1). Note! Size must be numUnnamedVars+1!
	lrt::Array<Bank*> banks;
	lrt::Vector<Task*> tasks;

	rint curTask;

	int creationCycle; ///< when was the bot created?
	int deactCycle; ///< latest deactivation of the bot, or 0 if never deactivated

	/// appends itself to sim->bots and puts itself into sim->field.
	Bot(rint instrSet, rint numBanks, rint mobile,
		rint posX, rint posY, rint dir, rint generation, 
		Program* owner);
	/// removes itself from sim->field. It has to be removed from sim->bots however.
	~Bot();

	ExecReturnType exec();
};

/** One task in a robot. Each robot always has at least one task running, but
  * can have more with some option sets. */
class Task {
public:
	Bot* bot;
	Simulation* sim; ///< Abkrzung

	rint vars[taskVarLength];
	unsigned long instrStarted; ///< is too large for vars
	Instr* curInstr;
	bool seized;

	/** adds itself to bot->tasks.
	  * @param creator The creator task, which will not be overwritten by this task. Can be 0.
	  */               
	Task(Bot* bot, Task* creator, rint dir);
	/// removes itself from bot->tasks.
	~Task();

	ExecReturnType exec();
	ExecReturnType reboot();


	/** Fetches the robot on the reference field of this bot, given by its
	  * active task <tt>this</tt>. */
	Bot* getRefBot();
	/** Calculates the reference robot position, given the current active 
	  * task <tt>this</tt>. */
	void getRefBotPos(rint& x, rint& y);
};

/** A code bank. Every different code bank exists only once in the simulation, 
  * the robots just store pointers to them (since they are not modifyable). */
class Bank {
public:
	Program* owner;

	lrt::String name;
	lrt::Vector<Instr*> instr;
	bool isSecret;

	lrt::String print();

	Bank(Program* const owner, const lrt::String& name);
	/// deletes instrs
	~Bank();
	
};

/** Implements one instruction in a robot's program. */
class Instr {
public:

	struct Op {
		bool isRemote;
		OpType type;
		rint ptr;
	};

	virtual ~Instr();

	/** Executes this instruction on the given bot (represented by its
	  * active task). 
	  * If this function doesn't return <tt>execOK</tt>, an error has
	  * occured and the robot given by <tt>task</tt> should be destroyed. */
	virtual ExecReturnType exec(Task* task) = 0;
	/** Return the number of cycles this instruction takes, 
	  * considering the given bot (represented by its active task).<br>
	  * This includes checking for errors as well! This function just
	  * returns <tt>cycles</tt> and will probably not be enough for most
	  * instructions. <br>
	  * If this function returns a negative value, an error has occured
	  * during checking and the bot should be destroyed. */
	virtual rint getNumCycles(Task* task);
	/** Returns the timing mode for this instruction.
	  * The method of class <tt>Instr</tt> returns <tt>timingNumExec</tt>.
	  * @see TimingMode */
	virtual TimingMode getTimingMode();

	/** Return the instruction name here. */
	virtual lrt::String getName() = 0;
	/** Returns a String representation of this instruction. 
	  * @param instrPos Position (in its bank) of the instruction.
	  */
	virtual lrt::String print(int instrPos = 0);

protected:
	/** Initializes the operand array. */
	Instr(int numParams);
	/** The operand array. */
	Op* ops;
	/** The number of ops this instruction has. */
	int numOps;
	/** The number of cycles this instruction takes. */
	rint cycles;
	/** Gets the current value of the op at <tt>opNum</tt>, for the bot 
	  * given by its active task <tt>task</tt>. 
	  * @param absolute For instructions needing absolute references (BJUMP, AJUMP).
	  */
	rint getValue(Task* task, int opNum, bool absolute = false);
	/** Sets the game (bot, task) variable given by the op at <tt>opNum</tt>
	  * of the bot given by <tt>task</tt> to the given value. */
	void setValue(Task* task, int opNum, rint value);
	/** Sets the ops up. */
	void initOps(const lrt::Array<Op>& ops);
	/** Prints the op. */
	lrt::String printOp(int opNum, int instrPos);
	/** Prints all ops in a comma-separated list. */
	lrt::String printOps(int instrPos);
};


/** This interface can be implemented if you want to plug into a running
  * simulation. */
class SimSupervisor {
public:
	struct GameState {
		GameEventType event;
		rint affectedProgram;
	};

	virtual ~SimSupervisor() {}

	/** Return the preferred position of your supervisor in the simulation's
	  * supervisor list. (The list order implies the execution order.)
	  * The method of class <tt>SimSupervisor</tt> returns <tt>supPosDontCare</tt>,
	  * which means that the supervisor does not rely on a certain list position. */
	virtual SimSuperPosition getPreferredPosition();
	/** Return the number of cycles after which your supervisor should
	  * be called again. */
	virtual unsigned long getStepCycles() = 0;
	/** Init your supervisor to this new simulation. The method of class
	  * <tt>SimSupervisor</tt> just does nothing. This method will be called on
	  * every registered SimSupervisor exactly once before the new simulation starts. */
	virtual void initSim(Simulation* const curSim);
	/** Execute your supervisor. This method will be called about every 
	  * <tt>getStepCycles()</tt> cycles. 
	  * @return The new simulation state. */
	virtual GameState exec(Simulation* const curSim) = 0;
	/** Process the end of a simulation. The method of class <tt>SimSupervisor</tt>
	  * just does nothing. This method will be called on every SimSupervisor exactly
	  * once after the current simulation has finished. 
	  * @param curSim The finished simulation. 
	  * @param simResult The simulation result. */
	virtual void exitSim(Simulation* const curSim, const GameState& simResult);

protected:
	SimSupervisor() {}
};

/** Another interface for supervising simulations. 
  */
class ExecSupervisor {
public:
	/** Init your supervisor to this new simulation. The method of class
	  * <tt>ExecSupervisor</tt> just does nothing. This method will be called on
	  * every registered ExecSupervisor exactly once before the new simulation starts. */
	virtual void initSim(Simulation* const curSim);
	/** This method will be called whenever an instruction is executed.
	  * @param instr The instruction to be executed.
	  * @param bank The bank to which that instruction belongs. 
	  * @param task The active task of the robot on which it will be executed. 
	  */
	virtual void onExec(Instr* instr, Bank* bank, Task* task) = 0;
	/** Process the end of a simulation. The method of class <tt>ExecSupervisor</tt>
	  * just does nothing. This method will be called on every ExecSupervisor exactly
	  * once after the current simulation has finished. 
	  * @param curSim The finished simulation. 
	  * @param simResult The simulation result. */
	virtual void exitSim(Simulation* const curSim);

	virtual ~ExecSupervisor() {}

protected:
	ExecSupervisor() {}
};

/** Loads a robot from a certain type of files. */
class ProgramLoader {
public:
	virtual ProgramLoader* create(const lrt::String& filename, const Globals& glob) = 0;
	virtual ~ProgramLoader();

	/// returns 0 if error. Checks "Language" version
	lrt::StringMap<Program::HeaderField>* getHeaders();
	/// returns 0 if error
	lrt::Vector<Bank*>* getBanks();

	virtual bool canLoad(const lrt::String& filename) = 0;

	/// returns a one-line format name, preferrably with extension (if applicable)
	virtual lrt::String getFormatName() = 0;

	/// Returns the file name from which the program was / will be loaded. 
	/// If the returned string is empty, means that this loader does not load from any file. 
	lrt::String getFileName() { return filename; }

	const ErrorHandler* getErrorHandler(); 
	void setErrorHandler(const ErrorHandler* handler);

	Program* owner;
protected:
	struct LoadReturnType {
		bool success;
		lrt::String message;
	};

	enum LoadState {
		lsError = -1,			///< tried to load, but got error
		lsNotLoaded = 0,		///< didn't try to load yet
		lsHeadersLoaded = 1,	///< loaded headers successfully
		lsLoaded = 2			///< loaded whole file successfully
	};

	LoadState state;
	lrt::Vector<Bank*>* banks;
	lrt::StringMap<Program::HeaderField>* headers;
	const Globals& glob;
	const ErrorHandler* handler; ///< error handler for load errors

	lrt::String filename; ///< robot file name
	
	ProgramLoader(const Globals& glob, const ErrorHandler* handler, const lrt::String& filename);
	virtual LoadReturnType loadHeaders() = 0;
	virtual LoadReturnType load() = 0;
	/// Checks if the given "RCxxx" RoboCom language version string is supported.
	bool isLanguageVersionSupported(const lrt::String& language);
};



} // namespace

#endif

