/** @file robotour.cpp
  * Interpretes command line parameters, and holds RoboTour together.
  */

#include <rtstring.h>
#include <rtcollect.h>
#include <rtlist.h>
#include <rtnetwork.h> // for __NETWORK__ define
#include <rtfile.h>

#include "robvars.h"
#include "robglob.h"
#include "robfrontend.h"
#include "robconsole.h"
#include "robmanage.h"
#include "robloader.h"
#include "robbase.h"

#ifdef __RBI__
#include "robbinary.h"
#endif

#ifdef __TELE__
#include "robtele.h"
#endif

#ifdef __WXWIN__
#include "robwxfe.h"
#endif

// Have Resources => Have Fun!
#include "resources.rh"

/** Since rtmain.h includes system includes, we may not import any namespace in
  * this file. Only in this file, really. */

enum WhichFrontend {
	UseTeleFrontend,
	UseWxFrontend,
	UseConsoleFrontend
};

WhichFrontend selectFrontend(const lrt::Array<lrt::String>& args)
{
#if defined(__NETWORK__) && defined(__TELE__)
	bool useTele = false;
	for(int at = 0; at < args.length(); at++)
		if(args[at].lowerCase() == "-tele")
			useTele = true;
	if(useTele)
		return UseTeleFrontend;
#endif // __NETWORK__ && __TELE__
#if defined(__WXWIN__)
	bool useWx = false;
	for(int aw = 0; aw < args.length(); aw++)
		if(args[aw].lowerCase() == "-vis")
			useWx = true;
	if(!args.length())
		useWx = true; 
	if(useWx)
		return UseWxFrontend;
#endif // __WXWIN__

	return UseConsoleFrontend;
}

rt::Frontend* createFrontend(WhichFrontend fe)
{
#if defined(__NETWORK__) && defined(__TELE__)
	if(fe == UseTeleFrontend)
		return new rt::TeleFrontend;
#endif
#if defined(__WXWIN__)
	if(fe == UseWxFrontend)
		return new rt::WxFrontend;
#endif
	return new rt::ConsoleFrontend;
}

int rtRun(const lrt::Array<lrt::String>& args)
{
	initAllResources();
	lrt::System::setAppName("RoboTour");

	rt::Frontend* frontend = createFrontend(selectFrontend(args));

	// create example loaders
	rt::Program::exampleLoaders += new rt::RobLoader(frontend);
#ifdef __RBI__
	rt::Program::exampleLoaders += new rt::RbiLoader(frontend);
#endif

	do{
		frontend->reset();
		if(!frontend->interpreteParams(args))
			goto rtMain_done;
		rt::SimOptions options(frontend);
		frontend->fillOptions(options);


		rt::SimManager* manager = frontend->getManager();
		manager->setup(&options);
		manager->run();
	
	}while(frontend->doAllAgain);

rtMain_done:
	delete frontend; deleteAll(rt::Program::exampleLoaders);
	return 0;
}

#ifndef __WXWIN__ // no wx-windows

// always include this file as the last one
#include <rtmain.h>

int rtMain(const lrt::Array<lrt::String>& args)
{
	return rtRun(args);
}

# else // with wx-windows

#include <wx/wx.h>
#include <wx/thread.h>
#include "robwxvis.h"
#include "robwxutil.h"

lrt::Array<lrt::String>* theArgs = 0; 

class RtWxApp : public wxApp {
public:
	int exitCode;
	virtual bool OnInit();
	virtual int OnExit();
};
DECLARE_APP(RtWxApp);

class SimThread : public wxThread {
public:
	virtual ExitCode Entry()
	{
		// Run simulations
		wxGetApp().exitCode = rtRun(*theArgs);
		// now RoboTour is done, so that, if the user closes the app, we'll leave. 
		wxTheApp->SetExitOnFrameDelete(true);
		// but if the frame *was* already closed, that won't help any more...
		if(!RobVisFrame::theVisFrame)
		{ // we'll have to exit manually now
#ifdef __WXMSW__ // special workaround: if no windows present, ExitMainLoop won't help within MSW
			if(!wxTheApp->GetTopWindow()) 
				System::exit(wxGetApp().exitCode);
			else
#endif // MSW
			wxTheApp->ExitMainLoop(); // should quit the program
		}
		// tell vis frame that we're done
		RobVisFrame::guiInfo.haveSimThread = false; 
		// finished
		return 0; 
	}
};

bool RtWxApp::OnInit()
{
	exitCode = 0; 
    SetAppName("RoboTour");
	SetVendorName("RoboComTeam");
	SetExitOnFrameDelete(false);
	RobVisFrame::theVisFrame = new RobVisFrame();
	RobVisFrame::theVisFrame->Show();
	SetTopWindow(RobVisFrame::theVisFrame);

	wxThread* simThread = new SimThread();
	simThread->Create();
	simThread->SetPriority(0);
	simThread->Run();
    return true;
}
int RtWxApp::OnExit()
{
	return exitCode; 
}

IMPLEMENT_APP_NO_MAIN(RtWxApp)

int main(int argc, char** argv)
{
	lrt::File::initialize(argv[0]);
	// convertStdArgsToLrtArgs()
	theArgs = new lrt::Array<lrt::String>(argc-1); 

	for(int i = 1; i < argc; i++)
	{
		(*theArgs)[i-1] = argv[i];
	}

	if(selectFrontend(*theArgs) == UseWxFrontend)
		return Wx::Startup();
	else
		return rtRun(*theArgs);
}

#endif // WX


