// robprofile.cpp: Implementierung der Klasse ProfilePlugin.
//
//////////////////////////////////////////////////////////////////////

#include "robprofile.h"
#include "robinstr.h"
#include "robhtml.h"

#include <rtfile.h>

using namespace lrt;

namespace rt {

FpInterpretePos ProfilePlugin::getInterpretePos()
{
	return fpInterPosBefore;
}

String ProfilePlugin::getName()
{
	return "Robot Code Profiler";
}

String ProfilePlugin::getHelpText()
{
	return "  -prof T    Create a profile of type T for all robots. Possible types are \n"
		   "             count (number of instr. exec.), time (spent during them) + fail.\n";
}

bool ProfilePlugin::interpreteParams(const Array<String>& params, Array<bool>& used)
{
	for(int i = 0; i < params.length(); i++)
		if(params[i].lowerCase() == "-prof") // Do profile.
		{
			used[i] = true;
			// get my profile type param
			if(used[i+1]) 
				parent->handleSystemError(1, "Parameter missing for argument -prof");
			used[i+1] = true;
			String type = params[i+1].lowerCase();
			if(type == "count") {
				mode = profCount;
			}
			else if(type == "time") {
				mode = profTime;
			}
			else if(type == "fail") {
				mode = profFailure;
			}
			else 
				parent->handleSystemError(1, "Invalid parameter for argument -prof");

			active = true;
		}
	return true;
}

void ProfilePlugin::fillOptions(SimOptions& options)
{
	if(active) {
		options.execSupervisors += new ProfileSupervisor(parent, mode);
	}
}

ProfilePlugin::ProfilePlugin(Frontend* parent) : FrontendPlugin(parent), mode(profCount)
{
}

///////////////////////// ProfileSimSupervisor

ProfileSupervisor::ProfileSupervisor(Frontend* parent, ProfileMode mode) : parent(parent), mode(mode)
{
}

void ProfileSupervisor::initSim(Simulation* const curSim)
{
	for(ProfileMap::Iterator it = profData.begin(); it.hasElement(); )
	{
		if(!isReachable((*it).getKey(), (*it).getValue().name, curSim)) {
			//System::println("No longer reachable: Bank '" + (*it).getValue().name + "', clearing profile info...");
			it.remove();
		}
		else {
			//System::println("Still reachable: Bank '" + (*it).getValue().name + "', keeping profile info...");
			++it;
		}
	}

}

void ProfileSupervisor::onExec(Instr* instr, Bank* bank, Task* task)
{
	ProfileBankData& bankData = profData.get(bank);
	if(!bankData.name.length()) bankData.name = bank->name;

	unsigned int val = 0;
	if(bankData.data.isSet(instr))
		val = bankData.data.get(instr);
	
	switch(mode) {
	  case profCount:
		val += 1;
		break;
	  case profTime:
		val += task->vars[taskCyclesDone];
		break;
	  case profFailure:
		{
		  unsigned int vTotal = (val & 0xFFFF0000) >> 16;
		  unsigned int vFailed = (val & 0xFFFF);
		  trool isFailed = detectFailure(instr, bank, task);
		  if(isFailed == trError) break;
		  vTotal++;
		  if(isFailed == trTrue) vFailed++;
		  if(vTotal >= 20000) { vTotal /= 2; vFailed /= 2; }
		  val = vFailed | (vTotal << 16);
		}
	    break;
	}

	bankData.data.put(instr, val); // first execution of this instruction
}

void ProfileSupervisor::exitSim(Simulation* const curSim)
{
	// for-each program in the simulation: print it, accompanied by the recorded numbers
	for(int prog = 0; prog < curSim->getNumPrograms(); prog++)
	{
		printProgram(curSim->getProgram(prog));
	}
}

void ProfileSupervisor::printProgram(Program* prog)
{
	ProgramLoader* loader = prog->getLoader();
	if(!loader) { parent->handleWarning("Cannot get loader. Cannot write program profile."); return; }
	String progFileName = loader->getFileName();
	if(!progFileName.length()) { parent->handleWarning("Program without file. Cannot write program profile."); return; }
	// Construct profile file name. The profile is saved in the robot folder, and has the
	// same name as the robot, but with a 'profile.' prefix. 
	File progFile(progFileName);
	String profileName = progFile.getParent() + "profile." + progFile.getFileName();
	// Open the profile file for writing.
	FileOutputStream os(profileName);
	if(os.fail()) { parent->handleWarning("Cannot create / write to profile file " + profileName + "."); return; }

	String name = prog->headers["Name"].value;
	System::println("Writing profile of robot '" + name + "' to file " + profileName + " ...");
	os.write("Profile of '" + name + "':\n");
	String desc;
	switch(mode) {
	  case profCount:
		desc = "Contains number of executions of each instruction.";
		break;
	  case profTime:
		desc = "Contains amount of cycles spent during all executions of each instruction.";
		break;
	  case profFailure:
		desc = "Contains permilleage of instruction executions that failed.";
		break;
	}
	os.write(desc + "\n");

	const Vector<Bank*>& banks = prog->getBanks();
	for(int b = 0; b < banks.length(); b++)
	{
		Bank* curBank = banks[b];
		if(curBank->isSecret) // don't print RBI banks!
			continue;
		os.write(String("\nBank ") + curBank->name + " (" + (b+1) + ")\n");
		printInstrs(curBank, &os);
	}

	os.close(); 

	printHtml(profileName); 
}

void ProfileSupervisor::printInstrs(Bank* bank, OutputStream* os)
{
	ProfileBankData& bankData = profData.get(bank);
	
	for(int i = 0; i < bank->instr.length(); i++)
	{
		Instr* curInstr = bank->instr[i];
		unsigned int data = 0;
		if(bankData.data.isSet(curInstr))
			data = bankData.data.get(curInstr);
		unsigned int val = 0;
		
		switch(mode) {
		  case profTime:
		  case profCount:
		    val = data;
			break;
		  case profFailure:
			{
			  int vFailed = (data & 0xFFFF), vTotal = ((data & 0xFFFF0000) >> 16);
			  if(vTotal) val =  vFailed * 1000 / vTotal;
			}
			break;
		}
		os->write(String((int)val) + "\t " + curInstr->print(i) + "\n");
	}
}

class ProfileLine 
{
public:
	ProfileLine() : haveNumber(false), number(0) {}

	bool read(ParseInputStream* in)
	{
		if(in->eos())
			return false; 

		String line = in->getLine(false);
		String lineTrimmed = line.trim();
		if((lineTrimmed.length() > 0) && (lineTrimmed[0] >= '0') && (lineTrimmed[0] <= '9'))
		{
			// number found!
			haveNumber = true; 
			Array<String> nt = lineTrimmed.split(" \t", "", 2);
			number = nt[0].intValue(0);
			text = nt[1];
		}
		else
		{
			// no number found
			haveNumber = false;
			number = 0;
			text = line; 
		}
		return true; 
	}

	String text;
	int number;
	bool haveNumber; 
};


void ProfileSupervisor::printHtml(const String& profileFile)
{
	String outFile = profileFile; 
	int dotIndex; 
	if((dotIndex = outFile.lastIndexOf('.')) > 0)
		outFile = outFile.substring(0, dotIndex) + ".html";
	else
		outFile += ".html"; 

	FileInputStream* fin = new FileInputStream(profileFile, true);
	if(fin->fail()) { parent->handleWarning("cannot open profile file"); delete fin; return; }
	ParseInputStream pin(fin, ParseInputStream::whitespace);

	FileOutputStream fout(outFile);
	if(fout.fail())
		return; 

	// read the file
	Vector<ProfileLine> lines;
	{
		ProfileLine line; 
		while(line.read(&pin))
			lines += line; 
	}
	fin->close();
	// calculate largest number
	int max = 1;
	for(int m = 0; m < lines.length(); m++)
		max = Math::max(max, lines[m].number);

	// write output file
	fout.write("<html><head><title>Profile for " + profileFile + "</title></head>\n"
               "<body style=\"font-family:lucidatypewriter,lucida console,courier new,courier,monospaced,monospace\">\n"
               "<table border=0 cellpadding=0 cellspacing=0>\n");

	// process the lines
	for(int l = 0; l < lines.length(); l++)
	{
		ProfileLine& line = lines[l];
		int colnum = (int)(Math::sqrt((double)line.number / (double)max) * 255);
		if(colnum) colnum = Math::max(colnum, 15);

		String colhex(255 - colnum, 16, 2);
		colhex = colhex.replace(' ', '0');
		String colstr = "#FF" + colhex + colhex; 

		if(line.text.trim().lowerCase().startsWith("bank"))
			colstr = "#B8B8FF";

		fout.write("<tr bgcolor=\"" + colstr + "\"><td align=\"right\">");
		if(line.haveNumber) fout.write(String(line.number));
		fout.write("&nbsp;&nbsp;&nbsp;<td>" + HtmlCreator::toHtml(line.text) + "</tr>\n");
	}

	// footer
	fout.write(String("</table>\n</body>\n</html>\n"));
}

bool ProfileSupervisor::isReachable(Bank* b, const String& name, Simulation* sim)
{
	for(int p = 0; p < sim->getNumPrograms(); p++)
	{
		const Vector<Bank*>& banks = sim->getProgram(p)->getBanks();
		for(int bb = 0; bb < banks.length(); bb++)
		{
			if(banks[bb] == b)
			{
				if(name == banks[bb]->name)
					return true;
				else {
					System::println("WARNING: identical bank pointer, but name has changed from '" 
					    + name + "' to '" + banks[bb]->name + "'!!!");
					return false;
				}
			}
		}
	}
	return false;
}

ProfileSupervisor::trool ProfileSupervisor::detectFailure(Instr* instr, Bank* bank, Task* task)
{
	String name = instr->getName().lowerCase();
	Bot* refBot = task->getRefBot();
	if((name == "create") || (name == "move")) 
		return(refBot ? trTrue : trFalse); // create and move fail if refBot exists (cannot build/go there)
	else if((name == "trans") || (name == "rtrans"))
		return(refBot ? trFalse : trTrue); // XTrans fails if no refBot exists (trans to nowhere)
	else
		return trError; // cannot detect failure, or instruction cannot fail, or whatever. 
}

} // namespace

