/** @file robsupervis.cpp
  * Implements the standard sim supervisors.
  */

#include "robsupervis.h"
#include "robbase.h"

#include <rtcollect.h>
#include <rtstring.h>
#include <rtmath.h>

using namespace lrt; 

namespace rt {

///// //// /// // / StdBotCreator / // /// /////
StdBotCreator::StdBotCreator()
{
	retState.event = gameNothing;
}

StdBotCreator::~StdBotCreator() {}

SimSuperPosition StdBotCreator::getPreferredPosition()
{
	return supPosBotCreator;
}

unsigned long StdBotCreator::getStepCycles()
{ // we DON'T want to be called during the game
	return 1000000;
}

void StdBotCreator::initSim(Simulation* const sim)
{
	const Globals& glob = sim->glob;

	for(int p = 0; p < sim->getNumPrograms(); p++) {
		Program* prog = sim->getProgram(p);
		const Vector<Bank*>& banks = prog->getBanks();
		if(banks.length() == 0) // probably, bot couldn't be loaded
			continue;

		// find an empty position on the field
		rint dir = Math::rand(0, 3);
		rint x, y;

		x = Math::rand(0, sim->fieldWidth - 1);
		y = Math::rand(0, sim->fieldHeight - 1);

		while(sim->field[x + y * sim->fieldWidth] != 0)
		{
			x++;
			if( x >= sim->fieldWidth)
			{
				x = 0;
				y++;
				y %= sim->fieldHeight;
			}
		}

		// create motherbot
		Bot* motherBot = new Bot(2, banks.length(), glob.firstRobotMobile, 
			x, y, dir, 0, prog); 
		for(int b = 0; b < banks.length(); b++) // copy banks
			motherBot->banks[b] = banks[b];
		motherBot->vars[botActive] = 1; // activate
	}
}

SimSupervisor::GameState StdBotCreator::exec(Simulation* const sim)
{
	return retState;
}

//// /// // / StdVictorySupervisor / // /// ////
StdVictorySupervisor::StdVictorySupervisor()
{
	retState.event = gameNothing;
}

StdVictorySupervisor::~StdVictorySupervisor()
{
}

SimSuperPosition StdVictorySupervisor::getPreferredPosition()
{ // supPos is only really interesting for init() handlers
	return supPosDontCare;
}

unsigned long StdVictorySupervisor::getStepCycles()
{ // we need to be called every cycle!
	return 1;
}

void StdVictorySupervisor::initSim(Simulation* const sim)
{
	retState.event = gameNothing;
}

SimSupervisor::GameState StdVictorySupervisor::exec(Simulation* const sim)
{
	int numProgs = sim->getNumPrograms();
	// no programs -> kill sim
	if(numProgs == 0) { retState.event = gameTie; return retState; }

	// fetch number of bots
	int allNumBots = sim->getAllNumBots();
	// no bots => "No program left in simulation" (tie)
	if(allNumBots == 0) { retState.event = gameTie; return retState; }

	// only one program -> let sim time out (so that the player can watch his prog)
	else if(numProgs == 1) { 
		retState.event = gameNothing; return retState; }
	
	// two or more programs, so let's work!
	for(int p = 0; p < numProgs; p++) {
		Program* prog = sim->getProgram(p);
		if(prog->vars[progMybots] == allNumBots) // one program has got all bots => won!
		{
			retState.event = gameFinished;
			retState.affectedProgram = p;
			return retState;
		}
	}
	// no one won
	retState.event = gameNothing;
	return retState;
}



} // namespace
