#include "robvis.h"
#include "robwxvis.h"
#include "robwxfe.h"

using namespace lrt;

namespace rt {

// mini helpie function: checks if we have a frame to do something for
bool HaveVisFrame()
{
	return !RobVisFrame::guiInfo.isDestroyed; 
}

// helpey function: waits until the event was really processes
// ALWAYS use this one!
void PostAndWait(wxEvent& evt, int maxWait)
{
	if(!HaveVisFrame())
		return; 

	static int numProcessors = wxThread::GetCPUCount();

	int numEventsSent = (++RobVisFrame::guiInfo.numEventsSent);
	// On Multiprocessor systems, and if this event is not a critical one, 
	// continue calculating while max. ONE event is still being processed by the GUI thread
	// (Don't do it on SingleProcessor systems, because it will make the GUI un-reactive)
	RoboTourEvent* evtAsRT = wxDynamicCast(&evt, RoboTourEvent);
	bool critical = (evtAsRT && (evtAsRT->retData == 0));
	if((numProcessors > 1) && critical && (maxWait >= 0)) 
		numEventsSent--; // relaxed waiting condition, which is faster

	wxPostEvent(RobVisFrame::theVisFrame, evt);
	
	int msWait = 0; 
	while(HaveVisFrame() && (RobVisFrame::guiInfo.restart != gtourExit)  
	  && (RobVisFrame::guiInfo.numEvents < numEventsSent) && ((maxWait < 0) || (msWait < maxWait)))
		wxThread::Sleep(1); // not yet finished
}


FpInterpretePos VisPlugin::getInterpretePos()
{
	return fpInterPosBefore; 
}

String VisPlugin::getName()
{
	return "Robot Visualization Engine";
}

String VisPlugin::getHelpText()
{
	return "  -vis       Enable built-in graphical visualization during simulation";
}
bool VisPlugin::interpreteParams(const Array<String>& params, Array<bool>& used)
{
	if(dynamic_cast<WxFrontend*>(parent))
		active = true; 

	for(int i = 0; i < params.length(); i++)
		if(params[i] == "-vis") 
		{
			used[i] = true; 
			active = true; 
		}
	return true; 
}

void VisPlugin::fillOptions(SimOptions& options)
{
	if(active) {
		VisDebugSupervisor* debugsup = new VisDebugSupervisor(parent);
		VisSupervisor* vsup = new VisSupervisor(debugsup); 
		options.supervisors += vsup;
		options.supervisors += debugsup;
		options.tourDisps += new VisTourDisplayer(vsup);

		RoboTourEvent evt(EVT_RT_SHOW_TYPE);
		PostAndWait(evt);
	}
}

VisPlugin::VisPlugin(Frontend* parent) : FrontendPlugin(parent)
{
}

VisPlugin::~VisPlugin()
{
}


VisSupervisor::VisSupervisor(VisDebugSupervisor* debugger) : initInfo(0,0), debugger(debugger), watch(new wxStopWatch)
{
}

VisSupervisor::~VisSupervisor()
{
	delete watch;
}

unsigned long VisSupervisor::getStepCycles()
{
	// observe the special case with 'Exit': if abortPressed and the vis frame is closed already 
	// we still have to process it
	if(!HaveVisFrame() && !RobVisFrame::guiInfo.abortPressed) 
		return 100000; // Frame closed => nothing to do

	if(RobVisFrame::guiInfo.slowMode)
		return 1;
	else
		return RobVisFrame::guiInfo.speed;
}

void VisSupervisor::initSim(Simulation* const curSim)
{
	curSimNum++;

	if(!HaveVisFrame())
		return;

	while(HaveVisFrame() && !RobVisFrame::guiInfo.startLocked && !RobVisFrame::guiInfo.startPressed && !RobVisFrame::guiInfo.abortPressed)
		processGuiRequests(curSim);
	RobVisFrame::guiInfo.startPressed = false; 

	RoboTourEvent evt(EVT_RT_INITSIM_TYPE);
	evt.LoadSimData(curSim, curSimNum);
	evt.LoadTourData(curSim, tourStatus, initInfo);
	PostAndWait(evt);
}

SimSupervisor::GameState VisSupervisor::exec(Simulation* const curSim)
{
	GameState state; 
	state.affectedProgram = 0;
	state.event = gameNothing; 

	if(RobVisFrame::guiInfo.abortPressed)
	{
		RobVisFrame::guiInfo.abortPressed = false;
		state.event = gameTie;
		return state;
	}

	if(!HaveVisFrame())
		return state;

	bool delayFinished = true;
	do {
		processGuiRequests(curSim);
		
		// additional delay may be required because we're in slow or pause mode
		if(RobVisFrame::guiInfo.pausePressed)
			delayFinished = false;
		else if(RobVisFrame::guiInfo.slowMode && (watch->Time() < RobVisFrame::guiInfo.speed))
			delayFinished = false;
		else
			delayFinished = true;

	} while(HaveVisFrame() && !delayFinished && !RobVisFrame::guiInfo.abortPressed);
	watch->Start();

	RoboTourEvent evt(EVT_RT_STEPSIM_TYPE);
	evt.LoadSimData(curSim, curSimNum);
	evt.LoadBotData(curSim);
	evt.LoadTourData(curSim, tourStatus, initInfo);
	PostAndWait(evt);

	return state; 
}

void VisSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
	if(!HaveVisFrame())
		return;
	RoboTourEvent evt(EVT_RT_EXITSIM_TYPE);
	evt.LoadSimData(curSim, curSimNum);
	if(simResult.event == gameTie || simResult.event == gameAbort)
		evt.simData.winner = -2;
	else if(simResult.event == gameFinished) {
		evt.simData.winner = simResult.affectedProgram;
		evt.simData.winnerName = curSim->getProgram(simResult.affectedProgram)->headers["name"].value;
	}
	PostAndWait(evt);
}

void VisSupervisor::updateTour(const Array<TourResult>& tourStatus)
{
	this->tourStatus.clear();
	this->tourStatus += tourStatus; 
}

void VisSupervisor::processGuiRequests(Simulation* sim)
{
	if(RobVisFrame::guiInfo.reqPlayerInfo >= 0)
	{
		// create player info 
		RoboTourEvent evt(EVT_RT_PLAYER_TYPE);
		evt.LoadPlayerData(RobVisFrame::guiInfo.reqPlayerInfo, sim, tourStatus);
		PostAndWait(evt);
		RobVisFrame::guiInfo.reqPlayerInfo = -1; 
	}
	// other requests here...
	else // no request pending
	{
		wxThread::Sleep(1); // so that time passes by
	}
	if(debugger)
		debugger->processGuiRequests(sim);
}

//////////////////// VisTourDisplayer ///////////
VisTourDisplayer::VisTourDisplayer(VisSupervisor* super)
	: super(super)
{
}

TourStatusType VisTourDisplayer::update(const Array<TourResult>& status)
{
	super->updateTour(status);

	if(RobVisFrame::guiInfo.abortTourPressed)
	{
		RobVisFrame::guiInfo.abortPressed = false;
		RobVisFrame::guiInfo.abortTourPressed = false;
		return tourAbort; 
	}

	if(!HaveVisFrame())
		return tourContinue;

	RoboTourEvent evt(EVT_RT_STEPTOUR_TYPE);
	evt.LoadTourData(0 /* no simulation */, status, super->initInfo);
	PostAndWait(evt);

	return tourContinue; 
}



////////////////////////////// VisDebugSupervisor ////////

/*class VisDebugSupervisor : public SimSupervisor, public BotErrorHandler 
{
public:
	VisDebugSupervisor(Frontend* parent);
	~VisDebugSupervisor();
	virtual unsigned long getStepCycles();
	virtual void initSim(Simulation* const sim);
	virtual GameState exec(Simulation* const curSim);
	virtual void exitSim(Simulation* const curSim, const GameState& simResult);

private: 
	void processGuiRequests(Simulation* sim); // my own (debug) requests

	Map<int, DebugInfo> debugInfo;
	WxFrontend* parent; // can be 0 (if another frontend is being used)
};*/

VisDebugSupervisor::VisDebugSupervisor(Frontend* fparent) : nextDebugID(1)
{
	parent = dynamic_cast<WxFrontend*>(fparent);
	if(parent)
		parent->addBotErrorHandler(this);
}

VisDebugSupervisor::~VisDebugSupervisor()
{
	if(parent)
		parent->removeBotErrorHandler(this);
}

unsigned long VisDebugSupervisor::getStepCycles()
{
	if(!HaveVisFrame())
		return 100000;
	else
		return 1;
}

void VisDebugSupervisor::initSim(Simulation* const sim)
{
}

SimSupervisor::GameState VisDebugSupervisor::exec(Simulation* const curSim)
{
	int oldID = nextDebugID;
	processGuiRequests(curSim);
	for(Map<int,DebugInfo>::Iterator iter = debugInfo.begin(); iter.hasElement();)
	{
		DebugInfo& di = iter.get().getValue();
		bool keepDI = true; // keep the DI in the map after this iteration?

		if(di.debugID < oldID) // is an old request, and was not yet sent
		{
			keepDI = false; // need to find it!
			if(!di.alive) lrt::System::exit(201, "dead bot in debugInfo");
			// Simulation.find(bot);
			for(List<Bot*>::Node* n = curSim->bots.first(); n; n = n->getNext())
			{
				if(di.botID == (int)n->accessElement()) // here it is
				{
					if(di.updateFrom(n->accessElement(), true)) // data has changed
					{
						DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
						evt.data = di;
						PostAndWait(evt);
					}
					keepDI = true;
					break;
				}
			}
		}
		if(!keepDI)
			iter.remove();
		else
			++iter;
	}

	SimSupervisor::GameState result;
	result.affectedProgram = 0; result.event = gameNothing;
	return result;
}

void VisDebugSupervisor::exitSim(Simulation* const curSim, const GameState& simResult)
{
	RobVisFrame::guiInfo.reqDebug.clear(); // requests from the old simulation are no longer relevant in the next one

	for(Map<int,DebugInfo>::Iterator iter = debugInfo.begin(); iter.hasElement(); iter.remove())
	{
		DebugInfo& di = iter.get().getValue();
		di.alive = false;
		DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
		evt.data = di;
		PostAndWait(evt);
	}
}

void VisDebugSupervisor::processGuiRequests(Simulation* sim) // my own (debug) requests
{
	while(true)
	{
		Request* req_ = RobVisFrame::guiInfo.reqDebug.removeFirst();
		if(!req_) break; // no more requests;
		DebugRequest* req = dynamic_cast<DebugRequest*>(req_);
		if(req)
		{
			if(req->debugX >= 0 && req->debugY >= 0 && req->debugX < sim->fieldWidth && req->debugY < sim->fieldHeight)
			{
				// find bot at field
				Bot* bot = sim->field[req->debugX + req->debugY * sim->fieldWidth];
				// start debugging it
				if(bot) {
					DebugInfo& di = debugInfo[nextDebugID]; // insert into map
					di.updateFrom(bot, true);
					di.debugID = nextDebugID;
					DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
					evt.data = di;
					PostAndWait(evt);
					nextDebugID++;
				}
			}

			if(req->debugCloseID >= 0)
			{
				if(debugInfo.isSet(req->debugCloseID))
				{
					debugInfo.remove(req->debugCloseID);
				}
			}

		}
		delete req_;
	}
}

void VisDebugSupervisor::handleBotError(const Bot* affectedBot, ExecReturnType type)
{
	// debugInfo.find(affectedBot);
	for(Map<int,DebugInfo>::Iterator iter = debugInfo.begin(); iter.hasElement();)
	{
		if(iter.get().getValue().botID == (int)affectedBot)
		{
			DebugInfo& di = iter.get().getValue();
			di.alive = false;
			DebugEvent evt(EVT_DEBUG_UPDATE_TYPE);
			evt.data = di;
			PostAndWait(evt);
			iter.remove();
		}
		else
			++iter;
	}

}



} // namespace
