#include "robwxfe.h"
#include "robvis.h"
#include "robwxvis.h"

#include "robstrings.h"
#include "robsound.h"
#include "robprofile.h"

namespace rt {

WxFrontend::WxFrontend() 
{
	doAllAgain = true; // we want to support multiple tournaments in a row
}

WxFrontend::~WxFrontend()
{
}

void WxFrontend::handleBotError(const Bot* affectedBot, ExecReturnType type) const
{
	// display it in the visualization
	if(HaveVisFrame()) {
		RoboTourEvent evt(EVT_RT_MSGROBERR_TYPE);
		evt.msgData.id = type;
		evt.msgData.msg = String("robot death: ") + affectedBot->vars[botPosX] + 
			"/" + affectedBot->vars[botPosY] + ": " + failMsg[-type];
		evt.msgData.program = affectedBot->owner->programNum;
		evt.LoadSimData(affectedBot->owner->sim, -1 /* don't know the sim num right now! */);
		PostAndWait(evt, 100);
	}
	// also display it on the console (eventually)
	ConsoleFrontend::handleBotError(affectedBot, type);

	// and inform the listeners
	for(const lrt::List<BotErrorHandler*>::Node* n = botErrorHandlers.first(); n; n = n->getNext())
		n->accessElement()->handleBotError(affectedBot, type);
}

void WxFrontend::handleLoadError(const String& affectedFile, const String& message) const
{
	if(HaveVisFrame()) {
		RoboTourEvent evt(EVT_RT_MSGERROR_TYPE);
		evt.msgData.msg = "load error: " + affectedFile + ": " + message;
		PostAndWait(evt, 100);
	}
	ConsoleFrontend::handleLoadError(affectedFile, message);
}

void WxFrontend::handleSystemError(int num, const String& message) const
{
	if(HaveVisFrame()) {
		wxMessageBox(wxString::Format("fatal error: %d - %s", num, message.cStr()), "RoboTour Fatal Error", wxOK | wxICON_ERROR);
		wxTheApp->ExitMainLoop();
		wxThread::Sleep(500); // give wx time to shut down
		System::exit(num);
	}
	else
		ConsoleFrontend::handleSystemError(num, message);
}

void WxFrontend::handleWarning(const String& message) const
{
	if(HaveVisFrame()) {
		RoboTourEvent evt(EVT_RT_MSGWARN_TYPE);
		evt.msgData.msg = "warning: " + message;
		PostAndWait(evt, 100);
	}
	ConsoleFrontend::handleWarning(message);
}

bool WxFrontend::interpreteParamsImpl(const Array<String>& params, Array<bool>& used)
{
	// interprete command line parameters
	bool consRet = ConsoleFrontend::interpreteParamsImpl(params, used);
	// enough parameters given => just start with them, but only once
	// (otherwise the same tournament would be repeated on and on)
	if(files.length() && consRet) {
		doAllAgain = false; 
		return consRet;
	}

	SoundPlugin* sp = (SoundPlugin*) findPlugin("Sound Plugin");
	ProfilePlugin* pp = (ProfilePlugin*) findPlugin("Robot Code Profiler");

	// wait until the user decides to setup a new tournament
	RobVisFrame::guiInfo.abortTourPressed = false; 
	RobVisFrame::guiInfo.abortPressed = false;
	do {
		switch(RobVisFrame::guiInfo.restart) {
		  case gtourExit:
			doAllAgain = false;
			return false;
		  case gtourNotYet:
			break;
		  case gtourNew:
		  {
			TourSetupInfo info;
			info.files = files;
			info.optionFile = optionFile;
			info.numRepeats = numRepeats;
			info.type = (TourSetupInfo::TourType) fightType;
			if(sp && sp->canPlay()) { info.haveSound = true; info.sound = sp->isActive(); }
			if(pp) { info.haveProfiler = true; info.profileType = pp->getMode(); }

			RoboTourEvent evt(EVT_RT_SETUPDLG_TYPE);
			evt.retData = &info;
			PostAndWait(evt, WaitInfinite);
			// ok, now we want to setup one tournament, but for the next one, we'll ask again...
			RobVisFrame::guiInfo.restart = gtourNotYet; 
			if(info.ok)
			{
				files += info.files;
				optionFile = info.optionFile;
				numRepeats = info.numRepeats;
				fightType = (FightType) info.type;
				if(sp && sp->canPlay()) { sp->setActive(info.sound); }
				if(pp) { pp->setActive(info.profile); pp->setMode((ProfileMode) info.profileType); }
				return true; 
			}
			else break;
		  }
		}
 		wxThread::Sleep(1);
	}while(HaveVisFrame() && (RobVisFrame::guiInfo.restart != gtourExit));

	return false; 
}

void WxFrontend::addBotErrorHandler(BotErrorHandler* handler)
{
	botErrorHandlers.append(handler);
}

void WxFrontend::removeBotErrorHandler(BotErrorHandler* handler)
{
	for(lrt::List<rt::BotErrorHandler*>::Node* n = botErrorHandlers.first(); n;)
	{
		if(n->accessElement() == handler)
			n = botErrorHandlers.remove(n);
		else
			n = n->getNext();
	}
}


} // namespace
