/*
 * rtfile.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

//////////////// Common elements of File.

#include "rtfile.h"
#include "rtstring.h"
#include "rtstreams.h"
#include "rtmath.h"

namespace lrt {

File *File::executableFile = 0;
File *File::homeFolder = 0;
File *File::settingsFolder = 0;
File *File::currentFolder = 0;
	
File::File() : names(0), attribs(0)
{
}

File::File(const String &name) : attribs(new FileAttributes())
{
	//this(*currentFolder, name);
	//	File(*currentFolder, name);

	String fname = name.replace('/', separatorChar);
	names = new Array<String>(fname.split(separator, "")); 
	if(!isAbsolute(name))
	  currentFolder->resolve(this, fname);
	if(fname.endsWith(separator))
		attribs->isFolder = ATTRIB_TRUE;

}

File::File(const File &other) : names(new Array<String>(*other.names)), 
	attribs(new FileAttributes(*other.attribs))
{
}

File::File(const File &parent, const String &child) : attribs(new FileAttributes())
{
	String fname = child.replace('/', separatorChar);
	names = new Array<String>(fname.split(separator, "")); 
	if(!isAbsolute(child))
	  parent.resolve(this, fname);
	if(fname.endsWith(separator))
		attribs->isFolder = ATTRIB_TRUE;
}

File &File::operator =(const File& other)
{
	set(new Array<String>(*(other.names)), new FileAttributes(*(other.attribs)));
	return *this;
}

File::~File()
{
	delete names;
	delete attribs;
}

bool File::createFolder(const File& theFolder)
{
	File folder(theFolder);
	if(folder.exists()) return true;
	bool ret = true;
	int end = folder.names->length();
	if(!folder.isFolder()) end--;
	if(end <= 0) return true;

	String curFn = getAbsolutePrefix();
	for(int i = 0; i < end; i++)
	{
		curFn += folder.names->operator[](i);
		curFn += separatorChar;
		File cur(curFn);
		if(!cur.exists())
			ret &= createSingleFolder(cur);
	}
	return ret;
}

void File::setCurrentFolder(const File &newFolder)
{
	delete currentFolder;
	if(newFolder.isFolder())
	  currentFolder = new File(newFolder);
	else
	  currentFolder = new File(newFolder.getParentFile());
}

const String File::getCurrentFolder()
{
	return currentFolder->getName();
}

const File File::getExecutableFile()
{
	return *executableFile;
}

const File File::getHomeFolder()
{
	return *homeFolder;
}

const File File::getSettingsFolder()
{
	return *settingsFolder;
}

void File::finalize()
{
	delete executableFile;
	delete currentFolder;
	delete homeFolder;
	delete settingsFolder;
}

String File::getName() const
{
	String ret = getAbsolutePrefix();
	ret += String::join(*names, separator);
	if(isFolder()) ret += separator;
	return ret;
}

String File::getFileName() const
{
	if(names->length() == 0)
		return getAbsolutePrefix();
	else
		return (*names)[names->length() - 1];
}

String File::getLocalName(const File& relFile) const
{
	File myRel = relFile;
	if(!relFile.isFolder()) 
		myRel = relFile.getParentFile();

	int numMatching = 0;
	int smallLength = Math::min(names->length(), myRel.names->length());
	for(int i = 0; i < smallLength; i++)
	{
		if((caseSensitive && (names->operator[](i).compare(myRel.names->operator[](i))))
			|| (!caseSensitive && (names->operator[](i).compareIgnoreCase(myRel.names->operator[](i)))))
			break;
		numMatching++;
	}
	if(numMatching == 0) // no matching path parts => return absolute
		return getName();

	String ret;

	// create doubledots "../"
	for(int dd = numMatching; dd < myRel.names->length(); dd++)
	{
		ret += "..";
		ret += separatorChar;
	}

	// create subpaths
	for(int b = numMatching; b < names->length(); b++)
	{
		ret += (*names)[b];
		if((b < names->length() - 1) || isFolder())
		    ret += separatorChar;
	}
	
	if(ret == "") // everything matches!
		ret = ".";

	return ret;
}

String File::getParent() const
{
	int upTo = ((names->length() == minNameCount) ? names->length() - 1  : names->length() - 2);
	String ret = getAbsolutePrefix();
	ret += String::join(*names, separator, upTo);
	ret += separator;
	return ret;
}

File File::getParentFile() const
{
	int newLen = ((names->length() == minNameCount) ? names->length() : names->length() - 1);
	Array<String>*nNames = new Array<String>(newLen);
	for(int i = 0; i < newLen; i++)
		(*nNames)[i] = (*names)[i];
	FileAttributes *nAttr = new FileAttributes(*attribs);
	nAttr->isFolder = ATTRIB_TRUE;
	return File(nNames, nAttr);
}

bool File::exists() 
{
	if(attribs->exists == ATTRIB_UNKNOWN)
		fetchAttributes();
	return (attribs->exists == ATTRIB_TRUE);
}

bool File::isFolder() const
{
	return (attribs->isFolder == ATTRIB_TRUE);
}

bool File::isFile() const
{
	return (attribs->isFolder == ATTRIB_FALSE);
}

bool File::canRead() 
{
	if(attribs->canRead == ATTRIB_UNKNOWN)
		fetchAttributes();
	return (attribs->canRead == ATTRIB_TRUE);
}

bool File::canWrite() 
{
	if(attribs->canWrite == ATTRIB_UNKNOWN)
		fetchAttributes();
	return (attribs->canWrite == ATTRIB_TRUE);
}

int File::getSize()
{
	if(attribs->size < 0)
		fetchAttributes();
	return attribs->size;
}

Time File::getLastModified()
{
	// can't easily determine if att->lastModified is valid, so use that one:
	if(attribs->exists == ATTRIB_UNKNOWN)
		fetchAttributes();
	return attribs->lastModified;
}


InputStream *File::openRead(bool textMode)
{
	return new FileInputStream(*this, textMode);
}

OutputStream *File::openWrite()
{
	return new FileOutputStream(*this, false);
}

OutputStream *File::openAppend()
{
	return new FileOutputStream(*this, true);
}

Array<File> File::list()
{
	return listImpl(0);
}

Array<File> File::list(const IFilenameFilter* filter)
{
	return listImpl(filter);
}



/////// private

//! This constructor does not copy anything!! 
//! I.e. you must not use names & attribs after calling it again
File::File(Array<String>* names, FileAttributes *attribs) : 
	names(names), attribs(attribs)
{
}

//! set does not copy anything!
//! I.e. you must have copied the parameters before!
void File::set(Array<String>* names, FileAttributes *attribs)
{
	delete this->names;
	delete this->attribs;
	this->names = names;
	this->attribs = attribs;
}

//! set does not copy anything!
//! I.e. you must have copied the parameters before!
void File::set(Array<String>* names)
{
	delete this->names;
	this->names = names;
}

void File::resolve(File *file, const String& fname) const
{
	if(!resolveExtra(file, fname))
	{
		if(file->names->length() == 0) file->set(new Array<String>(*names), new FileAttributes(*attribs));
		int endParent = names->length(), startChild = 0;
		int childLength = file->names->length();
		while((startChild < childLength) && 
			((file->names->operator[](startChild) == ".") || (file->names->operator[](startChild) == "..")))
		{
			if(file->names->operator[](startChild) == "..")
				if(endParent > minNameCount) endParent--;
			startChild++;
		}
		Array<String> *nNames = new Array<String>(endParent + childLength - startChild);
		for(int i = 0; i < endParent; i++)
			(*nNames)[i] = (*names)[i];
		for(int j = 0; j < childLength - startChild; j++)
			(*nNames)[endParent + j] = (*file->names)[startChild + j];
		file->set(nNames);
		if(startChild == childLength) // child vanished in parent => if parent is folder, child too
		{
			file->attribs->isFolder = attribs->isFolder;
			if(endParent < names->length()) // we've even had to cut something from parent => certainly a folder now
				file->attribs->isFolder = ATTRIB_TRUE;
		}
	}
}

File::FileAttributes::FileAttributes() : exists(ATTRIB_UNKNOWN), 
	isFolder(ATTRIB_UNKNOWN), canWrite(ATTRIB_UNKNOWN), canRead(ATTRIB_UNKNOWN), 
	size(-1), lastModified()
{
}

File::FileAttributes::FileAttributes(const FileAttributes& att) : exists(att.exists),
    isFolder(att.isFolder), canRead(att.canRead), canWrite(att.canWrite),
	size(att.size), lastModified(att.lastModified)
{
}


///////////////// StarFilenameFilter //////////////

Array<File> StarFilenameFilter::getFiles(const String& search)
{
	String mySearch(search.replace('/', File::separatorChar));
	int sepIn = mySearch.lastIndexOf(File::separatorChar);
	File folder = (sepIn < 0) ? File::getCurrentFolder() : File(mySearch.substring(0, sepIn + 1));
	if(!folder.exists()) return Array<File>(0);
	StarFilenameFilter* filter = new StarFilenameFilter(mySearch.substring(sepIn + 1));
	Array<File> files(folder.list( filter ));
	delete filter;
	return files;
}

bool StarFilenameFilter::accept(const File& parent, const String& child) const
{
	String myChild(File::caseSensitive ? child : child.lowerCase());

	if(findQM(myChild, queryComps[0]) != 0) return false;
	int pos = 0;
	for(int i = 1; i < queryComps.length() - 1; i++)
	{
		pos = findQM(myChild, queryComps[i], pos);
		if(pos < 0) return false;
		pos += queryComps[i].length();
	}

	const String& lastQC = queryComps[queryComps.length() - 1];
	int lastQCSearchPos = myChild.length() - lastQC.length();
	if(findQM(myChild, lastQC, lastQCSearchPos) != lastQCSearchPos) return false;

	return true;
}

StarFilenameFilter::~StarFilenameFilter() {}

StarFilenameFilter::StarFilenameFilter(const String& query) : queryComps(0)
{
	String myQuery(File::caseSensitive ? query : query.lowerCase());

	int starPos, oldStarPos = -1;

	// first star
	starPos = myQuery.indexOf('*'); 
	if(starPos < 0) { // no star in query
		queryComps += myQuery; // startsWith check
		queryComps += myQuery; // endsWith check
		return;
	}

	// more stars
	while((starPos = myQuery.indexOf('*', oldStarPos + 1)) >= 0) {
		queryComps += myQuery.substring(oldStarPos + 1, starPos);
		oldStarPos = starPos;
	}

	// endsWith check
	queryComps += myQuery.substring(oldStarPos + 1);
}

int StarFilenameFilter::findQM(const String& a, const String& b, int fromPos) const 
{
	if(b.length() > a.length()) return -1; // cannot be contained in this string
	int curIndex = Math::max(fromPos, 0);
	while(curIndex <= a.length() - b.length()) 
	{
		char chA, chB;
		for(int i = 0; i < b.length(); i++)
		{
			chA = a[curIndex + i];
			chB = b[i];
			if((chA != '?') && (chB != '?') && (chA != chB))
				goto findQM_nomatch;
		}
		return curIndex;

findQM_nomatch:
		curIndex++;
	}
	return -1;
}


} // namespace

#ifdef __SYMBIAN32__
#include "rtfile.epoc.cpp"
#else
#ifdef __WIN32__
  #include "rtfile.win32.cpp"
#else
  #ifdef __UNIX__
    #include "rtfile.unix.cpp"
  #endif
#endif
#endif

