/*
 * rtsystem.epoc.cpp
 * 
 * Copyright (c) 2000-2005 by Florian Fischer (florianfischer@gmx.de)
 * and Martin Trautmann (martintrautmann@gmx.de) 
 * 
 * This file may be distributed and/or modified under the terms of the 
 * GNU General Public License version 2 as published by the Free Software 
 * Foundation. 
 * 
 * This file is provided AS IS with NO WARRANTY OF ANY KIND, INCLUDING THE
 * WARRANTY OF DESIGN, MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 * 
 */

///////// Symbian EPOC implementation of System functions

#include <e32std.h>
#include <e32cons.h>
#include "rttuiedit.h"

#include "rtetools.h"

#ifdef __WINS__
#ifndef _UNICODE  // actually, if Epoc Version 5, but that's not possible
int atexit(void (__cdecl *func )( void ))
{
	return 0;
}
#endif
#endif


namespace lrt {

const TTime TIMEepoch(TDateTime(1970, EJanuary, 0, 0, 0, 0, 0));

Time::Time(const String& str) : sec(0), msec(0)
{
	Array<String> parts = str.split(" :", "");
	if(parts.length() != 7) // there should be 7 parts
		return;
	
	TDateTime dt;

	// Y, M, D, H, M, S, S
	dt.Set(parts[6].intValue(1970), TMonth(getMonthNum(parts[1])),
		parts[2].intValue(1), parts[3].intValue(0), parts[4].intValue(0),
		parts[5].intValue(0), 0);

	TTime time(dt);

	TTimeIntervalSeconds ti;
	time.SecondsFrom(TIMEepoch, ti);
	sec = ti.Int();
}

Time Time::getCurrentTime()
{
	Time ret;
	TTime t;
	t.HomeTime();
	TTimeIntervalMicroSeconds ti = t.MicroSecondsFrom(TIMEepoch);
	TInt64 tii = ti.Int64();
	tii /= TInt64(1000);
	ret.msec = (tii % 1000).GetTInt();
	tii /= 1000;
	ret.sec = tii.GetTInt();
	return ret;
}

String Time::toString()
{
	TInt64 tt = sec;
	tt *= 1000000;
	TTime time = TIMEepoch;
	time += TTimeIntervalMicroSeconds(tt);
	TBuf<40> buf;
	time.FormatL(buf, _L("%F%*E %*N %D %H:%T:%S %Y"));
	return MYSTRING(buf);
}

void System::message(const String& str)
{
	if(interactive) {
		User::InfoPrint(ESTRING(str));
	}
	else
		println(str);
}

void System::exit(int errno)
{
	//read();
	finalize();
	User::Exit(errno);
}


void System::exit(int errno, const String &errmsg)
{
	if(!interactive)
	  println(errmsg);
	finalize();
	if(interactive)
	  User::Panic(ESTRING(errmsg), errno);
	User::Exit(errno);
}

// is not supported
void System::setOnBreak(System::BreakFunction, bool)
{
}

// System
CTrapCleanup *System::cleanup = 0;

// Console
CConsoleBase *System::console = 0;
CLineEdit *System::edit = 0;

void System::initialize()
{
	console=Console::NewL(ESTRING((*appName)),
		TSize(KConsFullScreen,KConsFullScreen));
	edit=CLineEdit::NewL(console,10);
	init = true;
}

void System::printInternal(const char *cStr, bool isLong, bool addLine)
{
	//TPtrC8 eStr = _L8(cStr);
	_LIT(nl,"\n");
#ifdef _UNICODE
	TBuf<128> buf;
	int i = 0;
	while(cStr[i] != '\0') {
		buf.Append(cStr[i]);
		if((i >> 7) == 127) {
			console->Write(buf);
			buf.SetLength(0);
		}
		i++;
	}
	console->Write(buf);
#else
	console->Write(_L8(cStr));
#endif
	if(addLine)
		console->Write(nl);
}

void System::print(const char *cStr)
{
	if(!init) initialize();
	printInternal(cStr, false, false);	
}

void System::print(const String &str)
{
	print(str.cStr());
}

void System::println()
{
	if(!init) initialize();
	console->Printf(_L("\n"));
}

void System::println(const String &str)
{
	if(!init) initialize();
	printInternal(str.cStr(), (str.length() > 200), true);
}

char System::read()
{
	if(!init) initialize();
	return console->Getch();
}

String System::readLine(const String& query)
{
	if(!init) initialize();
	TBuf<80> input;
	edit->Edit(ESTRING(query), &input);
	return MYSTRING(input);
}

void System::setPos(int x, int y)
{
	if(!init) initialize();
	console->SetPos(x, y);
}

void System::copyMemory(const void* src, void* dest, int length)
{
#ifndef __NOCHECK__
	if(length < 0)
		System::exit(-1, "length < 0 in System::copyMemory()");
#endif
	Mem::Copy(dest, src, length);
}

void System::fillMemory(void* ptr, int length, char data)
{
#ifndef __NOCHECK__
	if(length < 0)
		System::exit(-1, "length < 0 in System::fillMemory()");
#endif
	Mem::Fill(ptr, length, data);
}

void System::finalize()
{
	delete edit;
	delete console;
	delete cleanup;
	init = false;
}

} // namespace

