/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib-object.h>
#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <dvd.h>

/*
 * AC3 Frame Size Code Table
 *
 * To get the correct table do
 * Ac3FSCTable = AC3_FSC_TABLE[Ac3SyncInfo.framesize_code]
 *
 * To determine frame size
 * guint16 framesize = Ac3FSCTable.sr[Ac3SyncInfo.samplerate_code]
 *
*/
typedef struct _Ac3FSCTable {
	guint16 sr[3];
	guint16 kbs;
} Ac3FSCTable;

static const Ac3FSCTable AC3_FSC_TABLE[38] = {
	{ {   64,   69,   96},  32},
	{ {   64,   70,   96},  32},
	{ {   80,   87,  120},  40},
	{ {   80,   88,  120},  40},
	{ {   96,  104,  144},  48},
	{ {   96,  105,  144},  48},
	{ {  112,  121,  168},  56},
	{ {  112,  122,  168},  56},
	{ {  128,  139,  192},  64},
	{ {  128,  140,  192},  64},
	{ {  160,  174,  240},  80},
	{ {  160,  175,  240},  80},
	{ {  192,  208,  288},  96},
	{ {  192,  209,  288},  96},
	{ {  224,  243,  336}, 112},
	{ {  224,  244,  336}, 112},
	{ {  256,  278,  384}, 128},
	{ {  256,  279,  384}, 128},
	{ {  320,  348,  480}, 160},
	{ {  320,  349,  480}, 160},
	{ {  384,  417,  576}, 192},
	{ {  384,  418,  576}, 192},
	{ {  448,  487,  672}, 224},
	{ {  448,  488,  672}, 224},
	{ {  512,  557,  768}, 256},
	{ {  512,  558,  768}, 256},
	{ {  640,  696,  960}, 320},
	{ {  640,  697,  960}, 320},
	{ {  768,  835, 1152}, 384},
	{ {  768,  836, 1152}, 384},
	{ {  896,  975, 1344}, 448},
	{ {  896,  976, 1344}, 448},
	{ { 1024, 1114, 1536}, 512},
	{ { 1024, 1115, 1536}, 512},
	{ { 1152, 1253, 1728}, 576},
	{ { 1152, 1254, 1728}, 576},
	{ { 1280, 1393, 1920}, 640},
	{ { 1280, 1394, 1920}, 640}
};

#define AC3_SYNCFRAME_ID 0x0b77

static DvdDecoderClass *dvd_decoder_ac3_parent_class = NULL;

static void     dvd_decoder_ac3_class_init	(DvdDecoderAc3Class	*class);
static void     dvd_decoder_ac3_instance_init	(GTypeInstance	*instance,
						 gpointer	 g_class);
static void     dvd_decoder_ac3_dispose		(GObject	*object);

static void	dvd_decoder_ac3_read_startcode	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	dvd_decoder_ac3_read_header	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static void	dvd_decoder_ac3_read_frame	(DvdDecoder	 *decoder,
						 guint8		**start,
						 guint8		 *end,
						 guint32	  pts);
static guint32	bitstream_readbits_uint32	(guint8			  data,
						 guint8			 *data_bit,
						 guint32		 *variable_uint,
						 const guint8		  variable_length,
						 guint			 *variable_bit);
static gboolean	read_ac3_sync_header		(DvdAc3SyncHeader 	 *header,
						 guint8			  data,
						 guint			 *header_byte,
						 DvdDecoderAC3State	 *state);
static gboolean	read_ac3_bsi_header		(DvdAc3BSHeader 	 *header,
						 guint8			  data,
						 guint8			 *data_bit,
						 guint			 *header_byte,
						 guint			 *header_variable,
						 guint32		 *variable_uint,
						 guint			 *variable_bit);
static void	ac3_print_syncinfo		(DvdAc3SyncHeader	 *sync_info);
static void	ac3_print_bsi			(DvdAc3BSHeader		 *info);

static void
dvd_decoder_ac3_class_init	(DvdDecoderAc3Class *class)
{
	GObjectClass *object_class = (GObjectClass *) class;
	
	dvd_decoder_ac3_parent_class = g_type_class_ref (DVD_DECODER_TYPE);
	
	object_class->dispose = dvd_decoder_ac3_dispose;
}

static void
dvd_decoder_ac3_instance_init	(GTypeInstance	*instance,
				 gpointer	 g_class)
{
	DvdDecoder *decoder;
	DvdDecoderAc3 *ac3_decoder;
	
	decoder = DVD_DECODER (instance);
	ac3_decoder = DVD_DECODER_AC3 (instance);
	decoder->stream_type = DVD_STREAM_AC3_AUDIO;
	
	/* these functions called by parent DvdDecoder */
	decoder->read_startcode = dvd_decoder_ac3_read_startcode;
	decoder->read_header    = dvd_decoder_ac3_read_header;
	decoder->read_frame     = dvd_decoder_ac3_read_frame;
	
	ac3_decoder->state = DVD_DECODER_AC3_STATE_SEEK_START;
	ac3_decoder->sync_hdr.sync_word = 0xffff;
	ac3_decoder->header_byte = 0;
	ac3_decoder->header_variable = 0;
	ac3_decoder->data_bit = 0;
	ac3_decoder->variable_uint = 0;
	ac3_decoder->variable_bit = 0;
}


static void
dvd_decoder_ac3_dispose		(GObject	*object)
{
	DvdDecoderAc3 *decoder;
	
	decoder = DVD_DECODER_AC3 (object);
	
	G_OBJECT_CLASS (dvd_decoder_ac3_parent_class)->dispose (object);
}

/**
 * dvd_decoder_ac3_get_type
 * @return The GType for the DvdDecoderAc3 class.
 */
GType
dvd_decoder_ac3_get_type	(void)
{
	static GType decoder_ac3_type = 0;

	if (decoder_ac3_type == 0) {
		GTypeInfo decoder_ac3_info = {
			sizeof (DvdDecoderAc3Class),
			NULL,
			NULL,
			(GClassInitFunc) dvd_decoder_ac3_class_init,
			NULL,
			NULL, /* class_data */
			sizeof (DvdDecoderAc3),
			0, /* n_preallocs */
			(GInstanceInitFunc) dvd_decoder_ac3_instance_init
	    	};
		decoder_ac3_type = g_type_register_static (DVD_DECODER_TYPE,
							   "DvdDecoderAc3",
							    &decoder_ac3_info, 0);
	}
	return decoder_ac3_type;
}

/* appends bit/s to variable_uint form data MSB first */
/* assumes variable_uint is zero to begin with ( only sets "on" bits ) */ 
/* returns the total number of bits read into variable_uint */
/* if data_bit equalls 8 on return, reset to 0 and supply next data byte */
static guint32
bitstream_readbits_uint32	(guint8		 data,
				 guint8		*data_bit,
				 guint32	*variable_uint,
				 const guint8	 variable_length,
				 guint		*variable_bit)
{
	g_assert (*variable_bit < variable_length);
	g_assert (*data_bit < 8);
	g_assert (variable_length < 32);
	
	while ((variable_length > *variable_bit) && (*data_bit < 8)) {
		if ((data & (0x80 >> *data_bit)) > 0) {
			*variable_uint |= (1 << (variable_length - *variable_bit - 1));
		}
		(*data_bit)++;
		(*variable_bit)++;
	}
	return *variable_bit;
}

/*
 * Reads a sync header provided one byte at a time.
 *
 * Increments byte_no until header is complete.
 *
 * Returns TRUE and sets byte_no to zero when header is complete.
 */
static gboolean
read_ac3_sync_header		(DvdAc3SyncHeader 	 *header,
				 guint8			  data,
				 guint			 *header_byte,
				 DvdDecoderAC3State	 *state)
{
	gboolean header_complete;
	
	header_complete = FALSE;
	
	switch (*header_byte) {
	case 0:
		/*g_message ("reading ac3 sync header");*/
		/*memset (header, '0', sizeof (DvdAc3SyncHeader));*/
		header->sync_word = data;
		(*header_byte)++;
		break;
	case 1:
		header->sync_word = (header->sync_word << 8);
		header->sync_word += data;
		if (header->sync_word == AC3_SYNCFRAME_ID) {
			(*header_byte)++;
		} else {
			if (*state == DVD_DECODER_AC3_STATE_SEEK_START) {
				g_warning ("Junk before AC3 Sync header");
			} else {
				g_warning ("AC3 Sync worn = 0x%x", header->sync_word);
				*state = DVD_DECODER_AC3_STATE_ERROR;
			}
		}
		break;
	case 2:
		header->crc1 = data;
		(*header_byte)++;
		break;
	case 3:
		header->crc1 = (header->crc1 << 8) + data;
		(*header_byte)++;
		break;
	case 4:
		if (header->samplerate_code == 0) {
			header->samplerate_code = (data >> 6);
			header->framesize_code = (data & 0x3f);
		} else {
			if (header->samplerate_code != (data >> 6)) {
				g_warning ("diffrerent samplerate old=%d, new=%d", header->samplerate_code, data >> 6);
				header->samplerate_code = data >> 6;
			}
			if (header->framesize_code != (data & 0x3f)) {
				g_warning ("diffrerent framesize old=%d, new=%d", header->framesize_code, (data & 0x3f));
				header->framesize_code = (data & 0x3f);
				/*ac3_decoder->bytes_per_frame = (AC3_FSC_TABLE[ac3_decoder->sync_hdr.framesize_code].sr[ac3_decoder->sync_hdr.samplerate_code] * 2);*/
			}
			
		}
		header_complete = TRUE;
		break;
	default:
		g_assert_not_reached ();
	}
	
	return header_complete;	
}

static gboolean
read_ac3_bsi_header		(DvdAc3BSHeader 	*header,
				 guint8			 data,
				 guint8			*data_bit,
				 guint			*header_byte,
				 guint			*header_variable,
				 guint32		*variable_uint,
				 guint			*variable_bit)
{
	gboolean header_complete;
	
	enum {
		BSIHV_ID = 0,
		BSIHV_BSMODE,
		BSIHV_ACMODE,
		BSIHV_CMIXLEVEL,
		BSIHV_SURMIXLEVEL,
		BSIHV_DSURMODE,
		BSIHV_LFEON,
		BSIHV_DIALNORM,
		BSIHV_COMPRE,
		BSIHV_COMPR,
		BSIHV_LANGCODE,
		BSIHV_LANGCOD,
		BSIHV_AUDPRODIE,
		BSIHV_MIXLEVEL,
		BSIHV_ROOMTYPE,
		BSIHV_DIALNORM2,
		BSIHV_COMPRE2,
		BSIHV_COMPR2,
		BSIHV_LANGCODE2,
		BSIHV_LANGCOD2,
		BSIHV_AUDPRODIE2,
		BSIHV_MIXLEVEL2,
		BSIHV_ROOMTYPE2,
		BSIHV_COPYRIGHT,
		BSIHV_ORIGINAL,
		BSIHV_COMPLETE
	};
	const guint HEADER_VARIABLE_BITS[BSIHV_COMPLETE] = {5, 3, 3, 2, 2, 2, 1, 5, 1, 8, 1, 8, 1, 5, 2, 5, 1, 8, 1, 8, 1, 5, 2, 1, 1};
	
	header_complete = FALSE;
	
	while ((*data_bit < 8) &&
	       (header_complete == FALSE)) {
		if (bitstream_readbits_uint32 (data,
					       data_bit,
					       variable_uint,
					       HEADER_VARIABLE_BITS[*header_variable],
					       variable_bit) == HEADER_VARIABLE_BITS[*header_variable]) {
				(*variable_bit) = 0;
			switch (*header_variable) {
			case BSIHV_ID:
				/*g_message ("reading ac3 bsi header");*/
				memset (header, '0', sizeof (DvdAc3BSHeader));
				header->id = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_BSMODE;
				break;
			case BSIHV_BSMODE:
				header->bsmode = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_ACMODE;
				
				break;
			case BSIHV_ACMODE:
				header->acmode = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_CMIXLEVEL;
				break;
			case BSIHV_CMIXLEVEL:
				
				if ((header->acmode & 0x1) && (header->acmode != 0x1)) {
					header->cmixlevel = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_SURMIXLEVEL;
				} else {
					*header_variable = BSIHV_SURMIXLEVEL;
					continue;
				}
				break;
			case BSIHV_SURMIXLEVEL:
				if (header->acmode & 0x4) {
					header->surmixlevel = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_DSURMODE;
				} else {
					*header_variable = BSIHV_DSURMODE;
					continue;
				}
				break;
			case BSIHV_DSURMODE:
				if (header->acmode & 0x2) {
					header->dsurmode = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_LFEON;
				} else {
					*header_variable = BSIHV_LFEON;
					continue;
				}
				break;
			case BSIHV_LFEON:
				header->lfeon = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_DIALNORM;
				break;
			case BSIHV_DIALNORM:
				header->dialnorm = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_COMPRE;
				break;
			case BSIHV_COMPRE:
				header->compre = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_COMPR;
				break;
			case BSIHV_COMPR:
				if (header->compre > 0) {
					header->compr = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_LANGCODE;
				} else {
					*header_variable = BSIHV_LANGCODE;
					continue;
				}
				break;
			case BSIHV_LANGCODE:
				header->langcode = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_LANGCOD;
				break;
			case BSIHV_LANGCOD:
				if (header->langcode > 0) {
					header->langcod = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_AUDPRODIE;
				} else {
					*header_variable = BSIHV_AUDPRODIE;
					continue;
				}
				break;
			case BSIHV_AUDPRODIE:
				header->audprodie = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_MIXLEVEL;
				break;
			case BSIHV_MIXLEVEL:
				if (header->audprodie > 0) {
					header->mixlevel = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_ROOMTYPE;
				} else {
					*header_variable = BSIHV_DIALNORM2;
				}
				break;
			case BSIHV_ROOMTYPE:
				/* only get here if (header->audprodie > 0) - see above case */
				header->roomtype = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_DIALNORM2;
				break;
			case BSIHV_DIALNORM2:
				if (header->acmode == 0) {
					header->dialnorm2 = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_COMPRE2;
				} else {
					*header_variable = BSIHV_COPYRIGHT;
				}
				break;
			case BSIHV_COMPRE2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				header->compre2 = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_COMPR2;
				break;
			case BSIHV_COMPR2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				if (header->compre2 > 0) {
					header->compr2 = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_LANGCODE2;
				} else {
					*header_variable = BSIHV_LANGCODE2;
					continue;
				}
				break;
			case BSIHV_LANGCODE2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				header->langcode2 = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_LANGCOD2;
				break;
			case BSIHV_LANGCOD2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				if (header->langcode2 > 0) {
					header->langcod2 = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_AUDPRODIE2;
				} else {
					*header_variable = BSIHV_AUDPRODIE2;
					continue;
				}
				break;
			case BSIHV_AUDPRODIE2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				header->audprodie2 = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_MIXLEVEL2;
				break;
			case BSIHV_MIXLEVEL2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				if (header->audprodie2 > 0) {
					header->mixlevel2 = *variable_uint;
					*variable_uint = 0;
					*header_variable = BSIHV_ROOMTYPE2;
				} else {
					*header_variable = BSIHV_COPYRIGHT;
				}
				break;
			case BSIHV_ROOMTYPE2:
				/* only get here if (header->acmod == 0) - see case BSIHV_DIALNORM2 */
				/* and if (header->audprodie2 > 0) - see case BSIHV_MIXLEVEL2 */
				header->roomtype2 = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_COPYRIGHT;
				break;
			case BSIHV_COPYRIGHT:
				header->copyright = *variable_uint;
				*variable_uint = 0;
				*header_variable = BSIHV_ORIGINAL;
				break;
			case BSIHV_ORIGINAL:
				header->original = *variable_uint;
				*variable_uint = 0;
				*header_variable = 0;
				header_complete = TRUE;
				break;
			case BSIHV_COMPLETE:
				g_assert_not_reached ();
				break;
			default:
				g_assert_not_reached ();
			}
		}
	}

	/* timecode: Time code (first and second) halves exist, 2 bits */
	/* timecod1: Time code first half, 14 bits */
		/* first 5 bits are hours (0-23) */
		/* next 6 bits are minutes (0-59) */
		/* next 3 bits are 8 second increments (0-7) */
	/* timecod2: Time code second, 14 bits half */
		/* first 3 bits are seconds (0-7) */
		/* next 5 bits are frames (0-29) */
		/* last 6 bits are fractions of 1/64 of a frame (0-63) */
	/* addbsie: Additional bit stream hdrrmation exists, 1 bit */
	/* addbsil: Additional bit stream hdrrmation length, 6 bits */
	/* addbsi: Additional bit stream hdrrmation, [(addbsil+1) × 8] bits */
	return header_complete;
}

static void
dvd_decoder_ac3_read_startcode	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	guint8 *iter;
	DvdDecoderAc3 *ac3_decoder;
	
	ac3_decoder = DVD_DECODER_AC3 (decoder);
	
	
	for (iter = *start;
	     iter < end;
	     iter++) {
		gboolean header_read;
		
		header_read = read_ac3_sync_header (&ac3_decoder->sync_hdr,
						    *iter,
						    &ac3_decoder->header_byte,
						    &ac3_decoder->state);
		if (header_read == TRUE) {
			dvd_decoder_ensure_buffer_size (decoder, ac3_decoder->bytes_per_frame);
			dvd_decoder_write_frame_byte (decoder, ac3_decoder->sync_hdr.sync_word >> 8);
			dvd_decoder_write_frame_byte (decoder, ac3_decoder->sync_hdr.sync_word & 0xff);
			dvd_decoder_write_frame_byte (decoder, ac3_decoder->sync_hdr.crc1 >> 8);
			dvd_decoder_write_frame_byte (decoder, ac3_decoder->sync_hdr.crc1 &0xff);
			dvd_decoder_write_frame_byte (decoder, *iter);
			/* there are 6 x 256 samples per ac3 frame */
			/* at 48000 samples per second that is 2880 90kHz clocks or 31.25 frames / sec */
			/* so if pts is specified in mid frame, how did the multiplexor determine pts?? */
			/* a). Start of broken frame */
			/* b). Start of next frame */
			/* c). Somewhere in between. */
			/* testing has indicated that b is correct */
			dvd_decoder_set_start_pts (decoder, pts);
			dvd_decoder_set_current_pts (decoder, pts);
			dvd_decoder_set_state (decoder, DVD_DECODER_STATE_HEADER);
			ac3_decoder->clocks_per_frame = 2880;
			dvd_decoder_set_frame_clocks (decoder, ac3_decoder->clocks_per_frame * 300);
			ac3_decoder->bytes_per_frame = (AC3_FSC_TABLE[ac3_decoder->sync_hdr.framesize_code].sr[ac3_decoder->sync_hdr.samplerate_code] * 2);
			ac3_decoder->state = DVD_DECODER_AC3_STATE_BSI_HDR;
			ac3_decoder->header_byte = 0;
			iter++;
			g_message ("found AC3 audio bitstream - %d bytes and %d clocks per frame",
				   ac3_decoder->bytes_per_frame, ac3_decoder->clocks_per_frame);
			ac3_print_syncinfo (&ac3_decoder->sync_hdr);
			g_assert (decoder->buffer_pos == 5);
			break;
		}
	}
	*start = iter;
}


static void
dvd_decoder_ac3_read_header	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	DvdDecoderAc3 *ac3_decoder;
	guint8 *iter;
	
	ac3_decoder = DVD_DECODER_AC3 (decoder);
	
	
	for (iter = *start;
	     iter < end;
	     iter++) {
		gboolean header_read;
		
		if (ac3_decoder->state == DVD_DECODER_AC3_STATE_SYNC_HDR) {
			header_read = read_ac3_sync_header (&ac3_decoder->sync_hdr,
							    *iter,
							    &ac3_decoder->header_byte,
							    &ac3_decoder->state);
			dvd_decoder_write_frame_byte (decoder, *iter);
			
		} else if (ac3_decoder->state == DVD_DECODER_AC3_STATE_BSI_HDR) {
			header_read = read_ac3_bsi_header (&ac3_decoder->bs_hdr,
							   *iter,
							   &ac3_decoder->data_bit,
							   &ac3_decoder->header_byte,
							   &ac3_decoder->header_variable,
							   &ac3_decoder->variable_uint,
							   &ac3_decoder->variable_bit);
			if ((ac3_decoder->data_bit == 8) ||
			    (header_read == TRUE)) {
				dvd_decoder_write_frame_byte (decoder, *iter);
				ac3_decoder->header_byte++;
				ac3_decoder->data_bit = 0;
				/*g_message ("Ac3 BSI header bytes = %d", ac3_decoder->header_byte);*/
			}
		} else if (ac3_decoder->state == DVD_DECODER_AC3_STATE_ERROR) {
			g_message ("next byte = 0x%x", *(iter++));
			g_warning ("AC3 Sync header not found - Broken bitstream");
			/* error in bitstream */
			*start = end;
			return;
		} else {
			g_assert_not_reached ();
		}
		
		if (header_read == TRUE) {
			
			ac3_decoder->header_byte = 0;
			if (ac3_decoder->state == DVD_DECODER_AC3_STATE_SYNC_HDR) {
				dvd_decoder_add_to_current_pts (decoder, ac3_decoder->clocks_per_frame);
				ac3_decoder->state = DVD_DECODER_AC3_STATE_BSI_HDR;
				g_assert (decoder->buffer_pos == 5);
			} else { /* DVD_DECODER_AC3_STATE_BSI_HDR */
				
				/*ac3_print_bsi (&ac3_decoder->bs_hdr); */
				
				ac3_decoder->state = DVD_DECODER_AC3_STATE_STREAM;
				dvd_decoder_set_state (decoder, DVD_DECODER_STATE_FRAME);
			}
			iter++;
			break;
		}
	}
	*start = iter;
}

static void
dvd_decoder_ac3_read_frame	(DvdDecoder	 *decoder,
				 guint8		**start,
				 guint8		 *end,
				 guint32	  pts)
{
	DvdDecoderAc3 *ac3_decoder;
	guint16 length;
	guint16 data_length;
	
	ac3_decoder = DVD_DECODER_AC3 (decoder);
	
	data_length = end - *start;
	length = ac3_decoder->bytes_per_frame - decoder->buffer_pos;
	if (length > data_length) {
		dvd_decoder_write_frame_bytes (decoder, *start, data_length);
		*start = end;
	} else {
		dvd_decoder_write_frame_bytes (decoder, *start, length);
		dvd_decoder_output_frame (decoder);
		*start = (*start) + length;
		ac3_decoder->state = DVD_DECODER_AC3_STATE_SYNC_HDR;
		dvd_decoder_set_state (decoder, DVD_DECODER_STATE_HEADER);
	}
}

const gchar *BIT_SREAM_MODE[9] = {
	"Main audio service - complete main (CM)",
	"Main audio service - music and effects (ME)",
	"Associated service - visually impaired (VI)",
	"Associated service - hearing impaired (HI)",
	"Associated service - dialogue (D)",
	"Associated service - commentary (C)",
	"Associated service - emergency (E)",
	"Associated service - voice over (VO)",
	"Main audio service - karaoke"
};

const gchar *AUDIO_CODING_MODE[8] = {
	"1+1 2ch [Ch1, Ch2]",
	"1/0 1ch [C]",
	"2/0 2ch [L, R]",
	"3/0 3ch [L, C, R]",
	"2/1 3ch [L, R, S]",
	"3/1 4ch [L, C, R, S]",
	"2/2 4ch [L, R, SL, SR]",
	"3/2 5ch [L, C, R, SL, SR]"
};

const gchar *CENTER_MIX_LEVEL[4]= {
	"-3.0dB",
	"-4.5dB",
	"-6.0dB",
	"Reserevd"
};

const gchar *SURROUND_MIX_LEVEL[4]= {
	"-3.0dB",
	"-6.0dB",
	"-0.0dB",
	"Reserevd"
};

const gchar *DOLBY_SURROUND_MODE[4]= {
	"Not Indicated",
	"Not Dolby Surround encoded",
	"Dolby Surround encoded",
	"Reserved / Not Indicated"
};

const gchar *ROOM_TYPE[4] = {
	"not indicated",
	"large room, X curve monitor",
	"small room, flat monitor",
	"reserved"
};

static void
ac3_print_syncinfo	(DvdAc3SyncHeader	 *sync_info)
{

	guint16 kbs_rate;
	
	kbs_rate = AC3_FSC_TABLE[sync_info->framesize_code].kbs;
	g_print ("Bytes Per Syncframe           = %d\n", AC3_FSC_TABLE[sync_info->framesize_code].sr[sync_info->samplerate_code] * 2);
	g_print ("Bit Rate                      = %dkb/s\n", kbs_rate);
/* DVD ac3 audio is always 48kHz! */
	switch (sync_info->samplerate_code) {
	case 0:
		g_print ("Sample Rate                   = 48kHz\n");
		break;
	case 1:
		g_print ("Sample Rate                   = 44.1kHz\n");
		break;
	case 2:
		g_print ("Sample Rate                   = 32kHz\n");
		break;
	default:
		g_assert_not_reached ();
	}
	
}

static void
ac3_print_bsi		(DvdAc3BSHeader	 *info)
{
	g_print ("AC3 bit stream id             = %d\n", info->id);
	g_print ("Bit Stream Mode               = %s\n", BIT_SREAM_MODE[info->bsmode]);
	g_print ("Audio Coding Mode             = %s\n", AUDIO_CODING_MODE[info->acmode]);
	if ((info->acmode & 0x1) && (info->acmode != 0x1)) {
		g_print ("Center Mix Level              = %s\n", CENTER_MIX_LEVEL[info->cmixlevel]);
	}
	if (info->acmode & 0x4) {
		g_print ("Surround Mix Level            = %s\n", SURROUND_MIX_LEVEL[info->surmixlevel]);
	}
	if (info->acmode & 0x2) {
		g_print ("Dolby Surround Mode           = %s\n", DOLBY_SURROUND_MODE[info->dsurmode]);
	}
	g_print ("Low frequency effects channel = %s\n", info->lfeon ? "YES" : "NO");
	if (info->audprodie == 1) {
		g_print ("Room Type                     = %s\n", ROOM_TYPE[info->roomtype]);
	}
	if (info->acmode == 0) {
		if (info->audprodie2 == 1) {
			g_print ("Room Type (second channel)    = %s\n", ROOM_TYPE[info->roomtype]);
		}
	}
	g_print ("Copyright bit is set          = %s\n", info->copyright ? "YES" : "NO");
	g_print ("Original bit stream           = %s\n", info->original ? "YES" : "NO");
}

