/*
 *  Part of the shrinkta program, a dvd backup tool
 *
 *  Copyright (C) 2005  Daryl Gray
 *  E-Mail Daryl Gray darylgray1@dodo.com.au
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
*/
#include <inttypes.h>
#include <config.h>
#include <glib/gi18n.h>
#include <glib-object.h>
#include <string.h>
#include "dvd.h"

static struct {
	gchar code[3];
	gchar name[20];
} language_str[] = {
	{ "  ", N_("Not Specified") },  { "aa", N_("Afar") },		{ "ab", N_("Abkhazian") },
	{ "af", N_("Afrikaans") },	{ "am", N_("Amharic") },	{ "ar", N_("Arabic") },
	{ "as", N_("Assamese") },	{ "ay", N_("Aymara") },		{ "az", N_("Azerbaijani") },
	{ "ba", N_("Bashkir") },	{ "be", N_("Byelorussian") },	{ "bg", N_("Bulgarian") },
	{ "bh", N_("Bihari") },		{ "bi", N_("Bislama") },	{ "bn", N_("Bengali; Bangla") },
	{ "bo", N_("Tibetan") },	{ "br", N_("Breton") },		{ "ca", N_("Catalan") },
	{ "co", N_("Corsican") },	{ "cs", N_("Czech") },		{ "cy", N_("Welsh") },
	{ "da", N_("Dansk") },		{ "de", N_("Deutsch") },	{ "dz", N_("Bhutani") },
	{ "el", N_("Greek") },		{ "en", N_("English") },	{ "eo", N_("Esperanto") },
	{ "es", N_("Espanol") },	{ "et", N_("Estonian") },	{ "eu", N_("Basque") },
	{ "fa", N_("Persian") },	{ "fi", N_("Suomi") },		{ "fj", N_("Fiji") },
	{ "fo", N_("Faroese") },	{ "fr", N_("Francais") },	{ "fy", N_("Frisian") },
	{ "ga", N_("Gaelic") },		{ "gd", N_("Scots Gaelic") },   { "gl", N_("Galician") },
	{ "gn", N_("Guarani") },	{ "gu", N_("Gujarati") },	{ "ha", N_("Hausa") },
	{ "he", N_("Hebrew") },		{ "hi", N_("Hindi") },		{ "hr", N_("Hrvatski") },
	{ "hu", N_("Magyar") },		{ "hy", N_("Armenian") },	{ "ia", N_("Interlingua") },
	{ "id", N_("Indonesian") },	{ "ie", N_("Interlingue") },	{ "ik", N_("Inupiak") },
	{ "in", N_("Indonesian") },	{ "is", N_("Islenska") },	{ "it", N_("Italiano") },
	{ "iu", N_("Inuktitut") },	{ "iw", N_("Hebrew") },		{ "ja", N_("Japanese") },
	{ "ji", N_("Yiddish") },	{ "jw", N_("Javanese") },	{ "ka", N_("Georgian") },
	{ "kk", N_("Kazakh") },		{ "kl", N_("Greenlandic") },	{ "km", N_("Cambodian") },
	{ "kn", N_("Kannada") },	{ "ko", N_("Korean") },		{ "ks", N_("Kashmiri") },
	{ "ku", N_("Kurdish") },	{ "ky", N_("Kirghiz") },	{ "la", N_("Latin") },
	{ "ln", N_("Lingala") },	{ "lo", N_("Laothian") },	{ "lt", N_("Lithuanian") },
	{ "lv", N_("Latvian, Lettish")},{ "mg", N_("Malagasy") },	{ "mi", N_("Maori") },
	{ "mk", N_("Macedonian") },	{ "ml", N_("Malayalam") },	{ "mn", N_("Mongolian") },
	{ "mo", N_("Moldavian") },	{ "mr", N_("Marathi") },	{ "ms", N_("Malay") },
	{ "mt", N_("Maltese") },	{ "my", N_("Burmese") },	{ "na", N_("Nauru") },
	{ "ne", N_("Nepali") },		{ "nl", N_("Nederlands") },	{ "no", N_("Norsk") },
	{ "oc", N_("Occitan") },	{ "om", N_("Oromo") },		{ "or", N_("Oriya") },
	{ "pa", N_("Punjabi") },	{ "pl", N_("Polish") },		{ "ps", N_("Pashto, Pushto") },
	{ "pt", N_("Portugues") },	{ "qu", N_("Quechua") },	{ "rm", N_("Rhaeto-Romance") },
	{ "rn", N_("Kirundi") },	{ "ro", N_("Romanian") },	{ "ru", N_("Russian") },
	{ "rw", N_("Kinyarwanda") },	{ "sa", N_("Sanskrit") },	{ "sd", N_("Sindhi") },
	{ "sg", N_("Sangho") },		{ "sh", N_("Serbo-Croatian") },	{ "si", N_("Sinhalese") },
	{ "sk", N_("Slovak") },		{ "sl", N_("Slovenian") },	{ "sm", N_("Samoan") },
	{ "sn", N_("Shona") },		{ "so", N_("Somali") },		{ "sq", N_("Albanian") },
	{ "sr", N_("Serbian") },	{ "ss", N_("Siswati") },	{ "st", N_("Sesotho") },
	{ "su", N_("Sundanese") },	{ "sv", N_("Svenska") },	{ "sw", N_("Swahili") },
	{ "ta", N_("Tamil") },		{ "te", N_("Telugu") },		{ "tg", N_("Tajik") },
	{ "th", N_("Thai") },		{ "ti", N_("Tigrinya") },	{ "tk", N_("Turkmen") },
	{ "tl", N_("Tagalog") },	{ "tn", N_("Setswana") },	{ "to", N_("Tonga") },
	{ "tr", N_("Turkish") },	{ "ts", N_("Tsonga") },		{ "tt", N_("Tatar") },
	{ "tw", N_("Twi") },		{ "ug", N_("Uighur") },		{ "uk", N_("Ukrainian") },
	{ "ur", N_("Urdu") },		{ "uz", N_("Uzbek") },		{ "vi", N_("Vietnamese") },
	{ "vo", N_("Volapuk") },	{ "wo", N_("Wolof") },		{ "xh", N_("Xhosa") },
	{ "yi", N_("Yiddish") },	{ "yo", N_("Yoruba") },		{ "za", N_("Zhuang") },
	{ "zh", N_("Chinese") },	{ "zu", N_("Zulu") },		{ {0,0,0}, N_("Unknown") }
};

DvdLanguage
dvd_language_get_from_code	(const guchar	*code)
{
	DvdLanguage lang;
	
	for (lang = DVD_LANG_UNDEFINED;
	     lang < DVD_LANG_UNKNOWN;
	     lang++) {
		if (memcmp (language_str[lang].code, code, 2) == 0) {
			break;
		}
	}
	return lang;
}

DvdLanguage
dvd_language_get_from_code16	(guint16	 code16)
{
	guchar lang_code[3];
	DvdLanguage lang;
		
	lang_code[0] = code16 >> 8;
	lang_code[1] = code16 & 0xff;
	lang_code[2] = '\0';
	if (lang_code[0] == 0) {
		lang =  DVD_LANG_UNKNOWN;
	} else {
		lang = dvd_language_get_from_code (lang_code);
	}
	return lang;
}

const gchar
*dvd_language_get_code		(DvdLanguage	 lang)
{
	return language_str[lang].code;
}

const gchar
*dvd_language_get_name		(DvdLanguage	 lang)
{
	return gettext (language_str[lang].name);
}

