/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance;

import java.awt.*;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;
import java.awt.image.*;
import java.util.*;
import java.util.List;

import javax.swing.*;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.layout.TransitionLayout;
import org.jvnet.substance.border.FlatBorderPainter;
import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.button.BaseButtonShaper;
import org.jvnet.substance.color.*;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.theme.SubstanceMixTheme;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.theme.SubstanceTheme.ThemeKind;
import org.jvnet.substance.utils.*;
import org.jvnet.substance.utils.ComponentState.ColorSchemeKind;
import org.jvnet.substance.utils.SubstanceConstants.Side;
import org.jvnet.substance.watermark.SubstanceWatermark;

/**
 * Provides utility functions for creating various images for <b>Substance </b>
 * look and feel. This class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public final class SubstanceImageCreator {
	/**
	 * Paints border instance of specified dimensions and status.
	 * 
	 * @param c
	 *            Component.
	 * @param graphics
	 *            Graphics context.
	 * @param x
	 *            Component left X (in graphics context).
	 * @param y
	 *            Component top Y (in graphics context).
	 * @param width
	 *            Border width.
	 * @param height
	 *            Border height.
	 * @param radius
	 *            Border radius.
	 * @param borderScheme
	 *            Border color scheme.
	 */
	public static void paintBorder(Component c, Graphics2D graphics, int x,
			int y, int width, int height, float radius, ColorScheme borderScheme) {

		SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
				.getBorderPainter(c, null);
		graphics.translate(x, y);
		int borderDelta = (int) Math.floor(SubstanceSizeUtils
				.getBorderStrokeWidth(SubstanceSizeUtils
						.getComponentFontSize(c)) / 2.0);
		Shape contour = BaseButtonShaper.getBaseOutline(width, height, radius,
				null, borderDelta);
		int borderThickness = (int) SubstanceSizeUtils
				.getBorderStrokeWidth(SubstanceSizeUtils
						.getComponentFontSize(c));
		GeneralPath contourInner = BaseButtonShaper.getBaseOutline(width,
				height, radius, null, borderThickness + borderDelta);
		borderPainter.paintBorder(graphics, c, width, height, contour,
				contourInner, borderScheme, borderScheme, 0, false);
		graphics.translate(-x, -y);
	}

	/**
	 * Paints border instance of specified dimensions and status.
	 * 
	 * @param c
	 *            Component.
	 * @param graphics
	 *            Graphics context.
	 * @param x
	 *            Component left X (in graphics context).
	 * @param y
	 *            Component top Y (in graphics context).
	 * @param width
	 *            Border width.
	 * @param height
	 *            Border height.
	 * @param radius
	 *            Border radius.
	 * @param theme
	 *            Border theme.
	 */
	public static void paintBorder(Component c, Graphics2D graphics, int x,
			int y, int width, int height, float radius, SubstanceTheme theme) {
		SubstanceImageCreator.paintBorder(c, graphics, x, y, width, height,
				radius, theme.getBorderTheme().getColorScheme());
	}

	/**
	 * Retrieves check mark image for {@link SubstanceCheckBoxUI}.
	 * 
	 * @param dimension
	 *            Check mark dimension.
	 * @param isEnabled
	 *            Enabled status.
	 * @param theme
	 *            Theme for the check mark.
	 * @param checkMarkVisibility
	 *            Checkmark visibility in 0.0-1.0 range.
	 * @return Check mark image.
	 */
	private static BufferedImage getCheckMark(int dimension, boolean isEnabled,
			SubstanceTheme theme, float checkMarkVisibility) {
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(dimension,
				dimension);

		// get graphics and set hints
		Graphics2D graphics = (Graphics2D) result.getGraphics();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		// create curved checkbox path
		GeneralPath path = new GeneralPath();

		path.moveTo(0.25f * dimension, 0.5f * dimension);
		path.quadTo(0.37f * dimension, 0.6f * dimension, 0.47f * dimension,
				0.8f * dimension);
		path.quadTo(0.55f * dimension, 0.5f * dimension, 0.85f * dimension, 0f);

		// compute the x-based clip for the visibility
		float xClipStart = 0.15f * dimension;
		float xClipEnd = 0.95f * dimension;

		float xClipRealEnd = xClipStart + (xClipEnd - xClipStart)
				* checkMarkVisibility;

		graphics.setClip(0, 0, (int) Math.ceil(xClipRealEnd), dimension);

		graphics
				.setColor(SubstanceCoreUtilities.getMarkColor(theme, isEnabled));
		Stroke stroke = new BasicStroke((float) 0.15 * dimension,
				BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND);
		graphics.setStroke(stroke);
		graphics.draw(path);

		if (isEnabled) {
			graphics.setColor(theme.getBorderTheme().getColorScheme()
					.getDarkColor());
			Stroke stroke2 = new BasicStroke((float) 0.05 * dimension,
					BasicStroke.CAP_ROUND, BasicStroke.JOIN_ROUND);
			graphics.setStroke(stroke2);
			graphics.draw(path);
		}

		return result;
	}

	/**
	 * Returns arrow icon for the specified parameters.
	 * 
	 * @param fontSize
	 *            Font size.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Icon theme.
	 * @return Arrow icon.
	 */
	public static Icon getArrowIcon(int fontSize, int direction,
			SubstanceTheme theme) {
		return new ImageIcon(getArrow(SubstanceSizeUtils
				.getArrowIconWidth(fontSize), SubstanceSizeUtils
				.getArrowIconHeight(fontSize), SubstanceSizeUtils
				.getArrowStrokeWidth(fontSize), direction, theme));
	}

	/**
	 * Returns arrow icon for the specified parameters.
	 * 
	 * @param fontSize
	 *            Font size.
	 * @param deltaWidth
	 *            Arrow width delta.
	 * @param deltaHeight
	 *            Arrow height delta.
	 * @param deltaStrokeWidth
	 *            Arrow stroke width delta.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Icon theme.
	 * @return Arrow icon.
	 */
	public static Icon getArrowIcon(int fontSize, int deltaWidth,
			int deltaHeight, float deltaStrokeWidth, int direction,
			SubstanceTheme theme) {
		int arrowWidth = SubstanceSizeUtils.getArrowIconWidth(fontSize);
		if (deltaWidth != 0) {
			arrowWidth += SubstanceSizeUtils.getAdjustedSize(fontSize,
					deltaWidth, 6, -1, true);// - 1;
		}
		if (arrowWidth % 2 == 0)
			arrowWidth--;
		int arrowHeight = SubstanceSizeUtils.getArrowIconHeight(fontSize);
		if (deltaHeight != 0) {
			arrowHeight += SubstanceSizeUtils.getAdjustedSize(fontSize,
					deltaHeight, 4, -1, false);
		}
		float arrowStrokeWidth = SubstanceSizeUtils
				.getArrowStrokeWidth(fontSize);
		if (deltaStrokeWidth != 0) {
			arrowStrokeWidth += SubstanceSizeUtils.getAdjustedSize(fontSize,
					deltaStrokeWidth, 1, 0.04f);
		}
		return new ImageIcon(getArrow(arrowWidth, arrowHeight,
				arrowStrokeWidth, direction, theme));
	}

	/**
	 * Retrieves arrow icon.
	 * 
	 * @param width
	 *            Arrow width.
	 * @param height
	 *            Arrow height.
	 * @param strokeWidth
	 *            Stroke width.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Theme for the arrow.
	 * @return Arrow image.
	 * @see SwingConstants#NORTH
	 * @see SwingConstants#WEST
	 * @see SwingConstants#SOUTH
	 * @see SwingConstants#EAST
	 */
	public static Icon getArrowIcon(int width, int height, float strokeWidth,
			int direction, SubstanceTheme theme) {
		return new ImageIcon(getArrow(width, height, strokeWidth, direction,
				theme));
	}

	/**
	 * Retrieves arrow image.
	 * 
	 * @param width
	 *            Arrow width.
	 * @param height
	 *            Arrow height.
	 * @param strokeWidth
	 *            Stroke width.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Theme for the arrow.
	 * @return Arrow image.
	 * @see SwingConstants#NORTH
	 * @see SwingConstants#WEST
	 * @see SwingConstants#SOUTH
	 * @see SwingConstants#EAST
	 */
	public static BufferedImage getArrow(int width, int height,
			float strokeWidth, int direction, SubstanceTheme theme) {
		BufferedImage arrowImage = SubstanceCoreUtilities.getBlankImage(width,
				height);

		// System.out.println(width + ":" + height + ":" + strokeWidth);

		// get graphics and set hints
		Graphics2D graphics = (Graphics2D) arrowImage.getGraphics();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		Color arrowColor = SubstanceCoreUtilities.getMarkColor(theme
				.getActiveTheme(), true);

		graphics.setColor(arrowColor);
		Stroke stroke = new BasicStroke(strokeWidth, BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_MITER);

		graphics.setStroke(stroke);

		if (direction == SwingConstants.CENTER) {
			GeneralPath gp = new GeneralPath();
			float h2 = height / 2.f;
			float w2 = (float) 0.5 * (width - 1);
			gp.moveTo(0, h2 - 1f);
			gp.lineTo(w2, 2);
			gp.lineTo(width - 1, h2 - 1f);
			gp.moveTo(width - 1, h2 + 1f);
			gp.lineTo(w2, height - 2);
			gp.lineTo(0, h2 + 1f);
			graphics.draw(gp);
			return arrowImage;

		} else {

			GeneralPath gp = new GeneralPath();
			gp.moveTo(0, 0);
			gp.lineTo((float) 0.5 * (width - 1), height - 2);
			gp.lineTo(width - 1, 0);
			graphics.draw(gp);
			// graphics.drawPolyline(new int[] { 0, width / 2, width - 1 }, new
			// int[] {
			// 0, height - 2, 0 }, 3);

			int quadrantCounterClockwise = 0;
			switch (direction) {
			case SwingConstants.NORTH:
				quadrantCounterClockwise = 2;
				break;
			case SwingConstants.WEST:
				quadrantCounterClockwise = 1;
				break;
			case SwingConstants.SOUTH:
				quadrantCounterClockwise = 0;
				break;
			case SwingConstants.EAST:
				quadrantCounterClockwise = 3;
				break;
			}
			BufferedImage rotatedImage = SubstanceImageCreator.getRotated(
					arrowImage, quadrantCounterClockwise);

			return rotatedImage;
		}
	}

	/**
	 * Returns double arrow icon for the specified parameters.
	 * 
	 * @param fontSize
	 *            Font size.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Icon theme.
	 * @return Double arrow icon.
	 */
	public static Icon getDoubleArrowIcon(int fontSize, int direction,
			SubstanceTheme theme) {
		return getDoubleArrowIcon(fontSize, 0, 0, 0, direction, theme);
	}

	/**
	 * Returns double arrow icon for the specified parameters.
	 * 
	 * @param fontSize
	 *            Font size.
	 * @param deltaWidth
	 *            Arrow width delta.
	 * @param deltaHeight
	 *            Arrow height delta.
	 * @param deltaStrokeWidth
	 *            Arrow stroke width delta.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Icon theme.
	 * @return Double arrow icon.
	 */
	public static Icon getDoubleArrowIcon(int fontSize, int deltaWidth,
			int deltaHeight, float deltaStrokeWidth, int direction,
			SubstanceTheme theme) {
		int arrowWidth = SubstanceSizeUtils
				.getArrowIconWidth(SubstanceSizeUtils.getControlFontSize())
				+ deltaWidth;
		int arrowHeight = SubstanceSizeUtils
				.getArrowIconHeight(SubstanceSizeUtils.getControlFontSize())
				+ deltaHeight;
		float arrowStrokeWidth = SubstanceSizeUtils
				.getDoubleArrowStrokeWidth(SubstanceSizeUtils
						.getControlFontSize())
				+ deltaStrokeWidth;
		return getDoubleArrowIcon(arrowWidth, arrowHeight, arrowStrokeWidth,
				direction, theme);
	}

	/**
	 * Retrieves arrow icon.
	 * 
	 * @param width
	 *            Arrow width.
	 * @param height
	 *            Arrow height.
	 * @param strokeWidth
	 *            Stroke width.
	 * @param direction
	 *            Arrow direction.
	 * @param theme
	 *            Theme for the arrow.
	 * @return Arrow image.
	 * @see SwingConstants#NORTH
	 * @see SwingConstants#WEST
	 * @see SwingConstants#SOUTH
	 * @see SwingConstants#EAST
	 */
	public static Icon getDoubleArrowIcon(int width, int height,
			float strokeWidth, int direction, SubstanceTheme theme) {
		int delta = 3 + SubstanceSizeUtils.getExtraPadding(SubstanceSizeUtils
				.getControlFontSize());
		BufferedImage downArrowImage = SubstanceCoreUtilities.getBlankImage(
				width, height + delta - 2);

		// get graphics and set hints
		Graphics2D graphics = (Graphics2D) downArrowImage.getGraphics();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		Color arrowColor = SubstanceCoreUtilities.getMarkColor(theme
				.getActiveTheme(), true);
		graphics.setColor(arrowColor);
		Stroke stroke = new BasicStroke(strokeWidth, BasicStroke.CAP_BUTT,
				BasicStroke.JOIN_MITER);

		graphics.setStroke(stroke);
		GeneralPath gp = new GeneralPath();
		gp.moveTo(0, 0);
		gp.lineTo((float) 0.5 * (width - 1), height - delta - 1);
		gp.lineTo(width - 1, 0);
		graphics.draw(gp);

		GeneralPath gp2 = new GeneralPath();
		gp2.moveTo(0, delta);
		gp2.lineTo((float) 0.5 * (width - 1), height - 1);
		gp2.lineTo(width - 1, delta);
		graphics.draw(gp2);

		int quadrantCounterClockwise = 0;
		switch (direction) {
		case SwingConstants.NORTH:
			quadrantCounterClockwise = 2;
			break;
		case SwingConstants.WEST:
			quadrantCounterClockwise = 1;
			break;
		case SwingConstants.SOUTH:
			quadrantCounterClockwise = 0;
			break;
		case SwingConstants.EAST:
			quadrantCounterClockwise = 3;
			break;
		}
		BufferedImage arrowImage = SubstanceImageCreator.getRotated(
				downArrowImage, quadrantCounterClockwise);

		return new ImageIcon(arrowImage);
	}

	/**
	 * Returns rotated image.
	 * 
	 * @param bi
	 *            Image to rotate.
	 * @param quadrantClockwise
	 *            Amount of quadrants to rotate in clockwise directio. The
	 *            rotation angle is 90 times this value.
	 * @return Rotated image.
	 */
	public static BufferedImage getRotated(final BufferedImage bi,
			int quadrantClockwise) {
		quadrantClockwise = quadrantClockwise % 4;
		int width = bi.getWidth();
		int height = bi.getHeight();
		if ((quadrantClockwise == 1) || (quadrantClockwise == 3)) {
			width = bi.getHeight();
			height = bi.getWidth();
		}
		BufferedImage biRot = SubstanceCoreUtilities.getBlankImage(width,
				height);
		switch (quadrantClockwise) {
		case 0:
			return bi;
		case 1:
			// 90 deg
			for (int col = 0; col < width; col++) {
				for (int row = 0; row < height; row++) {
					biRot.setRGB(col, row, bi.getRGB(row, width - col - 1));
				}
			}
			return biRot;
		case 2:
			// 180 deg
			for (int col = 0; col < width; col++) {
				for (int row = 0; row < height; row++) {
					biRot.setRGB(col, row, bi.getRGB(width - col - 1, height
							- row - 1));
				}
			}
			return biRot;
		case 3:
			// 270 deg
			for (int col = 0; col < width; col++) {
				for (int row = 0; row < height; row++) {
					biRot.setRGB(col, row, bi.getRGB(height - row - 1, col));
				}
			}
			return biRot;
		}
		return null;
	}

	/**
	 * Translated the specified icon to grey scale.
	 * 
	 * @param icon
	 *            Icon.
	 * @return Greyscale version of the specified icon.
	 */
	public static Icon toGreyscale(Icon icon) {
		if (icon == null) {
			return null;
		}

		int width = icon.getIconWidth();
		int height = icon.getIconHeight();

		BufferedImage result = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_ARGB);

		// set completely transparent
		for (int col = 0; col < width; col++) {
			for (int row = 0; row < height; row++) {
				result.setRGB(col, row, 0x0);
			}
		}

		icon.paintIcon(null, result.getGraphics(), 0, 0);
		for (int col = 0; col < width; col++) {
			for (int row = 0; row < height; row++) {
				int color = result.getRGB(col, row);
				int transp = (color >>> 24) & 0xFF;

				int newComp = SubstanceColorUtilities.getColorBrightness(color);
				int newColor = (transp << 24) | (newComp << 16)
						| (newComp << 8) | newComp;

				result.setRGB(col, row, newColor);
			}
		}

		return new ImageIcon(result);
	}

	/**
	 * Makes the specified icon transparent.
	 * 
	 * @param c
	 *            Component.
	 * @param icon
	 *            Icon.
	 * @param alpha
	 *            The opaqueness of the resulting image. The closer this value
	 *            is to 0.0, the more transparent resulting image will be.
	 * @return Transparent version of the specified icon.
	 */
	public static Icon makeTransparent(Component c, Icon icon, double alpha) {
		if (icon == null) {
			return null;
		}

		int width = icon.getIconWidth();
		int height = icon.getIconHeight();

		BufferedImage result = new BufferedImage(width, height,
				BufferedImage.TYPE_INT_ARGB);

		// set completely transparent
		for (int col = 0; col < width; col++) {
			for (int row = 0; row < height; row++) {
				result.setRGB(col, row, 0x0);
			}
		}

		icon.paintIcon(c, result.getGraphics(), 0, 0);
		for (int col = 0; col < width; col++) {
			for (int row = 0; row < height; row++) {
				int color = result.getRGB(col, row);
				int transp = (int) (alpha * ((color >>> 24) & 0xFF));
				int r = (color >>> 16) & 0xFF;
				int g = (color >>> 8) & 0xFF;
				int b = (color >>> 0) & 0xFF;

				int newColor = (transp << 24) | (r << 16) | (g << 8) | b;

				result.setRGB(col, row, newColor);
			}
		}

		return new ImageIcon(result);
	}

	/**
	 * Retrieves radio button of the specified size that matches the specified
	 * component state.
	 * 
	 * @param dimension
	 *            Radio button size.
	 * @param componentState
	 *            Component state.
	 * @param offsetX
	 *            Offset in X axis.
	 * @return Radio button of the specified size that matches the specified
	 *         component state.
	 */
	public static BufferedImage getRadioButton(int dimension,
			ComponentState componentState, int offsetX) {
		ComponentState.ColorSchemeKind kind = componentState
				.getColorSchemeKind();
		SubstanceTheme theme = SubstanceCoreUtilities.getComponentTheme(null,
				kind);
		//
		// SubstanceCoreUtilities.getTheme(
		// SubstanceLookAndFeel.getTheme(), kind).getColorScheme();
		return SubstanceImageCreator.getRadioButton(null, dimension,
				componentState, offsetX, theme, theme, 0, componentState
						.isKindActive(FadeKind.SELECTION) ? 1.0f : 0.0f);
	}

	/**
	 * Retrieves radio button of the specified size that matches the specified
	 * parameters.
	 * 
	 * @param component
	 *            Component.
	 * @param dimension
	 *            Radio button dimension.
	 * @param componentState
	 *            Component state.
	 * @param offsetX
	 *            Offset on X axis - should be positive in order to see the
	 *            entire radio button.
	 * @param theme1
	 *            Theme 1.
	 * @param theme2
	 *            Theme 2.
	 * @param interpolationCyclePos10
	 *            Interpolation cycle.
	 * @param checkMarkVisibility
	 *            Checkmark visibility in 0.0-1.0 range.
	 * @return Radio button of the specified size that matches the specified
	 *         parameters.
	 */
	public static BufferedImage getRadioButton(JComponent component,
			int dimension, ComponentState componentState, int offsetX,
			SubstanceTheme theme1, SubstanceTheme theme2,
			float interpolationCyclePos10, float checkMarkVisibility) {

		// ComponentState.ColorSchemeKind kind = componentState
		// .getColorSchemeKind();

		float cyclePos = (theme1 != theme2) ? interpolationCyclePos10
				: componentState.getCycleCount();

		Shape contour = new Ellipse2D.Float(0, 0, dimension - 1, dimension - 1);
		// GeneralPath contour = BaseButtonShaper.getBaseOutline(dimension,
		// dimension, (dimension - 1) / 2.0f, null);
		SubstanceGradientPainter painter = SubstanceLookAndFeel
				.getCurrentGradientPainter();
		// if ((componentState == ComponentState.PRESSED_SELECTED)
		// || (componentState == ComponentState.PRESSED_UNSELECTED)) {
		// painter = new FlatGradientPainter();
		// }
		if (componentState.getColorSchemeKind() != ColorSchemeKind.CURRENT) {
			painter = SubstanceCoreUtilities.getTheme(component, true)
					.getNonActivePainter();
		}
		SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
				.getBorderPainter(component, painter);
		BufferedImage background = painter.getContourBackground(dimension,
				dimension, contour, false, theme1.getColorScheme(), theme2
						.getColorScheme(), cyclePos, true, theme1 != theme2);

		float borderThickness = SubstanceSizeUtils
				.getBorderStrokeWidth(dimension);
		int delta = (int) (borderThickness - 0.6);
		// System.out.println(dimension + ":" + borderThickness + ":" + delta);

		Shape contourBorder = new Ellipse2D.Float(delta, delta, dimension - 2
				* delta - 1, dimension - 2 * delta - 1);

		// BufferedImage background = getContourBackground(dimension, dimension,
		// contour, colorScheme, cyclePos, true);

		BufferedImage offBackground = SubstanceCoreUtilities.getBlankImage(
				dimension + offsetX, dimension);
		Graphics2D graphics = (Graphics2D) offBackground.getGraphics();
		float alpha = SubstanceCoreUtilities.getTheme(component, true)
				.getThemeAlpha(component, componentState);
		graphics.setComposite(AlphaComposite.getInstance(
				AlphaComposite.SRC_OVER, alpha));

		// graphics.drawImage(background, offsetX, 0, null);
		graphics.translate(offsetX, 0);
		graphics.drawImage(background, 0, 0, null);
		Shape contourInner = new Ellipse2D.Float(borderThickness,
				borderThickness, dimension - 2 * borderThickness - 1, dimension
						- 2 * borderThickness - 1);
		borderPainter.paintBorder(graphics, component, dimension, dimension,
				contourBorder, contourInner, theme1.getBorderTheme()
						.getColorScheme(), theme2.getBorderTheme()
						.getColorScheme(), cyclePos, theme1 != theme2);
		graphics.translate(-offsetX, 0);

		float rc = dimension / 2.0f;
		float radius = dimension / 4.5f;
		graphics.translate(offsetX, 0);

		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		if (componentState.isKindActive(FadeKind.SELECTION)
				|| (checkMarkVisibility > 0.0)) {
			// mark
			Shape markOval = new Ellipse2D.Double(rc - radius, rc - radius,
					2 * radius, 2 * radius);
			float innerRadius = dimension / 7.0f;
			Shape innerMarkOval = new Ellipse2D.Double(rc - innerRadius, rc
					- innerRadius, 2 * innerRadius, 2 * innerRadius);

			graphics.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, alpha * checkMarkVisibility));
			drawRadioMark(graphics,
					SubstanceCoreUtilities.getMarkColor(theme1, componentState
							.getColorSchemeKind() != ColorSchemeKind.DISABLED),
					markOval);
			if (componentState.isKindActive(FadeKind.ENABLE))
				drawRadioMark(graphics, theme1.getBorderTheme()
						.getColorScheme().getDarkColor(), innerMarkOval);
			graphics.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, alpha * checkMarkVisibility
							* cyclePos / 10.0f));
			drawRadioMark(graphics,
					SubstanceCoreUtilities.getMarkColor(theme2, componentState
							.getColorSchemeKind() != ColorSchemeKind.DISABLED),
					markOval);
			if (componentState.isKindActive(FadeKind.ENABLE))
				drawRadioMark(graphics, theme2.getBorderTheme()
						.getColorScheme().getDarkColor(), innerMarkOval);
		} else {
			// draw ghost mark holder
			graphics.setPaint(new GradientPaint(rc + radius, rc - radius,
					theme1.getColorScheme().getDarkColor(), rc - radius, rc
							+ radius, theme1.getColorScheme().getLightColor()));
			Shape markOval = new Ellipse2D.Double(rc - radius, rc - radius,
					2 * radius, 2 * radius);
			graphics.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, alpha * 0.3f));
			graphics.fill(markOval);
		}
		graphics.translate(-offsetX, 0);

		return offBackground;
	}

	/**
	 * Draws radio mark.
	 * 
	 * @param graphics
	 *            Graphics context.
	 * @param color
	 *            Radio mark color.
	 * @param markOval
	 *            Radio mark shape.
	 */
	private static void drawRadioMark(Graphics2D graphics, Color color,
			Shape markOval) {
		graphics.setColor(color);
		graphics.fill(markOval);
	}

	/**
	 * Retrieves check box of the specified size that matches the specified
	 * component state.
	 * 
	 * @param dimension
	 *            Check box size.
	 * @param componentState
	 *            Component state.
	 * @return Check box of the specified size that matches the specified
	 *         component state.
	 */
	public static BufferedImage getCheckBox(int dimension,
			ComponentState componentState) {

		ComponentState.ColorSchemeKind kind = componentState
				.getColorSchemeKind();
		SubstanceTheme theme = SubstanceCoreUtilities.getComponentTheme(null,
				kind);
		return SubstanceImageCreator.getCheckBox(null, dimension,
				componentState, theme, theme, 0, componentState
						.isKindActive(FadeKind.SELECTION) ? 1.0f : 0.0f);
	}

	/**
	 * Retrieves check box of the specified size that matches the specified
	 * component state.
	 * 
	 * @param component
	 *            Component.
	 * @param dimension
	 *            Check box size.
	 * @param componentState
	 *            Component state.
	 * @return Check box of the specified size that matches the specified
	 *         component state.
	 * @param theme1
	 *            Theme 1.
	 * @param theme2
	 *            Theme 2.
	 * @param interpolationCyclePos10
	 *            Interpolation cycle.
	 * @param checkMarkVisibility
	 *            Checkmark visibility in 0.0-1.0 range.
	 */
	public static BufferedImage getCheckBox(JComponent component,
			int dimension, ComponentState componentState,
			SubstanceTheme theme1, SubstanceTheme theme2,
			float interpolationCyclePos10, float checkMarkVisibility) {

		int offset = SubstanceSizeUtils.getAdjustedSize(SubstanceSizeUtils
				.getComponentFontSize(component), 3, 9, 1, false);
		// 3 + SubstanceSizeUtils.getExtraPadding(3) / 3;
		int delta = offset;// + SubstanceSizeUtils.getExtraPadding(dimension,
		// 3);
		float cornerRadius = SubstanceSizeUtils
				.getClassicButtonCornerRadius(SubstanceSizeUtils
						.getComponentFontSize(component));
		if (dimension <= 10) {
			offset = 2;
			cornerRadius = 2;
		}

		float cyclePos = (theme1 != theme2) ? interpolationCyclePos10
				: componentState.getCycleCount();

		int contourDim = dimension - delta;
		int borderDelta = (int) Math.floor(SubstanceSizeUtils
				.getBorderStrokeWidth(SubstanceSizeUtils
						.getComponentFontSize(component)) / 2.0);
		GeneralPath contour = BaseButtonShaper.getBaseOutline(contourDim,
				contourDim, cornerRadius, null, borderDelta);

		SubstanceGradientPainter painter = SubstanceLookAndFeel
				.getCurrentGradientPainter();
		// if ((componentState == ComponentState.PRESSED_SELECTED)
		// || (componentState == ComponentState.PRESSED_UNSELECTED)) {
		// painter = new FlatGradientPainter();
		// }
		if (componentState.getColorSchemeKind() != ColorSchemeKind.CURRENT) {
			painter = SubstanceCoreUtilities.getTheme(component, true)
					.getNonActivePainter();
		}
		SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
				.getBorderPainter(component, painter);

		BufferedImage background = painter.getContourBackground(contourDim,
				contourDim, contour, false, theme1.getColorScheme(), theme2
						.getColorScheme(), cyclePos, true, theme1 != theme2);

		BufferedImage offBackground = SubstanceCoreUtilities.getBlankImage(
				dimension, dimension);
		Graphics2D graphics = (Graphics2D) offBackground.getGraphics();
		float alpha = SubstanceCoreUtilities.getTheme(component, true)
				.getThemeAlpha(component, componentState);
		graphics.setComposite(AlphaComposite.getInstance(
				AlphaComposite.SRC_OVER, alpha));

		graphics.translate(delta, delta - 1);
		graphics.drawImage(background, 0, 0, null);
		int borderThickness = (int) SubstanceSizeUtils
				.getBorderStrokeWidth(dimension);
		GeneralPath contourInner = BaseButtonShaper.getBaseOutline(contourDim,
				contourDim, cornerRadius, null, borderThickness + borderDelta);
		borderPainter.paintBorder(graphics, component, contourDim, contourDim,
				contour, contourInner,
				theme1.getBorderTheme().getColorScheme(), theme2
						.getBorderTheme().getColorScheme(), cyclePos,
				theme1 != theme2);
		graphics.translate(-delta, 1 - delta);
		if (componentState.isKindActive(FadeKind.SELECTION)
				|| (checkMarkVisibility > 0.0)) {
			BufferedImage checkMark1 = SubstanceImageCreator.getCheckMark(
					dimension - 2 * offset / 3, componentState
							.isKindActive(FadeKind.ENABLE), theme1,
					checkMarkVisibility);
			graphics.drawImage(checkMark1, 1 + 2 * offset / 3,
					(dimension < 14) ? 0 : -1, null);
			BufferedImage checkMark2 = SubstanceImageCreator.getCheckMark(
					dimension - 2 * offset / 3, componentState
							.isKindActive(FadeKind.ENABLE), theme2,
					checkMarkVisibility);
			graphics.setComposite(AlphaComposite.getInstance(
					AlphaComposite.SRC_OVER, alpha * cyclePos / 10.0f));
			graphics.drawImage(checkMark2, 1 + 2 * offset / 3,
					(dimension < 14) ? 0 : -1, null);
		}

		return offBackground;
	}

	/**
	 * Retrieves composite background for the specified parameters. The
	 * composite background consists of three layers:
	 * <ol>
	 * <li>Layer that matches the <code>increased</code> state.
	 * <li>Layer that matches the <code>decreased</code> state.
	 * <li>Regular layer with rounded background.
	 * </ol>
	 * The layers are drawn in the following order:
	 * <ol>
	 * <li>The left half of the first layer
	 * <li>The right half of the second layer
	 * <li>The third layer
	 * </ol>
	 * Combined together, the layers create the image for scrollbar track with
	 * continuation of the arrow increase and decrease buttons.
	 * 
	 * @param component
	 *            Component.
	 * @param width
	 *            Image width.
	 * @param height
	 *            Image height.
	 * @param cornerRadius
	 *            Corner radius.
	 * @param decrButton
	 *            The <code>decrease</code> button.
	 * @param incrButton
	 *            The <code>increase</code> button.
	 * @param flipSides
	 *            If <code>true</code>, the drawn halves of the first and the
	 *            second layers above will be swapped.
	 * @return Composite background for the specified parameters.
	 */
	public static BufferedImage getCompositeRoundedBackground(
			JComponent component, int width, int height, int cornerRadius,
			AbstractButton decrButton, AbstractButton incrButton,
			boolean flipSides) {

		BufferedImage decrLayer = null;
		BufferedImage incrLayer = null;

		int delta = 3;
		Set<Side> left = new HashSet<Side>();
		left.add(Side.LEFT);
		Set<Side> right = new HashSet<Side>();
		right.add(Side.RIGHT);
		SubstanceGradientPainter painter = new SimplisticSoftBorderReverseGradientPainter();

		if (decrButton != null) {
			decrLayer = PairwiseButtonBackgroundDelegate.getPairwiseBackground(
					decrButton, painter, width + 2 * delta, height,
					flipSides ? Side.RIGHT : Side.LEFT, true);
		}

		if (incrButton != null) {
			incrLayer = PairwiseButtonBackgroundDelegate.getPairwiseBackground(
					incrButton, painter, width + 2 * delta, height,
					flipSides ? Side.LEFT : Side.RIGHT, true);
		}

		BufferedImage result = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) result.getGraphics();
		if (!flipSides) {
			if (decrLayer != null) {
				graphics.drawImage(decrLayer, -delta, 0, width / 2, height, 0,
						0, width / 2, height, null);
			}
			if (incrLayer != null) {
				graphics.drawImage(incrLayer, width / 2, 0, width, height,
						width / 2, 0, width, height, null);
			}
		} else {
			if (incrLayer != null) {
				graphics.drawImage(incrLayer, -delta, 0, width / 2, height, 0,
						0, width / 2, height, null);
			}
			if (decrLayer != null) {
				graphics.drawImage(decrLayer, width / 2, 0, width, height,
						width / 2, 0, width, height, null);
			}
		}

		return result;
	}

	/**
	 * Returns a one-pixel high line of the specified width that has gradient
	 * based on the parameters.
	 * 
	 * @param width
	 *            The width of the resulting image.
	 * @param colorLeft
	 *            The color of the leftmost pixel.
	 * @param colorRight
	 *            The color of the rightmost pixel.
	 * @param waypoints
	 *            Each entry in this map specifies color for some
	 *            <code>waypoint</code>. The pixels between the waypoints
	 *            will be colored based on the interpolation of the two closest
	 *            waypoints.
	 * @return One-pixel high line of the specified width that has gradient
	 *         based on the parameters.
	 */
	public static BufferedImage getOneLineGradient(int width, Color colorLeft,
			Color colorRight, Map<Integer, Color> waypoints) {
		BufferedImage image = new BufferedImage(width, 1,
				BufferedImage.TYPE_INT_ARGB);

		Graphics2D graphics = (Graphics2D) image.getGraphics();

		java.util.List<Integer> waypointMarkers = new ArrayList<Integer>();
		if (waypoints != null) {
			for (Integer marker : waypoints.keySet()) {
				waypointMarkers.add(marker);
			}
		}
		Collections.sort(waypointMarkers);

		int[] markers = new int[waypointMarkers.size() + 2];
		Color[] colors = new Color[waypointMarkers.size() + 2];
		markers[0] = 0;
		colors[0] = colorLeft;
		int index = 1;
		for (Integer marker : waypointMarkers) {
			markers[index] = marker;
			colors[index] = waypoints.get(marker);
			index++;
		}
		markers[index] = width - 1;
		colors[index] = colorRight;

		for (int i = 0; i < (markers.length - 1); i++) {
			GradientPaint gradient = new GradientPaint(markers[i], 0,
					colors[i], markers[i + 1], 0, colors[i + 1]);
			graphics.setPaint(gradient);
			graphics.drawLine(markers[i], 0, markers[i + 1], 0);
		}

		return image;
	}

	/**
	 * Paints a one-pixel high line of the specified width that has gradient
	 * based on the parameters.
	 * 
	 * @param graphics
	 *            Graphics context.
	 * @param x
	 *            X start coordinate.
	 * @param y
	 *            Y start coordinate.
	 * @param dimension
	 *            The dimension of the resulting image. If
	 *            <code>isVertical</code> parameter is <code>true</code>,
	 *            the resulting painting will be 1 pixel high and
	 *            <code>dimension</code> pixels wide, otherwise it will be
	 *            <code>dimension</code> pixels high and 1 pixel wide.
	 * @param isVertical
	 *            Indication of horizontal / vertical orientation.
	 * @param colorLeft
	 *            The color of the leftmost pixel.
	 * @param colorRight
	 *            The color of the rightmost pixel.
	 * @param waypoints
	 *            Each entry in this map specifies color for some
	 *            <code>waypoint</code>. The pixels between the waypoints
	 *            will be colored based on the interpolation of the two closest
	 *            waypoints.
	 */
	public static void paintOneLineGradient(Graphics2D graphics, int x, int y,
			int dimension, boolean isVertical, Color colorLeft,
			Color colorRight, Map<Integer, Color> waypoints) {

		graphics.translate(x, y);
		java.util.List<Integer> waypointMarkers = new ArrayList<Integer>();
		if (waypoints != null) {
			for (Integer marker : waypoints.keySet()) {
				waypointMarkers.add(marker);
			}
		}
		Collections.sort(waypointMarkers);

		int[] markers = new int[waypointMarkers.size() + 2];
		Color[] colors = new Color[waypointMarkers.size() + 2];
		markers[0] = 0;
		colors[0] = colorLeft;
		int index = 1;
		for (Integer marker : waypointMarkers) {
			markers[index] = marker;
			colors[index] = waypoints.get(marker);
			index++;
		}
		markers[index] = dimension - 1;
		colors[index] = colorRight;

		if (!isVertical) {
			for (int i = 0; i < (markers.length - 1); i++) {
				GradientPaint gradient = new GradientPaint(markers[i], 0,
						colors[i], markers[i + 1], 0, colors[i + 1]);
				graphics.setPaint(gradient);
				graphics
						.fillRect(markers[i], 0, markers[i + 1] - markers[i], 1);
			}
		} else {
			for (int i = 0; i < (markers.length - 1); i++) {
				GradientPaint gradient = new GradientPaint(0, markers[i],
						colors[i], 0, markers[i + 1], colors[i + 1]);
				graphics.setPaint(gradient);
				graphics
						.fillRect(0, markers[i], 1, markers[i + 1] - markers[i]);
			}
		}
		graphics.translate(-x, -y);
	}

	/**
	 * Overlays light-colored echo below the specified image.
	 * 
	 * @param image
	 *            The input image.
	 * @param echoAlpha
	 *            Alpha channel for the echo image.
	 * @param offsetX
	 *            X offset of the echo.
	 * @param offsetY
	 *            Y offset of the echo.
	 * @return Image with overlayed echo.
	 */
	public static BufferedImage overlayEcho(BufferedImage image,
			float echoAlpha, int offsetX, int offsetY) {
		int width = image.getWidth();
		int height = image.getHeight();

		// blur the original image
		// ConvolveOp convolve = new ConvolveOp(new Kernel(3, 3, new float[] {
		// .4f, .4f, .4f, .4f, .0f, .4f, .4f, .4f, .4f }),
		// ConvolveOp.EDGE_NO_OP, null);
		offsetX = offsetY = 0;
		BufferedImage negated = getNegated(image);
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setComposite(AlphaComposite.getInstance(
				AlphaComposite.SRC_OVER, 0.2f * echoAlpha * echoAlpha
						* echoAlpha));
		graphics.drawImage(negated, offsetX - 1, offsetY - 1, null);
		graphics.drawImage(negated, offsetX + 1, offsetY - 1, null);
		graphics.drawImage(negated, offsetX - 1, offsetY + 1, null);
		graphics.drawImage(negated, offsetX + 1, offsetY + 1, null);
		graphics.setComposite(AlphaComposite.getInstance(
				AlphaComposite.SRC_OVER, 0.7f * echoAlpha * echoAlpha
						* echoAlpha));
		graphics.drawImage(negated, offsetX, offsetY - 1, null);
		graphics.drawImage(negated, offsetX, offsetY + 1, null);
		graphics.drawImage(negated, offsetX - 1, offsetY, null);
		graphics.drawImage(negated, offsetX + 1, offsetY, null);

		graphics.setComposite(AlphaComposite.getInstance(
				AlphaComposite.SRC_OVER, 1.0f));
		graphics.drawImage(image, 0, 0, null);

		graphics.dispose();
		return result;
		//
		// BufferedImage blurred = convolve.filter(image, null);
		// BufferedImage negatedBlurred = getNegated(blurred);
		// BufferedImage blurredNegated = convolve.filter(negated, null);
		// // put the original image
		// // blurredNegated.getGraphics().drawImage(image, 0, 0, null);
		// return negated;
	}

	/**
	 * Returns <code>minimize</code> icon.
	 * 
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Minimize</code> icon.
	 */
	public static Icon getMinimizeIcon(SubstanceTheme theme) {
		int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
		return getMinimizeIcon(iSize, theme);
	}

	/**
	 * Returns <code>minimize</code> icon.
	 * 
	 * @param iSize
	 *            Icon dimension.
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Minimize</code> icon.
	 */
	public static Icon getMinimizeIcon(int iSize, SubstanceTheme theme) {
		BufferedImage image = SubstanceCoreUtilities
				.getBlankImage(iSize, iSize);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		int start = (iSize / 4) - 2;
		int end = (3 * iSize / 4);// - 1;
		int size = end - start - 3;
		Color color = theme.getColorScheme().getForegroundColor();
		graphics.setColor(color);
		graphics.fillRect(start + 2, end - 2, size, 3);
		return new ImageIcon(SubstanceImageCreator.overlayEcho(image,
				SubstanceColorUtilities.getColorStrength(color), 1, 1));
	}

	/**
	 * Returns <code>restore</code> icon.
	 * 
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Restore</code> icon.
	 */
	public static Icon getRestoreIcon(SubstanceTheme theme) {
		int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
		BufferedImage image = SubstanceCoreUtilities
				.getBlankImage(iSize, iSize);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		int start = (iSize / 4) - 2;
		int end = (3 * iSize / 4) - 1;
		int size = end - start - 3;
		Color color = theme.getColorScheme().getForegroundColor();
		graphics.setColor(color);
		graphics.drawRect(start, end - size + 1, size, size);
		graphics.drawLine(start, end - size + 2, start + size, end - size + 2);
		graphics.fillRect(end - size, start + 1, size + 1, 2);
		graphics.drawLine(end, start + 1, end, start + size + 1);
		graphics.drawLine(start + size + 2, start + size + 1, end, start + size
				+ 1);
		return new ImageIcon(SubstanceImageCreator.overlayEcho(image,
				SubstanceColorUtilities.getColorStrength(color), 1, 1));
	}

	/**
	 * Returns <code>maximize</code> icon.
	 * 
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Maximize</code> icon.
	 */
	public static Icon getMaximizeIcon(SubstanceTheme theme) {
		int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
		return getMaximizeIcon(iSize, theme);
	}

	/**
	 * Returns <code>maximize</code> icon.
	 * 
	 * @param iSize
	 *            Icon dimension.
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Maximize</code> icon.
	 */
	public static Icon getMaximizeIcon(int iSize, SubstanceTheme theme) {
		BufferedImage image = SubstanceCoreUtilities
				.getBlankImage(iSize, iSize);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		int start = (iSize / 4) - 1;
		int end = iSize - start - 1;// (3 * iSize / 4);
		Color color = theme.getColorScheme().getForegroundColor();
		graphics.setColor(color);
		graphics.drawRect(start, start, end - start, end - start);
		graphics.drawLine(start, start + 1, end, start + 1);
		return new ImageIcon(SubstanceImageCreator.overlayEcho(image,
				SubstanceColorUtilities.getColorStrength(color), 1, 1));
	}

	/**
	 * Returns <code>close</code> icon.
	 * 
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Close</code> icon.
	 */
	public static Icon getCloseIcon(SubstanceTheme theme) {
		return SubstanceImageCreator.getCloseIcon(SubstanceSizeUtils
				.getTitlePaneIconSize(), theme);
	}

	/**
	 * Returns <code>close</code> icon.
	 * 
	 * @param iSize
	 *            Icon dimension.
	 * @param theme
	 *            Theme for the icon.
	 * @return <code>Close</code> icon.
	 */
	public static Icon getCloseIcon(int iSize, SubstanceTheme theme) {
		return getCloseIcon(iSize, theme.getColorScheme());
	}

	/**
	 * Returns <code>close</code> icon.
	 * 
	 * @param iSize
	 *            Icon dimension.
	 * @param colorScheme
	 *            Color scheme for the icon.
	 * @return <code>Close</code> icon.
	 */
	public static Icon getCloseIcon(int iSize, ColorScheme colorScheme) {
		BufferedImage image = SubstanceCoreUtilities
				.getBlankImage(iSize, iSize);
		Graphics2D graphics = (Graphics2D) image.getGraphics();
		if (iSize < 15) {
			graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
					RenderingHints.VALUE_ANTIALIAS_ON);
		} else {
			graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
					RenderingHints.VALUE_ANTIALIAS_OFF);
		}
		int start = (iSize / 4);// - 1;
		int end = (3 * iSize / 4);

		// System.out.println(iSize + ":" + start + ":" + end);

		Stroke stroke = new BasicStroke(SubstanceSizeUtils
				.getFocusStrokeWidth(iSize), BasicStroke.CAP_ROUND,
				BasicStroke.JOIN_ROUND);

		graphics.setStroke(stroke);
		Color color = colorScheme.getForegroundColor();
		graphics.setColor(color);
		graphics.drawLine(start, start, end, end);
		graphics.drawLine(start, end, end, start);
		return new ImageIcon(SubstanceImageCreator.overlayEcho(image,
				SubstanceColorUtilities.getColorStrength(color), 1, 1));
	}

	/**
	 * Paints rectangular gradient background.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param startX
	 *            Background starting X coord.
	 * @param startY
	 *            Background starting Y coord.
	 * @param width
	 *            Background width.
	 * @param height
	 *            Background height.
	 * @param colorScheme
	 *            Color scheme for the background.
	 * @param borderAlpha
	 *            Border alpha.
	 * @param isVertical
	 *            if <code>true</code>, the gradient will be vertical, if
	 *            <code>false</code>, the gradient will be horizontal.
	 */
	public static void paintRectangularBackground(Graphics g, int startX,
			int startY, int width, int height, ColorScheme colorScheme,
			float borderAlpha, boolean isVertical) {
		Graphics2D graphics = (Graphics2D) g.create();
		graphics.translate(startX, startY);

		if (colorScheme instanceof MixColorScheme) {
			ColorScheme[] schemes = ((MixColorScheme) colorScheme)
					.getOrigSchemes();
			if (!isVertical) {
				BufferedImage[] components = new BufferedImage[schemes.length];
				for (int i = 0; i < schemes.length; i++) {

					Map<Integer, Color> gradColors = new HashMap<Integer, Color>();
					gradColors.put((int) (0.4 * height), schemes[i]
							.getLightColor());
					gradColors.put((int) (0.5 * height), schemes[i]
							.getMidColor());

					BufferedImage horLine = SubstanceImageCreator
							.getOneLineGradient(height, schemes[i]
									.getUltraLightColor(), schemes[i]
									.getUltraLightColor(), gradColors);
					BufferedImage verLine = SubstanceImageCreator.getRotated(
							horLine, 1);

					BufferedImage schemeImage = SubstanceCoreUtilities
							.getBlankImage(width, height);
					Graphics2D schemeGraphics = (Graphics2D) schemeImage
							.getGraphics();
					for (int col = 0; col < width; col++) {
						schemeGraphics.drawImage(verLine, col, 0, null);
					}
					components[i] = schemeImage;
				}

				// Let the blending begin
				BufferedImage current = components[0];
				for (int i = 1; i < components.length; i++) {
					double start = (i - 0.3) / components.length;
					double end = (i + 0.3) / components.length;
					current = SubstanceCoreUtilities.blendImagesHorizontal(
							current, components[i], start, end);
				}

				// for (int row = 0; row < height; row++) {
				graphics.drawImage(current, 0, 0, null);
				// }
			} else {
				BufferedImage[] components = new BufferedImage[schemes.length];
				for (int i = 0; i < schemes.length; i++) {
					Map<Integer, Color> gradColors = new HashMap<Integer, Color>();
					gradColors.put((int) (0.4 * height), schemes[i]
							.getLightColor());
					gradColors.put((int) (0.5 * height), schemes[i]
							.getMidColor());
					components[i] = SubstanceImageCreator.getOneLineGradient(
							height, schemes[i].getUltraLightColor(), schemes[i]
									.getUltraLightColor(), gradColors);
				}

				// Let the blending begin
				BufferedImage current = components[0];
				for (int i = 1; i < components.length; i++) {
					double start = (i - 0.3) / components.length;
					double end = (i + 0.3) / components.length;
					current = SubstanceCoreUtilities.blendImagesHorizontal(
							current, components[i], start, end);
				}

				current = SubstanceImageCreator.getRotated(current, 1);

				for (int column = 0; column < width; column++) {
					graphics.drawImage(current, column, 0, null);
				}
			}
		} else {
			if (!isVertical) {
				Map<Integer, Color> gradColors = new HashMap<Integer, Color>();
				gradColors.put((int) (0.4 * height), colorScheme
						.getLightColor());
				gradColors.put((int) (0.5 * height), colorScheme.getMidColor());

				BufferedImage horLine = SubstanceImageCreator
						.getOneLineGradient(height, colorScheme
								.getUltraLightColor(), colorScheme
								.getUltraLightColor(), gradColors);
				BufferedImage verLine = SubstanceImageCreator.getRotated(
						horLine, 1);
				for (int col = 0; col < width; col++) {
					graphics.drawImage(verLine, col, 0, null);
				}
			} else {
				Map<Integer, Color> gradColors = new HashMap<Integer, Color>();
				gradColors
						.put((int) (0.4 * width), colorScheme.getLightColor());
				gradColors.put((int) (0.5 * width), colorScheme.getMidColor());

				BufferedImage horLine = SubstanceImageCreator
						.getOneLineGradient(width, colorScheme
								.getUltraLightColor(), colorScheme
								.getUltraLightColor(), gradColors);
				for (int row = 1; row < height; row++) {
					graphics.drawImage(horLine, 0, row, null);
				}
			}
		}

		if (borderAlpha > 0.0f) {
			Graphics2D g2d = (Graphics2D) graphics.create();
			g2d.setComposite(TransitionLayout.getAlphaComposite(null,
					borderAlpha, graphics));
			SubstanceImageCreator.paintBorder(null, g2d, 0, 0, width, height,
					0, colorScheme);
			g2d.dispose();
		}
		graphics.dispose();
	}

	/**
	 * Paints rectangular gradient background with spots and optional replicated
	 * stripe image.
	 * 
	 * @param g
	 *            Graphics context.
	 * @param startX
	 *            X start coordinate.
	 * @param startY
	 *            Y start coordinate.
	 * @param width
	 *            Background width.
	 * @param height
	 *            Background height.
	 * @param colorScheme
	 *            Color scheme for the background.
	 * @param stripeImage
	 *            Stripe image to replicate.
	 * @param stripeOffset
	 *            Offset of the first stripe replication.
	 * @param borderAlpha
	 *            Border alpha.
	 * @param isVertical
	 *            Indication of horizontal / vertical orientation.
	 */
	public static void paintRectangularStripedBackground(Graphics g,
			int startX, int startY, int width, int height,
			ColorScheme colorScheme, BufferedImage stripeImage,
			int stripeOffset, float borderAlpha, boolean isVertical) {
		Graphics2D graphics = (Graphics2D) g.create(startX, startY, width,
				height);
		if (!isVertical) {
			Map<Integer, Color> gradColors = new HashMap<Integer, Color>();
			gradColors.put((int) (0.2 * height), colorScheme.getLightColor());
			gradColors.put((int) (0.5 * height), colorScheme.getMidColor());
			gradColors.put((int) (0.8 * height), colorScheme.getLightColor());

			BufferedImage horLine = SubstanceImageCreator.getOneLineGradient(
					height, colorScheme.getDarkColor(), colorScheme
							.getDarkColor(), gradColors);
			BufferedImage verLine = SubstanceImageCreator
					.getRotated(horLine, 1);
			for (int col = 0; col < width; col++) {
				graphics.drawImage(verLine, col, 0, null);
			}

			if (stripeImage != null) {
				int stripeSize = stripeImage.getHeight();
				int stripeCount = width / stripeSize;
				stripeOffset = stripeOffset % (2 * stripeSize);
				for (int stripe = -2; stripe <= stripeCount; stripe += 2) {
					int stripePos = stripe * stripeSize + stripeOffset;

					graphics.drawImage(stripeImage, stripePos, 0, null);
				}
			}
		} else {
			Map<Integer, Color> gradColors = new HashMap<Integer, Color>();
			gradColors.put((int) (0.2 * width), colorScheme.getLightColor());
			gradColors.put((int) (0.5 * width), colorScheme.getMidColor());
			gradColors.put((int) (0.8 * width), colorScheme.getLightColor());

			BufferedImage horLine = SubstanceImageCreator.getOneLineGradient(
					width, colorScheme.getDarkColor(), colorScheme
							.getDarkColor(), gradColors);
			for (int row = 0; row < height; row++) {
				graphics.drawImage(horLine, 0, row, null);
			}

			if (stripeImage != null) {
				int stripeSize = stripeImage.getWidth();
				int stripeCount = height / stripeSize;
				stripeOffset = stripeOffset % (2 * stripeSize);
				for (int stripe = -2; stripe <= stripeCount; stripe += 2) {
					int stripePos = stripe * stripeSize + stripeOffset;

					graphics.drawImage(stripeImage, 0, stripePos, null);
				}
			}
		}

		if (borderAlpha > 0.0f) {
			Graphics2D g2d = (Graphics2D) graphics.create();
			g2d.setComposite(TransitionLayout.getAlphaComposite(null,
					borderAlpha, graphics));
			SubstanceImageCreator.paintBorder(null, g2d, 0, 0, width, height,
					0, colorScheme);
			g2d.dispose();
		}
		graphics.dispose();
	}

	/**
	 * Returns diagonal stripe image.
	 * 
	 * @param baseSize
	 *            Stripe base in pixels.
	 * @param color
	 *            Stripe color.
	 * @return Diagonal stripe image.
	 */
	public static BufferedImage getStripe(int baseSize, Color color) {
		int width = (int) (1.8 * baseSize);
		int height = baseSize;
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) result.getGraphics();

		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		Polygon polygon = new Polygon();
		polygon.addPoint(0, 0);
		polygon.addPoint(width - 1 - baseSize, 0);
		polygon.addPoint(width - 1, height - 1);
		polygon.addPoint(baseSize, height - 1);

		graphics.setColor(color);
		graphics.fillPolygon(polygon);
		graphics.drawPolygon(polygon);

		float[] BLUR = { 0.10f, 0.10f, 0.10f, 0.10f, 0.30f, 0.10f, 0.10f,
				0.10f, 0.10f };
		ConvolveOp vBlurOp = new ConvolveOp(new Kernel(3, 3, BLUR));
		BufferedImage blurred = vBlurOp.filter(result, null);

		return blurred;
	}

	/**
	 * Returns drag bumps image.
	 * 
	 * @param c
	 *            Component.
	 * @param theme
	 *            Theme.
	 * @param alwaysUseActive
	 *            Indicates whether the active theme should always be used.
	 * @param width
	 *            Drag bumps width.
	 * @param height
	 *            Drag bumps height.
	 * @param isCrowded
	 *            Indicates whether the bumps should be painted closely.
	 * @return Drag bumps image.
	 */
	public static BufferedImage getDragImage(Component c, SubstanceTheme theme,
			boolean alwaysUseActive, int width, int height, boolean isCrowded,
			int maxNumberOfStripes) {
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(width,
				height);
		Graphics2D graphics = (Graphics2D) result.getGraphics();

		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		boolean isDark = SubstanceCoreUtilities.isThemeDark(theme
				.getActiveTheme());
		ColorScheme colorScheme = (alwaysUseActive) ? theme.getActiveTheme()
				.getColorScheme() : theme.getDefaultTheme().getColorScheme();
		Color back1 = isDark ? colorScheme.getLightColor()
				: SubstanceColorUtilities.getInterpolatedColor(colorScheme
						.getLightColor(), colorScheme.getDarkColor(), 0.8);
		Color back2 = isDark ? colorScheme.getExtraLightColor()
				: SubstanceColorUtilities.getInterpolatedColor(colorScheme
						.getMidColor(), colorScheme.getDarkColor(), 0.4);
		Color fore = isDark ? colorScheme.getDarkColor() : colorScheme
				.getUltraLightColor();

		int bumpDotDiameter = SubstanceSizeUtils
				.getDragBumpDiameter(SubstanceSizeUtils.getComponentFontSize(c));
		int bumpCellSize = (int) (1.5 * bumpDotDiameter + 1);
		if (isCrowded)
			bumpCellSize--;
		int bumpRows = Math.max(1, height / bumpCellSize - 1);
		int bumpColumns = Math.max(1, (width - 2) / bumpCellSize);
		if (maxNumberOfStripes > 0) {
			if (height > width)
				bumpColumns = Math.min(bumpColumns, maxNumberOfStripes);
			else
				bumpRows = Math.min(bumpRows, maxNumberOfStripes);
		}

		int bumpRowOffset = (height - bumpCellSize * bumpRows) / 2;
		int bumpColOffset = 1 + (width - bumpCellSize * bumpColumns) / 2;

		for (int col = 0; col < bumpColumns; col++) {
			int cx = bumpColOffset + col * bumpCellSize;
			boolean isEvenCol = (col % 2 == 0);
			int offsetY = isEvenCol ? 0 : bumpDotDiameter;
			for (int row = 0; row < bumpRows; row++) {
				int cy = offsetY + bumpRowOffset + row * bumpCellSize;
				graphics.setColor(fore);
				graphics.fillOval(cx + 1, cy + 1, bumpDotDiameter,
						bumpDotDiameter);
				// graphics.setColor(back1);
				graphics
						.setPaint(new GradientPaint(cx, cy, back1, cx
								+ bumpDotDiameter - 1,
								cy + bumpDotDiameter - 1, back2));
				graphics.fillOval(cx, cy, bumpDotDiameter, bumpDotDiameter);
			}
		}
		return result;
	}

	/**
	 * Returns resize grip image.
	 * 
	 * @param c
	 *            Component.
	 * @param theme
	 *            Theme.
	 * @param alwaysUseActive
	 *            Indicates whether the active theme should always be used.
	 * @param dimension
	 *            Resize grip width.
	 * @param isCrowded
	 *            Indicates whether the grips should be painted closely.
	 * @return Resize grip image.
	 */
	public static BufferedImage getResizeGripImage(Component c,
			SubstanceTheme theme, boolean alwaysUseActive, int dimension,
			boolean isCrowded) {
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(dimension,
				dimension);
		Graphics2D graphics = (Graphics2D) result.getGraphics();

		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		boolean isDark = SubstanceCoreUtilities.isThemeDark(theme
				.getActiveTheme());
		ColorScheme colorScheme = (alwaysUseActive/* || isDark */) ? theme
				.getActiveTheme().getColorScheme() : theme.getDefaultTheme()
				.getColorScheme();
		// SubstanceCoreUtilities
		// .getActiveScheme(null) : SubstanceCoreUtilities
		// .getDefaultScheme(null);
		Color back1 = isDark ? colorScheme.getLightColor()
				: SubstanceColorUtilities.getInterpolatedColor(colorScheme
						.getLightColor(), colorScheme.getDarkColor(), 0.8);
		Color back2 = isDark ? colorScheme.getExtraLightColor()
				: SubstanceColorUtilities.getInterpolatedColor(colorScheme
						.getMidColor(), colorScheme.getDarkColor(), 0.4);
		Color fore = isDark ? colorScheme.getDarkColor() : colorScheme
				.getUltraLightColor();

		int bumpDotDiameter = SubstanceSizeUtils
				.getDragBumpDiameter(SubstanceSizeUtils.getComponentFontSize(c));
		int bumpCellSize = (int) (1.5 * bumpDotDiameter + 1);
		if (isCrowded)
			bumpCellSize--;
		int bumpLines = dimension / bumpCellSize;

		int bumpOffset = (dimension - bumpCellSize * bumpLines) / 2;

		for (int col = 0; col < bumpLines; col++) {
			int cx = bumpOffset + col * bumpCellSize;
			for (int row = (bumpLines - col - 1); row < bumpLines; row++) {
				int cy = bumpOffset + row * bumpCellSize;
				graphics.setColor(fore);
				graphics.fillOval(cx + 1, cy + 1, bumpDotDiameter,
						bumpDotDiameter);
				// graphics.setColor(back1);
				graphics
						.setPaint(new GradientPaint(cx, cy, back1, cx
								+ bumpDotDiameter - 1,
								cy + bumpDotDiameter - 1, back2));
				graphics.fillOval(cx, cy, bumpDotDiameter, bumpDotDiameter);
			}
		}
		return result;
	}

	/**
	 * Retrieves tree icon.
	 * 
	 * @param fontSize
	 *            Font size.
	 * @param tree
	 *            Tree.
	 * @param colorScheme
	 *            Color scheme.
	 * @param isDark
	 *            Indication whether the color scheme is drak.
	 * @param isCollapsed
	 *            Collapsed state.
	 * @return Tree icon.
	 */
	public static BufferedImage getTreeIcon(int fontSize, JTree tree,
			ColorScheme colorScheme, boolean isDark, boolean isCollapsed) {
		int dim = SubstanceSizeUtils.getTreeIconSize(fontSize);
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(dim + 2,
				dim);
		Graphics2D graphics = (Graphics2D) result.getGraphics();
		// int extraPadding = SubstanceSizeUtils.getExtraPadding(fontSize);
		// int extraPadding2 = 2 * extraPadding;

		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_OFF);

		Graphics2D g2 = (Graphics2D) graphics.create();
		g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_OFF);
		SimplisticGradientPainter gradPainter = new SimplisticSoftBorderReverseGradientPainter();
		FlatBorderPainter fbp = new FlatBorderPainter();

		int borderDelta = (int) Math.floor(SubstanceSizeUtils
				.getBorderStrokeWidth(SubstanceSizeUtils
						.getComponentFontSize(tree)) / 2.0);
		Shape contour = BaseButtonShaper.getBaseOutline(dim - 1, dim - 1,
				SubstanceSizeUtils.getClassicButtonCornerRadius(dim), null,
				borderDelta);

		g2.translate(1, 1);
		ColorScheme fillScheme = new ShiftColorScheme(colorScheme, colorScheme
				.getExtraLightColor(), 0.7);
		g2.drawImage(gradPainter.getContourBackground(dim - 1, dim - 1,
				contour, false, fillScheme, fillScheme, 0, false, false), 0, 0,
				null);
		ColorScheme borderScheme = new ShiftColorScheme(colorScheme,
				isDark ? colorScheme.getUltraLightColor() : colorScheme
						.getLightColor(), 0.95);
		// g2.setStroke(new BasicStroke(SubstanceSizeUtils
		// .getBorderStrokeWidth(fontSize), BasicStroke.CAP_BUTT,
		// BasicStroke.JOIN_ROUND));
		fbp.paintBorder(g2, tree, dim - 1, dim - 1, contour, null,
				borderScheme, borderScheme, 0, false);

		g2.dispose();

		// Polygon pol = new Polygon();
		// pol.addPoint(2, 1);
		// pol.addPoint(8 + extraPadding2, 1);
		// pol.addPoint(9 + extraPadding2, 2);
		// pol.addPoint(9 + extraPadding2, 8 + extraPadding2);
		// pol.addPoint(8 + extraPadding2, 9 + extraPadding2);
		// pol.addPoint(2, 9 + extraPadding2);
		// pol.addPoint(1, 8 + extraPadding2);
		// pol.addPoint(1, 2);

		// Color tlFillColor = isDark ? colorScheme.getMidColor().brighter()
		// : Color.white;
		// Color brFillColor = isDark ? colorScheme.getDarkColor().brighter()
		// : colorScheme.getLightColor();
		// graphics.setPaint(new GradientPaint(0, 0, tlFillColor,
		// 9 + extraPadding2, 9 + extraPadding2, brFillColor));
		// // graphics.fillPolygon(pol);
		// Color borderColor = isDark ? colorScheme.getUltraLightColor()
		// : colorScheme.getMidColor();
		Color signColor = isDark ? colorScheme.getUltraLightColor().brighter()
				.brighter() : colorScheme.getUltraDarkColor();
		// graphics.setColor(borderColor);
		// // graphics.drawPolygon(pol);

		// graphics.setStroke(new BasicStroke(SubstanceSizeUtils
		// .getBorderStrokeWidth(fontSize), BasicStroke.CAP_BUTT,
		// BasicStroke.JOIN_ROUND));
		graphics.setColor(signColor);
		int mid = dim / 2;
		int length = 5 * dim / 12;
		graphics.drawLine(mid - length / 2, dim / 2, mid + length / 2, dim / 2);
		if (isCollapsed) {
			graphics.drawLine(dim / 2, mid - length / 2, dim / 2, mid + length
					/ 2);
		}

		return result;
	}

	/**
	 * Retrieves a single crayon of the specified color and dimensions for the
	 * crayon panel in color chooser.
	 * 
	 * @param mainColor
	 *            Crayon main color.
	 * @param width
	 *            Crayon width.
	 * @param height
	 *            Crayon height.
	 * @return Crayon image.
	 */
	public static BufferedImage getSingleCrayon(Color mainColor, int width,
			int height) {
		BufferedImage image = SubstanceCoreUtilities.getBlankImage(width,
				height);

		int baseTop = (int) (0.2 * height);

		Graphics2D graphics = (Graphics2D) image.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		int r = mainColor.getRed();
		int g = mainColor.getGreen();
		int b = mainColor.getBlue();
		// light coefficient
		double lc = 0.8;
		int lr = (int) (r + (255 - r) * lc);
		int lg = (int) (g + (255 - g) * lc);
		int lb = (int) (b + (255 - b) * lc);
		// dark coefficient
		double dc = 0.05;
		int dr = (int) ((1.0 - dc) * r);
		int dg = (int) ((1.0 - dc) * g);
		int db = (int) ((1.0 - dc) * b);

		Color lightColor = new Color(lr, lg, lb);
		Color darkColor = new Color(dr, dg, db);

		Map<Integer, Color> fillColorsColor = new HashMap<Integer, Color>();
		fillColorsColor.put((int) (0.3 * width), darkColor);
		fillColorsColor.put((int) (0.5 * width), darkColor);
		fillColorsColor.put((int) (0.9 * width), lightColor);

		Image fillLineColor = SubstanceImageCreator.getOneLineGradient(width,
				lightColor, lightColor, fillColorsColor);

		for (int y = baseTop; y < height; y++) {
			graphics.drawImage(fillLineColor, 0, y, null);
		}

		int dbwr = lr;
		int dbwg = lg;
		int dbwb = lb;
		int lbwr = 128 + dr / 4;
		int lbwg = 128 + dg / 4;
		int lbwb = 128 + db / 4;
		Map<Integer, Color> fillColorsBW = new HashMap<Integer, Color>();
		fillColorsBW.put((int) (0.3 * width), new Color(dbwr, dbwg, dbwb));
		fillColorsBW.put((int) (0.5 * width), new Color(dbwr, dbwg, dbwb));
		fillColorsBW.put((int) (0.9 * width), new Color(lbwr, lbwg, lbwb));

		Image fillLineBW = SubstanceImageCreator.getOneLineGradient(width,
				new Color(lbwr, lbwg, lbwb), new Color(lbwr, lbwg, lbwb),
				fillColorsBW);

		int stripeTop = (int) (0.35 * height);
		int stripeHeight = (int) (0.04 * height);
		for (int y = stripeTop; y < (stripeTop + stripeHeight); y++) {
			graphics.drawImage(fillLineBW, 0, y, null);
		}
		graphics.setColor(new Color(lbwr, lbwg, lbwb));
		graphics.drawRect(0, stripeTop, width - 1, stripeHeight);

		// create cap path
		GeneralPath capPath = new GeneralPath();
		capPath.moveTo(0.5f * width - 3, 4);
		capPath.quadTo(0.5f * width, 0, 0.5f * width + 3, 4);
		capPath.lineTo(width - 3, baseTop);
		capPath.lineTo(2, baseTop);
		capPath.lineTo(0.5f * width - 3, 4);

		graphics.setClip(capPath);

		graphics.setPaint(new GradientPaint(0, baseTop / 2, lightColor,
				(int) (0.6 * width), baseTop, mainColor));
		graphics.fillRect(0, 0, width / 2, baseTop);
		graphics.setPaint(new GradientPaint(width, baseTop / 2, lightColor,
				(int) (0.4 * width), baseTop, mainColor));
		graphics.fillRect(width / 2, 0, width / 2, baseTop);

		graphics.setStroke(new BasicStroke((float) 1.3, BasicStroke.CAP_ROUND,
				BasicStroke.JOIN_ROUND));

		graphics.setClip(null);
		graphics
				.setColor(new Color(64 + dr / 2, 64 + dg / 2, 64 + db / 2, 200));
		graphics.drawRect(0, baseTop, width - 1, height - baseTop - 1);
		graphics.draw(capPath);

		graphics.dispose();

		return image;
	}

	/**
	 * Crayon colors.
	 */
	private final static int[] crayonColors = { 0x800000, // Cayenne
			0x808000, // Asparagus
			0x008000, // Clover
			0x008080, // Teal
			0x000080, // Midnight
			0x800080, // Plum
			0x7f7f7f, // Tin
			0x808080, // Nickel

			0x804000, // Mocha
			0x408000, // Fern
			0x008040, // Moss
			0x004080, // Ocean
			0x400080, // Eggplant
			0x800040, // Maroon
			0x666666, // Steel
			0x999999, // Aluminium

			0xff0000, // Maraschino
			0xffff00, // Lemon
			0x00ff00, // Spring
			0x00ffff, // Turquoise
			0x0000ff, // Blueberry
			0xff00ff, // Magenta
			0x4c4c4c, // Iron
			0xb3b3b3, // Magnesium

			0xff8000, // Tangerine
			0x80ff00, // Lime
			0x00ff80, // Sea Foam
			0x0080ff, // Aqua
			0x8000ff, // Grape
			0xff0080, // Strawberry
			0x333333, // Tungsten
			0xcccccc, // Silver

			0xff6666, // Salmon
			0xffff66, // Banana
			0x66ff66, // Flora
			0x66ffff, // Ice
			0x6666ff, // Orchid
			0xff66ff, // Bubblegum
			0x191919, // Lead
			0xe6e6e6, // Mercury

			0xffcc66, // Cantaloupe
			0xccff66, // Honeydew
			0x66ffcc, // Spindrift
			0x66ccff, // Sky
			0xcc66ff, // Lavender
			0xff6fcf, // Carnation
			0x000000, // Licorice
			0xffffff, // Snow
	};

	/**
	 * Retrieves crayon X offset.
	 * 
	 * @param i
	 *            Crayon index.
	 * @return Crayon X offset.
	 */
	private static int crayonX(int i) {
		return (i % 8) * 22 + 4 + ((i / 8) % 2) * 11;
	}

	/**
	 * Retrieves crayon Y offset.
	 * 
	 * @param i
	 *            Crayon index.
	 * @return Crayon Y offset.
	 */
	private static int crayonY(int i) {
		return (i / 8) * 20 + 23;
	}

	/**
	 * Retrieves crayons image for the crayon panel of color chooser.
	 * 
	 * @return Crayons image.
	 */
	public static Image getCrayonsImage() {
		int iw = 195;
		int ih = 208;
		Image image = SubstanceCoreUtilities.getBlankImage(iw, ih);
		Graphics2D graphics = (Graphics2D) image.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		graphics.setColor(new Color(240, 240, 240));
		graphics.fillRect(0, 0, iw, ih);

		for (int i = 0; i < SubstanceImageCreator.crayonColors.length; i++) {
			Color crayonColor = new Color(
					0xff000000 | SubstanceImageCreator.crayonColors[i]);
			Image crayonImage = SubstanceImageCreator.getSingleCrayon(
					crayonColor, 22, 120);
			graphics.drawImage(crayonImage, SubstanceImageCreator.crayonX(i),
					SubstanceImageCreator.crayonY(i), null);
		}

		graphics.setColor(new Color(190, 190, 190));
		graphics.drawRoundRect(0, 1, iw - 1, ih - 2, 4, 4);

		graphics.dispose();
		return image;
	}

	// /**
	// * Returns an error marker of specified dimension with an <code>X</code>
	// * inside. The resulting image is a circular icon with a diagonal cross
	// and
	// * border.
	// *
	// * @param dimension
	// * The diameter of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Error marker of specified dimension with an <code>X</code>
	// * inside
	// */
	// public static BufferedImage getErrorMarker(int dimension,
	// ColorScheme colorScheme) {
	// // new RGB image with transparency channel
	// BufferedImage image = new BufferedImage(dimension, dimension,
	// BufferedImage.TYPE_INT_ARGB);
	//
	// // create new graphics and set anti-aliasing hint
	// Graphics2D graphics = (Graphics2D) image.getGraphics().create();
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// // background fill
	// graphics.setColor(colorScheme.getMidColor());
	// graphics.fillOval(0, 0, dimension - 1, dimension - 1);
	//
	// // create spot in the upper-left corner using temporary graphics
	// // with clip set to the icon outline
	// GradientPaint spot = new GradientPaint(0, 0, new Color(255, 255, 255,
	// 200), dimension, dimension, new Color(255, 255, 255, 0));
	// Graphics2D tempGraphics = (Graphics2D) graphics.create();
	// tempGraphics.setPaint(spot);
	// tempGraphics.setClip(new Ellipse2D.Double(0, 0, dimension - 1,
	// dimension - 1));
	// tempGraphics.fillRect(0, 0, dimension, dimension);
	// tempGraphics.dispose();
	//
	// // draw outline of the icon
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.drawOval(0, 0, dimension - 1, dimension - 1);
	//
	// // draw the X sign using two paths
	// float dimOuter = (float) (0.5f * Math.pow(dimension, 0.75));
	// float dimInner = (float) (0.28f * Math.pow(dimension, 0.75));
	// float ds = 0.28f * (dimension - 1);
	// float de = 0.72f * (dimension - 1);
	//
	// // create the paths
	// GeneralPath gp1 = new GeneralPath();
	// gp1.moveTo(ds, ds);
	// gp1.lineTo(de, de);
	// GeneralPath gp2 = new GeneralPath();
	// gp2.moveTo(de, ds);
	// gp2.lineTo(ds, de);
	// graphics.setStroke(new BasicStroke(dimOuter, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.draw(gp1);
	// graphics.draw(gp2);
	// graphics.setStroke(new BasicStroke(dimInner, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraLightColor().brighter());
	// graphics.draw(gp1);
	// graphics.draw(gp2);
	//
	// // dispose
	// graphics.dispose();
	// return image;
	// }
	//
	// /**
	// * Returns an error marker of specified dimension with an <code>X</code>
	// * inside. The resulting image is a circular icon with a diagonal cross
	// and
	// * border.
	// *
	// * @param dimension
	// * The diameter of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Error marker of specified dimension with an <code>X</code>
	// * inside
	// */
	// public static Icon getErrorMarkerIcon(int dimension, ColorScheme
	// colorScheme) {
	// return new ImageIcon(SubstanceImageCreator.getErrorMarker(dimension,
	// colorScheme));
	// }
	//
	// /**
	// * Returns a warning marker of specified dimension with an <code>!</code>
	// * inside. The resulting image is a triangular icon with an exclamation
	// * mark.
	// *
	// * @param dimension
	// * The side of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Warning marker of specified dimension with an <code>!</code>
	// * inside.
	// */
	// public static BufferedImage getWarningMarker(int dimension,
	// ColorScheme colorScheme) {
	// // new RGB image with transparency channel
	// BufferedImage image = new BufferedImage(dimension, dimension,
	// BufferedImage.TYPE_INT_ARGB);
	//
	// // create new graphics and set anti-aliasing hint
	// Graphics2D graphics = (Graphics2D) image.getGraphics().create();
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// // create path for the outline
	// GeneralPath iconOutlinePath = new GeneralPath();
	// float d = dimension - 1;
	// float d32 = (float) (0.1 * d * Math.sqrt(3.0) / 2.0);
	// float height = (float) (1.1 * d * Math.sqrt(3.0) / 2.0);
	// iconOutlinePath.moveTo(0.45f * d, d32);
	// iconOutlinePath.quadTo(0.5f * d, 0, 0.55f * d, d32);
	// iconOutlinePath.lineTo(0.95f * d, height - d32);
	// iconOutlinePath.quadTo(d, height, 0.9f * d, height);
	// iconOutlinePath.lineTo(0.1f * d, height);
	// iconOutlinePath.quadTo(0, height, 0.05f * d, height - d32);
	// iconOutlinePath.lineTo(0.45f * d, d32);
	//
	// // fill inside
	// graphics.setColor(colorScheme.getMidColor());
	// graphics.fill(iconOutlinePath);
	//
	// // create spot in the upper-left corner using temporary graphics
	// // with clip set to the icon outline
	// GradientPaint spot = new GradientPaint(0, 0, new Color(255, 255, 255,
	// 200), dimension, dimension, new Color(255, 255, 255, 0));
	// Graphics2D tempGraphics = (Graphics2D) graphics.create();
	// tempGraphics.setPaint(spot);
	// tempGraphics.setClip(iconOutlinePath);
	// tempGraphics.fillRect(0, 0, dimension, dimension);
	// tempGraphics.dispose();
	//
	// // draw outline of the icon
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.draw(iconOutlinePath);
	//
	// // draw the ! sign
	// float dimOuter = (float) (0.5f * Math.pow(dimension, 0.75));
	// float dimInner = (float) (0.28f * Math.pow(dimension, 0.75));
	// GeneralPath markerPath = new GeneralPath();
	// markerPath.moveTo((float) 0.5 * d, (float) 0.3 * height);
	// markerPath.lineTo((float) 0.5 * d, (float) 0.6 * height);
	// markerPath.moveTo((float) 0.5 * d, (float) 0.85 * height);
	// markerPath.lineTo((float) 0.5 * d, (float) 0.85 * height);
	// graphics.setStroke(new BasicStroke(dimOuter, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.draw(markerPath);
	// graphics.setStroke(new BasicStroke(dimInner, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraLightColor().brighter());
	// graphics.draw(markerPath);
	//
	// // dispose
	// graphics.dispose();
	// return image;
	// }
	//
	// /**
	// * Returns an info marker of specified dimension with an <code>!</code>
	// * inside. The resulting image is a triangular icon with an exclamation
	// * mark.
	// *
	// * @param dimension
	// * The side of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Info marker of specified dimension with an <code>!</code>
	// * inside.
	// */
	// public static BufferedImage getInfoMarker(int dimension,
	// ColorScheme colorScheme) {
	// // new RGB image with transparency channel
	// BufferedImage image = new BufferedImage(dimension, dimension,
	// BufferedImage.TYPE_INT_ARGB);
	//
	// // create new graphics and set anti-aliasing hint
	// Graphics2D graphics = (Graphics2D) image.getGraphics().create();
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// // create path for the outline
	// GeneralPath iconOutlinePath = new GeneralPath();
	// float d = dimension - 1;
	// float d32 = (float) (0.1 * d * Math.sqrt(3.0) / 2.0);
	// float height = (float) (1.1 * d * Math.sqrt(3.0) / 2.0);
	// iconOutlinePath.moveTo(0.45f * d, d32);
	// iconOutlinePath.quadTo(0.5f * d, 0, 0.55f * d, d32);
	// iconOutlinePath.lineTo(0.95f * d, height - d32);
	// iconOutlinePath.quadTo(d, height, 0.9f * d, height);
	// iconOutlinePath.lineTo(0.1f * d, height);
	// iconOutlinePath.quadTo(0, height, 0.05f * d, height - d32);
	// iconOutlinePath.lineTo(0.45f * d, d32);
	//
	// // fill inside
	// graphics.setColor(colorScheme.getMidColor());
	// graphics.fill(iconOutlinePath);
	//
	// // create spot in the upper-left corner using temporary graphics
	// // with clip set to the icon outline
	// GradientPaint spot = new GradientPaint(0, 0, new Color(255, 255, 255,
	// 200), dimension, dimension, new Color(255, 255, 255, 0));
	// Graphics2D tempGraphics = (Graphics2D) graphics.create();
	// tempGraphics.setPaint(spot);
	// tempGraphics.setClip(iconOutlinePath);
	// tempGraphics.fillRect(0, 0, dimension, dimension);
	// tempGraphics.dispose();
	//
	// // draw outline of the icon
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.draw(iconOutlinePath);
	//
	// // draw the ! sign
	// float dimOuter = (float) (0.5f * Math.pow(dimension, 0.75));
	// float dimInner = (float) (0.28f * Math.pow(dimension, 0.75));
	// GeneralPath markerPath = new GeneralPath();
	// markerPath.moveTo((float) 0.5 * d, (float) 0.3 * height);
	// markerPath.lineTo((float) 0.5 * d, (float) 0.3 * height);
	// markerPath.moveTo((float) 0.5 * d, (float) 0.55 * height);
	// markerPath.lineTo((float) 0.5 * d, (float) 0.8 * height);
	// graphics.setStroke(new BasicStroke(dimOuter, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.draw(markerPath);
	// graphics.setStroke(new BasicStroke(dimInner, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraLightColor().brighter());
	// graphics.draw(markerPath);
	//
	// // dispose
	// graphics.dispose();
	// return image;
	// }
	//
	// /**
	// * Returns an info marker of specified dimension with an <code>!</code>
	// * inside. The resulting image is a triangular icon with an exclamation
	// * mark.
	// *
	// * @param dimension
	// * The diameter of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Info marker of specified dimension with an <code>!</code>
	// * inside
	// */
	// public static Icon getInfoMarkerIcon(int dimension, ColorScheme
	// colorScheme) {
	// return new ImageIcon(SubstanceImageCreator.getInfoMarker(dimension,
	// colorScheme));
	// }
	//
	// /**
	// * Returns a warning marker of specified dimension with an <code>!</code>
	// * inside. The resulting image is a triangular icon with an exclamation
	// * mark.
	// *
	// * @param dimension
	// * The side of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Warning marker of specified dimension with an <code>!</code>
	// * inside.
	// */
	// public static Icon getWarningMarkerIcon(int dimension,
	// ColorScheme colorScheme) {
	// return new ImageIcon(SubstanceImageCreator.getWarningMarker(dimension,
	// colorScheme));
	// }
	//
	// /**
	// * Returns an error marker of specified dimension with an <code>X</code>
	// * inside. The resulting image is a circular icon with a question mark.
	// *
	// * @param dimension
	// * The diameter of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Error marker of specified dimension with an <code>?</code>
	// * inside
	// */
	// public static BufferedImage getQuestionMarker(int dimension,
	// ColorScheme colorScheme) {
	// // new RGB image with transparency channel
	// BufferedImage image = new BufferedImage(dimension, dimension,
	// BufferedImage.TYPE_INT_ARGB);
	//
	// // create new graphics and set anti-aliasing hint
	// Graphics2D graphics = (Graphics2D) image.getGraphics().create();
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// // background fill
	// graphics.setColor(colorScheme.getMidColor());
	// graphics.fillOval(0, 0, dimension - 1, dimension - 1);
	//
	// // create spot in the upper-left corner using temporary graphics
	// // with clip set to the icon outline
	// GradientPaint spot = new GradientPaint(0, 0, new Color(255, 255, 255,
	// 200), dimension, dimension, new Color(255, 255, 255, 0));
	// Graphics2D tempGraphics = (Graphics2D) graphics.create();
	// tempGraphics.setPaint(spot);
	// tempGraphics.setClip(new Ellipse2D.Double(0, 0, dimension - 1,
	// dimension - 1));
	// tempGraphics.fillRect(0, 0, dimension, dimension);
	// tempGraphics.dispose();
	//
	// // draw outline of the icon
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.drawOval(0, 0, dimension - 1, dimension - 1);
	//
	// // draw the ? sign
	// float d = dimension - 1;
	// float dimOuter = (float) (0.5f * Math.pow(dimension, 0.75));
	// float dimInner = (float) (0.28f * Math.pow(dimension, 0.75));
	//
	// // create the paths
	// GeneralPath markerPath = new GeneralPath();
	// markerPath.moveTo((float) 0.3 * d, (float) 0.32 * d);
	// markerPath.quadTo((float) 0.3 * d, (float) 0.18 * d, (float) 0.5 * d,
	// (float) 0.18 * d);
	// markerPath.quadTo((float) 0.7 * d, (float) 0.18 * d, (float) 0.7 * d,
	// (float) 0.32 * d);
	// markerPath.quadTo((float) 0.7 * d, (float) 0.45 * d, (float) 0.6 * d,
	// (float) 0.45 * d);
	// markerPath.quadTo((float) 0.5 * d, (float) 0.45 * d, (float) 0.5 * d,
	// (float) 0.6 * d);
	// markerPath.moveTo((float) 0.5 * d, (float) 0.85 * d);
	// markerPath.lineTo((float) 0.5 * d, (float) 0.85 * d);
	// graphics.setStroke(new BasicStroke(dimOuter, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraDarkColor());
	// graphics.draw(markerPath);
	// graphics.setStroke(new BasicStroke(dimInner, BasicStroke.CAP_ROUND,
	// BasicStroke.JOIN_ROUND));
	// graphics.setColor(colorScheme.getUltraLightColor().brighter());
	// graphics.draw(markerPath);
	//
	// // dispose
	// graphics.dispose();
	// return image;
	// }
	//
	// /**
	// * Returns an error marker of specified dimension with an <code>X</code>
	// * inside. The resulting image is a circular icon with a question mark.
	// *
	// * @param dimension
	// * The diameter of the resulting marker.
	// * @param colorScheme
	// * Color scheme for the marker.
	// * @return Error marker of specified dimension with an <code>?</code>
	// * inside
	// */
	// public static Icon getQuestionMarkerIcon(int dimension,
	// ColorScheme colorScheme) {
	// return new ImageIcon(SubstanceImageCreator.getQuestionMarker(dimension,
	// colorScheme));
	// }
	//
	/**
	 * Returns small icon representation of the specified integer value. The
	 * remainder of dividing the integer by 16 is translated to four circles
	 * arranged in 2*2 grid.
	 * 
	 * @param value
	 *            Integer value to represent.
	 * @param theme
	 *            Icon theme.
	 * @return Icon representation of the specified integer value.
	 */
	public static Icon getHexaMarker(int value, SubstanceTheme theme) {
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(9, 9);

		value %= 16;
		Color offColor = null;
		Color onColor = null;
		if (theme == null) {
			return new ImageIcon(result);
		}
		ColorScheme colorScheme = theme.getColorScheme();
		boolean isDark = SubstanceCoreUtilities.isThemeDark(theme);
		offColor = isDark ? colorScheme.getMidColor() : colorScheme
				.getMidColor().darker();
		onColor = isDark ? SubstanceColorUtilities.getInterpolatedColor(
				colorScheme.getUltraLightColor(), Color.white, 0.2)
				: colorScheme.getUltraDarkColor().darker();

		boolean bit1 = ((value & 0x1) != 0);
		boolean bit2 = ((value & 0x2) != 0);
		boolean bit3 = ((value & 0x4) != 0);
		boolean bit4 = ((value & 0x8) != 0);

		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		graphics.setColor(bit1 ? onColor : offColor);
		graphics.fillOval(5, 5, 4, 4);
		graphics.setColor(bit2 ? onColor : offColor);
		graphics.fillOval(5, 0, 4, 4);
		graphics.setColor(bit3 ? onColor : offColor);
		graphics.fillOval(0, 5, 4, 4);
		graphics.setColor(bit4 ? onColor : offColor);
		graphics.fillOval(0, 0, 4, 4);

		graphics.dispose();
		return new ImageIcon(result);
	}

	/**
	 * Returns big icon representation of the specified integer value. The
	 * remainder of dividing the integer by 16 is translated to four circles
	 * arranged in 2*2 grid.
	 * 
	 * @param value
	 *            Integer value to represent.
	 * @param theme
	 *            Icon theme.
	 * @return Icon representation of the specified integer value.
	 */
	public static Image getBigHexaMarker(int value, SubstanceTheme theme) {
		int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(iSize,
				iSize);

		value %= 16;
		Color offColor = null;
		Color onColor = null;
		if (theme == null)
			return result;
		ColorScheme colorScheme = theme.getColorScheme();
		boolean isDark = SubstanceCoreUtilities.isThemeDark(theme);
		offColor = isDark ? colorScheme.getMidColor() : colorScheme
				.getMidColor().darker();
		onColor = isDark ? SubstanceColorUtilities.getInterpolatedColor(
				colorScheme.getUltraLightColor(), Color.white, 0.2)
				: colorScheme.getUltraDarkColor().darker();

		boolean bit1 = ((value & 0x1) != 0);
		boolean bit2 = ((value & 0x2) != 0);
		boolean bit3 = ((value & 0x4) != 0);
		boolean bit4 = ((value & 0x8) != 0);

		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		graphics.setColor(bit1 ? onColor : offColor);
		graphics.fillOval(9, 9, 6, 6);
		graphics.setColor(bit2 ? onColor : offColor);
		graphics.fillOval(9, 2, 6, 6);
		graphics.setColor(bit3 ? onColor : offColor);
		graphics.fillOval(2, 9, 6, 6);
		graphics.setColor(bit4 ? onColor : offColor);
		graphics.fillOval(2, 2, 6, 6);

		graphics.dispose();
		return result;
	}

	/**
	 * Returns search icon.
	 * 
	 * @param dimension
	 *            Icon dimension.
	 * @param theme
	 *            Icon theme.
	 * @param leftToRight
	 *            LTR indication of the resulting icon.
	 * @return Search icon.
	 */
	public static Icon getSearchIcon(int dimension, SubstanceTheme theme,
			boolean leftToRight) {
		// BufferedImage themeSearchIcon = SubstanceImageCreator.getThemeImage(
		// new ImageIcon(SubstanceImageCreator.class.getClassLoader()
		// .getResource("resource/system-search.png")), theme);
		// if (leftToRight) {
		// themeSearchIcon = getHorizontalFlip(themeSearchIcon);
		// }
		// return new ImageIcon(themeSearchIcon);
		//
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(dimension,
				dimension);

		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		Color color = SubstanceCoreUtilities.getMarkColor(theme, true);
		// theme.getColorScheme().getForegroundColor();
		graphics.setColor(color);

		graphics.setStroke(new BasicStroke(1.5f));
		if (leftToRight) {
			int xc = (int) (0.6 * dimension);
			int yc = (int) (0.45 * dimension);
			int r = (int) (0.3 * dimension);

			graphics.drawOval(xc - r, yc - r, 2 * r, 2 * r);

			graphics.setStroke(new BasicStroke(3.0f));
			GeneralPath handle = new GeneralPath();
			handle.moveTo((float) (xc - r / Math.sqrt(2.0)), (float) (yc + r
					/ Math.sqrt(2.0)));
			handle.lineTo(1.8f, dimension - 2.2f);
			graphics.draw(handle);
		} else {
			int xc = (int) (0.4 * dimension);
			int yc = (int) (0.45 * dimension);
			int r = (int) (0.3 * dimension);

			graphics.drawOval(xc - r, yc - r, 2 * r, 2 * r);

			graphics.setStroke(new BasicStroke(3.0f));
			GeneralPath handle = new GeneralPath();
			handle.moveTo((float) (xc + r / Math.sqrt(2.0)), (float) (yc + r
					/ Math.sqrt(2.0)));
			handle.lineTo(dimension - 2.5f, dimension - 2.2f);
			graphics.draw(handle);
		}

		graphics.dispose();
		return new ImageIcon(result);
		// return new ImageIcon(getThemeImage(new ImageIcon(result), theme
		// .getDefaultTheme()));
	}

	/**
	 * Returns an icon that matches the specified theme.
	 * 
	 * @param theme
	 *            Theme instance.
	 * @return Icon that matches the specified theme.
	 */
	public static Icon getThemeIcon(SubstanceTheme theme) {
		int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(iSize,
				iSize);
		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		boolean isMixed = (theme != null)
				&& (((theme.getKind() == ThemeKind.MIXED)));
		if (!isMixed) {
			Color color1 = (theme == null) ? Color.red : theme.getColorScheme()
					.getUltraDarkColor();
			Color color2 = (theme == null) ? Color.green : theme
					.getColorScheme().getMidColor();
			Color color3 = (theme == null) ? Color.blue : theme
					.getColorScheme().getExtraLightColor();

			graphics.setColor(color1);
			graphics.fillOval(5, 2, 6, 6);
			graphics.setColor(color1.darker());
			graphics.drawOval(5, 2, 6, 6);

			graphics.setColor(color2);
			graphics.fillOval(1, 9, 6, 6);
			graphics.setColor(color2.darker());
			graphics.drawOval(1, 9, 6, 6);

			graphics.setColor(color3);
			graphics.fillOval(9, 9, 6, 6);
			graphics.setColor(color3.darker());
			graphics.drawOval(9, 9, 6, 6);
		} else {
			SubstanceTheme[] origThemes = ((SubstanceMixTheme) theme)
					.getOriginalThemes();
			SubstanceTheme theme1 = origThemes[0];
			SubstanceTheme theme2 = origThemes[origThemes.length - 1];

			Color color1_1 = theme1.getColorScheme().getUltraDarkColor();
			Color color2_1 = theme1.getColorScheme().getMidColor();
			Color color3_1 = theme1.getColorScheme().getExtraLightColor();
			Color color1_2 = theme2.getColorScheme().getUltraDarkColor();
			Color color2_2 = theme2.getColorScheme().getMidColor();
			Color color3_2 = theme2.getColorScheme().getExtraLightColor();

			graphics.setClip(5, 2, 3, 7);
			graphics.setColor(color1_1);
			graphics.fillOval(5, 2, 6, 7);
			graphics.setColor(color1_1.darker());
			graphics.drawOval(5, 2, 6, 6);

			graphics.setClip(8, 2, 4, 7);
			graphics.setColor(color1_2);
			graphics.fillOval(5, 2, 6, 6);
			graphics.setColor(color1_2.darker());
			graphics.drawOval(5, 2, 6, 6);

			graphics.setClip(1, 9, 3, 7);
			graphics.setColor(color2_1);
			graphics.fillOval(1, 9, 6, 6);
			graphics.setColor(color2_1.darker());
			graphics.drawOval(1, 9, 6, 6);

			graphics.setClip(4, 9, 4, 7);
			graphics.setColor(color2_2);
			graphics.fillOval(1, 9, 6, 6);
			graphics.setColor(color2_2.darker());
			graphics.drawOval(1, 9, 6, 6);

			graphics.setClip(9, 9, 3, 7);
			graphics.setColor(color3_1);
			graphics.fillOval(9, 9, 6, 6);
			graphics.setColor(color3_1.darker());
			graphics.drawOval(9, 9, 6, 6);

			graphics.setClip(12, 9, 4, 7);
			graphics.setColor(color3_2);
			graphics.fillOval(9, 9, 6, 6);
			graphics.setColor(color3_2.darker());
			graphics.drawOval(9, 9, 6, 6);
		}

		graphics.dispose();
		return new ImageIcon(result);
	}

	/**
	 * Returns an icon that matches the specified watermark.
	 * 
	 * @param watermark
	 *            Watermark instance.
	 * @return Icon that matches the specified watermark.
	 */
	public static Icon getWatermarkIcon(SubstanceWatermark watermark) {
		int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(iSize,
				iSize);
		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_ON);

		watermark.previewWatermark(graphics, 0, 0, iSize, iSize);
		graphics.dispose();
		return new ImageIcon(result);
	}

	/**
	 * Returns a lock icon that matches the specified scheme.
	 * 
	 * @param scheme
	 *            Scheme instance.
	 * @return Lock icon that matches the specified scheme.
	 */
	public static Icon getSmallLockIcon(ColorScheme scheme) {
		int extraPadding = SubstanceSizeUtils
				.getExtraPadding(SubstanceSizeUtils.getControlFontSize());
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(
				6 + 2 * extraPadding, 8 + 2 * extraPadding);

		Color fore = scheme.getForegroundColor();
		Color fill = new Color(208, 208, 48);

		Graphics2D graphics = (Graphics2D) result.getGraphics().create();
		graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
				RenderingHints.VALUE_ANTIALIAS_OFF);

		graphics.setColor(fill);
		// lock fill
		graphics.fillRect(1, 2 + extraPadding, 4 + 2 * extraPadding,
				4 + 2 * extraPadding);
		graphics.setColor(fore);
		// lock left side
		graphics.drawLine(0, 3 + extraPadding, 0, 7 + 2 * extraPadding);
		// lock right side
		graphics.drawLine(5 + 2 * extraPadding, 3 + extraPadding,
				5 + 2 * extraPadding, 7 + 2 * extraPadding);
		// lock bottom side
		graphics.drawLine(0, 7 + 2 * extraPadding, 5 + 2 * extraPadding,
				7 + 2 * extraPadding);
		// lock top side
		graphics.drawLine(1, 2 + extraPadding, 4 + 2 * extraPadding,
				2 + extraPadding);
		// lock left handle
		graphics.drawLine(1 + extraPadding, 1, 1 + extraPadding,
				2 + extraPadding);
		// lock right handle
		graphics.drawLine(4 + extraPadding, 1, 4 + extraPadding,
				2 + extraPadding);
		// lock top handle
		graphics.drawLine(2 + extraPadding, 0, 3 + extraPadding, 0);
		// lock keyhole
		graphics.fillRect(2 + extraPadding, 4 + extraPadding, 2,
				2 + extraPadding);

		graphics.dispose();
		return new ImageIcon(result);
	}

	/**
	 * Returns the negative of the specified image.
	 * 
	 * @param bi
	 *            Image.
	 * @return The negative of the specified image.
	 */
	public static BufferedImage getNegated(BufferedImage bi) {
		int w = bi.getWidth();
		int h = bi.getHeight();
		BufferedImage result = SubstanceCoreUtilities.getBlankImage(w, h);
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
				result.setRGB(i, j, SubstanceColorUtilities.getNegativeColor(bi
						.getRGB(i, j)));
			}
		}
		return result;
	}

	// /**
	// * Returns <code>pin down</code> icon.
	// *
	// * @param theme
	// * Theme for the icon.
	// * @return <code>Pin down</code> icon.
	// */
	// public static Icon getPinDownIcon(SubstanceTheme theme) {
	// int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
	// BufferedImage image = SubstanceCoreUtilities
	// .getBlankImage(iSize, iSize);
	// Graphics2D graphics = (Graphics2D) image.getGraphics();
	// int start = (iSize / 4);
	// int end = (3 * iSize / 4) - 1;
	// int center = (start + end) / 2;
	// graphics.setColor(theme.getColorScheme().getForegroundColor());
	// graphics.drawRect(start, start, end - start, end - start);
	// graphics.fillRect(center, center, 2, 2);
	// return new ImageIcon(SubstanceImageCreator.overlayEcho(image, 1, 1));
	// }

	// /**
	// * Returns <code>pin down</code> icon.
	// *
	// * @param theme
	// * Theme for the icon.
	// * @return <code>Pin down</code> icon.
	// */
	// public static Icon getPinUpIcon(SubstanceTheme theme) {
	// int iSize = SubstanceSizeUtils.getTitlePaneIconSize();
	// BufferedImage image = SubstanceCoreUtilities
	// .getBlankImage(iSize, iSize);
	// Graphics2D graphics = (Graphics2D) image.getGraphics();
	// int start = (iSize / 4) - 1;
	// int end = (3 * iSize / 4);
	// int center = (start + end) / 2;
	// graphics.setColor(theme.getColorScheme().getForegroundColor());
	//
	// graphics.drawLine(start, start, center - 1, start);
	// graphics.drawLine(center + 2, start, end, start);
	// graphics.drawLine(end, start, end, center - 1);
	// graphics.drawLine(end, center + 2, end, end);
	// graphics.drawLine(end, end, center + 2, end);
	// graphics.drawLine(center - 1, end, start, end);
	// graphics.drawLine(start, end, start, center + 2);
	// graphics.drawLine(start, center - 1, start, start);
	// return new ImageIcon(SubstanceImageCreator.overlayEcho(image, 1, 1));
	// }

	// /**
	// * Retrieves computer icon.
	// *
	// * @return Computer icon.
	// */
	// public static BufferedImage getComputerIcon() {
	// BufferedImage result = SubstanceCoreUtilities.getBlankImage(18, 18);
	// Graphics2D graphics = (Graphics2D) result.getGraphics();
	//
	// ColorScheme colorScheme = SubstanceCoreUtilities.getActiveScheme(null);
	//
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// // tower
	// graphics.setPaint(new GradientPaint(0, 0, colorScheme
	// .getUltraLightColor(), 17, 17, colorScheme.getMidColor()));
	// graphics.fillRect(1, 4, 4, 11);
	// graphics.setPaint(new GradientPaint(0, 0, colorScheme.getDarkColor(),
	// 17, 17, colorScheme.getUltraDarkColor()));
	// graphics.drawRect(1, 4, 4, 11);
	// graphics.drawLine(2, 6, 4, 6);
	// graphics.drawLine(2, 8, 4, 8);
	// graphics.drawLine(3, 12, 3, 12);
	//
	// // screen
	// graphics.setPaint(new GradientPaint(0, 0, colorScheme
	// .getUltraLightColor(), 17, 17, colorScheme.getMidColor()));
	// graphics.fillRoundRect(8, 4, 9, 8, 2, 2);
	// graphics.setPaint(new GradientPaint(0, 0, colorScheme.getDarkColor(),
	// 17, 17, colorScheme.getUltraDarkColor()));
	// graphics.drawRoundRect(7, 4, 9, 8, 2, 2);
	// graphics.drawRoundRect(9, 6, 5, 4, 2, 2);
	// graphics.fillRect(11, 12, 2, 3);
	// graphics.drawLine(9, 15, 14, 15);
	//
	// float[] BLUR = { 0.03f, 0.03f, 0.03f, 0.03f, 0.86f, 0.03f, 0.03f,
	// 0.03f, 0.03f };
	// ConvolveOp vBlurOp = new ConvolveOp(new Kernel(3, 3, BLUR));
	// BufferedImage blurred = vBlurOp.filter(result, null);
	// return blurred;
	// }
	//
	// /**
	// * Retrieves disk icon.
	// *
	// * @param isDark
	// * Indication whether the color scheme is drak.
	// * @return Disk icon.
	// */
	// public static BufferedImage getDiskIcon(boolean isDark) {
	// BufferedImage result = SubstanceCoreUtilities.getBlankImage(18, 18);
	// Graphics2D graphics = (Graphics2D) result.getGraphics();
	//
	// ColorScheme colorScheme = SubstanceCoreUtilities.getActiveScheme(null);
	//
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// Color gradColor1 = isDark ? colorScheme.getUltraLightColor().brighter()
	// : colorScheme.getUltraLightColor();
	// Color gradColor2 = isDark ? colorScheme.getUltraLightColor()
	// : colorScheme.getMidColor();
	// graphics.setPaint(new GradientPaint(0, 0, gradColor1, 17, 17,
	// gradColor2));
	// graphics.fillRect(3, 4, 10, 10);
	// graphics.fillArc(3, 2, 10, 4, 0, 180);
	// graphics.fillArc(3, 12, 10, 4, 180, 180);
	//
	// Color gradColor3 = isDark ? colorScheme.getUltraLightColor()
	// : colorScheme.getDarkColor();
	// Color gradColor4 = isDark ? colorScheme.getLightColor() : colorScheme
	// .getUltraDarkColor();
	// graphics.setPaint(new GradientPaint(0, 0, gradColor3, 17, 17,
	// gradColor4));
	//
	// graphics.drawOval(3, 2, 10, 4);
	// graphics.drawArc(3, 12, 10, 4, 180, 180);
	// graphics.drawArc(3, 9, 10, 4, 180, 180);
	// graphics.drawArc(3, 6, 10, 4, 180, 180);
	// graphics.drawLine(3, 4, 3, 14);
	// graphics.drawLine(13, 4, 13, 14);
	//
	// return result;
	// }
	//
	// /**
	// * Retrieves floppy icon.
	// *
	// * @param isDark
	// * Indication whether the color scheme is drak.
	// * @return Floppy icon.
	// */
	// public static BufferedImage getFloppyIcon(boolean isDark) {
	// BufferedImage result = SubstanceCoreUtilities.getBlankImage(18, 18);
	// Graphics2D graphics = (Graphics2D) result.getGraphics();
	//
	// ColorScheme colorScheme = SubstanceCoreUtilities.getActiveScheme(null);
	//
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_ON);
	//
	// Polygon outline = new Polygon();
	// outline.addPoint(3, 3);
	// outline.addPoint(13, 3);
	// outline.addPoint(14, 4);
	// outline.addPoint(14, 15);
	// outline.addPoint(2, 15);
	// outline.addPoint(2, 4);
	//
	// Color gradColor1 = isDark ? colorScheme.getUltraLightColor().brighter()
	// : colorScheme.getUltraLightColor();
	// Color gradColor2 = isDark ? colorScheme.getUltraLightColor()
	// : colorScheme.getMidColor();
	// graphics.setPaint(new GradientPaint(0, 0, gradColor1, 17, 17,
	// gradColor2));
	// graphics.fillPolygon(outline);
	//
	// Color gradColor3 = isDark ? colorScheme.getUltraLightColor()
	// : colorScheme.getDarkColor();
	// Color gradColor4 = isDark ? colorScheme.getLightColor() : colorScheme
	// .getUltraDarkColor();
	// graphics.setPaint(new GradientPaint(0, 0, gradColor3, 17, 17,
	// gradColor4));
	//
	// graphics.drawPolygon(outline);
	//
	// graphics.drawRect(5, 3, 6, 3);
	// graphics.fillRect(9, 3, 3, 3);
	//
	// graphics.drawRect(4, 9, 8, 6);
	// graphics.drawLine(6, 11, 10, 11);
	// graphics.drawLine(6, 13, 10, 13);
	//
	// return result;
	// }
	//
	// /**
	// * Retrieves list view icon.
	// *
	// * @param isDark
	// * Indication whether the color scheme is drak.
	// * @return List view icon.
	// */
	// public static BufferedImage getListViewIcon(boolean isDark) {
	// BufferedImage result = SubstanceCoreUtilities.getBlankImage(16, 16);
	// Graphics2D graphics = (Graphics2D) result.getGraphics();
	//
	// ColorScheme colorScheme = SubstanceCoreUtilities.getActiveScheme(null);
	//
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_OFF);
	//
	// Color color = colorScheme.getForegroundColor();
	// graphics.setColor(color);
	// graphics.drawRect(1, 1, 4, 4);
	// graphics.drawLine(7, 4, 7, 4);
	// graphics.drawRect(9, 1, 4, 4);
	// graphics.drawLine(15, 4, 15, 4);
	// graphics.drawRect(1, 9, 4, 4);
	// graphics.drawLine(7, 11, 7, 11);
	// graphics.drawRect(9, 9, 4, 4);
	// graphics.drawLine(15, 11, 15, 11);
	//
	// return result;
	// }
	//
	// /**
	// * Retrieves details view icon.
	// *
	// * @param isDark
	// * Indication whether the color scheme is drak.
	// * @return Details view icon.
	// */
	// public static BufferedImage getDetailsViewIcon(boolean isDark) {
	// BufferedImage result = SubstanceCoreUtilities.getBlankImage(16, 16);
	// Graphics2D graphics = (Graphics2D) result.getGraphics();
	//
	// ColorScheme colorScheme = SubstanceCoreUtilities.getActiveScheme(null);
	//
	// graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
	// RenderingHints.VALUE_ANTIALIAS_OFF);
	//
	// Color color = colorScheme.getForegroundColor();
	// graphics.setColor(color);
	// graphics.drawRect(1, 1, 4, 4);
	// graphics.drawLine(7, 4, 14, 4);
	// graphics.drawRect(1, 9, 4, 4);
	// graphics.drawLine(7, 11, 14, 11);
	//
	// return result;
	// }

	/**
	 * Creates a new version of the specified icon that is rendered in the
	 * colors of the specified theme.
	 * 
	 * @param comp
	 *            Component.
	 * @param original
	 *            The original icon.
	 * @param theme
	 *            Theme.
	 * @param toSaturate
	 *            Indicates whether the resulting image should be saturated.
	 * @return Theme-based version of the original icon.
	 */
	public static BufferedImage getThemeImage(Component comp, Icon original,
			SubstanceTheme theme, boolean toSaturate) {
		int w = original.getIconWidth();
		int h = original.getIconHeight();
		BufferedImage origImage = SubstanceCoreUtilities.getBlankImage(w, h);
		original.paintIcon(comp, origImage.getGraphics(), 0, 0);

		return getThemeImage(origImage, theme, toSaturate);
	}

	/**
	 * Creates a new version of the specified image that is rendered in the
	 * colors of the specified theme.
	 * 
	 * @param original
	 *            The original image.
	 * @param theme
	 *            Theme.
	 * @param toSaturate
	 *            Indicates whether the resulting image should be saturated.
	 * @return Theme-based version of the original icon.
	 */
	public static BufferedImage getThemeImage(BufferedImage original,
			SubstanceTheme theme, boolean toSaturate) {
		int w = original.getWidth();
		int h = original.getHeight();
		ColorScheme scheme = theme.getActiveTheme().getColorScheme();
		if (toSaturate) {
			scheme = new ShiftColorScheme(
					new SaturatedColorScheme(scheme, 0.2), scheme
							.getDarkColor(), 0.2);
		}

		// collect the brightness factors of the color scheme
		Map<Integer, Color> schemeColorMapping = new TreeMap<Integer, Color>();
		schemeColorMapping.put(SubstanceColorUtilities
				.getColorBrightness(scheme.getUltraLightColor().getRGB()),
				scheme.getUltraLightColor());
		schemeColorMapping.put(SubstanceColorUtilities
				.getColorBrightness(scheme.getExtraLightColor().getRGB()),
				scheme.getExtraLightColor());
		schemeColorMapping.put(SubstanceColorUtilities
				.getColorBrightness(scheme.getLightColor().getRGB()), scheme
				.getLightColor());
		schemeColorMapping.put(SubstanceColorUtilities
				.getColorBrightness(scheme.getMidColor().getRGB()), scheme
				.getMidColor());
		schemeColorMapping.put(SubstanceColorUtilities
				.getColorBrightness(scheme.getDarkColor().getRGB()), scheme
				.getDarkColor());
		schemeColorMapping.put(SubstanceColorUtilities
				.getColorBrightness(scheme.getUltraDarkColor().getRGB()),
				scheme.getUltraDarkColor());

		List<Integer> schemeBrightness = new ArrayList<Integer>();
		schemeBrightness.addAll(schemeColorMapping.keySet());
		Collections.sort(schemeBrightness);

		int lowestSchemeBrightness = schemeBrightness.get(0);
		int highestSchemeBrightness = schemeBrightness.get(schemeBrightness
				.size() - 1);

		Map<Integer, Color> stretchedColorMapping = new TreeMap<Integer, Color>();
		for (Map.Entry<Integer, Color> entry : schemeColorMapping.entrySet()) {
			int brightness = entry.getKey();
			int stretched = 255 - 255 * (highestSchemeBrightness - brightness)
					/ (highestSchemeBrightness - lowestSchemeBrightness);
			stretchedColorMapping.put(stretched, entry.getValue());
		}
		schemeBrightness = new ArrayList<Integer>();
		schemeBrightness.addAll(stretchedColorMapping.keySet());
		Collections.sort(schemeBrightness);

		boolean isDark = SubstanceCoreUtilities.isThemeDark(theme);

		BufferedImage result = SubstanceCoreUtilities.getBlankImage(w, h);
		for (int i = 0; i < w; i++) {
			for (int j = 0; j < h; j++) {
				int color = original.getRGB(i, j);
				int transp = (color >>> 24) & 0xFF;
				// int r = (color >>> 16) & 0xFF;
				// int g = (color >>> 8) & 0xFF;
				// int b = (color >>> 0) & 0xFF;

				int brightness = SubstanceColorUtilities
						.getColorBrightness(color);
				Color pixelColor = null;
				if (schemeBrightness.contains(brightness)) {
					pixelColor = stretchedColorMapping.get(brightness);
				} else {
					int currIndex = 0;
					while (true) {
						int currStopValue = schemeBrightness.get(currIndex);
						int nextStopValue = schemeBrightness.get(currIndex + 1);
						if ((brightness > currStopValue)
								&& (brightness < nextStopValue)) {
							// interpolate
							Color currStopColor = stretchedColorMapping
									.get(currStopValue);
							Color nextStopColor = stretchedColorMapping
									.get(nextStopValue);
							pixelColor = SubstanceColorUtilities
									.getInterpolatedColor(
											currStopColor,
											nextStopColor,
											1.0
													- (double) (brightness - currStopValue)
													/ (double) (nextStopValue - currStopValue));
							break;
						}
						currIndex++;
					}
				}
				result.setRGB(i, j, SubstanceColorUtilities.getAlphaColor(
						pixelColor, transp).getRGB());

				// Color c = null;
				// int iLuminance = 255 - (int) (0.3 * r + 0.59 * g + 0.11 * b);
				// if (!isDark) {
				// if (iLuminance < 52) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getLightColor(),
				// scheme.getExtraLightColor(),
				// iLuminance / 51.0));
				// } else if (iLuminance < 103) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getMidColor(),
				// scheme.getLightColor(),
				// (iLuminance - 52) / 51.0));
				// } else if (iLuminance < 154) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getDarkColor(),
				// scheme.getMidColor(),
				// (iLuminance - 103) / 51.0));
				// } else if (iLuminance < 205) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getUltraDarkColor(),
				// scheme.getDarkColor(),
				// (iLuminance - 154) / 51.0));
				// } else
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(
				// scheme.getForegroundColor(), scheme
				// .getUltraDarkColor(),
				// (iLuminance - 205) / 51.0));
				// } else {
				// if (iLuminance < 176) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(
				// scheme.getUltraLightColor(), scheme
				// .getForegroundColor(),
				// iLuminance / 176.0));
				// } else if (iLuminance < 196) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(
				// scheme.getExtraLightColor(), scheme
				// .getUltraLightColor(),
				// (iLuminance - 176) / 20.0));
				// } else if (iLuminance < 216) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getLightColor(),
				// scheme.getExtraLightColor(),
				// (iLuminance - 196) / 20.0));
				// } else if (iLuminance < 236) {
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getMidColor(),
				// scheme.getLightColor(),
				// (iLuminance - 216) / 20.0));
				// } else
				// c = new Color(SubstanceColorUtilities
				// .getInterpolatedRGB(scheme.getDarkColor(),
				// scheme.getMidColor(),
				// (iLuminance - 236) / 20.0));
				// }
				// result.setRGB(i, j, SubstanceColorUtilities.getAlphaColor(c,
				// transp).getRGB());
			}
		}
		return result;
	}
	//
	// public static BufferedImage getHorizontalFlip(BufferedImage orig) {
	// int w = orig.getWidth();
	// int h = orig.getHeight();
	// BufferedImage result = SubstanceCoreUtilities.getBlankImage(w, h);
	// for (int i = 0; i < w; i++) {
	// for (int j = 0; j < h; j++) {
	// result.setRGB(w - i - 1, j, orig.getRGB(i, j));
	// }
	// }
	// return result;
	// }
}
