/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance;

import java.awt.Graphics;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.plaf.ComponentUI;
import javax.swing.plaf.basic.BasicGraphicsUtils;
import javax.swing.plaf.basic.BasicLabelUI;

import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * UI for labels in <b>Substance</b> look and feel.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceLabelUI extends BasicLabelUI {
	/**
	 * Background delegate.
	 */
	private SubstanceFillBackgroundDelegate bgDelegate;

	/**
	 * Name of the client property that points to the original icon.
	 */
	private final static String ORIG_ICON = "substancelaf.internal.label.origIcon";

	/**
	 * Name of the client property that marks the label during icon replacement.
	 */
	private final static String REPLACING_ICON = "substancelaf.internal.label.replacingIcon";

	/**
	 * Property change listener.
	 */
	protected PropertyChangeListener substancePropertyChangeListener;

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.ComponentUI#createUI(javax.swing.JComponent)
	 */
	public static ComponentUI createUI(JComponent b) {
		return new SubstanceLabelUI();
	}

	/**
	 * Creates a new UI delegate for labels.
	 */
	public SubstanceLabelUI() {
		super();
		bgDelegate = new SubstanceFillBackgroundDelegate();
	}

	/**
	 * Replaces the label icon with a themed version of that icon.
	 * 
	 * @param label
	 *            Label.
	 */
	protected void replaceThemedIcon(final JLabel label) {
		if (!SubstanceCoreUtilities.useThemedDefaultIcon(label))
			return;

		final Icon icon = label.getIcon();
		if ((icon != null)
				&& (!Boolean.TRUE.equals(label
						.getClientProperty(REPLACING_ICON)))) {
			SwingUtilities.invokeLater(new Runnable() {
				public void run() {
					label.putClientProperty(ORIG_ICON, label.getIcon());
					label.putClientProperty(REPLACING_ICON, Boolean.TRUE);
					label.setIcon(new ImageIcon(SubstanceImageCreator
							.getThemeImage(label, icon, SubstanceCoreUtilities
									.getDefaultTheme(label, true), false)));
					label.putClientProperty(REPLACING_ICON, null);
				}
			});
		}
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLabelUI#installDefaults(javax.swing.JLabel)
	 */
	@Override
	protected void installDefaults(JLabel c) {
		super.installDefaults(c);
		this.replaceThemedIcon(c);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLabelUI#uninstallDefaults(javax.swing.JLabel)
	 */
	@Override
	protected void uninstallDefaults(JLabel c) {
		Icon origIcon = (Icon) c.getClientProperty(ORIG_ICON);
		if (origIcon != null) {
			c.putClientProperty(REPLACING_ICON, Boolean.TRUE);
			c.setIcon(origIcon);
			c.putClientProperty(REPLACING_ICON, null);
		}

		super.uninstallDefaults(c);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLabelUI#installListeners(javax.swing.JLabel)
	 */
	@Override
	protected void installListeners(final JLabel c) {
		super.installListeners(c);

		this.substancePropertyChangeListener = new PropertyChangeListener() {
			public void propertyChange(PropertyChangeEvent evt) {
				if ("icon".equals(evt.getPropertyName())) {
					replaceThemedIcon(c);
				}

				if (SubstanceLookAndFeel.USE_THEMED_DEFAULT_ICONS.equals(evt
						.getPropertyName())) {
					replaceThemedIcon(c);
				}
			}
		};
		c.addPropertyChangeListener(this.substancePropertyChangeListener);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLabelUI#uninstallListeners(javax.swing.JLabel)
	 */
	@Override
	protected void uninstallListeners(JLabel c) {
		c.removePropertyChangeListener(this.substancePropertyChangeListener);
		this.substancePropertyChangeListener = null;

		super.uninstallListeners(c);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.plaf.basic.BasicLabelUI#paintDisabledText(javax.swing.JLabel,
	 *      java.awt.Graphics, java.lang.String, int, int)
	 */
	@Override
	protected void paintDisabledText(JLabel l, Graphics g, String s, int textX,
			int textY) {
		int accChar = l.getDisplayedMnemonicIndex();
		g.setColor(UIManager.getColor("Label.disabledText"));
		BasicGraphicsUtils.drawStringUnderlineCharAt(g, s, accChar, textX,
				textY);
	}

	@Override
	public void update(Graphics g, JComponent c) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel))
			return;
		// important to synchronize on the button as we are
		// about to fiddle with its opaqueness
		// synchronized (c) {
		// this.bgDelegate.update(g, c);
		// remove opaqueness
		// boolean isOpaque = c.isOpaque();
		// c.setOpaque(false);
		bgDelegate.updateIfOpaque(g, c);
		super.paint(g, c);
		// restore opaqueness
		// c.setOpaque(isOpaque);
		// }
	}

}
