/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.title;

import java.awt.*;

import javax.swing.*;

import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.SubstanceCoreUtilities;

/**
 * Implementation of {@link SubstanceHeaderPainter} that uses matte painting on
 * title panes.
 * 
 * @author Kirill Grouchnikov
 */
public class MatteHeaderPainter extends BaseHeaderPainter {
	/**
	 * The display name for the header painters of this class.
	 */
	public static final String DISPLAY_NAME = "Matte";

	/**
	 * The secondary painter. It is used for painting areas that do not have the
	 * painting of title pane (such as menu background, table headers etc).
	 */
	private SubstanceTitlePainter secondaryPainter;

	/**
	 * Indicates whether this painter is used to paint the toolbar background.
	 */
	private boolean isPaintingToolbars;

	/**
	 * Indicates whether this painter will cause drop shadows to appear on the
	 * top portion of the toolbars that are <b>not</b> painted by this painter.
	 */
	private boolean isPaintingToolbarDropShadows;

	/**
	 * Creates a new matte header painter that is used to paint title panes,
	 * menubars <b>and</b> toolbars.
	 */
	public MatteHeaderPainter() {
		this(true, true);
	}

	/**
	 * Creates a new matte header painter
	 * 
	 * @param isPaintingToolbars
	 *            If <code>true</code>, this painter will be used to paint
	 *            toolbars in addition to painting title panes and menubars.
	 * @param isPaintingGeneralDecorationAreas
	 *            If <code>true</code>, this painter will be used to paint
	 *            additional general decoration areas.
	 */
	public MatteHeaderPainter(boolean isPaintingToolbars,
			boolean isPaintingGeneralDecorationAreas) {
		super(isPaintingToolbars, isPaintingGeneralDecorationAreas);
		this.secondaryPainter = new Glass3DTitlePainter();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.painter.SubstanceGradientPainter#getDisplayName()
	 */
	public String getDisplayName() {
		return DISPLAY_NAME;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceTitlePainter#paintTitleBackground(java.awt.Graphics2D,
	 *      java.awt.Component, int, int, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme, float)
	 */
	public void paintTitleBackground(Graphics2D graphics, Component comp,
			int width, int height, int leftTitleX, int rightTitleX,
			SubstanceTheme theme, float borderAlpha) {
		ColorScheme scheme = theme.getColorScheme();
		Graphics2D temp = (Graphics2D) graphics.create();
		this.fill(temp, scheme, 0, 0, 0, width, height);
		temp.dispose();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceHeaderPainter#paintExtraBackground(java.awt.Graphics2D,
	 *      java.awt.Container, java.awt.Component, int, int,
	 *      org.jvnet.substance.theme.SubstanceTheme, boolean)
	 */
	public void paintExtraBackground(Graphics2D graphics, Container parent,
			Component comp, int width, int height, SubstanceTheme theme,
			boolean hasDarkBorder) {

		JRootPane rootPane = SwingUtilities.getRootPane(parent);
		// fix for bug 234 - Window doesn't have a root pane.
		int dx = 0;
		int dy = 0;
		JComponent titlePane = null;

		if (rootPane != null) {
			titlePane = SubstanceCoreUtilities.getTitlePane(rootPane);

			if (titlePane != null) {
				if (comp.isShowing() && titlePane.isShowing()) {
					dx += (comp.getLocationOnScreen().x - titlePane
							.getLocationOnScreen().x);
					dy += (comp.getLocationOnScreen().y - titlePane
							.getLocationOnScreen().y);
				} else {
					// have to traverse the hierarchy
					Component c = comp;
					dx = 0;
					dy = 0;
					while (c != rootPane) {
						dx += c.getX();
						dy += c.getY();
						c = c.getParent();
					}
					c = titlePane;
					while (c != rootPane) {
						dx -= c.getX();
						dy -= c.getY();
						c = c.getParent();
					}
				}
			}
		}

		// int pWidth = (titlePane == null) ? parent.getWidth() : titlePane
		// .getWidth();
		// int pHeight = parent.getHeight();
		//
		ColorScheme scheme = theme.getColorScheme();
		Graphics2D temp = (Graphics2D) graphics.create();
		this.fill(temp, scheme, dy, 0, 0, width, height);
		temp.dispose();
	}

	/**
	 * Fills the relevant part with the gradient fill.
	 * 
	 * @param graphics
	 *            Graphics.
	 * @param scheme
	 *            Color scheme to use.
	 * @param offsetY
	 *            Vertical offset.
	 * @param x
	 *            X coordinate of the fill area.
	 * @param y
	 *            Y coordinate of the fill area.
	 * @param width
	 *            Fill area width.
	 * @param height
	 *            Fill area height.
	 */
	protected void fill(Graphics2D graphics, ColorScheme scheme, int offsetY,
			int x, int y, int width, int height) {
		int bandHeight = 20;
		// 0 - 20 : ultralight -> extra light
		// 20 - 40 : extra light -> light
		// 40 - 60 : light -> medium
		// 60 - 80 : medium -> dark
		// 80 - 100 : dark -> ultra dark
		// 100 - : ultra dark fill

		int startY = y + offsetY;
		int endY = y + offsetY + height;

		int currStart = 0;
		int currEnd = bandHeight;

		if (currEnd >= startY) {
			graphics.setPaint(new GradientPaint(x, currStart - offsetY, scheme
					.getUltraLightColor(), x, currEnd - offsetY, scheme
					.getExtraLightColor()));
			graphics.fillRect(x, currStart - offsetY, width, bandHeight);
		}
		currStart += bandHeight;
		currEnd += bandHeight;

		if (currStart > endY)
			return;

		if (currEnd >= startY) {
			graphics.setPaint(new GradientPaint(x, currStart - offsetY, scheme
					.getExtraLightColor(), x, currEnd - offsetY, scheme
					.getLightColor()));
			graphics.fillRect(x, currStart - offsetY, width, bandHeight);
		}
		currStart += bandHeight;
		currEnd += bandHeight;

		if (currStart > endY)
			return;

		if (currEnd >= startY) {
			graphics.setPaint(new GradientPaint(x, currStart - offsetY, scheme
					.getLightColor(), x, currEnd - offsetY, scheme
					.getMidColor()));
			graphics.fillRect(x, currStart - offsetY, width, bandHeight);
		}
		currStart += bandHeight;
		currEnd += bandHeight;

		if (currStart > endY)
			return;

		if (currEnd >= startY) {
			graphics
					.setPaint(new GradientPaint(x, currStart - offsetY, scheme
							.getMidColor(), x, currEnd - offsetY, scheme
							.getDarkColor()));
			graphics.fillRect(x, currStart - offsetY, width, bandHeight);
		}
		currStart += bandHeight;
		currEnd += bandHeight;

		if (currStart > endY)
			return;

		if (currEnd >= startY) {
			graphics.setPaint(new GradientPaint(x, currStart - offsetY, scheme
					.getDarkColor(), x, currEnd - offsetY, scheme
					.getUltraDarkColor()));
			graphics.fillRect(x, currStart - offsetY, width, bandHeight);
		}
		currStart += bandHeight;
		currEnd += bandHeight;

		if (currStart > endY)
			return;

		graphics.setColor(scheme.getUltraDarkColor());
		graphics.fillRect(x, currStart - offsetY, width, endY - currStart
				+ offsetY);
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see org.jvnet.substance.title.SubstanceHeaderPainter#getSecondaryPainter()
	 */
	public SubstanceTitlePainter getSecondaryPainter() {
		return this.secondaryPainter;
	}
}
