/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.utils;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.awt.image.BufferedImage;
import java.util.Map;
import java.util.Set;

import javax.swing.*;
import javax.swing.JInternalFrame.JDesktopIcon;

import org.jvnet.lafwidget.animation.*;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.border.SubstanceBorderPainter;
import org.jvnet.substance.button.*;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.combo.SubstanceComboBoxButton;
import org.jvnet.substance.painter.*;
import org.jvnet.substance.theme.SubstanceTheme;
import org.jvnet.substance.utils.ComponentState.ColorSchemeKind;
import org.jvnet.substance.utils.SubstanceConstants.Side;

/**
 * Delegate class for painting backgrounds of buttons in <b>Substance </b> look
 * and feel. This class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public class ButtonBackgroundDelegate {
	/**
	 * Cache for background images. Each time
	 * {@link #getBackground(AbstractButton, SubstanceButtonShaper, SubstanceGradientPainter, int, int)}
	 * is called, it checks <code>this</code> map to see if it already
	 * contains such background. If so, the background from the map is returned.
	 */
	private static Map<String, BufferedImage> regularBackgrounds = new SoftHashMap<String, BufferedImage>();

	/**
	 * Contains information on a button background.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static class ButtonBackground {
		/**
		 * Indicates whether the button is painted in active visual state.
		 */
		public boolean isPaintedActive;

		/**
		 * The button background image.
		 */
		public BufferedImage backgroundImage;

		/**
		 * Button background info object.
		 * 
		 * @param isPaintedActive
		 *            Indicates whether the button is painted in active visual
		 *            state.
		 * @param backgroundImage
		 *            The button background image.
		 */
		public ButtonBackground(boolean isPaintedActive,
				BufferedImage backgroundImage) {
			this.isPaintedActive = isPaintedActive;
			this.backgroundImage = backgroundImage;
		}

	}

	/**
	 * Resets image maps (used when setting new theme).
	 * 
	 * @see SubstanceLookAndFeel#setCurrentTheme(String)
	 * @see SubstanceLookAndFeel#setCurrentTheme(SubstanceTheme)
	 */
	public static synchronized void reset() {
		regularBackgrounds.clear();
		PairwiseButtonBackgroundDelegate.reset();
	}

	/**
	 * Retrieves the background for the specified button.
	 * 
	 * @param button
	 *            Button.
	 * @param shaper
	 *            Button shaper.
	 * @param painter
	 *            Button gradient painter.
	 * @param borderPainter
	 *            Button border painter.
	 * @param width
	 *            Button width.
	 * @param height
	 *            Button height.
	 * @return Button background.
	 */
	public static synchronized ButtonBackground getBackground(
			AbstractButton button, SubstanceButtonShaper shaper,
			SubstanceGradientPainter painter,
			SubstanceBorderPainter borderPainter, int width, int height) {
		ComponentState state = ComponentState.getState(button.getModel(),
				button);
		ComponentState prevState = SubstanceCoreUtilities
				.getPrevComponentState(button);
		// if (prevState != state)
		// button.putClientProperty(PREV_COMPONENT_STATE, state);

		boolean isPaintedActive = (state.getColorSchemeKind() == ColorSchemeKind.CURRENT);
		// compute cycle count (for animation)
		float cyclePos = state.getCycleCount();
		boolean isPulsating = false;
		if (button instanceof JButton) {
			JButton jb = (JButton) button;
			if (PulseTracker.isPulsating(jb)
					&& (state != ComponentState.PRESSED_SELECTED)
					&& (state != ComponentState.PRESSED_UNSELECTED)) {
				isPulsating = true;
				cyclePos = (int) (PulseTracker.getCycles(jb) % 20);
				if (cyclePos > 10) {
					cyclePos = 19 - cyclePos;
				}
				isPaintedActive = true;
			}
		}

		// compute color scheme
		SubstanceTheme theme = SubstanceCoreUtilities.getTheme(button, state,
				true, true);
		ColorScheme colorScheme = theme.getColorScheme();
		ColorScheme colorScheme2 = colorScheme;
		ColorScheme borderScheme = theme.getBorderTheme().getColorScheme();
		ColorScheme borderScheme2 = borderScheme;

		// see if need to use attention-drawing animation
		boolean isWindowModified = false;
		if (SubstanceCoreUtilities.isTitleCloseButton(button)) {
			// check if have windowModified property
			Component comp = button;
			while (comp != null) {
				if (comp instanceof JInternalFrame) {
					JInternalFrame jif = (JInternalFrame) comp;
					isWindowModified = Boolean.TRUE
							.equals(jif
									.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED));
					break;
				}
				if (comp instanceof JRootPane) {
					JRootPane jrp = (JRootPane) comp;
					isWindowModified = Boolean.TRUE
							.equals(jrp
									.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED));
					break;
				}
				if (comp instanceof JDesktopIcon) {
					JDesktopIcon jdi = (JDesktopIcon) comp;
					JInternalFrame jif = jdi.getInternalFrame();
					isWindowModified = Boolean.TRUE
							.equals(jif
									.getClientProperty(SubstanceLookAndFeel.WINDOW_MODIFIED));
					break;
				}
				comp = comp.getParent();
			}
			if (isWindowModified) {
				colorScheme2 = SubstanceTheme.YELLOW;
				colorScheme = SubstanceTheme.ORANGE;
				isPaintedActive = true;
			}
		}

		// see if need to use fade animation. Important - don't do it
		// on pulsating buttons (such as default or close buttons
		// of modified frames).
		if (!isWindowModified && !isPulsating) {
			// FadeTracker fadeTracker = FadeTracker.getInstance();

			FadeState fadeState = SubstanceFadeUtilities.getFadeState(button,
					FadeKind.ROLLOVER, FadeKind.SELECTION, FadeKind.PRESS);
			if (fadeState != null) {
				SubstanceTheme prevTheme = SubstanceCoreUtilities.getTheme(
						button, prevState, true, true);
				colorScheme = theme.getColorScheme();
				colorScheme2 = prevTheme.getColorScheme();
				borderScheme = theme.getBorderTheme().getColorScheme();
				borderScheme2 = prevTheme.getBorderTheme().getColorScheme();
				cyclePos = fadeState.getFadePosition();
				if (fadeState.isFadingIn())
					cyclePos = 10 - cyclePos;
				isPaintedActive = true;
			}
		}

		// compute a straight side
		Set<SubstanceConstants.Side> straightSides = SubstanceCoreUtilities
				.getSides(button, SubstanceLookAndFeel.BUTTON_SIDE_PROPERTY);
		String straightKey = "";
		for (Side sSide : straightSides) {
			straightKey += sSide.name() + "-";
		}

		boolean isRoundButton = StandardButtonShaper.isRoundButton(button);
		float radius = 0.0f;
		if (shaper instanceof RectangularButtonShaper) {
			radius = ((RectangularButtonShaper) shaper).getCornerRadius(button,
					null);
		}

		Set<Side> openSides = SubstanceCoreUtilities.getSides(button,
				SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY);
		String openKey = "";
		for (Side oSide : openSides) {
			openKey += oSide.name() + "-";
		}
		boolean isContentAreaFilled = button.isContentAreaFilled();
		boolean isBorderPainted = button.isBorderPainted();
		String key = width + ":" + height + ":" + prevState.name() + ":"
				+ state.name() + ":" + cyclePos + ":"
				+ SubstanceCoreUtilities.getSchemeId(colorScheme) + ":"
				+ SubstanceCoreUtilities.getSchemeId(colorScheme2) + ":"
				+ SubstanceCoreUtilities.getSchemeId(borderScheme) + ":"
				+ SubstanceCoreUtilities.getSchemeId(borderScheme2) + ":"
				+ shaper.getDisplayName() + ":" + painter.getDisplayName()
				+ ":" + borderPainter.getDisplayName() + ":" + straightKey
				+ ":" + openKey + ":" + button.getClass().getName() + ":"
				+ isRoundButton + ":" + radius + ":" + isContentAreaFilled
				+ ":" + isBorderPainted + ":" + button.getFont().getSize();

		if (regularBackgrounds.get(key) == null) {
			BufferedImage newBackground;

			int openDelta = (int) (Math.ceil(3.0 * SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(button))));
			int deltaLeft = openSides.contains(Side.LEFT) ? openDelta : 0;
			int deltaRight = openSides.contains(Side.RIGHT) ? openDelta : 0;
			int deltaTop = openSides.contains(Side.TOP) ? openDelta : 0;
			int deltaBottom = openSides.contains(Side.BOTTOM) ? openDelta : 0;

			// System.err.println(key);
			int borderDelta = (int) Math.floor(SubstanceSizeUtils
					.getBorderStrokeWidth(SubstanceSizeUtils
							.getComponentFontSize(button)) / 2.0);
			GeneralPath contour = shaper.getButtonOutline(button, new Insets(
					borderDelta, borderDelta, borderDelta, borderDelta), width
					+ deltaLeft + deltaRight, height + deltaTop + deltaBottom);

			newBackground = SubstanceCoreUtilities.getBlankImage(width, height);
			Graphics2D finalGraphics = (Graphics2D) newBackground.getGraphics();
			finalGraphics.translate(-deltaLeft, -deltaTop);
			if (isContentAreaFilled) {
				BufferedImage contourBackground = painter.getContourBackground(
						width + deltaLeft + deltaRight, height + deltaTop
								+ deltaBottom, contour, false, colorScheme,
						colorScheme2, cyclePos, true,
						colorScheme != colorScheme2);
				finalGraphics.drawImage(contourBackground, 0, 0, null);
			}

			if (isBorderPainted) {
				int borderThickness = (int) SubstanceSizeUtils
						.getBorderStrokeWidth(SubstanceSizeUtils
								.getComponentFontSize(button));
				GeneralPath contourInner = shaper.getButtonOutline(button,
						new Insets(borderDelta + borderThickness, borderDelta
								+ borderThickness, borderDelta
								+ borderThickness, borderDelta
								+ borderThickness), width + deltaLeft
								+ deltaRight, height + deltaTop + deltaBottom);
				borderPainter.paintBorder(finalGraphics, button, width
						+ deltaLeft + deltaRight, height + deltaTop
						+ deltaBottom, contour, contourInner, borderScheme,
						borderScheme2, cyclePos, borderScheme != borderScheme2);
			}

			regularBackgrounds.put(key, newBackground);
		}
		return new ButtonBackground(isPaintedActive, regularBackgrounds
				.get(key));
	}

	// /**
	// * Retrieves background image for the specified button in button pair
	// (such
	// * as scrollbar arrows, for example).
	// *
	// * @param button
	// * Button.
	// * @param painter
	// * Gradient painter.
	// * @param width
	// * Button width.
	// * @param height
	// * Button height.
	// * @param side
	// * Button orientation.
	// * @return Button background image.
	// */
	// private static synchronized ButtonBackground getPairwiseBackground(
	// AbstractButton button, SubstanceGradientPainter painter, int width,
	// int height, SubstanceConstants.Side side) {
	// ComponentState state = ComponentState.getState(button.getModel(),
	// button);
	// ComponentState.ColorSchemeKind kind = state.getColorSchemeKind();
	//
	// // if ((kind == ColorSchemeKind.CURRENT)
	// // || (kind == ColorSchemeKind.DISABLED))
	// return getPairwiseBackground(button, kind, painter, width, height, side);
	// //
	// // return getPairwiseBackground(button, kind, SubstanceCoreUtilities
	// // .getTheme(button, true).getNonActivePainter(), width, height,
	// // side);
	// // ButtonBackground bbRegular = getPairwiseBackground(button,
	// // ColorSchemeKind.REGULAR, painter, width, height, side);
	// // if (!Boolean.TRUE.equals(button
	// // .getClientProperty(SubstanceScrollBarUI.ADJACENT_TO_TRACK)))
	// // return bbRegular;
	// //
	// // ButtonBackground bbDisabled = getPairwiseBackground(button,
	// // ColorSchemeKind.DISABLED, painter, width, height, side);
	// //
	// // BufferedImage result = SubstanceCoreUtilities.getBlankImage(width,
	// // height);
	// // Graphics2D graphics = (Graphics2D) result.createGraphics();
	// // graphics.drawImage(bbDisabled.backgroundImage, 0, 0, null);
	// // graphics.drawImage(bbRegular.backgroundImage, 1, 1, width - 1,
	// // height - 1, 1, 1, width - 1, height - 1, null);
	// // graphics.dispose();
	// // return new ButtonBackground(bbRegular.isPaintedActive, result);
	// }
	//
	// /**
	// * Retrieves background image for the specified button in button pair
	// (such
	// * as scrollbar arrows, for example).
	// *
	// * @param button
	// * Button.
	// * @param kind
	// * Color scheme kind.
	// * @param painter
	// * Gradient painter.
	// * @param width
	// * Button width.
	// * @param height
	// * Button height.
	// * @param side
	// * Button orientation.
	// * @return Button background image.
	// */
	// private static synchronized ButtonBackground getPairwiseBackground(
	// AbstractButton button, ColorSchemeKind kind,
	// SubstanceGradientPainter painter, int width, int height,
	// SubstanceConstants.Side side) {
	// ComponentState state = ComponentState.getState(button.getModel(),
	// button);
	// // ComponentState.ColorSchemeKind kind = state.getColorSchemeKind();
	// boolean isPaintedActive = (kind == ColorSchemeKind.CURRENT);
	// float cyclePos = state.getCycleCount();
	//
	// ColorScheme colorScheme = SubstanceCoreUtilities.getComponentTheme(
	// button, kind, true).getColorScheme();
	// ColorScheme colorScheme2 = colorScheme;
	// FadeTracker fadeTracker = FadeTracker.getInstance();
	// if (fadeTracker.isTracked(button, FadeKind.ROLLOVER)) {
	// boolean isActive = SubstanceCoreUtilities
	// .isControlAlwaysPaintedActive(button, true);
	// ColorScheme defaultScheme = SubstanceCoreUtilities
	// .getDefaultScheme(button);
	// if (state == ComponentState.DEFAULT) {
	// // Came from rollover state
	// colorScheme = SubstanceCoreUtilities.getActiveScheme(button);
	// colorScheme2 = isActive ? colorScheme : defaultScheme;
	// cyclePos = 10 - fadeTracker
	// .getFade10(button, FadeKind.ROLLOVER);
	// isPaintedActive = true;
	// }
	// if (state == ComponentState.ROLLOVER_UNSELECTED) {
	// // Came from default state
	// colorScheme2 = colorScheme;
	// colorScheme = isActive ? colorScheme : defaultScheme;
	// cyclePos = fadeTracker.getFade10(button, FadeKind.ROLLOVER);
	// isPaintedActive = true;
	// }
	// }
	//
	// Set<Side> openSides = SubstanceCoreUtilities.getSides(button,
	// SubstanceLookAndFeel.BUTTON_OPEN_SIDE_PROPERTY);
	// String openKey = "";
	// for (Side oSide : openSides) {
	// openKey += oSide.name() + "-";
	// }
	// String key = width + ":" + height + ":" + side.toString() + ":"
	// + cyclePos + ":" + openKey + ":"
	// + SubstanceCoreUtilities.getSchemeId(colorScheme) + ":"
	// + SubstanceCoreUtilities.getSchemeId(colorScheme2) + ":"
	// + button.getClass().getName() + ":" + painter.getDisplayName();
	// if (!pairwiseBackgrounds.containsKey(key)) {
	// BufferedImage newBackground = null;
	//
	// // buttons will be rectangular to make two scrolls (horizontal
	// // and vertical) touch the corners.
	// int radius = 0;
	//
	// int deltaLeft = openSides.contains(Side.LEFT) ? 3 : 0;
	// int deltaRight = openSides.contains(Side.RIGHT) ? 3 : 0;
	// int deltaTop = openSides.contains(Side.TOP) ? 3 : 0;
	// int deltaBottom = openSides.contains(Side.BOTTOM) ? 3 : 0;
	//
	// GeneralPath contour = null;
	//
	// // SubstanceGradientPainter painter = new StandardGradientPainter();
	//
	// switch (side) {
	// case TOP:
	// case BOTTOM:
	// // arrow in vertical bar
	// contour = BaseButtonShaper.getBaseOutline(height + deltaTop
	// + deltaBottom, width + deltaLeft + deltaRight, radius,
	// null);
	//
	// newBackground = painter.getContourBackground(height + deltaTop
	// + deltaBottom, width + deltaLeft + deltaRight, contour,
	// false, colorScheme, colorScheme2, cyclePos, true,
	// colorScheme != colorScheme2);
	// newBackground = SubstanceImageCreator.getRotated(newBackground,
	// 3);
	// break;
	// case RIGHT:
	// case LEFT:
	// // arrow in horizontal bar
	// contour = BaseButtonShaper.getBaseOutline(width + deltaLeft
	// + deltaRight, height + deltaTop + deltaBottom, radius,
	// null);
	//
	// newBackground = painter.getContourBackground(width + deltaLeft
	// + deltaRight, height + deltaTop + deltaBottom, contour,
	// false, colorScheme, colorScheme2, cyclePos, true,
	// colorScheme != colorScheme2);
	// break;
	// }
	//
	// BufferedImage finalBackground = SubstanceCoreUtilities
	// .getBlankImage(width, height);
	// Graphics2D finalGraphics = (Graphics2D) finalBackground
	// .getGraphics();
	// finalGraphics.drawImage(newBackground, -deltaLeft, -deltaTop, null);
	// pairwiseBackgrounds.put(key, finalBackground);
	// }
	// return new ButtonBackground(isPaintedActive, pairwiseBackgrounds
	// .get(key));
	// }
	//
	/**
	 * Simple constructor.
	 */
	public ButtonBackgroundDelegate() {
		super();
	}

	/**
	 * Updates background of the specified button.
	 * 
	 * @param g
	 *            Graphic context.
	 * @param button
	 *            Button to update.
	 */
	public void updateBackground(Graphics g, AbstractButton button) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel))
			return;

		button.setOpaque(false);
		if (SubstanceCoreUtilities.isButtonNeverPainted(button))
			return;

		Graphics2D graphics = (Graphics2D) g.create();
		SubstanceGradientPainter painter = SubstanceCoreUtilities
				.getGradientPainter(button);
		// if (button.getModel().isPressed()) {
		// painter = new FlatGradientPainter();
		// }

		try {
			int width = button.getWidth();
			int height = button.getHeight();
			int y = 0;
			if (button instanceof SubstanceComboBoxButton) {
				Component parent = button.getParent();
				if (parent instanceof JComboBox) {
					JComboBox jcb = (JComboBox) parent;
					if (jcb.getBorder() != null) {
						y++;
						height -= 2;
					}
				}
			}

			if (SubstanceCoreUtilities.isScrollButton(button)) {
				Sideable sideable = (Sideable) button;
				PairwiseButtonBackgroundDelegate.updateBackground(g, button,
						sideable.getSide());
				return;
			}

			if (SubstanceCoreUtilities.isSpinnerButton(button)) {
				Sideable sideable = (Sideable) button;
				PairwiseButtonBackgroundDelegate.updateBackground(g, button,
						sideable.getSide());
				return;
			}

			SubstanceButtonShaper shaper = SubstanceCoreUtilities
					.getButtonShaper(button);
			SubstanceBorderPainter borderPainter = SubstanceCoreUtilities
					.getBorderPainter(button, painter);

			// if (MemoryAnalyzer.isRunning())
			// MemoryAnalyzer.enqueueUsage("For '" + button.getText()
			// + "' using shaper '" + shaper.getClass().getName() + "'");
			ButtonBackground bb = getBackground(button, shaper, painter,
					borderPainter, width, height);
			BufferedImage bgImage = bb.backgroundImage;
			ControlBackgroundComposite composite = SubstanceCoreUtilities
					.getControlBackgroundComposite(button);

			boolean isCustomComposite = (composite.getClass() != DefaultControlBackgroundComposite.class);

			// Here we can't set the composite directly on the graphics
			// since then we would override the currently set composite
			// (from TransitionLayout fades).
			BufferedImage result = bgImage;

			if (isCustomComposite) {
				result = SubstanceCoreUtilities.getBlankImage(bgImage
						.getWidth(), bgImage.getHeight());
			}
			Graphics2D resultGr = result.createGraphics();
			if (isCustomComposite) {
				resultGr.setComposite(composite.getBackgroundComposite(button,
						button.getParent(), -1, bb.isPaintedActive));
			}

			try {
				ComponentState state = ComponentState.getState(button
						.getModel(), button);

				// Two special cases here:
				// 1. Button has flat appearance.
				// 2. Button is disabled.
				// For both cases, we need an extra temporary image with custom
				// translucency.
				boolean isSpecial = SubstanceCoreUtilities
						.hasFlatAppearance(button)
						|| !state.isKindActive(FadeKind.ENABLE);
				if (isSpecial) {
					result = SubstanceCoreUtilities.getBlankImage(bgImage
							.getWidth(), bgImage.getHeight());
					resultGr = result.createGraphics();

					BufferedImage temp = SubstanceCoreUtilities.getBlankImage(
							bgImage.getWidth(), bgImage.getHeight());
					Graphics2D tempGr = temp.createGraphics();
					if (SubstanceCoreUtilities.hasFlatAppearance(button)) {
						// Special handling of flat buttons

						// System.out.println(button.getClass().getSimpleName()
						// + " --> " + state.name());
						if (FadeTracker.getInstance().isTracked(button,
								FadeKind.ROLLOVER)
								&& !state.isKindActive(FadeKind.SELECTION)
								&& state.isKindActive(FadeKind.ENABLE)) {
							float fadeCoef = FadeTracker.getInstance()
									.getFade10(button, FadeKind.ROLLOVER);
							tempGr.setComposite(AlphaComposite.getInstance(
									AlphaComposite.SRC_OVER, fadeCoef / 10.0f));
						} else {
							if (state == ComponentState.DEFAULT) {
								// System.out.println("Doing nothing");
								return;
							}
						}
						if (state != ComponentState.DISABLED_UNSELECTED)
							tempGr.drawImage(bgImage, 0, 0, null);
					} else {
						if (!state.isKindActive(FadeKind.ENABLE)) {
							tempGr
									.setComposite(AlphaComposite.getInstance(
											AlphaComposite.SRC_OVER,
											SubstanceCoreUtilities.getTheme(
													button, true)
													.getThemeAlpha(button,
															state)));
						}
						tempGr.drawImage(bgImage, 0, 0, null);
					}
					// clear the image
					resultGr.drawImage(temp, 0, 0, null);
					tempGr.dispose();
				} else {
					if (isCustomComposite) {
						resultGr.drawImage(bgImage, 0, 0, null);
					}
				}
				graphics.drawImage(result, 0, y, null);
			} finally {
				resultGr.dispose();
			}
		} finally {
			graphics.dispose();
		}
	}

	/**
	 * Checks whether the specified button has round corners.
	 * 
	 * @param button
	 *            Button to check.
	 * @return <code>true</code> if the specified button has round corners,
	 *         <code>false</code> otherwise.
	 */
	public static boolean isRoundButton(AbstractButton button) {
		return (!SubstanceCoreUtilities.isComboBoxButton(button))
				&& (!SubstanceCoreUtilities.isScrollButton(button))
				&& SubstanceCoreUtilities.hasText(button);
	}

	/**
	 * Returns <code>true</code> if the specified <i>x,y </i> location is
	 * contained within the look and feel's defined shape of the specified
	 * component. <code>x</code> and <code>y</code> are defined to be
	 * relative to the coordinate system of the specified component.
	 * 
	 * @param button
	 *            the component where the <i>x,y </i> location is being queried;
	 * @param x
	 *            the <i>x </i> coordinate of the point
	 * @param y
	 *            the <i>y </i> coordinate of the point
	 * @return <code>true</code> if the specified <i>x,y </i> location is
	 *         contained within the look and feel's defined shape of the
	 *         specified component, <code>false</code> otherwise.
	 */
	public static boolean contains(AbstractButton button, int x, int y) {
		// failsafe for LAF change
		if (!(UIManager.getLookAndFeel() instanceof SubstanceLookAndFeel)) {
			return false;
		}
		SubstanceButtonShaper shaper = SubstanceCoreUtilities
				.getButtonShaper(button);
		GeneralPath contour = shaper.getButtonOutline(button);
		return contour.contains(x, y);
	}

	/**
	 * Returns the memory usage string.
	 * 
	 * @return Memory usage string.
	 */
	static String getMemoryUsage() {
		StringBuffer sb = new StringBuffer();
		sb.append("SubstanceBackgroundDelegate: \n");
		sb.append("\t" + regularBackgrounds.size() + " regular");
		// + pairwiseBackgrounds.size() + " pairwise");
		return sb.toString();
	}

}
