/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.utils;

import javax.swing.*;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.substance.color.*;

/**
 * This enum is used in order to provide uniform transition effects on mouse
 * events. The effects include different visual appearance of the corresponding
 * control when the mouse hovers over it (rollover), when it's pressed or
 * selected, disabled etc.<br>
 * Each enum value represents a single state and contains information that is
 * used by the UI delegates in order to correctly paint the corresponding
 * controls. <br>
 * This class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public enum ComponentState {
	/**
	 * Disabled active. Used for disabled buttons that have been marked as
	 * <code>default</code>.
	 */
	DISABLED_ACTIVE(ColorSchemeKind.DISABLED, 0),

	/**
	 * Active. Used for enabled buttons that have been marked as
	 * <code>default</code>.
	 */
	ACTIVE(ColorSchemeKind.CURRENT, 0, FadeKind.ENABLE),

	/**
	 * Disabled selected.
	 */
	DISABLED_SELECTED(ColorSchemeKind.DISABLED, 10, FadeKind.SELECTION),

	/**
	 * Disabled and not selected.
	 */
	DISABLED_UNSELECTED(ColorSchemeKind.DISABLED, 0),

	/**
	 * Pressed selected.
	 */
	PRESSED_SELECTED(ColorSchemeKind.CURRENT, 5, FadeKind.ENABLE,
			FadeKind.SELECTION, FadeKind.PRESS),

	/**
	 * Pressed and not selected.
	 */
	PRESSED_UNSELECTED(ColorSchemeKind.CURRENT, 8, FadeKind.ENABLE,
			FadeKind.PRESS),

	/**
	 * Selected.
	 */
	SELECTED(ColorSchemeKind.CURRENT, 8, FadeKind.ENABLE, FadeKind.SELECTION),

	/**
	 * Selected and rolled over.
	 */
	ROLLOVER_SELECTED(ColorSchemeKind.CURRENT, 4, FadeKind.ENABLE,
			FadeKind.ROLLOVER, FadeKind.SELECTION),

	/**
	 * Armed.
	 */
	ARMED(ColorSchemeKind.CURRENT, 5, FadeKind.ENABLE, FadeKind.ARM),

	/**
	 * Armed and rolled over.
	 */
	ROLLOVER_ARMED(ColorSchemeKind.CURRENT, 7, FadeKind.ENABLE,
			FadeKind.ROLLOVER, FadeKind.ARM),

	/**
	 * Not selected and rolled over.
	 */
	ROLLOVER_UNSELECTED(ColorSchemeKind.CURRENT, 3, FadeKind.ENABLE,
			FadeKind.ROLLOVER),

	/**
	 * Default state.
	 */
	DEFAULT(ColorSchemeKind.REGULAR, 0, FadeKind.ENABLE);

	/**
	 * Enum for color scheme kind. Is used in order to decouple the actual
	 * current color scheme and the decision on whether to use it.
	 * 
	 * @author Kirill Grouchnikov
	 */
	public static enum ColorSchemeKind {
		/**
		 * Current color scheme (e.g. {@link AquaColorScheme}).
		 */
		CURRENT,

		/**
		 * Regular color scheme (usually {@link MetallicColorScheme}).
		 */
		REGULAR,

		/**
		 * Disabled color scheme (usually {@link LightGrayColorScheme}).
		 */
		DISABLED
	}

	/**
	 * The corresponding color scheme kind.
	 */
	private ColorSchemeKind colorSchemeKind;

	/**
	 * The corresponding cycle count. Should be a number between 0 and 10. This
	 * number is used to compute the foreground color of some component. The
	 * color is interpolated between two values (0 corresponds to usual color,
	 * 10 corresponds to very light version of the usual color).
	 */
	private int cycleCount;

	/**
	 * Active fade kinds for this state. For example, {@link #ROLLOVER_SELECTED}
	 * contains {@link FadeKind#ROLLOVER} and {@link FadeKind#SELECTION}.
	 */
	private FadeKind[] activeKinds;

	/**
	 * Simple constructor.
	 * 
	 * @param kind
	 *            The corresponding color scheme kind.
	 * @param count
	 *            The corresponding cycle count.
	 * @param activeKinds
	 *            Indicates active fade kinds for this state. For example,
	 *            {@link #ROLLOVER_SELECTED} should pass both
	 *            {@link FadeKind#ROLLOVER} and {@link FadeKind#SELECTION}.
	 */
	ComponentState(ColorSchemeKind kind, int count, FadeKind... activeKinds) {
		colorSchemeKind = kind;
		cycleCount = count;
		this.activeKinds = activeKinds;
	}

	/**
	 * Returns the corresponding color scheme kind
	 * 
	 * @return Corresponding color scheme kind
	 */
	public ColorSchemeKind getColorSchemeKind() {
		return colorSchemeKind;
	}

	/**
	 * Returns the corresponding cycle count.
	 * 
	 * @return Corresponding cycle count.
	 */
	public int getCycleCount() {
		return cycleCount;
	}

	/**
	 * Returns indication whether <code>this</code> component state is
	 * "active" under the specified fade kind. For example,
	 * {@link #ROLLOVER_SELECTED} will return <code>true</code> for both
	 * {@link FadeKind#ROLLOVER} and {@link FadeKind#SELECTION}.
	 * 
	 * @param fadeKind
	 *            Fade kind.
	 * @return <code>true</code> if <code>this</code> component state is
	 *         "active" under the specified fade kind (for example,
	 *         {@link #ROLLOVER_SELECTED} will return <code>true</code> for
	 *         both {@link FadeKind#ROLLOVER} and {@link FadeKind#SELECTION}),
	 *         <code>false</code> otherwise.
	 */
	public boolean isKindActive(FadeKind fadeKind) {
		if (activeKinds == null)
			return false;
		for (FadeKind fk : activeKinds)
			if (fadeKind == fk)
				return true;
		return false;
	}

	/**
	 * Retrieves component state based on the button model (required parameter)
	 * and button itself (optional parameter).
	 * 
	 * @param model
	 *            Button model (required).
	 * @param component
	 *            Button (optional).
	 * @return The matching component state.
	 */
	public static ComponentState getState(ButtonModel model,
			JComponent component) {
		return getState(model, component, false);
	}

	/**
	 * Retrieves component state based on the button model (required parameter)
	 * and button itself (optional parameter).
	 * 
	 * @param model
	 *            Button model (required).
	 * @param component
	 *            Button (optional).
	 * @param toIgnoreSelection
	 *            If <code>true</code>, the {@link ButtonModel#isSelected()}
	 *            will not be checked. This can be used for tracking transitions
	 *            on menu items that use <code>armed</code> state instead,
	 *            when we don't want to use different rollover themes for
	 *            selected and unselected checkbox and radio button menu items
	 *            (to preserve consistent visual appearence of highlights).
	 * @return The matching component state.
	 */
	public static ComponentState getState(ButtonModel model,
			JComponent component, boolean toIgnoreSelection) {

		// if (button instanceof JToggleButton) {
		// System.out.println(button.getWidth() + "*" + button.getHeight()
		// + ":" + (model.isEnabled() ? "enabled:" : "")
		// + (model.isArmed() ? "armed:" : "")
		// + (model.isSelected() ? "selected:" : "")
		// + (model.isRollover() ? "rollover:" : "")
		// + (button.isRolloverEnabled() ? "rollover enabled:" : "")
		// + (model.isPressed() ? "pressed:" : ""));
		// }
		//

		boolean isRollover = model.isRollover();
		// fix for defect 103 - no rollover effects on menu items
		// that are not in the selected menu path
		MenuElement[] selectedMenuPath = MenuSelectionManager.defaultManager()
				.getSelectedPath();
		for (MenuElement elem : selectedMenuPath) {
			if (elem == component) {
				isRollover = true;
				break;
			}
		}

		if (component != null) {
			if (component instanceof JButton) {
				JButton jb = (JButton) component;
				if (jb.isDefaultButton()) {
					if (model.isEnabled()) {
						if ((!model.isPressed()) && (!model.isArmed()))
							return ACTIVE;
					} else
						return DISABLED_ACTIVE;
				}
			}
		}

		boolean isRolloverEnabled = true;
		if (component instanceof AbstractButton)
			isRolloverEnabled = ((AbstractButton) component)
					.isRolloverEnabled();
		if (!model.isEnabled()) {
			if (model.isSelected())
				return DISABLED_SELECTED;
			return DISABLED_UNSELECTED;
		} else if (model.isPressed() && model.isArmed()) {
			if (model.isSelected())
				return PRESSED_SELECTED;
			return PRESSED_UNSELECTED;
		} else if (!toIgnoreSelection && model.isSelected()) {
			if (((component == null) || isRolloverEnabled) && isRollover)
				return ROLLOVER_SELECTED;
			return SELECTED;
		} else if (model.isArmed()) {
			if (((component == null) || isRolloverEnabled) && isRollover)
				return ROLLOVER_ARMED;
			return ARMED;
		} else if (((component == null) || isRolloverEnabled) && isRollover)
			return ROLLOVER_UNSELECTED;

		return DEFAULT;
	}

}
