/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *  o Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 *  o Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of
 *    its contributors may be used to endorse or promote products derived
 *    from this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */
package org.jvnet.substance.utils;

import java.awt.Color;
import java.awt.Component;

import javax.swing.AbstractButton;
import javax.swing.ButtonModel;

import org.jvnet.lafwidget.animation.FadeKind;
import org.jvnet.lafwidget.animation.FadeState;
import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.color.ColorScheme;
import org.jvnet.substance.color.ColorBlindColorScheme.BlindnessKind;
import org.jvnet.substance.theme.SubstanceComplexTheme;
import org.jvnet.substance.theme.SubstanceTheme;

/**
 * Various color-related utilities. This class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public class SubstanceColorUtilities {
	/**
	 * Returns the color of the top portion of border in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @return The color of the top portion of border in control backgrounds.
	 */
	public static Color getTopBorderColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef) {
		return SubstanceColorUtilities.getInterpolatedColor(scheme1
				.getUltraDarkColor(), scheme2.getUltraDarkColor(), cycleCoef);
	}

	/**
	 * Returns the color of the top portion of border.
	 * 
	 * @param scheme
	 *            The color scheme.
	 * @param isDark
	 *            Indicates whether the matching theme is dark.
	 * @return The color of the top portion of border.
	 */
	public static Color getTopBorderColor(ColorScheme scheme, boolean isDark) {
		return isDark ? SubstanceColorUtilities.getInterpolatedColor(scheme
				.getExtraLightColor(), scheme.getForegroundColor(), 0.5)
				: scheme.getDarkColor();
	}

	/**
	 * Returns the color of the bottom portion of border.
	 * 
	 * @param scheme
	 *            The color scheme.
	 * @param isDark
	 *            Indicates whether the matching theme is dark.
	 * @return The color of the bottom portion of border.
	 */
	public static Color getBottomBorderColor(ColorScheme scheme, boolean isDark) {
		return isDark ? SubstanceColorUtilities.getInterpolatedColor(scheme
				.getExtraLightColor(), scheme.getForegroundColor(), 0.8)
				: scheme.getMidColor();
	}

	/**
	 * Returns the color of the middle portion of border in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @return The color of the middle portion of border in control backgrounds.
	 */
	public static Color getMidBorderColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef) {
		// if (isDark) {
		// return SubstanceCoreUtilities.getInterpolatedColor(scheme1
		// .getExtraLightColor(), scheme2.getExtraLightColor(),
		// cycleCoef);
		// } else {
		return SubstanceColorUtilities.getInterpolatedColor(scheme1
				.getDarkColor(), scheme2.getDarkColor(), cycleCoef);
		// }
	}

	/**
	 * Returns the color of the bottom portion of border in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @return The color of the bottom portion of border in control backgrounds.
	 */
	public static Color getBottomBorderColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef) {
		// if (isDark) {
		// return SubstanceCoreUtilities.getInterpolatedColor(scheme1
		// .getLightColor(), scheme2.getLightColor(), cycleCoef);
		// } else {
		Color c1 = SubstanceColorUtilities.getInterpolatedColor(scheme1
				.getDarkColor(), scheme1.getMidColor(), 0.5);
		Color c2 = SubstanceColorUtilities.getInterpolatedColor(scheme2
				.getDarkColor(), scheme2.getMidColor(), 0.5);
		return SubstanceColorUtilities.getInterpolatedColor(c1, c2, cycleCoef);
		// }
	}

	/**
	 * Returns the color of the top portion of fill in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @param useCyclePosAsInterpolation
	 *            Indicates the algorithm to use for computing various colors.
	 *            If <code>true</code>, the <code>cyclePos</code> is used
	 *            to interpolate colors between different color components of
	 *            both color schemes. If <code>false</code>, the
	 *            <code>cyclePos</code> is used to interpolate colors between
	 *            different color components of the first color scheme.
	 * @return The color of the top portion of fill in control backgrounds.
	 */
	public static Color getTopFillColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef,
			boolean useCyclePosAsInterpolation) {
		if (!useCyclePosAsInterpolation) {
			Color c = SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getDarkColor(), scheme1.getMidColor(), 0.4);
			return SubstanceColorUtilities.getInterpolatedColor(c, scheme2
					.getLightColor(), cycleCoef);
		} else {
			Color c1 = SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getDarkColor(), scheme1.getMidColor(), 0.4);
			Color c2 = SubstanceColorUtilities.getInterpolatedColor(scheme2
					.getDarkColor(), scheme2.getMidColor(), 0.4);
			return SubstanceColorUtilities.getInterpolatedColor(c1, c2,
					cycleCoef);
		}
	}

	/**
	 * Returns the color of the middle portion of fill in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @param useCyclePosAsInterpolation
	 *            Indicates the algorithm to use for computing various colors.
	 *            If <code>true</code>, the <code>cyclePos</code> is used
	 *            to interpolate colors between different color components of
	 *            both color schemes. If <code>false</code>, the
	 *            <code>cyclePos</code> is used to interpolate colors between
	 *            different color components of the first color scheme.
	 * @return The color of the middle portion of fill in control backgrounds.
	 */
	public static Color getMidFillColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef,
			boolean useCyclePosAsInterpolation) {
		if (!useCyclePosAsInterpolation) {
			return SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getMidColor(), scheme2.getLightColor(), cycleCoef);
		} else {
			return SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getMidColor(), scheme2.getMidColor(), cycleCoef);
		}
	}

	/**
	 * Returns the color of the bottom portion of fill in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @param useCyclePosAsInterpolation
	 *            Indicates the algorithm to use for computing various colors.
	 *            If <code>true</code>, the <code>cyclePos</code> is used
	 *            to interpolate colors between different color components of
	 *            both color schemes. If <code>false</code>, the
	 *            <code>cyclePos</code> is used to interpolate colors between
	 *            different color components of the first color scheme.
	 * @return The color of the bottom portion of fill in control backgrounds.
	 */
	public static Color getBottomFillColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef,
			boolean useCyclePosAsInterpolation) {
		// if (isDark) {
		// if (!useCyclePosAsInterpolation) {
		// return SubstanceCoreUtilities.getInterpolatedColor(scheme1
		// .getDarkColor(), scheme2.getMidColor(), cycleCoef);
		// } else {
		// return SubstanceCoreUtilities.getInterpolatedColor(scheme1
		// .getDarkColor(), scheme2.getDarkColor(), cycleCoef);
		// }
		// } else {
		if (!useCyclePosAsInterpolation) {
			return SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getUltraLightColor(), scheme2.getExtraLightColor(),
					cycleCoef);
		} else {
			return SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getUltraLightColor(), scheme2.getUltraLightColor(),
					cycleCoef);
		}
		// }
	}

	/**
	 * Returns the color of the top portion of shine in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @param useCyclePosAsInterpolation
	 *            Indicates the algorithm to use for computing various colors.
	 *            If <code>true</code>, the <code>cyclePos</code> is used
	 *            to interpolate colors between different color components of
	 *            both color schemes. If <code>false</code>, the
	 *            <code>cyclePos</code> is used to interpolate colors between
	 *            different color components of the first color scheme.
	 * @return The color of the top portion of shine in control backgrounds.
	 */
	public static Color getTopShineColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef,
			boolean useCyclePosAsInterpolation) {
		return getBottomFillColor(scheme1, scheme2, cycleCoef,
				useCyclePosAsInterpolation);
	}

	/**
	 * Returns the color of the bottom portion of shine in control backgrounds.
	 * 
	 * @param scheme1
	 *            The first color scheme.
	 * @param scheme2
	 *            The second color scheme.
	 * @param cycleCoef
	 *            Cycle position. Is used for rollover and pulsation effects.
	 *            Must be in 0.0 .. 1.0 range.
	 * @param useCyclePosAsInterpolation
	 *            Indicates the algorithm to use for computing various colors.
	 *            If <code>true</code>, the <code>cyclePos</code> is used
	 *            to interpolate colors between different color components of
	 *            both color schemes. If <code>false</code>, the
	 *            <code>cyclePos</code> is used to interpolate colors between
	 *            different color components of the first color scheme.
	 * @return The color of the bottom portion of shine in control backgrounds.
	 */
	public static Color getBottomShineColor(ColorScheme scheme1,
			ColorScheme scheme2, double cycleCoef,
			boolean useCyclePosAsInterpolation) {
		// if (isDark) {
		// if (!useCyclePosAsInterpolation) {
		// return SubstanceCoreUtilities.getInterpolatedColor(scheme1
		// .getMidColor(), scheme2.getDarkColor(), cycleCoef);
		// } else {
		// return SubstanceCoreUtilities.getInterpolatedColor(scheme1
		// .getMidColor(), scheme2.getMidColor(), cycleCoef);
		// }
		// } else {
		if (!useCyclePosAsInterpolation) {
			return SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getLightColor(), scheme2.getUltraLightColor(), cycleCoef);
		} else {
			return SubstanceColorUtilities.getInterpolatedColor(scheme1
					.getLightColor(), scheme2.getLightColor(), cycleCoef);
		}
		// }
	}

	/**
	 * Interpolates color.
	 * 
	 * @param color1
	 *            The first color
	 * @param color2
	 *            The second color
	 * @param color1Likeness
	 *            The closer this value is to 0.0, the closer the resulting
	 *            color will be to <code>color2</code>.
	 * @return Interpolated RGB value.
	 */
	public static int getInterpolatedRGB(Color color1, Color color2,
			double color1Likeness) {
		int lr = color1.getRed();
		int lg = color1.getGreen();
		int lb = color1.getBlue();
		int la = color1.getAlpha();
		int dr = color2.getRed();
		int dg = color2.getGreen();
		int db = color2.getBlue();
		int da = color2.getAlpha();

		int r = (int) (color1Likeness * lr + (1.0 - color1Likeness) * dr);
		int g = (int) (color1Likeness * lg + (1.0 - color1Likeness) * dg);
		int b = (int) (color1Likeness * lb + (1.0 - color1Likeness) * db);
		int a = (int) (color1Likeness * la + (1.0 - color1Likeness) * da);

		r = Math.min(255, r);
		g = Math.min(255, g);
		b = Math.min(255, b);
		a = Math.min(255, a);
		r = Math.max(0, r);
		g = Math.max(0, g);
		b = Math.max(0, b);
		a = Math.max(0, a);
		return (a << 24) | (r << 16) | (g << 8) | b;
	}

	/**
	 * Interpolates color.
	 * 
	 * @param color1
	 *            The first color
	 * @param color2
	 *            The second color
	 * @param color1Likeness
	 *            The closer this value is to 0.0, the closer the resulting
	 *            color will be to <code>color2</code>.
	 * @return Interpolated color.
	 */
	public static Color getInterpolatedColor(Color color1, Color color2,
			double color1Likeness) {
		if (color1.equals(color2))
			return color1;
		if (color1Likeness == 1.0)
			return color1;
		if (color1Likeness == 0.0)
			return color2;
		return new Color(getInterpolatedRGB(color1, color2, color1Likeness),
				true);
	}

	/**
	 * Inverts the specified color.
	 * 
	 * @param color
	 *            The original color.
	 * @return The inverted color.
	 */
	public static Color invertColor(Color color) {
		return new Color(255 - color.getRed(), 255 - color.getGreen(),
				255 - color.getBlue(), color.getAlpha());
	}

	/**
	 * Returns background color for the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Background color for the specified theme.
	 */
	public static Color getBackgroundColor(SubstanceTheme theme) {
		if (theme instanceof SubstanceComplexTheme) {
			SubstanceComplexTheme sct = (SubstanceComplexTheme) theme;
			SubstanceTheme defTheme = sct.getDefaultTheme();
			switch (defTheme.getKind()) {
			case DARK:
				return defTheme.getColorScheme().getDarkColor().brighter();
			default:
				return defTheme.getColorScheme().getExtraLightColor();
			}
		}
		switch (theme.getKind()) {
		case DARK:
			return theme.getColorScheme().getDarkColor().brighter();
		default:
			return theme.getDefaultColorScheme().getExtraLightColor();
			// SubstanceTheme.getMetallic(theme.getKind())
			// .getExtraLightColor();
		}
	}

	/**
	 * Returns disabled foreground color for the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Disabled foreground color for the specified theme.
	 */
	public static Color getDisabledForegroundColor(SubstanceTheme theme) {
		return theme.getDisabledTheme().getForegroundColor();
		// switch (theme.getKind()) {
		// case DARK:
		// return getInterpolatedColor(theme.getDefaultColorScheme()
		// .getForegroundColor(), theme.getColorScheme()
		// .getDarkColor(), 0.45);
		// default:
		// return getInterpolatedColor(theme.getDefaultColorScheme()
		// .getMidColor(), theme.getDefaultColorScheme()
		// .getDarkColor(), 0.7);
		// }
	}

	/**
	 * Returns selection background color for the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Selection background color for the specified theme.
	 */
	public static Color getSelectionBackgroundColor(SubstanceTheme theme) {
		// fix for enhancement 182 - use highlight theme as
		// selection background.
		switch (theme.getKind()) {
		case DARK:
			return theme.getHighlightTheme(null, ComponentState.ACTIVE)
					.getColorScheme().getUltraLightColor();// .brighter()
			// .brighter();
		default:
			return theme.getHighlightTheme(null, ComponentState.ACTIVE)
					.getColorScheme().getExtraLightColor();
		}

	}

	/**
	 * Returns line color for the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Line color for the specified theme.
	 */
	public static Color getLineColor(SubstanceTheme theme) {
		switch (theme.getKind()) {
		case DARK:
			return theme.getColorScheme().getUltraLightColor();
		default:
			return theme.getColorScheme().getMidColor();
		}
	}

	/**
	 * Returns selection foreground color for the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Selection foreground color for the specified theme.
	 */
	public static Color getSelectionForegroundColor(SubstanceTheme theme) {
		switch (theme.getKind()) {
		case DARK:
			return theme.getActiveTheme().getColorScheme().getUltraDarkColor()
					.darker();
		default:
			return theme.getActiveTheme().getColorScheme().getUltraDarkColor();
		}
	}

	/**
	 * Returns the watermark stamp color for the current theme.
	 * 
	 * @return Watermark stamp color for the current theme.
	 */
	public static Color getWatermarkStampColor() {
		return getWatermarkStampColor(SubstanceLookAndFeel.getTheme());
	}

	/**
	 * Returns the watermark stamp color for the current theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Watermark stamp color for the current theme.
	 */
	public static Color getWatermarkStampColor(SubstanceTheme theme) {
		return theme.getWatermarkTheme().getWatermarkStampColor();
	}

	/**
	 * Returns a negative of the specified color.
	 * 
	 * @param color
	 *            Color.
	 * @return Negative of the specified color.
	 */
	public static Color getNegativeColor(Color color) {
		return new Color(255 - color.getRed(), 255 - color.getGreen(),
				255 - color.getBlue(), color.getAlpha());
	}

	/**
	 * Returns a negative of the specified color.
	 * 
	 * @param rgb
	 *            Color RGB.
	 * @return Negative of the specified color.
	 */
	public static int getNegativeColor(int rgb) {
		int transp = (rgb >>> 24) & 0xFF;
		int r = (rgb >>> 16) & 0xFF;
		int g = (rgb >>> 8) & 0xFF;
		int b = (rgb >>> 0) & 0xFF;

		return (transp << 24) | ((255 - r) << 16) | ((255 - g) << 8)
				| (255 - b);
	}

	/**
	 * Returns the watermark light color for the current theme.
	 * 
	 * @return Watermark light color for the current theme.
	 */
	public static Color getWatermarkLightColor() {
		return getWatermarkLightColor(1.0f);
	}

	/**
	 * Returns the watermark light color for the current theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Watermark light color for the current theme.
	 */
	public static Color getWatermarkLightColor(SubstanceTheme theme) {
		return getWatermarkLightColor(theme, 1.0f);
	}

	/**
	 * Returns the watermark light color for the current theme.
	 * 
	 * @param coef
	 *            Translucency coefficient.
	 * @return Watermark light color for the current theme.
	 */
	public static Color getWatermarkLightColor(float coef) {
		return getWatermarkLightColor(SubstanceLookAndFeel.getTheme(), coef);
	}

	/**
	 * Returns the watermark light color for the current theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @param coef
	 *            Translucency coefficient.
	 * @return Watermark light color for the current theme.
	 */
	public static Color getWatermarkLightColor(SubstanceTheme theme, float coef) {
		SubstanceTheme watermarkTheme = theme.getWatermarkTheme();
		int alpha = SubstanceCoreUtilities.isThemeDark(watermarkTheme) ? (int) (coef * 100)
				: (int) (coef * 255);
		if (alpha > 255)
			alpha = 255;
		return SubstanceColorUtilities.getAlphaColor(watermarkTheme
				.getColorScheme().getLightColor(), alpha);
	}

	/**
	 * Returns the watermark dark color for the current theme.
	 * 
	 * @return Watermark dark color for the current theme.
	 */
	public static Color getWatermarkDarkColor() {
		return getWatermarkDarkColor(SubstanceLookAndFeel.getTheme());
	}

	/**
	 * Returns the watermark dark color for the current theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Watermark dark color for the current theme.
	 */
	public static Color getWatermarkDarkColor(SubstanceTheme theme) {
		SubstanceTheme watermarkTheme = theme.getWatermarkTheme();
		return SubstanceCoreUtilities.isThemeDark(watermarkTheme) ? SubstanceColorUtilities
				.getAlphaColor(watermarkTheme.getColorScheme()
						.getUltraLightColor(), 128)
				: SubstanceColorUtilities.getAlphaColor(watermarkTheme
						.getColorScheme().getDarkColor(), 15);
	}

	/**
	 * Returns the watermark stamp color of the specified opacity for the
	 * current theme.
	 * 
	 * @param alpha
	 *            Alpha channel value.
	 * @return Watermark stamp color of the specified opacity for the current
	 *         theme.
	 */
	public static Color getWatermarkStampColor(int alpha) {
		SubstanceTheme watermarkTheme = SubstanceLookAndFeel.getTheme()
				.getWatermarkTheme();
		return SubstanceCoreUtilities.isThemeDark(watermarkTheme) ? SubstanceColorUtilities
				.getAlphaColor(watermarkTheme.getColorScheme()
						.getUltraLightColor(), alpha)
				: SubstanceColorUtilities.getAlphaColor(watermarkTheme
						.getColorScheme().getDarkColor(), alpha);
	}

	/**
	 * Returns a translucent of the specified color.
	 * 
	 * @param color
	 *            Color.
	 * @param alpha
	 *            Alpha channel value.
	 * @return Translucent of the specified color that matches the requested
	 *         alpha channel value.
	 */
	public static Color getAlphaColor(Color color, int alpha) {
		return new Color(color.getRed(), color.getGreen(), color.getBlue(),
				alpha);
	}

	/**
	 * Returns saturated version of the specified color.
	 * 
	 * @param color
	 *            Color.
	 * @param factor
	 *            Saturation factor.
	 * @return Saturated color.
	 */
	public static Color getSaturatedColor(Color color, double factor) {
		float[] hsbvals = new float[3];
		Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(),
				hsbvals);
		float sat = hsbvals[1];
		if (factor > 0.0) {
			sat = sat + (float) factor * (1.0f - sat);
		} else {
			sat = sat + (float) factor * sat;
		}
		return new Color(Color.HSBtoRGB(hsbvals[0], sat, hsbvals[2]));
	}

	/**
	 * Returns hue-shifted (in HSV space) version of the specified color.
	 * 
	 * @param color
	 *            Color.
	 * @param hueShift
	 *            hue shift factor.
	 * @return Hue-shifted (in HSV space) color.
	 */
	public static Color getHueShiftedColor(Color color, double hueShift) {
		float[] hsbvals = new float[3];
		Color.RGBtoHSB(color.getRed(), color.getGreen(), color.getBlue(),
				hsbvals);
		float hue = hsbvals[0];
		hue += hueShift;
		if (hue < 0.0)
			hue += 1.0;
		if (hue > 1.0)
			hue -= 1.0;
		return new Color(Color.HSBtoRGB(hue, hsbvals[1], hsbvals[2]));
	}

	/**
	 * Returns the foreground color of the specified theme.
	 * 
	 * @param theme
	 *            Theme.
	 * @return Theme foreground color.
	 */
	public static Color getForegroundColor(SubstanceTheme theme) {
		return theme.getColorScheme().getForegroundColor();
	}

	/**
	 * Returns lighter version of the specified color.
	 * 
	 * @param color
	 *            Color.
	 * @param diff
	 *            Difference factor (values closer to 0.0 will produce results
	 *            closer to white color).
	 * @return Lighter version of the specified color.
	 */
	public static Color getLighterColor(Color color, double diff) {
		int r = color.getRed() + (int) (diff * (255 - color.getRed()));
		int g = color.getGreen() + (int) (diff * (255 - color.getGreen()));
		int b = color.getBlue() + (int) (diff * (255 - color.getBlue()));
		return new Color(r, g, b);
	}

	/**
	 * Returns darker version of the specified color.
	 * 
	 * @param color
	 *            Color.
	 * @param diff
	 *            Difference factor (values closer to 1.0 will produce results
	 *            closer to black color).
	 * @return Darker version of the specified color.
	 */
	public static Color getDarkerColor(Color color, double diff) {
		int r = (int) ((1.0 - diff) * color.getRed());
		int g = (int) ((1.0 - diff) * color.getGreen());
		int b = (int) ((1.0 - diff) * color.getBlue());
		return new Color(r, g, b);
	}

	/**
	 * Converts the specified color into color-blind version.
	 * 
	 * @param orig
	 *            The original color.
	 * @param rgbToLms
	 *            RGB to LMS conversion matrix.
	 * @param kind
	 *            Color-blindness kind.
	 * @param lmsToRgb
	 *            LMS to RGB conversion matrix.
	 * @return Color-blind version of the original color.
	 */
	public static Color getColorBlindColor(Color orig, double[][] rgbToLms,
			BlindnessKind kind, double[][] lmsToRgb) {
		double r = orig.getRed();
		double g = orig.getGreen();
		double b = orig.getBlue();

		double[] rgbOrig = new double[] { r, g, b };
		double[] lms = mult3(rgbToLms, rgbOrig);
		double tmp = 0.0;

		double[] anchor = { 0.08008, 0.1579, 0.5897, 0.1284, 0.2237, 0.3636,
				0.9856, 0.7325, 0.001079, 0.0914, 0.007009, 0.0 };

		double[] rgbAnchor = {
				rgbToLms[0][0] + rgbToLms[0][1] + rgbToLms[0][2],
				rgbToLms[1][0] + rgbToLms[1][1] + rgbToLms[1][2],
				rgbToLms[2][0] + rgbToLms[2][1] + rgbToLms[2][2] };

		double a1, a2, b1, b2, c1, c2, inflection;

		switch (kind) {
		case PROTANOPIA:
			a1 = rgbAnchor[1] * anchor[8] - rgbAnchor[2] * anchor[7];
			b1 = rgbAnchor[2] * anchor[6] - rgbAnchor[0] * anchor[8];
			c1 = rgbAnchor[0] * anchor[7] - rgbAnchor[1] * anchor[6];
			a2 = rgbAnchor[1] * anchor[2] - rgbAnchor[2] * anchor[1];
			b2 = rgbAnchor[2] * anchor[0] - rgbAnchor[0] * anchor[2];
			c2 = rgbAnchor[0] * anchor[1] - rgbAnchor[1] * anchor[0];
			inflection = rgbAnchor[2] / rgbAnchor[1];
			tmp = lms[2] / lms[1];
			if (tmp < inflection)
				lms[0] = -(b1 * lms[1] + c1 * lms[2]) / a1;
			else
				lms[0] = -(b2 * lms[1] + c2 * lms[2]) / a2;
			break;

		case DEUTERANOPIA:
			a1 = rgbAnchor[1] * anchor[8] - rgbAnchor[2] * anchor[7];
			b1 = rgbAnchor[2] * anchor[6] - rgbAnchor[0] * anchor[8];
			c1 = rgbAnchor[0] * anchor[7] - rgbAnchor[1] * anchor[6];
			a2 = rgbAnchor[1] * anchor[2] - rgbAnchor[2] * anchor[1];
			b2 = rgbAnchor[2] * anchor[0] - rgbAnchor[0] * anchor[2];
			c2 = rgbAnchor[0] * anchor[1] - rgbAnchor[1] * anchor[0];
			inflection = rgbAnchor[2] / rgbAnchor[0];
			tmp = lms[2] / lms[0];
			/* See which side of the inflection line we fall... */
			if (tmp < inflection)
				lms[1] = -(a1 * lms[0] + c1 * lms[2]) / b1;
			else
				lms[1] = -(a2 * lms[0] + c2 * lms[2]) / b2;
			break;

		case TRITANOPIA:
			a1 = rgbAnchor[1] * anchor[11] - rgbAnchor[2] * anchor[10];
			b1 = rgbAnchor[2] * anchor[9] - rgbAnchor[0] * anchor[11];
			c1 = rgbAnchor[0] * anchor[10] - rgbAnchor[1] * anchor[9];
			a2 = rgbAnchor[1] * anchor[5] - rgbAnchor[2] * anchor[4];
			b2 = rgbAnchor[2] * anchor[3] - rgbAnchor[0] * anchor[5];
			c2 = rgbAnchor[0] * anchor[4] - rgbAnchor[1] * anchor[3];
			inflection = (rgbAnchor[1] / rgbAnchor[0]);
			tmp = lms[1] / lms[0];
			if (tmp < inflection)
				lms[2] = -(a1 * lms[0] + b1 * lms[1]) / c1;
			else
				lms[2] = -(a2 * lms[0] + b2 * lms[1]) / c2;
			break;

		default:
			break;
		}
		double[] rgbCb = mult3(lmsToRgb, lms);

		double nr = Math.min(255.0, Math.max(0.0, rgbCb[0]));
		double ng = Math.min(255.0, Math.max(0.0, rgbCb[1]));
		double nb = Math.min(255.0, Math.max(0.0, rgbCb[2]));
		return new Color((int) nr, (int) ng, (int) nb);
	}

	/**
	 * Multiplies the specified 3x3 matrix by the specified 3x1 vector.
	 * 
	 * @param matrix
	 *            Matrix.
	 * @param vector
	 *            Vector.
	 * @return Vector multiplication.
	 */
	private static double[] mult3(double[][] matrix, double[] vector) {
		double[] res = new double[3];
		res[0] = matrix[0][0] * vector[0] + matrix[0][1] * vector[1]
				+ matrix[0][2] * vector[2];
		res[1] = matrix[1][0] * vector[0] + matrix[1][1] * vector[1]
				+ matrix[1][2] * vector[2];
		res[2] = matrix[2][0] * vector[0] + matrix[2][1] * vector[1]
				+ matrix[2][2] * vector[2];
		return res;
	}

	/**
	 * Returns the brightness of the specified color.
	 * 
	 * @param rgb
	 *            RGB value of a color.
	 * @return The brightness of the specified color.
	 */
	public static int getColorBrightness(int rgb) {
		int oldR = (rgb >>> 16) & 0xFF;
		int oldG = (rgb >>> 8) & 0xFF;
		int oldB = (rgb >>> 0) & 0xFF;

		return (222 * oldR + 707 * oldG + 71 * oldB) / 1000;
	}

	/**
	 * Returns the color of the focus ring for the specified component.
	 * 
	 * @param comp
	 *            Component.
	 * @return The color of the focus ring for the specified component.
	 */
	public static Color getFocusColor(Component comp) {
		SubstanceTheme activeTheme = SubstanceCoreUtilities.getActiveTheme(
				comp, true);

		if (comp instanceof AbstractButton) {
			AbstractButton ab = (AbstractButton) comp;
			ButtonModel model = ab.getModel();

			ComponentState currState = ComponentState.getState(model, ab);
			SubstanceTheme currTheme = SubstanceCoreUtilities.getTheme(comp,
					currState, true, true);
			Color currColor = SubstanceCoreUtilities.isThemeDark(currTheme) ? currTheme
					.getColorScheme().getUltraDarkColor()
					: currTheme.getColorScheme().getDarkColor();

			FadeState fadeState = SubstanceFadeUtilities.getFadeState(comp,
					FadeKind.PRESS, FadeKind.SELECTION, FadeKind.ROLLOVER);
			if (fadeState != null) {
				// the component is currently animating
				ComponentState prevState = SubstanceCoreUtilities
						.getPrevComponentState(ab);

				SubstanceTheme prevTheme = SubstanceCoreUtilities.getTheme(
						comp, prevState, true, true);

				Color prevColor = SubstanceCoreUtilities.isThemeDark(prevTheme) ? prevTheme
						.getColorScheme().getUltraDarkColor()
						: prevTheme.getColorScheme().getDarkColor();

				float likeness = fadeState.getFadePosition() / 10.0f;
				if (fadeState.isFadingIn())
					likeness = 1.0f - likeness;
				return SubstanceColorUtilities.getInterpolatedColor(prevColor,
						currColor, likeness);
			} else {
				return currColor;
			}
		}

		Color color = SubstanceCoreUtilities.isThemeDark(SubstanceCoreUtilities
				.getActiveTheme(comp, true)) ? activeTheme.getColorScheme()
				.getUltraDarkColor() : activeTheme.getColorScheme()
				.getDarkColor();
		return color;
	}

	/**
	 * Returns the color strength.
	 * 
	 * @param color
	 *            Color.
	 * @return Color strength.
	 */
	public static float getColorStrength(Color color) {
		return Math.max(getColorBrightness(color.getRGB()),
				getColorBrightness(getNegativeColor(color.getRGB()))) / 255.0f;
	}
}
