/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package org.jvnet.substance.utils.menu;

import java.awt.*;
import java.awt.event.KeyEvent;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;

import javax.swing.*;
import javax.swing.plaf.ButtonUI;

import org.jvnet.substance.SubstanceMenu;

/**
 * A collection of ugly finctions from
 * {@link javax.swing.plaf.basic.BasicMenuItemUI} since they are private. This
 * class is <b>for internal use only</b>.
 * 
 * @author Kirill Grouchnikov
 */
public class MenuUtilities {
	public static class MenuPropertyListener implements PropertyChangeListener {
		protected JMenuItem menuItem;

		public MenuPropertyListener(JMenuItem menuItem) {
			this.menuItem = menuItem;
		}

		public void install() {
			this.menuItem.addPropertyChangeListener(this);
		}

		public void uninstall() {
			this.menuItem.removePropertyChangeListener(this);
		}

		public void propertyChange(PropertyChangeEvent evt) {
			if (!evt.getPropertyName().equals(
					SubstanceMenu.SINGLE_MENU_TEXT_OFFSET))
				SwingUtilities.invokeLater(new Runnable() {
					public void run() {
						menuItem.putClientProperty(
								SubstanceMenu.SINGLE_MENU_TEXT_OFFSET, null);
						if (menuItem.getParent() instanceof JPopupMenu) {
							((JPopupMenu) menuItem.getParent())
									.putClientProperty(
											SubstanceMenu.SINGLE_MENU_TEXT_OFFSET,
											null);
						}
					}
				});
		}

	}

	public static class MenuLayoutInfo {
		public Rectangle viewRect;
		public Rectangle iconRect;
		public Rectangle checkIconRect;
		public Rectangle textRect;
	}

	/* Client Property keys for the icon and text maximal offsets */
	static final StringBuffer MAX_ICON_OFFSET = new StringBuffer(
			"maxIconOffset");

	static final StringBuffer MAX_TEXT_OFFSET = new StringBuffer(
			"maxTextOffset");

	public static int getTextOffset(Graphics g, Component menuItemParent) {
		if (!(menuItemParent instanceof JPopupMenu)) {
			return 0;
		}

		return getTextOffset(g, null, menuItemParent);
		//		
		// JPopupMenu popupMenu = (JPopupMenu) menuItemParent;
		// int maxOffset = -1;
		// for (int i = 0; i < popupMenu.getComponentCount(); i++) {
		// Component comp = popupMenu.getComponent(i);
		// if (comp instanceof JMenuItem) {
		// JMenuItem childMenuItem = (JMenuItem) comp;
		// SubstanceMenu ui = (SubstanceMenu) childMenuItem.getUI();
		// maxOffset = Math.max(maxOffset, MenuUtilities.getTextOffset(g,
		// childMenuItem, ui.getAcceleratorFont(), ui
		// .getCheckIcon(), ui.getArrowIcon(), ui
		// .getDefaultTextIconGap()));
		// }
		// }
		// return maxOffset;
	}

	private static int getSingleItemTextOffset(Graphics g, JMenuItem menuItem) {
		// Check if we have this property on the menu item.
		Object prop = menuItem
				.getClientProperty(SubstanceMenu.SINGLE_MENU_TEXT_OFFSET);
		if (prop instanceof Integer) {
			// This property is reset on every property change
			// to the menu item. If the property exists, nothing
			// has changed since the last time it was computed.
			return (Integer) prop;
		}

		SubstanceMenu ui = (SubstanceMenu) menuItem.getUI();
		int result = MenuUtilities.getTextOffset(g, menuItem, ui
				.getAcceleratorFont(), ui.getCheckIcon(), ui.getArrowIcon(), ui
				.getDefaultTextIconGap());
		// Now that the offset has been computed, set it on the menu item.
		menuItem.putClientProperty(SubstanceMenu.SINGLE_MENU_TEXT_OFFSET,
				result);
		return result;
	}

	public static int getTextOffset(Graphics g, JMenuItem menuItem) {
		return getTextOffset(g, menuItem, menuItem.getParent());
	}

	public static int getTextOffset(Graphics g, JMenuItem menuItem,
			Component parent) {
		if (!(parent instanceof JPopupMenu)) {
			return getSingleItemTextOffset(g, menuItem);
		}

		// test if there's a computed property on the parent
		JPopupMenu popupMenu = (JPopupMenu) parent;
		Object prop = popupMenu
				.getClientProperty(SubstanceMenu.SINGLE_MENU_TEXT_OFFSET);
		if (prop instanceof Integer) {
			// This property is reset on every change to both the
			// popup (adding / removing child menu component) and
			// and each child component (on any property change).
			// So, if this property exists on the popup menu, nothing
			// has changed since the last time the offset was computed.
			return (Integer) prop;
		}
		int maxTextOffset = -1;
		int maxIconWidth = 0;
		int maxIconX = 0;
		int maxCheckIconX = 0;
		int maxCheckIconWidth = 0;
		// int maxViewRect = 0;
		for (int i = 0; i < popupMenu.getComponentCount(); i++) {
			Component comp = popupMenu.getComponent(i);
			if (comp instanceof JMenuItem) {
				JMenuItem childMenuItem = (JMenuItem) comp;
				ButtonUI bui = childMenuItem.getUI();
				if (!(bui instanceof SubstanceMenu))
					continue;
				SubstanceMenu ui = (SubstanceMenu) bui;
				MenuLayoutInfo mli = MenuUtilities.getMenuLayoutInfo(g,
						childMenuItem, ui.getAcceleratorFont(), ui
								.getCheckIcon(), ui.getArrowIcon(), ui
								.getDefaultTextIconGap());

				if (popupMenu.getComponentOrientation().isLeftToRight()) {
					if (mli.textRect != null) {
						maxTextOffset = Math.max(maxTextOffset, mli.textRect.x);
					}
					if (mli.iconRect != null) {
						maxIconWidth = Math
								.max(maxIconWidth, mli.iconRect.width
										+ ui.getDefaultTextIconGap());
						maxIconX = Math.max(maxIconX, mli.iconRect.x);
					}
					if (mli.checkIconRect != null) {
						maxCheckIconWidth = Math.max(maxCheckIconWidth,
								mli.checkIconRect.width + 2
										* ui.getDefaultTextIconGap());
						maxCheckIconX = Math.max(maxCheckIconX,
								mli.checkIconRect.x);
					}
				} else {
					if (mli.textRect != null) {
						maxTextOffset = Math.max(maxTextOffset, mli.viewRect.x
								+ mli.viewRect.width
								- (mli.textRect.x + mli.textRect.width));
					}
					if (mli.iconRect != null) {
						maxIconWidth = Math.max(maxIconWidth,
								mli.iconRect.width + 2
										* ui.getDefaultTextIconGap());
						maxIconX = Math.max(maxIconX, mli.iconRect.x
								+ mli.iconRect.width);
					}
					if (mli.checkIconRect != null) {
						maxCheckIconWidth = Math.max(maxCheckIconWidth,
								mli.checkIconRect.width
										+ ui.getDefaultTextIconGap());
						maxCheckIconX = Math.max(maxCheckIconX,
								mli.checkIconRect.x + mli.checkIconRect.width);
					}
					// maxViewRect = Math.max(maxViewRect, mli.viewRect.x
					// + mli.viewRect.width);
				}

				//
				// int childResult = MenuUtilities.getTextOffset(g,
				// childMenuItem,
				// ui.getAcceleratorFont(), ui.getCheckIcon(), ui
				// .getArrowIcon(), ui.getDefaultTextIconGap());
				// if (maxOffset < childResult)
				// maxOffset = childResult;
			}
		}
		// Now that the offset is computed, store it on the
		// popup menu.
		int maxOffset = Math.max(maxTextOffset, maxIconWidth
				+ maxCheckIconWidth);
		if (popupMenu.getComponentOrientation().isLeftToRight()) {
			maxOffset = Math.max(maxOffset, maxIconX + maxIconWidth);
			maxOffset = Math.max(maxOffset, maxCheckIconX + maxCheckIconWidth);
		} else {
			// maxOffset = Math.max(maxOffset, maxViewRect
			// - (maxIconX - maxIconWidth));
		}
		popupMenu.putClientProperty(SubstanceMenu.SINGLE_MENU_TEXT_OFFSET,
				maxOffset);
		return maxOffset;
	}

	public static int getTextOffset(Graphics g, JMenuItem menuItem,
			Font acceleratorFont, Icon checkIcon, Icon arrowIcon,
			int defaultTextIconGap) {

		// Dimension size = b.getSize();
		// int menuWidth = menuItem.getWidth();
		// int menuHeight = menuItem.getHeight();
		Insets i = menuItem.getInsets();

		Rectangle iconRect = new Rectangle();
		Rectangle textRect = new Rectangle();
		Rectangle acceleratorRect = new Rectangle();
		Rectangle checkIconRect = new Rectangle();
		Rectangle arrowIconRect = new Rectangle();
		Rectangle viewRect = new Rectangle(Short.MAX_VALUE, Short.MAX_VALUE);

		int menuWidth = menuItem.getWidth();
		int menuHeight = menuItem.getHeight();
		if ((menuWidth > 0) && (menuHeight > 0))
			viewRect.setBounds(0, 0, menuWidth, menuHeight);

		// fix for defect 108 - failed on b62 of Mustang and later on Hebrew
		// localized menu strings.
		// viewRect.setBounds(0, 0, menuWidth, menuHeight);

		viewRect.x += i.left;
		viewRect.y += i.top;
		viewRect.width -= (i.right + viewRect.x);
		viewRect.height -= (i.bottom + viewRect.y);

		Font f = menuItem.getFont();
		g.setFont(f);
		FontMetrics fm = g.getFontMetrics(f);
		FontMetrics fmAccel = g.getFontMetrics(acceleratorFont);

		// get Accelerator text
		KeyStroke accelerator = menuItem.getAccelerator();
		String acceleratorText = "";
		if (accelerator != null) {
			int modifiers = accelerator.getModifiers();
			if (modifiers > 0) {
				acceleratorText = KeyEvent.getKeyModifiersText(modifiers);
				// acceleratorText += "-";
				acceleratorText += UIManager
						.getString("MenuItem.acceleratorDelimiter");
			}

			int keyCode = accelerator.getKeyCode();
			if (keyCode != 0) {
				acceleratorText += KeyEvent.getKeyText(keyCode);
			} else {
				acceleratorText += accelerator.getKeyChar();
			}
		}

		// layout the text and icon
		layoutMenuItem(menuItem, fm, menuItem.getText(), fmAccel,
				acceleratorText, menuItem.getIcon(), checkIcon, arrowIcon,
				menuItem.getVerticalAlignment(), menuItem
						.getHorizontalAlignment(), menuItem
						.getVerticalTextPosition(), menuItem
						.getHorizontalTextPosition(), viewRect, iconRect,
				textRect, acceleratorRect, checkIconRect, arrowIconRect,
				menuItem.getText() == null ? 0 : defaultTextIconGap,
				defaultTextIconGap);

		if (menuItem.getComponentOrientation().isLeftToRight()) {
			return textRect.x;
		} else {
			// int result = viewRect.x + viewRect.width;
			// if (iconRect.width != 0) {
			// result -= (iconRect.x + defaultTextIconGap);
			// }
			// else {
			// if (checkIconRect.width != 0) {
			// result -= (checkIconRect.x + defaultTextIconGap);
			// }
			// }
			return viewRect.x + viewRect.width - iconRect.x
					- ((iconRect.width != 0) ? 0 : defaultTextIconGap);
		}
	}

	private static MenuLayoutInfo getMenuLayoutInfo(Graphics g,
			JMenuItem menuItem, Font acceleratorFont, Icon checkIcon,
			Icon arrowIcon, int defaultTextIconGap) {

		// Dimension size = b.getSize();
		// int menuWidth = menuItem.getWidth();
		// int menuHeight = menuItem.getHeight();
		Insets i = menuItem.getInsets();

		Rectangle iconRect = new Rectangle(0, 0, 0, 0);
		Rectangle textRect = new Rectangle(0, 0, 0, 0);
		Rectangle acceleratorRect = new Rectangle(0, 0, 0, 0);
		Rectangle checkIconRect = new Rectangle(0, 0, 0, 0);
		Rectangle arrowIconRect = new Rectangle(0, 0, 0, 0);
		Rectangle viewRect = new Rectangle(Short.MAX_VALUE, Short.MAX_VALUE);

		int menuWidth = menuItem.getWidth();
		int menuHeight = menuItem.getHeight();
		if ((menuWidth > 0) && (menuHeight > 0))
			viewRect.setBounds(0, 0, menuWidth, menuHeight);

		// fix for defect 108 - failed on b62 of Mustang and later on Hebrew
		// localized menu strings.
		// viewRect.setBounds(0, 0, menuWidth, menuHeight);

		viewRect.x += i.left;
		viewRect.y += i.top;
		viewRect.width -= (i.right + viewRect.x);
		viewRect.height -= (i.bottom + viewRect.y);

		Font f = menuItem.getFont();
		g.setFont(f);
		FontMetrics fm = g.getFontMetrics(f);
		FontMetrics fmAccel = g.getFontMetrics(acceleratorFont);

		// get Accelerator text
		KeyStroke accelerator = menuItem.getAccelerator();
		String acceleratorText = "";
		if (accelerator != null) {
			int modifiers = accelerator.getModifiers();
			if (modifiers > 0) {
				acceleratorText = KeyEvent.getKeyModifiersText(modifiers);
				// acceleratorText += "-";
				acceleratorText += UIManager
						.getString("MenuItem.acceleratorDelimiter");
			}

			int keyCode = accelerator.getKeyCode();
			if (keyCode != 0) {
				acceleratorText += KeyEvent.getKeyText(keyCode);
			} else {
				acceleratorText += accelerator.getKeyChar();
			}
		}

		// layout the text and icon
		layoutMenuItem(menuItem, fm, menuItem.getText(), fmAccel,
				acceleratorText, menuItem.getIcon(), checkIcon, arrowIcon,
				menuItem.getVerticalAlignment(), menuItem
						.getHorizontalAlignment(), menuItem
						.getVerticalTextPosition(), menuItem
						.getHorizontalTextPosition(), viewRect, iconRect,
				textRect, acceleratorRect, checkIconRect, arrowIconRect,
				menuItem.getText() == null ? 0 : defaultTextIconGap,
				defaultTextIconGap);

		MenuLayoutInfo mlInfo = new MenuLayoutInfo();
		mlInfo.checkIconRect = (checkIconRect.width > 0) ? checkIconRect : null;
		mlInfo.iconRect = (iconRect.width > 0) ? iconRect : null;
		mlInfo.textRect = (textRect.width > 0) ? textRect : null;
		mlInfo.viewRect = (viewRect.width > 0) ? viewRect : null;

		return mlInfo;
		//		
		// if (menuItem.getComponentOrientation().isLeftToRight()) {
		// return textRect.x;
		// } else {
		// // int result = viewRect.x + viewRect.width;
		// // if (iconRect.width != 0) {
		// // result -= (iconRect.x + defaultTextIconGap);
		// // }
		// // else {
		// // if (checkIconRect.width != 0) {
		// // result -= (checkIconRect.x + defaultTextIconGap);
		// // }
		// // }
		// return viewRect.x + viewRect.width - iconRect.x
		// - ((iconRect.width != 0) ? 0 : defaultTextIconGap);
		// }
	}

	/**
	 * Compute and return the location of the icons origin, the location of
	 * origin of the text baseline, and a possibly clipped version of the
	 * compound labels string. Locations are computed relative to the viewRect
	 * rectangle.
	 */
	private static String layoutMenuItem(JMenuItem menuItem, FontMetrics fm,
			String text, FontMetrics fmAccel, String acceleratorText,
			Icon icon, Icon checkIcon, Icon arrowIcon, int verticalAlignment,
			int horizontalAlignment, int verticalTextPosition,
			int horizontalTextPosition, Rectangle viewRect, Rectangle iconRect,
			Rectangle textRect, Rectangle acceleratorRect,
			Rectangle checkIconRect, Rectangle arrowIconRect, int textIconGap,
			int menuItemGap) {

		SwingUtilities.layoutCompoundLabel(menuItem, fm, text, icon,
				verticalAlignment, horizontalAlignment, verticalTextPosition,
				horizontalTextPosition, viewRect, iconRect, textRect,
				textIconGap);

		/*
		 * Initialize the acceelratorText bounds rectangle textRect. If a null
		 * or and empty String was specified we substitute "" here and use
		 * 0,0,0,0 for acceleratorTextRect.
		 */
		if ((acceleratorText == null) || acceleratorText.equals("")) {
			acceleratorRect.width = acceleratorRect.height = 0;
			acceleratorText = "";
		} else {
			acceleratorRect.width = fmAccel.stringWidth(acceleratorText);
			acceleratorRect.height = fmAccel.getHeight();
		}

		/*
		 * Initialize the checkIcon bounds rectangle's width & height.
		 */

		if (useCheckAndArrow(menuItem)) {
			if (checkIcon != null) {
				checkIconRect.width = checkIcon.getIconWidth();
				checkIconRect.height = checkIcon.getIconHeight();
			} else {
				checkIconRect.width = checkIconRect.height = 0;
			}

			/*
			 * Initialize the arrowIcon bounds rectangle width & height.
			 */

			if (arrowIcon != null) {
				arrowIconRect.width = arrowIcon.getIconWidth();
				arrowIconRect.height = arrowIcon.getIconHeight();
			} else {
				arrowIconRect.width = arrowIconRect.height = 0;
			}
		}

		Rectangle labelRect = iconRect.union(textRect);
		if (menuItem.getComponentOrientation().isLeftToRight()) {
			textRect.x += menuItemGap;
			iconRect.x += menuItemGap;

			// Position the Accelerator text rect
			acceleratorRect.x = viewRect.x + viewRect.width
					- arrowIconRect.width - menuItemGap - acceleratorRect.width;

			// Position the Check and Arrow Icons
			if (useCheckAndArrow(menuItem)) {
				checkIconRect.x = viewRect.x + menuItemGap;
				textRect.x += menuItemGap + checkIconRect.width;
				iconRect.x += menuItemGap + checkIconRect.width;
				arrowIconRect.x = viewRect.x + viewRect.width - menuItemGap
						- arrowIconRect.width;
			}
		} else {
			textRect.x -= menuItemGap;
			iconRect.x -= menuItemGap;

			// Position the Accelerator text rect
			acceleratorRect.x = viewRect.x + arrowIconRect.width + menuItemGap;

			// Position the Check and Arrow Icons
			if (useCheckAndArrow(menuItem)) {
				checkIconRect.x = viewRect.x + viewRect.width - menuItemGap
						- checkIconRect.width;
				textRect.x -= menuItemGap + checkIconRect.width;
				iconRect.x -= menuItemGap + checkIconRect.width;
				arrowIconRect.x = viewRect.x + menuItemGap;
			}
		}

		// Align the accelertor text and the check and arrow icons vertically
		// with the center of the label rect.
		acceleratorRect.y = labelRect.y + (labelRect.height / 2)
				- (acceleratorRect.height / 2);
		if (useCheckAndArrow(menuItem)) {
			arrowIconRect.y = labelRect.y + (labelRect.height / 2)
					- (arrowIconRect.height / 2);
			checkIconRect.y = labelRect.y + (labelRect.height / 2)
					- (checkIconRect.height / 2);
		}

		/*
		 * System.out.println("Layout: text="+menuItem.getText()+"\n\tv="
		 * +viewRect+"\n\tc="+checkIconRect+"\n\ti="
		 * +iconRect+"\n\tt="+textRect+"\n\tacc="
		 * +acceleratorRect+"\n\ta="+arrowIconRect+"\n");
		 */

		return text;
	}

	/*
	 * Returns false if the component is a JMenu and it is a top level menu (on
	 * the menubar).
	 */
	private static boolean useCheckAndArrow(JMenuItem menuItem) {
		boolean b = true;
		if ((menuItem instanceof JMenu)
				&& (((JMenu) menuItem).isTopLevelMenu())) {
			b = false;
		}
		return b;
	}

	public static void lineupTextRectangle(Graphics g, JMenuItem menuItem,
			Rectangle textRect, int defaultTextIconGap) {
		int textOffset = MenuUtilities.getTextOffset(g, menuItem);
		if (menuItem.getComponentOrientation().isLeftToRight()) {
			textRect.setLocation(textOffset, textRect.y);
		} else {
			int loc = menuItem.getWidth() - textOffset - textRect.width
					- defaultTextIconGap;
			textRect.setLocation(loc, textRect.y);
		}

	}
}
