/*
 * Copyright (c) 2005-2007 Substance Kirill Grouchnikov. All Rights Reserved.
 *
 * Redistribution and use in source and binary forms, with or without 
 * modification, are permitted provided that the following conditions are met:
 * 
 *  o Redistributions of source code must retain the above copyright notice, 
 *    this list of conditions and the following disclaimer. 
 *     
 *  o Redistributions in binary form must reproduce the above copyright notice, 
 *    this list of conditions and the following disclaimer in the documentation 
 *    and/or other materials provided with the distribution. 
 *     
 *  o Neither the name of Substance Kirill Grouchnikov nor the names of 
 *    its contributors may be used to endorse or promote products derived 
 *    from this software without specific prior written permission. 
 *     
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" 
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, 
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR 
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, 
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; 
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, 
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE 
 * OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, 
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE. 
 */
package test.check;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.*;
import java.util.List;

import javax.swing.*;
import javax.swing.border.TitledBorder;

import org.jvnet.substance.SubstanceLookAndFeel;
import org.jvnet.substance.painter.SpecularGradientPainter;
import org.jvnet.substance.theme.*;
import org.jvnet.substance.theme.SubstanceTheme.ThemeKind;

import test.Check;

/**
 * Test application panel for testing {@link JButton} component under various
 * {@link SubstanceMixTheme} combinations.
 * 
 * @author Kirill Grouchnikov
 */
public class MixedButtonsPanel extends JPanel {
	/**
	 * Contains all buttons that use bright themes.
	 */
	protected List<JButton> brightMixed;

	/**
	 * Contains all buttons that use cold themes.
	 */
	protected List<JButton> coldMixed;

	/**
	 * Maps the buttons to the color distances between the component mix themes.
	 */
	protected Map<JButton, Integer> colorDistances;

	/**
	 * Current page number.
	 */
	protected int pageNumber;

	/**
	 * Number of buttons per page.
	 */
	protected int pageSize = 40;

	/**
	 * Panel for bright buttons.
	 */
	protected JPanel brightButtons;

	/**
	 * Panel for cold buttons.
	 */
	protected JPanel coldButtons;

	/**
	 * Creates a new panel with buttons using mixed themes.
	 */
	public MixedButtonsPanel() {
		this.setLayout(new BorderLayout());

		this.brightMixed = new ArrayList<JButton>();
		this.coldMixed = new ArrayList<JButton>();
		for (ThemeInfo theme : SubstanceLookAndFeel.getAllThemes().values()) {
			if (theme.getThemeKind() == ThemeKind.MIXED)
				continue;
			if (theme.getThemeKind() == ThemeKind.DARK)
				continue;
			if (theme.getThemeKind() == ThemeKind.INVERTED)
				continue;
			if (theme.getThemeKind() == ThemeKind.NEGATED)
				continue;
			for (ThemeInfo theme2 : SubstanceLookAndFeel.getAllThemes()
					.values()) {
				if (theme.getThemeKind() != theme2.getThemeKind())
					continue;
				if (theme.getDisplayName().equals(theme2.getDisplayName()))
					continue;

				try {
					JButton mixedButton = new JButton("Test");
					SubstanceMixTheme mixedTheme = new SubstanceMixTheme(
							SubstanceTheme.createInstance(theme),
							SubstanceTheme.createInstance(theme2));
					mixedButton.putClientProperty(
							SubstanceLookAndFeel.THEME_PROPERTY, mixedTheme);
					mixedButton.putClientProperty(
							SubstanceLookAndFeel.PAINT_ACTIVE_PROPERTY,
							Boolean.TRUE);
					mixedButton.putClientProperty(
							SubstanceLookAndFeel.GRADIENT_PAINTER_PROPERTY,
							SpecularGradientPainter.class.getName());
					mixedButton.setToolTipText(mixedTheme.getDisplayName());
					// mixedButton.putClientProperty(
					// SubstanceLookAndFeel.BUTTON_NO_MIN_SIZE_PROPERTY,
					// Boolean.TRUE);
					// mixedButton.setPreferredSize(new Dimension(45, 16));
					if (theme.getThemeKind() == ThemeKind.BRIGHT)
						brightMixed.add(mixedButton);
					if (theme.getThemeKind() == ThemeKind.COLD)
						coldMixed.add(mixedButton);
				} catch (Exception exc) {
				}
			}
		}

		this.colorDistances = new HashMap<JButton, Integer>();
		for (JButton bright : brightMixed) {
			SubstanceMixTheme mix = (SubstanceMixTheme) bright
					.getClientProperty(SubstanceLookAndFeel.THEME_PROPERTY);
			SubstanceTheme[] origThemes = ((SubstanceMixTheme) mix)
					.getOriginalThemes();
			SubstanceTheme theme1 = origThemes[0];
			SubstanceTheme theme2 = origThemes[origThemes.length - 1];
			Color c1 = theme1.getColorScheme().getMidColor();
			Color c2 = theme2.getColorScheme().getMidColor();
			int dr = c1.getRed() - c2.getRed();
			int dg = c1.getGreen() - c2.getGreen();
			int db = c1.getBlue() - c2.getBlue();

			double distance = Math.sqrt(dr * dr + dg * dg + db * db);
			colorDistances.put(bright, (int) distance);
		}
		for (JButton cold : coldMixed) {
			SubstanceMixTheme mix = (SubstanceMixTheme) cold
					.getClientProperty(SubstanceLookAndFeel.THEME_PROPERTY);
			SubstanceTheme[] origThemes = ((SubstanceMixTheme) mix)
					.getOriginalThemes();
			SubstanceTheme theme1 = origThemes[0];
			SubstanceTheme theme2 = origThemes[origThemes.length - 1];
			Color c1 = theme1.getColorScheme().getMidColor();
			Color c2 = theme2.getColorScheme().getMidColor();
			int dr = c1.getRed() - c2.getRed();
			int dg = c1.getGreen() - c2.getGreen();
			int db = c1.getBlue() - c2.getBlue();

			double distance = Math.sqrt(dr * dr + dg * dg + db * db);
			colorDistances.put(cold, (int) distance);
		}

		Comparator<JButton> mixComparator = new Comparator<JButton>() {
			public int compare(JButton o1, JButton o2) {
				int dist1 = colorDistances.get(o1);
				int dist2 = colorDistances.get(o2);
				if (dist1 == dist2) {
					SubstanceMixTheme mix1 = (SubstanceMixTheme) o1
							.getClientProperty(SubstanceLookAndFeel.THEME_PROPERTY);
					SubstanceMixTheme mix2 = (SubstanceMixTheme) o2
							.getClientProperty(SubstanceLookAndFeel.THEME_PROPERTY);
					return mix1.getDisplayName().compareTo(
							mix2.getDisplayName());
				} else
					return dist1 - dist2;
			}
		};

		Collections.sort(brightMixed, mixComparator);
		// for (JButton bright : brightMixed) {
		// bright.setText("" + colorDistances.get(bright));
		// this.add(bright);
		// }
		Collections.sort(coldMixed, mixComparator);
		// for (JButton cold : coldMixed) {
		// cold.setText("" + colorDistances.get(cold));
		// this.add(cold);
		// }

		this.brightButtons = new JPanel(new FlowLayout());
		this.coldButtons = new JPanel(new FlowLayout());

		JPanel controlPanel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
		JButton nextPage = new JButton("Next " + this.pageSize);
		JButton prevPage = new JButton("Prev " + this.pageSize);

		nextPage.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				pageNumber++;
				syncPage();
			}
		});

		prevPage.addActionListener(new ActionListener() {
			public void actionPerformed(ActionEvent e) {
				pageNumber--;
				syncPage();
			}
		});

		this.pageNumber = 0;
		syncPage();

		controlPanel.add(prevPage);
		controlPanel.add(nextPage);

		this.add(controlPanel, BorderLayout.SOUTH);

		JPanel mainPanel = new JPanel(new GridLayout(1, 2));
		mainPanel.add(this.brightButtons);
		mainPanel.add(this.coldButtons);

		this.add(mainPanel, BorderLayout.CENTER);

		JPanel pillButtonPanel = new JPanel();
		JPanel pillButtonPanelTop = new JPanel(new FlowLayout(FlowLayout.LEFT));
		JPanel pillButtonPanelBottom = new JPanel(new FlowLayout(
				FlowLayout.LEFT));
		// pillButtonPanel.setLayout(new BoxLayout(pillButtonPanel,
		// BoxLayout.Y_AXIS));
		pillButtonPanel.setLayout(new GridLayout(2, 1));
		pillButtonPanel.add(pillButtonPanelTop);
		pillButtonPanel.add(pillButtonPanelBottom);

		pillButtonPanel.setBorder(new TitledBorder("Presets"));

		// pillButtonPanel.add(Box.createVerticalStrut(3));
		pillButtonPanelTop.add(Check.getPill(new SubstanceAquaTheme(),
				new SubstanceLightAquaTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceAquaTheme(),
				new SubstanceBottleGreenTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceBarbyPinkTheme(),
				new SubstanceRaspberryTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceBottleGreenTheme(),
				new SubstanceLimeGreenTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceBrownTheme(),
				new SubstanceSunGlareTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceSunsetTheme(),
				new SubstanceOrangeTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceLightAquaTheme(),
				new SubstancePurpleTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelTop.add(Check.getPill(new SubstanceBrownTheme(),
				new SubstanceTerracottaTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));

		pillButtonPanelBottom
				.add(Check.getMultiPill(new SubstanceAquaTheme(),
						new SubstanceLimeGreenTheme(),
						new SubstanceBottleGreenTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelBottom.add(Check.getMultiPill(
				new SubstanceBarbyPinkTheme(), new SubstancePurpleTheme(),
				new SubstanceRaspberryTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelBottom.add(Check.getMultiPill(
				new SubstanceRaspberryTheme(), new SubstanceSunsetTheme(),
				new SubstanceOrangeTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelBottom.add(Check.getMultiPill(new SubstanceBrownTheme(),
				new SubstanceSunGlareTheme(), new SubstanceOrangeTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelBottom.add(Check.getMultiPill(new SubstanceAquaTheme(),
				new SubstanceLightAquaTheme(), new SubstancePurpleTheme(),
				new SubstanceBarbyPinkTheme()));
		// pillButtonPanel.add(Box.createVerticalStrut(2));
		pillButtonPanelBottom.add(Check.getMultiPill(new SubstanceBrownTheme(),
				new SubstanceTerracottaTheme(), new SubstanceSunGlareTheme(),
				new SubstanceOrangeTheme(), new SubstanceSunsetTheme()));

		this.add(pillButtonPanel, BorderLayout.NORTH);
	}

	/**
	 * Synchronizes panel contents.
	 */
	protected void syncPage() {
		int startIndex = this.pageNumber * this.pageSize;
		int endIndex = startIndex + this.pageSize;

		this.brightButtons.removeAll();
		this.coldButtons.removeAll();

		for (int i = startIndex; i < endIndex; i++) {
			if ((i >= 0) && (i < this.brightMixed.size())) {
				JButton bright = brightMixed.get(i);
				bright.setText("" + colorDistances.get(bright));
				this.brightButtons.add(bright);
			}
			if ((i >= 0) && (i < this.coldMixed.size())) {
				JButton cold = coldMixed.get(i);
				cold.setText("" + colorDistances.get(cold));
				this.coldButtons.add(cold);
			}
		}
		this.revalidate();
		this.repaint();
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.JComponent#isOpaque()
	 */
	@Override
	public boolean isOpaque() {
		return false;
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see javax.swing.JComponent#paint(java.awt.Graphics)
	 */
	@Override
	public void paint(Graphics g) {
		long time0 = System.currentTimeMillis();
		super.paint(g);
		long time1 = System.currentTimeMillis();
		Check.out("Time to paint mixed is " + (time1 - time0));
	}
}
