package tools.jitterbug;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.net.URL;

import javax.imageio.ImageIO;
import javax.swing.*;
import javax.swing.event.*;

import org.jvnet.substance.api.SubstanceColorScheme;
import org.jvnet.substance.utils.SubstanceColorSchemeUtilities;

import com.jgoodies.forms.builder.DefaultFormBuilder;
import com.jgoodies.forms.layout.FormLayout;

public class JColorSchemeComponent extends JPanel {
	private JCheckBox isLight;
	private JTextField name;
	private JColorComponent ultraLight;
	private JColorComponent extraLight;
	private JColorComponent light;
	private JColorComponent mid;
	private JColorComponent dark;
	private JColorComponent ultraDark;
	private JColorComponent foreground;
	private ButtonGroup bg;
	private BufferedImage exclamation;

	public JColorSchemeComponent() {
		FormLayout layout = new FormLayout("fill:pref", "");
		DefaultFormBuilder formBuilder = new DefaultFormBuilder(layout, this);

		this.bg = new ButtonGroup();

		this.ultraLight = createColorComponent("ultra light");
		this.extraLight = createColorComponent("extra light");
		this.light = createColorComponent("light");
		this.mid = createColorComponent("mid");
		this.dark = createColorComponent("dark");
		this.ultraDark = createColorComponent("ultra dark");
		this.foreground = createColorComponent("foreground");

		JPanel header = new JPanel();
		FormLayout hLayout = new FormLayout(
				"fill:pref,4dlu,fill:pref,2dlu,fill:pref:grow", "");
		DefaultFormBuilder headerBuilder = new DefaultFormBuilder(hLayout,
				header);
		isLight = new JCheckBox("light");
		isLight.setSelected(true);
		isLight.setFocusable(false);
		name = new JTextField("") {
			@Override
			protected void paintBorder(Graphics g) {
				super.paintBorder(g);
				if ((this.getText() == null)
						|| (this.getText().trim().length() == 0)) {
					g.drawImage(exclamation, getWidth()
							- exclamation.getWidth(), 0, null);
				}
			}
		};
		name.setCaretPosition(name.getDocument().getLength());
		name.moveCaretPosition(0);
		headerBuilder.append(isLight);
		headerBuilder.append(new JLabel("Name:"));
		headerBuilder.append(name);

		name.getDocument().addDocumentListener(new DocumentListener() {
			@Override
			public void changedUpdate(DocumentEvent e) {
				fireStateChanged();
			}

			@Override
			public void insertUpdate(DocumentEvent e) {
				fireStateChanged();
			}

			@Override
			public void removeUpdate(DocumentEvent e) {
				fireStateChanged();
			}
		});
		isLight.getModel().addChangeListener(new ChangeListener() {
			@Override
			public void stateChanged(ChangeEvent e) {
				fireStateChanged();
			}
		});

		formBuilder.append(header);

		formBuilder.append(ultraLight);
		formBuilder.append(extraLight);
		formBuilder.append(light);
		formBuilder.append(mid);
		formBuilder.append(dark);
		formBuilder.append(ultraDark);
		formBuilder.append(foreground);

		try {
			this.exclamation = ImageIO.read(JColorSchemeComponent.class
					.getResourceAsStream("/tools/jitterbug/exclamation.png"));
		} catch (Exception exc) {
			exc.printStackTrace();
		}

	}

	private JColorComponent createColorComponent(String label) {
		JColorComponent result = new JColorComponent(label, null);
		result.addPropertyChangeListener(new PropertyChangeListener() {
			@Override
			public void propertyChange(PropertyChangeEvent evt) {
				if ("selectedColor".equals(evt.getPropertyName())) {
					fireStateChanged();
				}
			}
		});
		this.bg.add(result.getRadio());
		return result;
	}

	public void setContent(URL url) {
		try {
			SubstanceColorScheme scheme = SubstanceColorSchemeUtilities
					.getColorScheme(url);
			if (scheme == null) {
				return;
			}

			ultraLight.setColor(scheme.getUltraLightColor());
			extraLight.setColor(scheme.getExtraLightColor());
			light.setColor(scheme.getLightColor());
			mid.setColor(scheme.getMidColor());
			dark.setColor(scheme.getDarkColor());
			ultraDark.setColor(scheme.getUltraDarkColor());
			foreground.setColor(scheme.getForegroundColor());
			isLight.setSelected(!scheme.isDark());
			name.setText(scheme.getDisplayName());

			fireStateChanged();
		} catch (IllegalArgumentException iae) {
			JOptionPane.showMessageDialog(SwingUtilities
					.getWindowAncestor(this), iae.getMessage(),
					"Problem reading color scheme definition",
					JOptionPane.ERROR_MESSAGE);
		}
	}

	public boolean isLight() {
		return this.isLight.isSelected();
	}

	public String getDisplayName() {
		return this.name.getText();
	}

	public Color getUltraLightColor() {
		return ultraLight.getColor();
	}

	public Color getExtraLightColor() {
		return extraLight.getColor();
	}

	public Color getLightColor() {
		return light.getColor();
	}

	public Color getMidColor() {
		return mid.getColor();
	}

	public Color getDarkColor() {
		return dark.getColor();
	}

	public Color getUltraDarkColor() {
		return ultraDark.getColor();
	}

	public Color getForegroundColor() {
		return foreground.getColor();
	}

	public String getEncoded() {
		StringBuffer sb = new StringBuffer();
		sb.append(this.getDisplayName() + " {\n");
		sb.append("\tkind=" + (this.isLight() ? "Light" : "Dark") + "\n");
		sb.append("\tcolorUltraLight=" + this.ultraLight.getEncodedColor()
				+ "\n");
		sb.append("\tcolorExtraLight=" + this.extraLight.getEncodedColor()
				+ "\n");
		sb.append("\tcolorLight=" + this.light.getEncodedColor() + "\n");
		sb.append("\tcolorMid=" + this.mid.getEncodedColor() + "\n");
		sb.append("\tcolorDark=" + this.dark.getEncodedColor() + "\n");
		sb
				.append("\tcolorUltraDark=" + this.ultraDark.getEncodedColor()
						+ "\n");
		sb.append("\tcolorForeground=" + this.foreground.getEncodedColor()
				+ "\n");
		sb.append("}\n");
		return sb.toString();
	}

	/**
	 * Adds the specified change listener to track changes to this ribbon.
	 * 
	 * @param l
	 *            Change listener to add.
	 * @see #removeChangeListener(ChangeListener)
	 */
	public void addChangeListener(ChangeListener l) {
		this.listenerList.add(ChangeListener.class, l);
	}

	/**
	 * Removes the specified change listener from tracking changes to this
	 * ribbon.
	 * 
	 * @param l
	 *            Change listener to remove.
	 * @see #addChangeListener(ChangeListener)
	 */
	public void removeChangeListener(ChangeListener l) {
		this.listenerList.remove(ChangeListener.class, l);
	}

	/**
	 * Notifies all registered listener that the state of this ribbon has
	 * changed.
	 */
	protected void fireStateChanged() {
		// Guaranteed to return a non-null array
		Object[] listeners = this.listenerList.getListenerList();
		// Process the listeners last to first, notifying
		// those that are interested in this event
		ChangeEvent event = new ChangeEvent(this);
		for (int i = listeners.length - 2; i >= 0; i -= 2) {
			if (listeners[i] == ChangeListener.class) {
				((ChangeListener) listeners[i + 1]).stateChanged(event);
			}
		}
	}

	public boolean isDefined() {
		if ((this.name.getText() == null)
				|| (this.name.getText().trim().length() == 0))
			return false;
		if (!this.ultraLight.isDefined())
			return false;
		if (!this.extraLight.isDefined())
			return false;
		if (!this.light.isDefined())
			return false;
		if (!this.mid.isDefined())
			return false;
		if (!this.dark.isDefined())
			return false;
		if (!this.ultraDark.isDefined())
			return false;
		if (!this.foreground.isDefined())
			return false;
		return true;
	}

	public JColorComponent getSelectedColorComponent() {
		if (this.ultraLight.getRadio().isSelected())
			return this.ultraLight;
		if (this.extraLight.getRadio().isSelected())
			return this.extraLight;
		if (this.light.getRadio().isSelected())
			return this.light;
		if (this.mid.getRadio().isSelected())
			return this.mid;
		if (this.dark.getRadio().isSelected())
			return this.dark;
		if (this.ultraDark.getRadio().isSelected())
			return this.ultraDark;
		if (this.foreground.getRadio().isSelected())
			return this.foreground;
		return null;
	}
}
