/*  gtktiemu - a TI89/92/92+ emulator
 *  (c) Copyright 2000-2001, Romain Lievin and Thomas Corvazier
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/timeb.h>
#include <sys/types.h>
#include <unistd.h>

#include "platform.h"
#include "struct.h"
#include "specific.h"
#include "version.h"
#include "misc.h"
#include "files.h"
#include "rcfile.h"

#include "gtk_tiemu_cb.h"
#include "gtk_wizard_dbox.h"
#include "gtk_refresh.h"

#ifdef __WIN32__
# include <windows.h>
#endif

/* 
   The ti92/89 should approximately execute NB_INSTRUCTIONS_PER_LOOP in 
   TIME_LIMIT milliseconds
   If you think this values are a bit too big, you can slow down 
   the emulator by changing them 
*/
#define LCD_FREQUENCY            4      // in hertz
#define NB_INSTRUCTIONS_PER_LOOP 50000
#define TIME_LIMIT               30


gint romOk = 0;
extern int wizard_ok;
extern gchar *wizard_rom;


void updateLcd(void) 
{
  //set_screen_ptr(getLcdPtr());
  //update_screen();
}

Options options;        // general gtktiemu options
InstPaths inst_paths;   // installation paths
InfoUpdate info_update; // pbar, msg_box, refresh, ...

int gtk_main_window(void);

/* Main function */		
int main(int argc, char **argv) 
{
  struct timeb tCurrentTime,tLastTime;
  unsigned long int iCurrentTime,iLastTime; // 64 bits integers
  gint res=0;

  /*
    Do primary initializations
  */

  // Display program version
  version();

  // Initialize platform independant paths
  initialize_paths();

  // Get pointer on libti68k internal parameters
  init_options(&glob_inf);

  // Load default config
  load_default_config();

  // Parse the config file
  read_rc_file();

  // Scan the command line
  scan_cmdline(argc, argv);

  /*
    Initialize i18n support 
  */
  // Init locale & internationalization
  //DISPLAY("Locale to set: <%s>\n", options.locale);
  //DISPLAY("Locale set: <%s>\n", setlocale(LC_ALL, options.locale));
#if defined(__WIN32__)
  //g_win32_getlocale();
# else
  //setlocale (LC_ALL, options.locale));		
#endif
  //gtk_set_locale ();
  //setlocale (LC_ALL, "C");
  bindtextdomain (PACKAGE, inst_paths.locale_dir);	
  textdomain (PACKAGE);
  
  /*
    Check libticables library
  */
  DISPLAY(_("libticables library, version %s\n"),
	  ticable_get_version());
  if(strcmp(ticable_get_version(), LIB_CABLE_VERSION_REQUIRED) < 0)
    {
      DISPLAY("Library version <%s> mini required.\n", 
	      LIB_CABLE_VERSION_REQUIRED);
      exit(-1);
    }

  /*
    Check libticalcs library
  */
  DISPLAY(_("libticalcs library, version %s\n"),
	  ticalc_get_version());
  if(strcmp(ticalc_get_version(), LIB_CALC_VERSION_REQUIRED) < 0)
    {
      DISPLAY("Library version <%s> mini required.\n", 
	      LIB_CALC_VERSION_REQUIRED);
      exit(-1);
    }

  /* 
     Initialize the libticables library 
  */
  ticable_set_param(options.lp);

  /*
    Initialize refresh functions of the libticalcs library
  */
  gt_init_refresh_functions();
  
  /*
    Assign a GUI to the libti68k library and
    define a debugger to use
  */
  set_gui_callbacks();
  ti68k_defineDebugger(enter_gtk_debugger);

  /* 
     Load a ROM image
  */
  if(ti68k_loadRom(options.rom_file)) 
    {
      gtk_init(NULL, NULL);
      gtk_widget_show(create_step1_dbox());
      while(!wizard_ok)
	{
	  while( gtk_events_pending() ) { gtk_main_iteration(); }
	}
      g_free(options.rom_file);
      options.rom_file = g_strdup(wizard_rom);
      g_free(wizard_rom);
      ti68k_loadRom(options.rom_file);
    }

  /*
    Try to load an FLASH update if it exists
  */
  if(strcmp(options.tib_file, ""))
    ti68k_loadTib(options.tib_file);

  /* 
     Init m68k core
  */
  if(ti68k_initLib68k())
    return -1;
  
  /*
    Reload previous state
  */
  ti68k_loadState(options.ram_file);
  
  /*
    Close the start-up console (Win32)
  */
  //ticable_DISPLAY_settings(options.console);
  ticable_DISPLAY_settings(DSP_CLOSE); //DSP_CLOSE

  /*
    Create popup menu: must be called after initLib68k !
  */
  gtk_main_window();
  
  /* 
     Run the emulator itself 
  */
  while (1) 
    {
      // Update calculator display
      updateLcd();

      // Update GUI only if emulator core is suspended
      if(update_gui()) 
	{
	  cbk_update_keys();
	  continue;
	}

      // Run emulator core
      ftime(&tLastTime);
      if( (res = ti68k_doInstructions(NB_INSTRUCTIONS_PER_LOOP)) )
	{  // a bkpt has been encountered
	}
      else 
	{ // normal execution
	  /* added by tom for too fast CPU.
	     Any PII(I) should be able to emulate a 68k CPU at double speed.
	     We are using 64bits of precision for time, so our program 
	     should be able to run after the year 2060 !=) 
	  */
	  ftime(&tCurrentTime);
	  iLastTime    = tLastTime.time*1000+tLastTime.millitm;
	  iCurrentTime = tCurrentTime.time*1000+tCurrentTime.millitm;
	  if ((iCurrentTime - iLastTime) < TIME_LIMIT)
	  {
#if defined(__LINUX__)
	    usleep((TIME_LIMIT - iCurrentTime + iLastTime)*1000);
#elif defined(__WIN32__)
	    Sleep((TIME_LIMIT - iCurrentTime + iLastTime));
#endif
	  }
	}
    }

  /* 
     Close the emulator library and exit 
  */
  // Close m68k core library
  ti68k_closeLib68k();
  
  return 0;
}


/* 
   If GtkTiEmu is compiled in console mode (_CONSOLE), 
   then we use the 'main' entry point.
   If GtkTiEmu is compiled as a windowed application (_WINDOWS), 
   then we use the 'WinMain' entry point.
*/
#if defined(__WIN32__) && defined(_WINDOWS)// && !defined(_CONSOLE)
int APIENTRY WinMain(HINSTANCE hInstance,
                     HINSTANCE hPrevInstance,
                     LPSTR     lpCmdLine,
                     int       nCmdShow)
{
  /* Check whether a TiLP session is already running */
  HANDLE hMutex = CreateMutex(NULL, TRUE, "GtkTiEmu");
  if (GetLastError() == ERROR_ALREADY_EXISTS)
    {
      g_error("GtkTiEmu is already running.");
    }

//  ticable_DISPLAY_settings(DSP_ON);
  
  return main(__argc, __argv);
}
#endif


GtkWidget *window;

/* 
   A GTK callback: displays a popup menu if the auxiliary 
   window has been enabled
*/
gboolean
button_press_event        (GtkWidget       *widget,
                           GdkEventButton  *event,
                           gpointer         user_data)
{
  GtkWidget *menu;

  /* Displays the popup menu */
  if(event->button == 3)
    {
      menu = display_popup_menu();
      gtk_widget_grab_focus(menu);
      gdk_event_get();
      gtk_menu_popup(GTK_MENU(menu), 
		     NULL, NULL, NULL, NULL, 
		     event->button, event->time);
      suspend();
    }
  
  return FALSE;
}

/* 
   The GTK auxiliary window: may not be shown but used for 
   using GTK with SDL.
   A better way should be to use a GtkSDL plugin.
*/
int gtk_main_window(void)
{
  GtkWidget *eventbox;
  GtkWidget *label;
  
  gtk_init (NULL, NULL);
    
  /* The main window */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "GtkTiEmu");
  gtk_widget_set_usize(window, 250, 50);

  /* The event box */
  eventbox = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (window), eventbox);
  GTK_WIDGET_SET_FLAGS (eventbox, GTK_CAN_FOCUS);
  GTK_WIDGET_SET_FLAGS (eventbox, GTK_CAN_DEFAULT);
  gtk_widget_set_events (eventbox, GDK_BUTTON_PRESS_MASK | GDK_KEY_PRESS_MASK);
  gtk_widget_show (eventbox);

  label = gtk_label_new ("Click here to display a menu");
  gtk_container_add (GTK_CONTAINER (eventbox), label);
  gtk_widget_show (label);
  
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (gtk_main_quit),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (eventbox), "button_press_event",
                      GTK_SIGNAL_FUNC (button_press_event),
                      NULL);
  //gtk_widget_grab_focus (eventbox);
  gtk_widget_grab_default (eventbox);

#ifdef EXT_WIN  
  gtk_widget_show_all(window);
#endif

  return 0;
}
