package tijmp.ui;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.management.MemoryMXBean;
import java.lang.management.MemoryUsage;
import javax.swing.BorderFactory;
import javax.swing.JPanel;
import javax.swing.Timer;

/** A panel that shows a graph of the current memory usage. 
 */
public class MemoryGraph extends JPanel implements ActionListener {
    private MemoryMXBean mbean;
    private long[] usage = null;
    private Image image = null;
    private int currentCol = 0;
    private int maxColumns = -1;
    private long currentMaxCommitted = 0;
    private Timer timer;

    public MemoryGraph (MemoryMXBean mbean) {
	this.mbean = mbean;
	setPreferredSize (new Dimension (500, 100));
	setBorder (BorderFactory.createLineBorder (Color.BLACK, 1));
	setOpaque (true);
	setBackground (Color.WHITE);
    }

    public void start () {
	timer = new Timer (1000, this);
	timer.start ();
    }

    public void stop () {
	timer.stop ();
    }

    public void actionPerformed (ActionEvent e) {
	if (!isVisible ())
	    return;
	Insets insets = getInsets ();
	int currentWidth = getWidth () - insets.left - insets.right;	
	int currentHeight = getHeight () - insets.top - insets.bottom;
	if (usage == null || usage.length / 2 != currentWidth) {
	    // TODO: copy old contents...
	    usage = new long[currentWidth * 2];
	    image = createImage (currentWidth, currentHeight);
	    currentCol = 0;
	    maxColumns = currentWidth;
	}
	MemoryUsage heapUsage = mbean.getHeapMemoryUsage ();
	int currentLine = currentCol % maxColumns;
	int index = currentLine * 2;
	usage[index] = heapUsage.getUsed ();
	usage[index + 1] = heapUsage.getCommitted ();
	Graphics g = null;
	try {
	    g = image.getGraphics ();
	    if (usage[index + 1] > currentMaxCommitted) {
		currentMaxCommitted = usage[index + 1];
		doFullRedraw (g, insets, currentHeight, currentWidth);
	    } else {
		drawUsage (g, index, currentLine, currentHeight);
	    }
	} finally {
	    if (g != null)
		g.dispose ();
	}
	try {
	    g = getGraphics ();
	    paintComponent (g);
	} finally {
	    if (g != null)
		g.dispose ();
	}
	currentCol++;
    }

    @Override public void paintComponent (Graphics g) {
	Insets insets = getInsets ();
	int currentLine = currentCol % maxColumns;
	if (currentCol < maxColumns || currentLine == 0) {
	    g.drawImage (image, insets.left, insets.top, null);
	} else {
	    int yMax = image.getHeight (null) + insets.top;
	    int startSecond = insets.left + maxColumns - currentLine;
	    g.drawImage (image, insets.left, insets.top, 
			 startSecond, yMax, 
			 currentLine, 0, 
			 maxColumns, yMax, null);
	    g.drawImage (image, startSecond, insets.top, 
			 startSecond + currentLine, yMax, 
			 0, 0, 
			 currentLine, yMax, null);
	}
    }

    private void doFullRedraw (Graphics g, Insets insets, 
			       int currentHeight, int currentWidth) {
	g.setColor (getBackground ());
	g.fillRect (insets.left, insets.top, currentWidth, currentHeight);
	for (int col = 0; col < currentWidth; col++)
	    drawUsage (g, col * 2, col, currentHeight);
    }
    
    private void drawUsage (Graphics g, int index, int col, int currentHeight) {
	double cmc = currentMaxCommitted;
	if (usage[index + 1] == 0)
	    return;    // no commited memory = no value for column.
	int endH = (int)(currentHeight * (1 - usage[index] / cmc));
	int endC = (int)(currentHeight * (1 - usage[index + 1] / cmc));
	g.setColor (Color.BLUE);
	g.drawLine (col, currentHeight, col, endH);
	g.setColor (Color.PINK);
	g.drawLine (col, endH, col, endC);
	g.setColor (getBackground ());
	g.drawLine (col, endC, col, 0);
    }
}
