package tijmp.ui;

import java.awt.Dimension;
import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.io.IOException;
import java.io.PrintWriter;
import java.io.Writer;
import java.lang.management.LockInfo;
import java.lang.management.MonitorInfo;
import java.lang.management.ThreadInfo;
import java.lang.management.ThreadMXBean;
import java.util.ArrayList;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTabbedPane;
import javax.swing.JTable;
import javax.swing.ListSelectionModel;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;
import javax.swing.table.AbstractTableModel;
import javax.swing.table.TableColumn;
import tijmp.ProfilerHandler;

/** A class to show current memory usage. 
 */
public class ThreadInfoPanel extends JPanel {
    private final ProfilerHandler ph;

    public ThreadInfoPanel (ProfilerHandler ph) {
	this.ph = ph;
	final ThreadMXBean mbean = ph.getThreadMXBean ();
	JLabel ct = new JLabel ("Current Threads: " + 
				mbean.getThreadCount ());
	JLabel dt = new JLabel ("Daemon Threads: " + 
				mbean.getDaemonThreadCount ());
	JLabel st = new JLabel ("Started Threads: " + 
				mbean.getTotalStartedThreadCount ());
	JLabel pt = new JLabel ("Peak Threads: " + 
				mbean.getPeakThreadCount ());	
	JButton update = new JButton ("Update threads");
	final JCheckBox tCpuTime = 
	    new JCheckBox ("Thread Cpu time enabled: ", 
			   mbean.isThreadCpuTimeEnabled ());
	tCpuTime.addItemListener (new ItemListener () {
		public void itemStateChanged (ItemEvent e) {
		    mbean.setThreadCpuTimeEnabled (tCpuTime.isSelected ()); 
		}
	    });
	if (!mbean.isThreadCpuTimeSupported ())
	    tCpuTime.setEnabled (false);
	final JCheckBox tMonitorTime = 
	    new JCheckBox ("Thread contention monitoring: ",
			   mbean.isThreadContentionMonitoringEnabled ());
	tMonitorTime.addItemListener (new ItemListener () {
		public void itemStateChanged (ItemEvent e) {
		    boolean state = tMonitorTime.isSelected ();
		    mbean.setThreadContentionMonitoringEnabled (state);
		}
	    });
	if (!mbean.isThreadContentionMonitoringSupported ())
	    tMonitorTime.setEnabled (false);
	
	setLayout (new GridBagLayout ());
	GridBagConstraints c = new GridBagConstraints ();
	c.fill = GridBagConstraints.HORIZONTAL;
	c.gridx = 0; 
	c.gridy = 0; 
	c.insets = new Insets (2, 2, 2, 2);

	add (ct, c);
	c.gridy++;
	add (dt, c);

	c.gridy = 0;
	c.gridx = 1;
	add (st, c);
	c.gridy++;
	add (pt, c);

	c.gridy = 0;
	c.gridx = 2;
	add (tCpuTime, c);
	c.gridy++;
	add (tMonitorTime, c);
	
	c.gridy = 1;
	c.gridx = 3;
	c.anchor = GridBagConstraints.SOUTHEAST;
	c.fill = GridBagConstraints.NONE;
	add (update, c);

	long[] threadIds = mbean.getAllThreadIds ();
	ThreadInfo[] tinfos = mbean.getThreadInfo (threadIds);
	final TTM threadModel = new TTM (tinfos, mbean);
	final JTable threadTable = new JTable (threadModel);
	setThreadColumnProperties (threadTable);
	threadTable.getRowSorter ().toggleSortOrder (TTM.COL_CPU_TIME);
	threadTable.getRowSorter ().toggleSortOrder (TTM.COL_CPU_TIME);
	JScrollPane spThreads = new JScrollPane (threadTable);
	spThreads.setBorder (BorderFactory.createTitledBorder ("Current Threads"));

	final StackTraceModel stm = new StackTraceModel (null);
	final JTable stackTrace = new JTable (stm);
	JScrollPane spTrace = new JScrollPane (stackTrace);
	setTraceColumnProperties (stackTrace);
	
	final MonitorModel mtm = new MonitorModel (null);
	final JTable monitors = new JTable (mtm);
	JScrollPane spMonitors = new JScrollPane (monitors);

	final LocksModel ltm = new LocksModel (null);
	final JTable locks = new JTable (ltm);
	JScrollPane spLocks = new JScrollPane (locks);

	ListSelectionModel lsm = threadTable.getSelectionModel ();
	lsm.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
	lsm.addListSelectionListener (new ListSelectionListener () {
		public void valueChanged (ListSelectionEvent e) {
		    showStackTrace (threadTable, stm, mtm, mbean);
		}
	    }); 
	lsm.setSelectionInterval (0, 0);

	update.addActionListener (new ActionListener () {
		public void actionPerformed (ActionEvent e) {
		    threadModel.updateThreadInfos ();
		}
	    });

	JTabbedPane tabs = new JTabbedPane ();
	tabs.addTab ("Stack Trace", spTrace);
	tabs.addTab ("Monitors", spMonitors);
	tabs.addTab ("Locks", spLocks);

	c.gridx = 0;
	c.gridy = 3;
	c.gridwidth = 4;
	c.weightx = 1;
	c.weighty = 1;
	c.anchor = GridBagConstraints.NORTH;
	c.fill = GridBagConstraints.BOTH;
	add (spThreads, c);
	c.gridy++;
	c.weighty = 2;
	add (tabs, c);
    }

    private void showStackTrace (JTable threadTable, StackTraceModel stackTrace, 
				 MonitorModel monitors, ThreadMXBean mbean) {
	int row = threadTable.getSelectedRow ();
	if (row >= 0) {
	    row = threadTable.convertRowIndexToModel(row);
	    Long id = (Long)threadTable.getModel ().getValueAt (row, TTM.COL_ID);
	    long[] lids = new long[] { id };
	    ThreadInfo[] tis = mbean.getThreadInfo (lids, true, true);
	    if (tis != null && tis.length > 0 || tis[0] != null) {
		stackTrace.setTrace (tis[0]);
		monitors.setMonitors (tis[0]);
		return;
	    }
	}
	stackTrace.setTrace (null);
    }

    private void setThreadColumnProperties (JTable table) {
	table.setAutoResizeMode (JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
	table.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
	table.setAutoCreateRowSorter (true);
	TableColumn column = null;
	int totalWidth = 0;
	TimeRenderer tr = new TimeRenderer ();
	for (int i = 0; i < table.getModel ().getColumnCount (); i++) {
	    column = table.getColumnModel ().getColumn (i);
	    int width = 75;
	    if (i == TTM.COL_ID)
		width = 50;
	    else if (i == TTM.COL_NAME)
		width = 200;
	    else if (i == TTM.COL_STATE || i == TTM.COL_CPU_TIME)
		width = 100;
	    else if (i == TTM.COL_WAIT_ON)
		width = 350;

	    if (i == TTM.COL_CPU_TIME || 
		i == TTM.COL_BLOCK_TIME || 
		i == TTM.COL_WAIT_TIME)
		column.setCellRenderer (tr);
	    
	    column.setPreferredWidth (width);
	    totalWidth += width;
	}
	int rows = table.getModel ().getRowCount () + 1; // header 
        table.setPreferredScrollableViewportSize (new Dimension (totalWidth, 10 * rows));
    }

    private void setTraceColumnProperties (JTable table) {
	table.setAutoResizeMode (JTable.AUTO_RESIZE_SUBSEQUENT_COLUMNS);
	table.setSelectionMode (ListSelectionModel.SINGLE_SELECTION);
	TableColumn column = null;
	int totalWidth = 0;
	for (int i = 0; i < table.getModel ().getColumnCount (); i++) {
	    column = table.getColumnModel ().getColumn (i);
	    int width = 200;
	    if (i == StackTraceModel.COL_DEPTH)
		width = 30;
	    if (i == StackTraceModel.COL_CLASS)
		width = 300;
	    if (i == StackTraceModel.COL_LINE || 
		i == StackTraceModel.COL_NATIVE)
		width = 50;
	    column.setPreferredWidth (width);
	    totalWidth += width;	    
	}
	int rows = 20;
	Dimension d = new Dimension (totalWidth, 10 * rows);
        table.setPreferredScrollableViewportSize (d);
    }

    public void writeFullState (Writer w) throws IOException {
	PrintWriter pw = new PrintWriter (w);
	ThreadMXBean mbean = ph.getThreadMXBean ();
	ThreadInfo[] tinfos = mbean.dumpAllThreads (true, true);
	for (ThreadInfo ti : tinfos) {
	    writeThreadHeader (pw, mbean, ti);
	    int depth = 0;
	    for (StackTraceElement se : ti.getStackTrace ()) {
		List<MonitorInfo> lockedMonitors = 
		    getLockedMonitors (depth++, ti.getLockedMonitors ());
		writeStackElement (pw, se, lockedMonitors);
	    }
	    pw.println ();
	}
	pw.flush ();
    }

    private List<MonitorInfo> getLockedMonitors (int depth, 
						 MonitorInfo[] locked) {
	List<MonitorInfo> ret = new ArrayList<MonitorInfo> ();
	for (MonitorInfo mi : locked) {
	    if (mi.getLockedStackDepth () == depth)
		ret.add (mi);
	}
	return ret;
    }

    private void writeThreadHeader (PrintWriter pw, 
				    ThreadMXBean mbean, 
				    ThreadInfo ti) 
	throws IOException {
	pw.println (ti.getThreadName () + ", " + 
		    "id: " + ti.getThreadId () + ", " + 
		    "state: " + ti.getThreadState ().toString ());
	LockInfo li = ti.getLockInfo ();
	if (li != null)
	    pw.println ("waiting on: " + li.toString ());
	pw.println ("thread time: " + 
		    mbean.getThreadUserTime (ti.getThreadId ()) + ", " + 
		    "block count: " + ti.getBlockedCount () + ", " + 
		    "block time: " + ti.getBlockedTime () + ", " + 
		    "wait count: " + ti.getWaitedCount () + ", " + 
		    "wait time: " + ti.getWaitedTime ());
    }

    private void writeStackElement (PrintWriter pw, StackTraceElement se, 
				    List<MonitorInfo> locked) 
	throws IOException {
	pw.print ("    " + 
		  se.getClassName () + " " + 
		  se.getMethodName () + " " +
		  se.getFileName () + 
		  (se.isNativeMethod () ? 
		   "(native)" : 
		   ":" + se.getLineNumber ()));
	if (!locked.isEmpty ()) {
	    for (int i = 0, s = locked.size (); i < s; i++) {
		pw.println ();
		pw.print ("        locked: ");
		MonitorInfo mi = locked.get (i);
		pw.print (mi.getClassName () + ", id: " + 
			  Integer.toHexString (mi.getIdentityHashCode ()));
	    }
	}
	pw.println ();
    }
}

class TTM extends AbstractTableModel {
    private ThreadInfo[] tinfos;
    private ThreadMXBean mbean;
    
    private static final String[] columnNames = 
    {"Id", "Name", "State", "Waiting On", "cpu time",
     "Block count", "Blocked time", 
     "Wait count", "Waited time"};

    public static final int COL_ID = 0;
    public static final int COL_NAME = COL_ID + 1;
    public static final int COL_STATE = COL_NAME + 1;
    public static final int COL_WAIT_ON = COL_STATE + 1;
    public static final int COL_CPU_TIME = COL_WAIT_ON + 1;
    public static final int COL_BLOCK_COUNT = COL_CPU_TIME + 1;
    public static final int COL_BLOCK_TIME = COL_BLOCK_COUNT + 1;
    public static final int COL_WAIT_COUNT = COL_BLOCK_TIME + 1;
    public static final int COL_WAIT_TIME = COL_WAIT_COUNT + 1;

    public TTM (ThreadInfo[] tinfos, ThreadMXBean mbean) {
	this.tinfos = tinfos;
	this.mbean = mbean;
    }

    public void updateThreadInfos () {
	long[] threadIds = mbean.getAllThreadIds ();
	tinfos = mbean.getThreadInfo (threadIds);
	fireTableDataChanged ();
    }

    @Override public String getColumnName (int col) {
	return columnNames[col];
    }

    public int getRowCount () { 
	return tinfos.length;
    }

    public int getColumnCount () { 
	return columnNames.length; 
    }

    public Object getValueAt (int row, int col) {
	ThreadInfo ti = tinfos[row];
	switch (col) {
	case COL_ID:
	    return ti.getThreadId ();
	case COL_NAME:
	    return ti.getThreadName ();
	case COL_STATE:
	    return ti.getThreadState ().toString ();
	case COL_WAIT_ON:
	    LockInfo li = ti.getLockInfo ();
	    return li == null ? "" : li.toString ();
	case COL_CPU_TIME:
	    return mbean.getThreadUserTime (ti.getThreadId ());
	case COL_BLOCK_COUNT:
	    return ti.getBlockedCount ();
	case COL_BLOCK_TIME:
	    return ti.getBlockedTime ();
	case COL_WAIT_COUNT:
	    return ti.getWaitedCount ();
	case COL_WAIT_TIME:
	    return ti.getWaitedTime ();
	default:
	    throw new IllegalArgumentException ("unknow column: " + col);	
	}
    }    
    
    @Override public Class<?> getColumnClass (int col) {
	switch (col) {
	case COL_ID:          // fall through
	case COL_CPU_TIME:    // fall through
	case COL_BLOCK_COUNT: // fall through
	case COL_BLOCK_TIME:  // fall through
	case COL_WAIT_COUNT:  // fall through
	case COL_WAIT_TIME:   // fall through
	    return Long.class;
	case COL_NAME:
	    return String.class;
	default:
	    return String.class;
	}
    }

    @Override public boolean isCellEditable (int row, int col) { 
	return false; 
    }
    
    @Override public void setValueAt (Object value, int row, int col) {
	throw new IllegalStateException ("non editable table");
    }	
}

class StackTraceModel extends AbstractTableModel {
    private ThreadInfo ti;

    private static final String[] columnNames = 
    {"Depth", "Class", "Method", "File", "Line#", "Native"};

    public static final int COL_DEPTH = 0;
    public static final int COL_CLASS = COL_DEPTH + 1;
    public static final int COL_METHOD = COL_CLASS + 1;
    public static final int COL_FILE = COL_METHOD + 1;
    public static final int COL_LINE = COL_FILE + 1;
    public static final int COL_NATIVE = COL_LINE + 1;

    public StackTraceModel (ThreadInfo ti) {
	this.ti = ti;
    }

    public void setTrace (ThreadInfo ti) {
	this.ti = ti;
	fireTableDataChanged ();
    }
    
    @Override public String getColumnName (int col) {
	return columnNames[col];
    }

    public int getRowCount () {
	if (ti == null || ti.getStackTrace () == null)
	    return 0;
	return ti.getStackTrace ().length;
    }

    public int getColumnCount () {
	return columnNames.length; 
    }

    public Object getValueAt (int row, int col) {
	StackTraceElement ste = ti.getStackTrace ()[row];
	switch (col) {
	case COL_DEPTH: 
	    return Integer.valueOf (row + 1);
	case COL_CLASS:
	    return ste.getClassName ();
	case COL_METHOD:
	    return ste.getMethodName ();
	case COL_FILE:
	    return ste.getFileName ();
	case COL_LINE:
	    return ste.getLineNumber ();
	case COL_NATIVE:
	    return ste.isNativeMethod ();
	default: 
	    throw new IllegalArgumentException ("unknow column: " + col);
	}
    }

    @Override public Class<?> getColumnClass (int col) {
	switch (col) {
	case COL_DEPTH: // fall through
	case COL_LINE:
	    return Integer.class;
	case COL_NATIVE:
	    return Boolean.class;
	default:
	    return String.class;
	}
    }
}

class MonitorModel extends AbstractTableModel {
    private ThreadInfo ti;
    
    public static final int COL_CLASS = 0;
    public static final int COL_LOCK_CLASS = COL_CLASS + 1;
    public static final int COL_METHOD = COL_LOCK_CLASS + 1;
    public static final int COL_DEPTH = COL_METHOD + 1;
    
    private static final String[] columnNames = 
    {"Monitor Class", "Locked at", "Method", "Depth"};
    
    public MonitorModel (ThreadInfo ti) {
	this.ti = ti;
    }

    public void setMonitors (ThreadInfo ti) {
	this.ti = ti;
	fireTableDataChanged ();
    }

    @Override public String getColumnName (int col) {
	return columnNames[col];
    }

    public int getRowCount () {
	if (ti == null || ti.getLockedMonitors () == null)
	    return 0;
	return ti.getLockedMonitors ().length;
    }

    public int getColumnCount () {
	return columnNames.length; 
    }
    
    public Object getValueAt (int row, int col) {
	MonitorInfo mi = ti.getLockedMonitors ()[row];
	StackTraceElement ste = mi.getLockedStackFrame ();
	switch (col) {
	case COL_CLASS:
	    return mi.getClassName ();
	case COL_LOCK_CLASS:
	    return ste.getClassName ();
	case COL_METHOD:
	    return ste.getMethodName ();
	case COL_DEPTH:
	    return mi.getLockedStackDepth () + 1;
	default: 
	    throw new IllegalArgumentException ("unknow column: " + col);
	}
    }

    @Override public Class<?> getColumnClass (int col) {
	switch (col) {
	case COL_DEPTH:
	    return Integer.class;
	default:
	    return String.class;
	}
    }
}

class LocksModel extends AbstractTableModel {
    private ThreadInfo ti;
    
    public static final int COL_CLASS = 0;

    
    private static final String[] columnNames = 
    {"Lock Class"};
    
    public LocksModel (ThreadInfo ti) {
	this.ti = ti;
    }

    public void setLocks (ThreadInfo ti) {
	this.ti = ti;
	fireTableDataChanged ();
    }

    @Override public String getColumnName (int col) {
	return columnNames[col];
    }

    public int getRowCount () {
	if (ti == null || ti.getLockedSynchronizers () == null)
	    return 0;
	return ti.getLockedSynchronizers ().length;
    }

    public int getColumnCount () {
	return columnNames.length; 
    }
    
    public Object getValueAt (int row, int col) {
	LockInfo mi = ti.getLockedSynchronizers ()[row];
	switch (col) {
	case COL_CLASS:
	    return mi.getClassName ();
	default: 
	    throw new IllegalArgumentException ("unknow column: " + col);
	}
    }

    @Override public Class<?> getColumnClass (int col) {
	return String.class;
    }
}