package org.apache.turbine.util;

import de.fub.bytecode.Repository;
import de.fub.bytecode.classfile.ClassParser;
import de.fub.bytecode.classfile.Code;
import de.fub.bytecode.classfile.Constant;
import de.fub.bytecode.classfile.ConstantClass;
import de.fub.bytecode.classfile.ConstantPool;
import de.fub.bytecode.classfile.ConstantUtf8;
import de.fub.bytecode.classfile.JavaClass;
import de.fub.bytecode.classfile.LocalVariable;
import de.fub.bytecode.classfile.LocalVariableTable;
import de.fub.bytecode.classfile.Method;
import de.fub.bytecode.classfile.Utility;

import java.io.PrintStream;

import java.util.Collection;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.TreeMap;

/**
 * Utility class for creating a list of fully qualified
 * import statements.
 *
 * @author <a href="mailto:jvanzyl@apache.org">Jason van Zyl</a>
 *
 * Problems:
 *
 * Whatever[] : class for array declarations are not forming correctly.
 *              coming out like Lorg.apache.turbine.util.Whatever;;
 */
public class ImportStatements
{
    public ImportStatements(String className)
    {
        execute(className);
    }
    
    public void execute(String className)
    {
        Hashtable classes = new Hashtable();
    
        try
        {
            TreeMap classNames = new TreeMap();
            classNames.putAll(classReferences(className));
            classNames.putAll(localVariables(className));
            
            /*
             * Don't need reference to the class
             * being analysed.
             */
            classNames.remove(className);            
            
            Iterator iterator = classNames.values().iterator();
            StringBuffer sb = new StringBuffer();
            while (iterator.hasNext())
            {
                String signature = (String) iterator.next();
                
                /*
                 *  Get rid of imports that are of the form
                 * import java.lang.X;
                 */
                if (signature.indexOf("java.lang.") != -1)
                {
                    if (signature.substring("java.lang.".length()).indexOf(".") == -1)
                        continue;
                }
                
                if (basePackage(className).equals(basePackage(signature)))
                    continue;
                
                /*
                 * We also need to get rid of imports for classes
                 * in the same package as the class being
                 * analysed.
                 */
                
                sb.append("import ")
                  .append(signature)
                  .append(";")
                  .append("\n");
            }                
        
            System.out.println(sb);
        }
        catch(Exception e)
        {
            e.printStackTrace();
        }
    }

    private String basePackage(String className)
    {
        return className.substring(0, className.lastIndexOf(".") - 1);
    }

    /**
     * Resolve all class references in a class file by
     * examining the constant pool in the class.
     *
     * @param String className
     * @return Hashtable Fully qualified class references.
     */
    public static Hashtable classReferences(String className) throws Exception
    {
        Hashtable classReferences = new Hashtable();
        JavaClass javaClass;
        
        if ((javaClass = Repository.lookupClass(className)) == null)
        {
            javaClass = new ClassParser(className).parse();
        }            
        
        ConstantPool constantPool = javaClass.getConstantPool();
        Constant[] constantPoolArray = constantPool.getConstantPool();
                        
        for (int i = 0; i < constantPoolArray.length; i++)
        {
            if (constantPoolArray[i] instanceof ConstantClass)
            {
                int index = ((ConstantClass) 
                    constantPoolArray[i]).getNameIndex();
                
                ConstantUtf8 constant = (ConstantUtf8) 
                    constantPool.getConstant(index);
                
                String signature = new String(constant.getBytes());
                signature = signature.replace('/', '.');
                classReferences.put(signature,signature);
            }
        }
    
        return classReferences;
    }
    
    public static Hashtable localVariables(String name) throws Exception
    {
        Hashtable vars = new Hashtable();
        
        JavaClass javaClass;
        
        if ((javaClass = Repository.lookupClass(name)) == null)
        {
            javaClass = new ClassParser(name).parse();
        }            
        
        Method[] methods = javaClass.getMethods();
        
        for (int i = 0; i < methods.length; i++)
        {
            Code code = methods[i].getCode();
            if (code != null)
            {
                LocalVariableTable table = code.getLocalVariableTable();
                
                if (table == null)
                {
                    continue;
                }                    
                
                LocalVariable[] lvt = table.getLocalVariableTable();
                
                for (int j = 0; j < lvt.length; j++)
                {
                    String lvar = Utility.signatureToString(lvt[j].getSignature());
                    
                    if (lvar.indexOf(".") == -1)
                        continue;
                    
                    /*
                     * Get rid of the array signature
                     * if present we just need the base type
                     * signature.
                     */
                    if (lvar.endsWith("[]"))
                        lvar = lvar.substring(0, lvar.length() - 2);
                        
                    vars.put(lvar,lvar);
                }                    
            }                
        }
    
        return vars;
    }

    public static void main(String[] args)
    {
        ImportStatements is = new ImportStatements(args[0]);
    }
}
