package org.apache.turbine.services.intake.model;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and 
 *    "Apache Turbine" must not be used to endorse or promote products 
 *    derived from this software without prior written permission. For 
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without 
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// JDK classes
import java.util.*;
import java.lang.reflect.*;

import org.apache.regexp.RE;
import org.apache.regexp.RESyntaxException;

// Turbine classes
import org.apache.turbine.om.*;
import org.apache.turbine.util.ObjectUtils;
import org.apache.turbine.util.StringUtils;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.ParameterParser;
import org.apache.turbine.util.Log;
import org.apache.turbine.util.TurbineException;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.services.intake.xmlmodel.*;
import org.apache.turbine.services.intake.*;

/**
 * Base class for Intake generated input processing classes.
 *
 * @author <a href="mailto:jmcnally@collab.net>John McNally</a>
 * @version $Id: Field.java,v 1.9 2001/04/11 04:41:34 jmcnally Exp $
 */
public abstract class Field
{
    protected final String name;
    protected final String key;
    protected final String mapToObject;
    protected final Method getter;
    protected final Method setter;

    protected final RE regexp;
    protected final String regexpMessage; 
    protected final int minLength;
    protected final String minLengthMessage; 
    protected final int maxLength;
    protected final String maxLengthMessage; 
    protected final boolean maxLengthForce; 
    protected final String[] requires; 
    protected final String[] requiresMessage; 
    protected final String ifRequiredMessage; 
    protected final boolean isMultiValued; 
    protected Object onError;

    protected final Group group;
 
    protected boolean set_flag;
    protected boolean valid_flag;
    protected boolean required;
    protected boolean initialized;
    protected String message;

    protected OMTool omTool; 
    protected IntakeValue iv;
    protected Retrievable retrievable;

    /**
     * The object containing the request data
     */
    protected RunData data;    

    public Field(XmlField field, Group group)
        throws Exception
    {
        this.group = group;
        key = field.getKey();
        name = field.getName();
        isMultiValued  = field.isMultiValued();

        int requires_index = 0;
        for (int i=0; i<field.getRules().size(); i++) 
        {
            Rule rule = (Rule)field.getRules().get(i);
            if ( rule.getRequiresProp()!=null )
            {
                requires_index++;
            }
        }
        requires = new String[requires_index]; 
        requiresMessage  = new String[requires_index];
        
        requires_index = 0;
        RE tmpRE = null;
        String tmpREMessage = null;
        int tmpMinLength = 0;
        String tmpMinLengthMessage = null;            
        int tmpMaxLength = 0;
        String tmpMaxLengthMessage = null;
        boolean tmpMaxLengthForce = false;
        for (int i=0; i<field.getRules().size(); i++) 
        {
            Rule rule = (Rule)field.getRules().get(i);
            
            if ( rule.getMask() != null ) 
            {
                tmpRE = new RE(rule.getMask());            
                tmpREMessage = rule.getMessage();
            }

            if ( rule.getMinLength() != null )
            {
                tmpMinLength = Integer.parseInt(rule.getMinLength());
                tmpMinLengthMessage = rule.getMessage();
            }

            if ( rule.getMaxLength() != null )
            {
                tmpMaxLength = Integer.parseInt(rule.getMaxLength());
                tmpMaxLengthMessage = rule.getMessage();
                tmpMaxLengthForce = "force".equals(rule.getAction());
            }

            if ( field.getGroup()!=null && rule.getRequiresProp()!=null )
            {
                requires[requires_index] = rule.getRequiresProp();
                requiresMessage[requires_index] = rule.getMessage();
                requires_index++;
            }
        }
        regexp = tmpRE;
        regexpMessage  = tmpREMessage;
        minLength = tmpMinLength;
        minLengthMessage  = tmpMinLengthMessage;
        maxLength = tmpMaxLength;
        maxLengthMessage  = tmpMaxLengthMessage;
        maxLengthForce  = tmpMaxLengthForce;

        iv = getNewIntakeValue(name);
        mapToObject = field.getMapToObject();
        String propName = field.getMapToProperty();
        Method tmpGetter = null;
        Method tmpSetter = null;
        if ( mapToObject != null && mapToObject.length() != 0) 
        {
            tmpGetter = TurbineIntake.getFieldGetter(mapToObject, propName);
            tmpSetter = TurbineIntake.getFieldSetter(mapToObject, propName);
        }
        getter = tmpGetter;
        setter = tmpSetter;
        ifRequiredMessage = field.getIfRequiredMessage();
    }

 
    public Field init(RunData data) //, User user)
        throws TurbineException
    {
        this.data = data;
        valid_flag = true;
        // String omKey = TurbineIntake.getOMKey();
        // omTool = (OMTool)TurbineVelocity.getContext(data).get(omKey);
        
        /*
        IntakeValueMapWrap ivs = user.getTemp(IntakeValueMapWrap.KEY);
        if ( ivs == null )
        {
            ivs = new IntakeValueMapWrap();
            user.setTemp(IntakeValueMapWrap.KEY, ivs);
        }
        
        if (ivs.contains(key))
        {
            iv = ivs.get(key);
        }
        else 
        {
            iv = getNewIntakeValue(NAME);
            ivs.put(iv);
        }
        */

        
        ParameterParser pp = data.getParameters();
        if ( pp.containsKey(getKey()) && pp.getString(getKey()) != null )
        {
            set_flag = true;
            if (validate(pp))
            {
                // iv.reconcileNotValid(pp);
            }
        }
        else if ( pp.containsKey(getValueIfAbsent()) && 
                  pp.getString(getValueIfAbsent()) != null )
        {
            pp.add(getKey(), pp.getString(getValueIfAbsent()));
            set_flag = true;
            validate(pp);
        }

        initialized = true;
        return this;
    }

    public Field init(Retrievable obj)
    {
        if ( !initialized ) 
        {
            valid_flag = true;
        }
        retrievable = obj;
        return this;
    }

    
    /**
     * Get the value of required.
     * @return value of required.
     */
    public boolean isRequired() 
    {
        return required;
    }

    /**
     * Set the value of required.
     * @param v  Value to assign to required.
     */
    public void setRequired(boolean  v) 
    {
        setRequired(v, ifRequiredMessage);
    }
        
    /**
     * Set the value of required.
     *
     * @param v a <code>boolean</code> value
     * @param message, override the value from intake.xml
     */
    public void setRequired(boolean  v, String message) 
    {
        this.required = v;
        if (v && !set_flag) 
        {
            valid_flag=false;
            this.message = message;
        }
    }

    /**
     * Removes references to this group and its fields from the 
     * query parameters
     */
    public void removeFromRequest()
    {
        data.getParameters().remove(getKey());
    }


    /**
     * Disposes the object after use. The method is called
     * when the Group is returned to its pool.
     * if overridden, super.dispose() should be called.
     */
    public void dispose()
    {
        data = null;
        initialized = false;
        set_flag = false;
        valid_flag = false;
        required = false;
        message = null;
        iv.dispose();
    }

    protected IntakeValue getNewIntakeValue(String property)
    {
        return new IntakeValue(property, this);
    }

    /**
     * Get the key used to identify the field.
     * @return the query data key.
     */
    public String getKey()
    {
        if ( group == null ) 
        {
            return key;
        }
        else 
        {
            return group.getObjectKey() + key;
        }        
    }

    /**
     * Use in a hidden field assign a default value in the event the
     * field is absent from the query parameters.  Used to track checkboxes, 
     * since they only show up if checked.
     */
    public String getValueIfAbsent()
    {
        return getKey() + "_vifa_";
    }
    
    /**
     * Get the value of $field.Name.
     * @return value of $field.Name
     */
    public IntakeValue getValue() 
    {
        return iv;
    }
    
    public boolean isValid()
    {
        return valid_flag;
    }

    public boolean isSet()
    {
        return set_flag;
    }

    public String getMessage()
    {
        return message;
    }

    /**
     * Sets an error message.  The field is also marked as invalid.
     */
    public void setMessage(String message)
    {
         this.message = message;
         valid_flag = false;
    }

    /**
     * Compares request data with constraints and sets the valid flag.
     */
    protected boolean validate(ParameterParser pp)
        throws TurbineException
    {        
        if ( isMultiValued  ) 
        {
            String[] ss = pp.getStrings(getKey());
            // this definition of not set might need refined.  But
            // not sure the situation will arise.
            if ( ss.length == 0 || (ss.length == 1 && ss[0].length() == 0) ) 
            {
                set_flag = false;
                return true;
            }
            else 
            {            
                for (int i=0; i<ss.length; i++) 
                {
                    checkString(ss[i]);
                }
            }
        }
        else 
        {
            String s = pp.getString(getKey());
            if ( s.length() == 0 ) 
            {
                set_flag = false;
                return true;
            }
            else 
            {
                checkString(s);             
            }
        }

        if ( group != null && requires.length > 0 )
        {
            for (int i=0; i<requires.length; i++) 
            {
                boolean b = pp.containsKey(((Field)group.get(requires[i]))
                                           .getKey());
                valid_flag &= b;
                if (requiresMessage[i] != null && !b)
                {
                    message = requiresMessage[i];
                }                 
            }
        }

        doValidate(pp);
        return valid_flag;
    }

    /**
     * Compares request data with constraints and sets the valid flag.
     * To be implemented in subclasses
     */
    protected abstract void doValidate(ParameterParser pp);

    /**
     * Compares request data with constraints and sets the valid flag.
     */
    private void checkString(String val)
    {
        boolean b;
        if ( regexp != null )
        {
            b = regexp.match(val);
            valid_flag &= b;
            if (regexpMessage  != null && !b)
            {
                message = regexpMessage;
            } 
        }

        if ( minLength > 0 )
        {
            b = val.length() < minLength; 
            valid_flag &= !b;
            if (minLengthMessage  != null && b)
            {
                message = minLengthMessage;
            } 
        }

    }
    
    public String toString()
    {
        return iv.toString();
    }

}

    /* this stuff does not belong here delete it 
    public void getProperty($appData.BasePackage$field.MapToObject obj) 
    //    throws TurbineException
    {
            setValue(obj.get${field.MapToProperty}();
    }

    /* *
     * Calls a setter method on obj, if this field has been set.  Does not
     * check validity as it is assumed this will have been checked prior
     * to calling this method.
     * @exception throws a TurbineException if called and the input
     * was not valid.
     * /
    public void setProperty($appData.BasePackage$field.MapToObject obj) 
        throws TurbineException
    {
        if (!isValid()) 
        {
   	        throw new TurbineException(
                "Attempted to assign an invalid input.");
        }
        if (isSet())
        {
            try
            {
                obj.set${field.MapToProperty}(getValue());
            }
            catch ( Exception e)
            {
                throw new TurbineException("An exception prevented the" +
                    " mapping to $appData.BasePackage$field.MapToObject", e);
            }
        }
    }
    */
