package org.herac.tuxguitar.gui.marker;

import org.eclipse.swt.SWT;
import org.eclipse.swt.events.PaintEvent;
import org.eclipse.swt.events.PaintListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.graphics.Color;
import org.eclipse.swt.graphics.RGB;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.ColorDialog;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Shell;
import org.eclipse.swt.widgets.Spinner;
import org.eclipse.swt.widgets.Text;
import org.herac.tuxguitar.gui.TuxGuitar;
import org.herac.tuxguitar.gui.undo.undoables.JoinedUndoable;
import org.herac.tuxguitar.gui.undo.undoables.custom.UndoableChangeMarker;
import org.herac.tuxguitar.song.managers.SongManager;
import org.herac.tuxguitar.song.models.Marker;

public class MarkerEditor {
	public static final int STATUS_NEW = 1;
	public static final int STATUS_EDIT = 2;
	
	private static final int MINIMUN_CONTROL_WIDTH = 180;    
    private static final int MINIMUN_BUTTON_WIDTH = 80;
    private static final int MINIMUN_BUTTON_HEIGHT = 25;	
	
	private int status;
	private Marker marker;
	private Shell dialog;
	private Spinner measureSpinner;
	private Text titleText;
	private Button colorButton;

	public MarkerEditor(Marker marker) {
		this(marker,STATUS_NEW);
	}	
	
	public MarkerEditor(Marker marker,int status) {
		this.marker = (Marker)marker.clone();
		this.status = status;
	}		

	public void show(Shell shell) {
		dialog = new Shell(shell, SWT.DIALOG_TRIM | SWT.APPLICATION_MODAL);
		dialog.setLayout(new GridLayout());
		dialog.setText(TuxGuitar.getProperty("marker"));

		// ----------------------------------------------------------------------
		Composite composite = new Composite(dialog, SWT.NONE);
		composite.setLayout(new GridLayout(2, false));
		composite.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, true));

		// Measure Number
		final int measureCount = TuxGuitar.instance().getSongManager().countMeasures();
		Label measureLabel = new Label(composite, SWT.NULL);
		measureLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true,true));
		measureLabel.setText(TuxGuitar.getProperty("measure"));

		this.measureSpinner = new Spinner(composite, SWT.BORDER);
		this.measureSpinner.setLayoutData(getAlignmentData(MINIMUN_CONTROL_WIDTH,SWT.FILL));
		
		
		this.measureSpinner.setMinimum(1);
		this.measureSpinner.setMaximum(measureCount);
		this.measureSpinner.setSelection(this.marker.getMeasure());
		this.measureSpinner.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				int selection = measureSpinner.getSelection();
				if (selection < 1) {
					measureSpinner.setSelection(1);
				} else if (selection > measureCount) {
					measureSpinner.setSelection(measureCount);
				}
			}
		});

		// Title
		Label titleLabel = new Label(composite, SWT.NULL);
		titleLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, true));
		titleLabel.setText(TuxGuitar.getProperty("title"));
		this.titleText = new Text(composite, SWT.BORDER);
		this.titleText.setLayoutData(getAlignmentData(MINIMUN_CONTROL_WIDTH,SWT.FILL));
		this.titleText.setText(this.marker.getTitle());

		// Color
		Label colorLabel = new Label(composite, SWT.NULL);
		colorLabel.setLayoutData(new GridData(SWT.RIGHT, SWT.CENTER, true, true));
		colorLabel.setText(TuxGuitar.getProperty("color"));
		this.colorButton = new Button(composite, SWT.PUSH);
		this.colorButton.setLayoutData(getAlignmentData(MINIMUN_CONTROL_WIDTH,SWT.FILL));
		this.colorButton.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent event) {
				ColorDialog dlg = new ColorDialog(dialog);
				dlg.setRGB(dialog.getDisplay().getSystemColor(SWT.COLOR_BLACK).getRGB());
				dlg.setText(TuxGuitar.getProperty("choose-color"));
				RGB rgb = dlg.open();
				if (rgb != null) {
					marker.getColor().setR(rgb.red);
					marker.getColor().setG(rgb.green);
					marker.getColor().setB(rgb.blue);       
                    colorButton.redraw();					
				}
			}
		});		
		this.colorButton.addPaintListener(new PaintListener() {		
			public void paintControl(PaintEvent e) {
				Color color = new Color(dialog.getDisplay(), marker.getColor().getR(), marker.getColor().getG(), marker.getColor().getB());
		        e.gc.setBackground(color);
		        e.gc.fillRectangle(5,5,colorButton.getSize().x - 10,colorButton.getSize().y - 10);
		        color.dispose();
			}		
		});		
		// ------------------BUTTONS--------------------------
		Composite buttons = new Composite(dialog, SWT.NONE);
		buttons.setLayout(new GridLayout(2, false));
		buttons.setLayoutData(new GridData(SWT.END, SWT.FILL, true, true));

		GridData data = new GridData(SWT.FILL, SWT.FILL, true, true);
		data.minimumWidth = MINIMUN_BUTTON_WIDTH;
		data.minimumHeight = MINIMUN_BUTTON_HEIGHT;

		final Button buttonOK = new Button(buttons, SWT.PUSH);
		buttonOK.setText(TuxGuitar.getProperty("ok"));
		buttonOK.setLayoutData(data);
		buttonOK.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				updateMarker();
				dialog.dispose();
			}
		});

		Button buttonCancel = new Button(buttons, SWT.PUSH);
		buttonCancel.setText(TuxGuitar.getProperty("cancel"));
		buttonCancel.setLayoutData(data);
		buttonCancel.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent arg0) {
				dialog.dispose();
			}
		});

		dialog.pack();
		dialog.open();

		int x = shell.getBounds().x + (shell.getBounds().width - dialog.getSize().x) / 2;
		int y = shell.getBounds().y + (shell.getBounds().height - dialog.getSize().y) / 2;
		dialog.setLocation(x, y);

		while (!dialog.isDisposed()) {
			if (!dialog.getDisplay().readAndDispatch()) {
				dialog.getDisplay().sleep();
			}

		}

	}

    private GridData getAlignmentData(int minimumWidth,int horizontalAlignment){
    	GridData data = new GridData();
    	data.minimumWidth = minimumWidth;
    	data.horizontalAlignment = horizontalAlignment;
    	data.verticalAlignment = SWT.DEFAULT;
    	data.grabExcessHorizontalSpace = true;
    	data.grabExcessVerticalSpace = true;
    	return data;
    }	

	private void updateMarker() {	
		int oldMeasure = this.marker.getMeasure();
		this.marker.setMeasure(this.measureSpinner.getSelection());
		this.marker.setTitle(this.titleText.getText());
		this.marker = (Marker)this.marker.clone();
		
		SongManager manager = TuxGuitar.instance().getSongManager();
		
		//comienza el undoable
		JoinedUndoable joinedUndoable = new JoinedUndoable();
		
		if(this.status == STATUS_EDIT && oldMeasure != this.marker.getMeasure()){
			UndoableChangeMarker undoable = UndoableChangeMarker.startUndo(manager.getMarker(oldMeasure));			
			TuxGuitar.instance().getSongManager().removeMarker(oldMeasure);			
			joinedUndoable.addUndoableEdit(undoable.endUndo(null));
		}
		UndoableChangeMarker undoable = UndoableChangeMarker.startUndo(manager.getMarker(this.marker.getMeasure()));		
		TuxGuitar.instance().getSongManager().updateMarker(this.marker);
		joinedUndoable.addUndoableEdit(undoable.endUndo(this.marker));

		// termia el undoable
		TuxGuitar.instance().getTablatureEditor().getUndoManager().addEdit(joinedUndoable.endUndo());
	}
}
