#!/usr/bin/perl
# umx-probe-monitor.pl
# 
#  This program is a modified version of monitor-probe-using-X and 
# probes monitor uing Multi-X server. This program gets video driver and
# BusID from xorg.conf.new file.
#

use strict;
use warnings;
use Getopt::Long;

my $verbose = '';
my $pcibus = '';
my $Driver = '';
my %opt = ();

GetOptions ('verbose' => \$verbose);

if ( -e "/tmp/Monitor.Probed") {
    unlink "/tmp/Monitor.Probed";
}

#if ( !$pcibus || !$Driver ) {
#    die "usage: umx-probe-monitor.pl [--verbose] --pcibus <bus> --driver <driver>\n";
#}

#print "$pcibus, $Driver\n";

if (! -e "/root/xorg.conf.new" ) {
    print "No /root/xorg.conf.new\n";
    exit 1;
}
open(FILE, "/root/xorg.conf.new") || die("(EE) $!");

# Find "Driver" and "BusID" from each "Device" section 
my $found_device = 0;
my $tmp = "";
while (<FILE>) {
    # Remove white preceeding and tailing space
    s/^\s+//;
    s/\s+$//;

    if($found_device) {
        if (/^\s*Driver\s+/) {
            ($tmp, $Driver) = split(/\s+/);
            #print $tmp, " is ", $Driver, "\n";
        }
        elsif (/^\s*BusID\s+/) {
            ($tmp, $pcibus) = split(/\s+/);
            #print $tmp, " is ", $pcibus, "\n";
        }
        
        if ( "$Driver" && "$pcibus" ) {
            #print "found Driver and BusID\n\n";

            ##
            ## Probe Monitor on the current video card
            if (my $log = probe_using_X($Driver, $pcibus)) {
                ##
                ## Parse EDID data found
                if (parse_X_log_edids($log)) {
                    #print "Parsing EDID succeed.\n";
                    last;
                }
                else {
                    print "{EE} Parsing EDID failed.\n";
                }
                #my $ok = parse_X_log($log);
            } 
            else { # Faile to Probe EDID on the current video card.
                warn("{EE} X probing failed.\n");
            }

            $Driver = "";
            $pcibus = "";
        }
    }
    if (/^\s*Section\s+"Device"/) {
        $found_device = 1;
    }
    elsif (/^\s*EndSection/) {
        $found_device = 0;
    }
}

close(FILE);

##
## Probe Monitor
##
#my $log = probe_using_X($Driver, $pcibus) or warn("X probe failed\n"), exit 1;

##
## Parse EDID data
##
## TODO:
##  - Need to update for 64bit system. lib path may be different.
#my $ok = parse_X_log($log);

sub probe_using_X {
    my ($Driver, $pcibus) = @_;
    print "\nProbing monitor on $pcibus\n";

    my $tmp_conf = tmpfile();
    my $tmp_log = tmpfile();

    # Use vesa driver to probe each PCI entity
    output($tmp_conf, <<EOF);
Section "Files"
    ModulePath      "/opt/MX11R7/lib/xorg/modules"
    FontPath "/usr/lib/X11/fonts/misc:unscaled"    
EndSection

Section "Module"
        SubSection "extmod"
                Option "omit XFree86-DGA"
        EndSubSection
        Load "xtrap"
        Load "wfb"
        Load "GLcore"
        Load "extmod"
        Load "dbe"
        Load "record"
EndSection

Section "Device"
    Identifier "device"
    Driver "vesa"
    BusID $pcibus
EndSection

Section "Screen"
    Identifier "screen"
    Device "device"
    DefaultDepth 16
EndSection

Section "ServerLayout"
    Identifier "layout"
    Screen "screen"
EndSection
EOF

    my $ok = eval {
        local $SIG{ALRM} = sub { die "alarm\n" };
        alarm 10;
        my $ok = 
           system("/opt/MX11R7/bin/Xorg :0 -ac -probeonly -logfile $tmp_log -config $tmp_conf -isolateDevice $pcibus > /dev/null 2>&1") == 0;
        alarm 0;

        if ($ok) {
            return "true";
        }
        else {
            return;
        }

    };

    if ($@) {
        # timed out
        die unless $@ eq "alarm\n";
        print "{EE} X probing failed.\n";
        return;

    } else {
        if ($ok) {
	    my $log = cat_($tmp_log);
	    return $log;
        }
        else {
            return;
        }
    }
}

sub tmpfile() { 
    chomp(my $s = `mktemp /tmp/tmp.XXXXXXXXXX`); #- we could use simply mktemp with new mktemp
    eval "END { unlink '$s' }";
    $s;
}

sub output { my $f = shift; open(my $F, ">$f") or die "output in file $f failed: $!\n"; print $F $_ foreach @_; 1 }

sub cat_ { open(my $F, $_[0]) or return; my @l = <$F>; wantarray() ? @l : join '', @l }


sub parse_X_log {
    my ($log) = @_;

    parse_X_log_edids($log) and return 1;

    print "\n\n{DD} Ever reach here\n\n";

    0
      # i810 format: "Size of device %s is %d x %d\n" (i810/i830_driver.c)
      # with one of "CRT", "TV", "DFP (digital flat panel)", "LFP (local flat panel)", "CRT2 (second CRT)", "TV2 (second TV)", "DFP2 (second digital flat panel)", "LFP2 (second local flat panel)",
      # example: (II) I810(0): Size of device LFP (local flat panel) is 1024 x 768
      || $log =~ m!\bSize of device LFP \(local flat panel\) is (\d+) x (\d+)$!m 

      # ati format: "%dx%d panel (ID %d) detected.\n" (ati/atipreinit.c)
      # example: (--) ATI(0): 1024x768 panel (ID 3) detected.
      || $log =~ m!\b(\d+)x(\d+) panel \(ID \d+\) detected\.$!m

      # radeon format: "Panel Size from BIOS: %dx%d\n" (ati/radeon_bios.c)
      # example: (II) RADEON(0): Panel Size from BIOS: 1400x1050
      || $log =~ m!\bPanel Size from BIOS: (\d+)x(\d+)$!m 

      # nv format: "Panel size is %i x %i\n" (nv/nv_setup.c)
      # example: (--) NV(0): Panel size is 1280 x 800
      || $log =~ m!\bPanel size is (\d+) x (\d+)$!m

      # savage format: "%dx%d %s LCD panel detected %s\n" (savage/savage_driver.c)
      # with one of "TFT", "DSTN", "STN"
      #         and "and active", "but not active"
      # example: (--) SAVAGE(0): 1024x768 TFT LCD panel detected and active
      || $log =~ m!\b(\d+)x(\d+) \S+ LCD panel detected !m

      # neomagic format: "Panel is a %dx%d %s %s display\n" (neomagic/neo_driver.c)
      # with one of "color", "monochrome"
      #         and "TFT", "dual scan"
      # example: (--) NEOMAGIC(0): Panel is a 1024x768 color TFT display
      || $log =~ m!\bPanel is a (\d+)x(\d+) (?:color|monochrome) (?:TFT|dual scan) display$!m

      # siliconmotion format: "Detected panel size via BIOS: %d x %d\n" (siliconmotion/smi_driver.c)
      || $log =~ m!\bDetected panel size via BIOS: (\d+) x (\d+)$!m
      # siliconmotion format: "%s Panel Size = %dx%d\n" (siliconmotion/smi_driver.c)
      || $log =~ m! Panel Size = (\d+)x(\d+)$!m

      # trident format: "%s Panel %ix%i found\n" (trident/trident_driver.c)
      # with one of "TFT", "DSTN", "STN"
      || $log =~ m!\b(?:TFT|DSTN|STN) Panel (\d+)x(\d+) found$!m

      # via format: "Selected Panel Size is 640x480\n", ... (via/via_driver.c)
      || $log =~ m!\bSelected Panel Size is (\d+)x(\d+)$!m

      # (WW) intel(0): BIOS panel mode is bigger than probed programmed mode, continuing with BIOS mode.
      # (II) intel(0): BIOS mode:
      # (II) intel(0): Modeline "1280x800"x0.0   68.88  1280 1296 1344 1410  800 804 807 815 (48.9 kHz)
      || $log =~ m!BIOS panel mode is bigger than probed programmed mode, continuing with BIOS mode.*?BIOS mode:.*?Modeline "(\d+)x(\d+)"!s

      or return;

    my ($X, $Y) = ($1, $2);

    print $opt{perl} ? "[ { preferred_resolution => { X => $X, Y => $Y } } ]" : "${X}x${Y}", "\n";
    1;
}

sub parse_X_log_edids {
    my ($log) = @_;
    my @edids;

    while ($log =~ /: EDID \(in hex\):\n((.*\n){8})/g) {
    
	my @lines = split '\n', $1;

        # Each line should consist of hex-decimal charactor
        my $edid = join('', map { /:\s+([0-9a-f]{32})$/ && $1 } @lines);
	if (length $edid == 256) {
	    @edids = uniq(@edids, $edid);
	} else {
	    warn "bad EDID found\n";
	}
    }
    if (@edids) {
	print "[\n" if $opt{perl};
	foreach my $edid (@edids) {
	    open(my $F, '| ' . parse_edid());
	    print $F pack("C*", map { hex($_) } $edid =~ /(..)/g);
	    close $F;
	    print ",\n" if $opt{perl};
	}
	print "]\n" if $opt{perl};
    }
    @edids;
}

sub uniq { my %l; $l{$_} = 1 foreach @_; grep { delete $l{$_} } @_ }

sub parse_edid() {
    join(' ', '/opt/userful/bin/umx-parse-edid.pl', propagate_options('verbose', 'MonitorsDB', 'perl'));
}

sub propagate_options {
    my (@l) = @_;
    map { $_ eq 'verbose' ? '-v' : "--$_" } grep { $opt{$_} } @l;
}
