
#include "cg_local.h"

//===================================================================

enum {
	DEMOCAM_FIRSTPERSON,
	DEMOCAM_THIRDPERSON,
	DEMOCAM_POSITIONAL,
	DEMOCAM_PATH_LINEAR,
	DEMOCAM_PATH_SIN,
	DEMOCAM_ORBITAL,

	DEMOCAM_MAX_TYPES
};

static char *cam_TypeNames[] = {
	"FirstPerson",
	"ThirdPerson",
	"Positional",
	"Path_linear",
	"Path_sin",
	"orbital",
	NULL
};

typedef struct cg_democam_s {
	int					type;
	unsigned int		timeStamp;
	int					trackEnt;
	vec3_t				origin;
	vec3_t				angles;
	int					fov;
	struct cg_democam_s *next;
} cg_democam_t;

cg_democam_t	*cg_cams_headnode = NULL;
cg_democam_t	*currentcam, *nextcam;

cvar_t		*demoname;
char		*demoscriptname;
qboolean	showviewermodel;
qboolean	democam_editing_mode;
unsigned int	demo_initial_timestamp;
unsigned int	demo_time;
static vec3_t	cam_origin, cam_angles, cam_velocity;
static float	cam_fov = 90;

//===============
//CG_Democam_FindCurrent
//===============
static cg_democam_t *CG_Democam_FindCurrent( void ) {
	unsigned int	higher_time = 0;
	cg_democam_t *cam, *currentcam;

	cam = cg_cams_headnode;
	currentcam = NULL;
	while( cam != NULL ) {
		if( cam->timeStamp <= demo_time && cam->timeStamp > higher_time ) {
			higher_time = cam->timeStamp;
			currentcam = cam;
		}
		cam = cam->next;
	}

	return currentcam;
}

//===============
//CG_Democam_FindNext
//===============
static cg_democam_t *CG_Democam_FindNext( void ) {
	unsigned int	lower_time = 0xFFFFFFFF;
	cg_democam_t *cam, *nextcam;

	cam = cg_cams_headnode;
	nextcam = NULL;
	while( cam != NULL ) {
		if( cam->timeStamp > demo_time && cam->timeStamp < lower_time ) {
			lower_time = cam->timeStamp;
			nextcam = cam;
		}
		cam = cam->next;
	}

	return nextcam;
}

//===============
//CG_Democam_RegisterCam
//===============
static cg_democam_t *CG_Democam_RegisterCam( int type ) {
	cg_democam_t	*cam;

	demo_time = cg.time - demo_initial_timestamp; // update demo time

	cam = cg_cams_headnode;
	while( cam != NULL ) {
		if( cam->timeStamp == demo_time ) { // a cam exists with the very same timestamp
			CG_Printf( "warning: There was a cam with the same timestamp, it's being replaced\n" );
			break;
		}
		cam = cam->next;
	}

	if( cam == NULL ) {
		cam = CG_Malloc( sizeof(cg_democam_t) );
		cam->next = cg_cams_headnode;
		cg_cams_headnode = cam;
	}

	cam->timeStamp = demo_time;
	cam->type = type;
	VectorCopy( cam_origin, cam->origin );
	VectorCopy( cam_angles, cam->angles );
	if( type == DEMOCAM_ORBITAL ) // in orbital cams, the angles are the angular velocity
		VectorSet( cam->angles, 0, 96, 0 );
	if( type == DEMOCAM_FIRSTPERSON || type == DEMOCAM_THIRDPERSON )
		cam->fov = 0;
	else
		cam->fov = 90;

	return cam;
}

//===============
//CG_Democam_UnregisterCam
//===============
static void CG_Democam_UnregisterCam( cg_democam_t *cam ) {
	cg_democam_t *tcam;

	if( !cam )
		return;

	// headnode shortcut
	if( cg_cams_headnode == cam ) {
		cg_cams_headnode = cg_cams_headnode->next;
		CG_Free( cam );
		return;
	}

	// find the camera which has this one as next;
	tcam = cg_cams_headnode;
	while( tcam != NULL ) {
		if( tcam->next == cam ) {
			tcam->next = cam->next;
			
			CG_Free( cam );
			break;
		}
		tcam = tcam->next;
	}
}

//===============
//CG_Democam_FreeCams
//===============
void CG_Democam_FreeCams( void ) {
	while( cg_cams_headnode ) {
		CG_Democam_UnregisterCam( cg_cams_headnode );
	}

	cg_cams_headnode = NULL;
}

//===============
//CG_LoadRecamScriptFile
//===============
qboolean CG_LoadRecamScriptFile( char *filename ) {
	int		filelen, filehandle;
	qbyte	*buf = NULL;
	char	*ptr, *token;
	int		linecount;
	cg_democam_t	*cam = NULL;

	if( !filename ) {
		CG_Printf( "CG_LoadRecamScriptFile: no filename\n" );
		return qfalse;
	}

	filelen = trap_FS_FOpenFile( filename, &filehandle, FS_READ );
	if( !filehandle || filelen < 1 ) {
		trap_FS_FCloseFile( filehandle );
	}
	else {
		buf = CG_Malloc( filelen + 1 );
		filelen = trap_FS_Read( buf, filelen, filehandle );
		trap_FS_FCloseFile( filehandle );
	}

	if( !buf )
		return qfalse;

	// parse the script
	linecount = 0;
	ptr = ( char * )buf;
	while( ptr ) {
		token = COM_ParseExt( &ptr, qtrue );
		if( !token || !token[0] )
			break;

		switch( linecount ) {
			case 0:
				cam = CG_Democam_RegisterCam( atoi(token) );
				break;
			case 1:
				cam->timeStamp = (unsigned int)atoi(token);
				break;
			case 2:
				cam->origin[0] = atof(token);
				break;
			case 3:
				cam->origin[1] = atof(token);
				break;
			case 4:
				cam->origin[2] = atof(token);
				break;
			case 5:
				cam->angles[0] = atof(token);
				break;
			case 6:
				cam->angles[1] = atof(token);
				break;
			case 7:
				cam->angles[2] = atof(token);
				break;
			case 8:
				cam->trackEnt = atoi(token);
				break;
			case 9:
				cam->fov = atoi(token);
				break;
			default:
				CG_Error( "CG_LoadRecamScriptFile: bad switch\n" );
		}
		
		linecount++;
		if( linecount == 10 )
			linecount = 0;
	}

	CG_Free( buf );
	if( linecount != 0 ) {
		CG_Printf( "CG_LoadRecamScriptFile: Invalid script. Ignored\n" );
		CG_Democam_FreeCams();
		return qfalse;
	}

	return qtrue;
}

//===============
//CG_SaveRecamScriptFile
//===============
void CG_SaveRecamScriptFile( char *filename ) {
	cg_democam_t *cam;
	int			filehandle;

	if( !cg_cams_headnode ) {
		CG_Printf( "CG_SaveRecamScriptFile: no cameras to save\n" );
		return;
	}

	if( !filename ) {
		filename = demoscriptname;
		if( !filename )
			return;
	}

	if( trap_FS_FOpenFile( filename, &filehandle, FS_WRITE ) == -1 ) {
		CG_Printf( "CG_SaveRecamScriptFile: Couldn't create the file %s\n", demoscriptname );
		return;
	}

	trap_FS_Printf( filehandle, "// cam script file generated by Warsow\n" );
	trap_FS_Printf( filehandle, "// demo start time: %i\n", demo_initial_timestamp );

	cam = cg_cams_headnode;
	while( cam != NULL ) {
		trap_FS_Printf( filehandle, "%i %i %f %f %f %f %f %f %i %i\n",
			cam->type,
			cam->timeStamp,
			cam->origin[0],
			cam->origin[1],
			cam->origin[2],
			cam->angles[0],
			cam->angles[1],
			cam->angles[2],
			cam->trackEnt, 
			cam->fov
			);
		cam = cam->next;
	}

	trap_FS_FCloseFile( filehandle );
	CG_Printf( "cam file saved\n" );
}

//===================================================================

//===============
//CG_DrawEntityNumbers
//===============
static void CG_DrawEntityNumbers( void ) 
{
	float		zfar = 2048;
	int			i, entnum;
	centity_t	*cent;
	vec2_t		coords;
	vec3_t		dir;
	float		dist;
	trace_t		trace;
	vec3_t		eorigin;
	int			xoffset = 0, yoffset = 0;

	for( i = 0; i < cg.frame.numEntities; i++ ) {
		entnum = cg.frame.parsedEntities[i&(MAX_PARSE_ENTITIES-1)].number;
		if( entnum < 1 || entnum >= MAX_EDICTS )
			continue;
		cent = &cg_entities[entnum];
		if( cent->serverFrame != cg.frame.serverFrame )
			continue;

		if( !cent->current.modelindex )
			continue;

		// Kill if behind the view
		VectorLerp( cent->prev.origin, cg.lerpfrac, cent->current.origin, eorigin );
		VectorSubtract( eorigin, cam_origin, dir );
		dist = VectorNormalize2( dir, dir ) * cg.view_fracDistFOV;
		if( dist > zfar )
			continue;

		if( DotProduct( dir, cg.v_forward ) < 0 )
			continue;
		
		CG_Trace( &trace, cam_origin, vec3_origin, vec3_origin, eorigin, cent->current.number, MASK_OPAQUE );
		if( trace.fraction == 1.0f ) {
			// find the 3d point in 2d screen
			trap_R_TransformVectorToScreen( &cg.refdef, eorigin, coords );
			trap_SCR_DrawString( coords[0]+xoffset+1,
				(cg.refdef.height - coords[1])+yoffset+1,
				ALIGN_LEFT_MIDDLE, va("%i",cent->current.number), cgs.fontSystemSmall, colorBlack );
			trap_SCR_DrawString( coords[0]+xoffset,
				(cg.refdef.height - coords[1])+yoffset,
				ALIGN_LEFT_MIDDLE, va("%i",cent->current.number), cgs.fontSystemSmall, colorWhite );
		}
	}
}

//===============
//CG_Draw2Ddemocam
//===============
qboolean CG_Draw2Ddemocam( void ) {
	int	xpos, ypos;
	char			*cam_type_name;
	unsigned int	cam_timestamp;
	char			sfov[8], strack[8];

	if( !cgs.demoPlaying )
		return qtrue;

	if( democam_editing_mode ) 
	{
		// draw the numbers of every entity in the view
		CG_DrawEntityNumbers();

		// draw the cams info
		xpos = 8;
		ypos = 100;

		if( demoname && demoname->string ) {
			trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Demo: %s",demoname->string), cgs.fontSystemSmall, colorWhite );
			ypos += trap_SCR_strHeight( cgs.fontSystemSmall );
		}

		trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Play mode: %s%s%s", S_COLOR_ORANGE, CamIsFree ? "Free Fly" : "Preview", S_COLOR_WHITE ), cgs.fontSystemSmall, colorWhite );
		ypos += trap_SCR_strHeight( cgs.fontSystemSmall );

		trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Time: %i", demo_time), cgs.fontSystemSmall, colorWhite );
		ypos += trap_SCR_strHeight( cgs.fontSystemSmall );

		cam_type_name = "none";
		cam_timestamp = 0;
		
		if( currentcam ) {
			cam_type_name = cam_TypeNames[currentcam->type];
			cam_timestamp = currentcam->timeStamp;
			Q_snprintfz( strack, sizeof(strack), "%i", currentcam->trackEnt );
			Q_snprintfz( sfov, sizeof(sfov), "%i", currentcam->fov );
		} else {
			Q_strncpyz( strack, "NO", sizeof(strack) );
			Q_strncpyz( sfov, "NO", sizeof(sfov) );
		}

		trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Current cam: "S_COLOR_ORANGE"%s"S_COLOR_WHITE" Fov "S_COLOR_ORANGE"%s"S_COLOR_WHITE" Start %i Tracking "S_COLOR_ORANGE"%s"S_COLOR_WHITE, 
			cam_type_name, sfov, cam_timestamp, strack ),
			cgs.fontSystemSmall, colorWhite );
		ypos += trap_SCR_strHeight( cgs.fontSystemSmall );

		if( currentcam ) {
			trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Pitch: "S_COLOR_ORANGE"%.2f"S_COLOR_WHITE" Yaw: "S_COLOR_ORANGE"%.2f"S_COLOR_WHITE" Roll: "S_COLOR_ORANGE"%.2f"S_COLOR_WHITE, 
			currentcam->angles[PITCH], currentcam->angles[YAW], currentcam->angles[ROLL] ),
			cgs.fontSystemSmall, colorWhite );
		}
		ypos += trap_SCR_strHeight( cgs.fontSystemSmall );

		cam_type_name = "none";
		cam_timestamp = 0;
		Q_strncpyz( sfov, "NO", sizeof(sfov) );
		if( nextcam ) {
			cam_type_name = cam_TypeNames[nextcam->type];
			cam_timestamp = nextcam->timeStamp;
			Q_snprintfz( strack, sizeof(strack), "%i", nextcam->trackEnt );
			Q_snprintfz( sfov, sizeof(sfov), "%i", nextcam->fov );
		} else {
			Q_strncpyz( strack, "NO", sizeof(strack) );
			Q_strncpyz( sfov, "NO", sizeof(sfov) );
		}

		trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Next cam: "S_COLOR_ORANGE"%s"S_COLOR_WHITE" Fov "S_COLOR_ORANGE"%s"S_COLOR_WHITE" Start %i Tracking "S_COLOR_ORANGE"%s"S_COLOR_WHITE, 
			cam_type_name, sfov, cam_timestamp, strack ),
			cgs.fontSystemSmall, colorWhite );
		ypos += trap_SCR_strHeight( cgs.fontSystemSmall );

		if( nextcam ) {
			trap_SCR_DrawString( xpos, ypos, ALIGN_LEFT_TOP, va("Pitch: "S_COLOR_ORANGE"%.2f"S_COLOR_WHITE" Yaw: "S_COLOR_ORANGE"%.2f"S_COLOR_WHITE" Roll: "S_COLOR_ORANGE"%.2f"S_COLOR_WHITE, 
				nextcam->angles[PITCH], nextcam->angles[YAW], nextcam->angles[ROLL] ),
				cgs.fontSystemSmall, colorWhite );
		}
		ypos += trap_SCR_strHeight( cgs.fontSystemSmall );
	}

	if( currentcam ) {
		// show the hud in the following cammera modes
		if( currentcam->type != DEMOCAM_FIRSTPERSON )
			return qfalse;
	}
	return qtrue;
}

//===================================================================

//===============
//CG_DemoCam_LookAt
//===============
qboolean CG_DemoCam_LookAt( int trackEnt, vec3_t vieworg, vec3_t viewangles ) {
	centity_t *cent;
	vec3_t		dir;
	vec3_t		origin;

	if( trackEnt < 1 || trackEnt >= MAX_EDICTS )
		return qfalse;

	cent = &cg_entities[trackEnt];
	if( cent->serverFrame != cg.frame.serverFrame )
		return qfalse;

	// seems to be valid. Find the angles to look at this entity
	VectorLerp( cent->prev.origin, cg.lerpfrac, cent->current.origin, origin );
	VectorSubtract( origin, vieworg, dir );
	VectorNormalize( dir );
	VecToAngles( dir, viewangles );
	return qtrue;
}

//===============
//CG_DemoCam_RFViewerModel
//===============
qboolean CG_DemoCam_RFViewerModel( void ) {
	if( !cgs.demoPlaying )
		return qtrue;

	return !showviewermodel;
}

qboolean CG_Democam_OverrideChasedNum( void ) {
	if( !cgs.demoPlaying )
		return qfalse;

	if( currentcam && currentcam->type != DEMOCAM_FIRSTPERSON && currentcam->type != DEMOCAM_THIRDPERSON ) {
		cg.chasedNum = -1;
		return qtrue;
	}

	return qfalse;
}

//===============
//CG_Democam_CalcView
//===============
qboolean CG_Democam_CalcView( void ) {
	int				i;
	float			lerpfrac, ratio;
	qboolean		forceview = qfalse;
	static vec3_t	orbital_angles;
	static float	orbital_radius;

	if( currentcam ) {
		if( !nextcam )
			lerpfrac = 0;
		else
			lerpfrac = (float)(demo_time - currentcam->timeStamp) / (float)(nextcam->timeStamp - currentcam->timeStamp);

		cg.thirdPerson = qfalse;
		switch( currentcam->type ) {
			case DEMOCAM_FIRSTPERSON:
				showviewermodel = cg.thirdPerson;
				forceview = qfalse;
				break;
			case DEMOCAM_THIRDPERSON:
				vweap.active = qfalse;
				cg.thirdPerson = qtrue;
				showviewermodel = qtrue;
				forceview = qfalse;
				break;
			case DEMOCAM_POSITIONAL:
				vweap.active = qfalse;
				showviewermodel = qtrue;
				forceview = qtrue;
				VectorCopy( currentcam->origin, cam_origin );
				if( !CG_DemoCam_LookAt( currentcam->trackEnt, cam_origin, cam_angles ) ) {
					VectorCopy( currentcam->angles, cam_angles );
				}
				cam_fov = currentcam->fov;
				break;
			case DEMOCAM_PATH_LINEAR:
				vweap.active = qfalse;
				showviewermodel = qtrue;
				forceview = qtrue;
				if( !nextcam || nextcam->type == DEMOCAM_FIRSTPERSON || nextcam->type == DEMOCAM_THIRDPERSON ) {
					CG_Printf( "Warning: CG_DemoCam: path_linear cam without a valid next cam\n" );
					VectorCopy( currentcam->origin, cam_origin );
					if( !CG_DemoCam_LookAt( currentcam->trackEnt, cam_origin, cam_angles ) ) {
						VectorCopy( currentcam->angles, cam_angles );
					}
					cam_fov = currentcam->fov;
				} else {
					VectorLerp( currentcam->origin, lerpfrac, nextcam->origin, cam_origin );
					if( !CG_DemoCam_LookAt( currentcam->trackEnt, cam_origin, cam_angles ) ) {
						for( i = 0; i < 3; i++ ) cam_angles[i] = LerpAngle( currentcam->angles[i], nextcam->angles[i], lerpfrac );
					}
					cam_fov = (float)currentcam->fov + (float)(nextcam->fov - currentcam->fov) * lerpfrac;
				}
				break;
			case DEMOCAM_PATH_SIN:
				vweap.active = qfalse;
				showviewermodel = qtrue;
				forceview = qtrue;
				clamp( lerpfrac, 0, 1 );
				ratio = sin( DEG2RAD(lerpfrac*90) );
				if( !nextcam || nextcam->type == DEMOCAM_FIRSTPERSON || nextcam->type == DEMOCAM_THIRDPERSON ) {
					CG_Printf( "Warning: CG_DemoCam: path_linear cam without a valid next cam\n" );
					VectorCopy( currentcam->origin, cam_origin );
					if( !CG_DemoCam_LookAt( currentcam->trackEnt, cam_origin, cam_angles ) ) {
						VectorCopy( currentcam->angles, cam_angles );
					}
					cam_fov = currentcam->fov;
				} else {
					VectorLerp( currentcam->origin, ratio, nextcam->origin, cam_origin );
					if( !CG_DemoCam_LookAt( currentcam->trackEnt, cam_origin, cam_angles ) ) {
						for( i = 0; i < 3; i++ ) cam_angles[i] = LerpAngle( currentcam->angles[i], nextcam->angles[i], ratio );
					}
					cam_fov = (float)currentcam->fov + (float)(nextcam->fov - currentcam->fov) * ratio;
				}
				break;
			case DEMOCAM_ORBITAL:
				vweap.active = qfalse;
				showviewermodel = qtrue;
				forceview = qtrue;
				cam_fov = currentcam->fov;
				if( !currentcam->trackEnt || currentcam->trackEnt >= MAX_EDICTS ) {
					CG_Printf( "Warning: CG_DemoCam: orbital cam needs a track entity set\n" );
					VectorCopy( currentcam->origin, cam_origin );
					VectorClear( cam_angles );
				}
				else {
					vec3_t center, forward;
					
					// find the trackEnt origin
					VectorLerp( cg_entities[currentcam->trackEnt].prev.origin, cg.lerpfrac, cg_entities[currentcam->trackEnt].current.origin, center );
					if( !orbital_radius ) { // cam is just started, find distance from cam to trackEnt and keep it as radius
						VectorSubtract( currentcam->origin, center, forward );
						orbital_radius = VectorNormalize( forward );
						VecToAngles( forward, orbital_angles );
					}
					orbital_angles[PITCH] += currentcam->angles[PITCH] * cg.frameTime; AngleNormalize360(orbital_angles[PITCH]);
					orbital_angles[YAW] += currentcam->angles[YAW] * cg.frameTime; AngleNormalize360(orbital_angles[YAW]);
					orbital_angles[ROLL] += currentcam->angles[ROLL] * cg.frameTime; AngleNormalize360(orbital_angles[ROLL]);
					AngleVectors( orbital_angles, forward, NULL, NULL );
					VectorMA( center, orbital_radius, forward, cam_origin );
					// lookat
					VectorInverse( forward );
					VecToAngles( forward, cam_angles );
				}
				break;
			default:
				forceview = qfalse;
				break;
		}

		if( currentcam->type != DEMOCAM_ORBITAL ) {
			VectorClear( orbital_angles );
			orbital_radius = 0;
		}
	}

	// if not forceview, update the local cam with the refdef position
	if( !forceview ) {
		VectorCopy( cg.refdef.viewangles, cam_angles );
		VectorCopy( cg.refdef.vieworg, cam_origin );
		cam_fov = cg.player.fov;
	}

	return forceview;
}

//===============
//CG_DemoCam
//===============
qboolean CG_DemoCam( void )
{
	qboolean		forceview = qfalse;

	if( !cgs.demoPlaying )
		return qfalse;

	//if( cg_noDemoRecam->integer )
	//	return qfalse;

	if( !demo_initial_timestamp && cg.frame.valid )
		demo_initial_timestamp = cg.time;

	demo_time = cg.time - demo_initial_timestamp;

	// see if we have any cams to be played
	currentcam = CG_Democam_FindCurrent();
	nextcam = CG_Democam_FindNext();

	if( CamIsFree ) {
		pmove_t		pm;

		vweap.active = qfalse;
		showviewermodel = qtrue;

		// copy current state to pmove
		memset( &pm, 0, sizeof(pm) );
		pm.s.pm_type = PM_SPECTATOR;
		VectorCopy( cam_origin, pm.s.origin );
		VectorCopy( cam_velocity, pm.s.velocity );
		pm.s.gravity = 800; // FIXME
		pm.max_walljumps = GS_GameType_MaxWallJumps( cg.frame.playerState.stats[STAT_GAMETYPE] );

		// run frame
		trap_NET_GetUserCmd( trap_NET_GetCurrentUserCmdNum() - 1, &pm.cmd );
		pm.cmd.msec = cg.realFrameTime * 1000;
		pm.cmd.forwardfrac = ((float)pm.cmd.forwardmove/(float)pm.cmd.msec);
		pm.cmd.sidefrac = ((float)pm.cmd.sidemove/(float)pm.cmd.msec);
		pm.cmd.upfrac = ((float)pm.cmd.upmove/(float)pm.cmd.msec);
		Pmove( &pm );

		VectorCopy( pm.s.origin, cam_origin );
		VectorCopy( pm.viewangles, cam_angles );
		VectorCopy( pm.s.velocity, cam_velocity );

		forceview = qtrue;
	}
	else if( currentcam ) { // Is preview mode
		forceview = CG_Democam_CalcView();
	}

	if( forceview ) {
		VectorCopy( cam_angles, cg.refdef.viewangles );
		VectorCopy( cam_origin, cg.refdef.vieworg );
		AngleVectors( cg.refdef.viewangles, cg.v_forward, cg.v_right, cg.v_up );

		// set fov
		if( currentcam && currentcam->fov && !CamIsFree ) {
			cg.refdef.fov_x = cam_fov;
		}
		else { // player fov
			if( !cg_demo_truePOV->integer ) {
				cg.refdef.fov_x = (cg_fov->integer < 1 || cg_fov->integer > 160) ? 90 : cg_fov->integer;
			} else {
				cg.refdef.fov_x = cg.player.fov;
			}
		}

		// used for outline LODs computations
		cg.view_fracDistFOV = tan( cg.refdef.fov_x * (M_PI/180) * 0.5f );
		cg.view_fracWeapFOV = ( 1.0f / cg.view_fracDistFOV ) * tan( cg_gun_fov->integer * (M_PI/180) * 0.5f );
	}

	return forceview;
}

//===============
//CG_DemoFreeFly_Cmd_f
//===============
static void CG_DemoFreeFly_Cmd_f( void ) {
	if( trap_Cmd_Argc() > 1 ) {
		if( !Q_stricmp( trap_Cmd_Argv(1), "on" ) )
			CamIsFree = qtrue;
		else if( !Q_stricmp( trap_Cmd_Argv(1), "off" ) )
			CamIsFree = qfalse;
	}
	else
		CamIsFree = !CamIsFree;
	CG_Printf( "demo cam mode %s\n", CamIsFree ? "Free Fly" : "Preview" );
}

//===============
//CG_AddCam_Cmd_f
//===============
static void CG_AddCam_Cmd_f( void ) {
	int type, i;

	if( trap_Cmd_Argc() == 2 ) {
		// type
		type = -1;
		for( i = 0; cam_TypeNames[i] != NULL; i++ ) {
			if( !Q_stricmp( cam_TypeNames[i], trap_Cmd_Argv(1) ) ) {
				type = i;
				break;
			}
		}

		if( type != -1 ) { // valid. Register and return
			if( CG_Democam_RegisterCam( type ) != NULL ) {
				CG_Printf( "cam added\n" );

				// update current cam
				currentcam = CG_Democam_FindCurrent();
				nextcam = CG_Democam_FindNext();
				return;
			}
		}
	}

	// print help
	CG_Printf( " : Usage: AddCam <type>\n" );
	CG_Printf( " : Available types:\n" );
	for( i = 0; cam_TypeNames[i] != NULL; i++ )
		CG_Printf( " : %s\n", cam_TypeNames[i] );
}

//===============
//CG_DeleteCam_Cmd_f
//===============
static void CG_DeleteCam_Cmd_f( void ) {
	if( !currentcam ) {
		CG_Printf( "DeleteCam: No current cam to delete\n" );
		return;
	}

	CG_Democam_UnregisterCam( currentcam );
	// update pointer to current cam
	currentcam = CG_Democam_FindCurrent();
	nextcam = CG_Democam_FindNext();
	CG_Printf( "cam deleted\n" );
}

//===============
//CG_EditCam_Cmd_f
//===============
static void CG_EditCam_Cmd_f( void ) {
	if( !currentcam ) {
		CG_Printf( "Editcam: no current cam\n" );
		return;
	}

	if( trap_Cmd_Argc() >= 2 && Q_stricmp( trap_Cmd_Argv(1), "help" ) ) 
	{
		if( !Q_stricmp( trap_Cmd_Argv(1), "type" ) ) {
			int	type, i;
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam type <type name>\n" );
				return;
			}
			// type
			type = -1;
			for( i = 0; cam_TypeNames[i] != NULL; i++ ) {
				if( !Q_stricmp( cam_TypeNames[i], trap_Cmd_Argv(2) ) ) {
					type = i;
					break;
				}
			}

			if( type != -1 ) { // valid. Register and return
				currentcam->type = type;
				CG_Printf( "cam edited\n" );
				return;
			} else {
				CG_Printf( "invalid type name\n" );
			}
		}
		if( !Q_stricmp( trap_Cmd_Argv(1), "track" ) ) {
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam track <entity number> ( 0 for no tracking )\n" );
				return;
			}
			currentcam->trackEnt = atoi( trap_Cmd_Argv(2) );
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "fov" ) ) {
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam fov <value>\n" );
				return;
			}
			currentcam->fov = atoi( trap_Cmd_Argv(2) );
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "timeOffset" ) ) {
			unsigned int newtimestamp;
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam timeOffset <value>\n" );
				return;
			}
			newtimestamp = currentcam->timeStamp += atoi( trap_Cmd_Argv(2) );
			if( newtimestamp + cg.time <= demo_initial_timestamp )
				newtimestamp = 1;
			currentcam->timeStamp = newtimestamp;
			currentcam = CG_Democam_FindCurrent();
			nextcam = CG_Democam_FindNext();
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "origin" ) ) {
			VectorCopy( cg.refdef.vieworg, currentcam->origin );
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "angles" ) ) {
			VectorCopy( cg.refdef.viewangles, currentcam->angles );
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "pitch" ) ) {
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam pitch <value>\n" );
				return; 
			}
			currentcam->angles[PITCH] = atof( trap_Cmd_Argv(2) );
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "yaw" ) ) {
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam yaw <value>\n" );
				return; 
			}
			currentcam->angles[YAW] = atof( trap_Cmd_Argv(2) );
			CG_Printf( "cam edited\n" );
			return;
		}
		else if( !Q_stricmp( trap_Cmd_Argv(1), "roll" ) ) {
			if( trap_Cmd_Argc() < 3 ) { // not enough parameters, print help
				CG_Printf( "Usage: EditCam roll <value>\n" );
				return; 
			}
			currentcam->angles[ROLL] = atof( trap_Cmd_Argv(2) );
			CG_Printf( "cam edited\n" );
			return;
		}
	}

	// print help
	CG_Printf( " : Usage: EditCam <command>\n" );
	CG_Printf( " : Available commands:\n" );
	CG_Printf( " : type <type name>\n" );
	CG_Printf( " : track <entity number> ( 0 for no track )\n" );
	CG_Printf( " : fov <value> ( only for not player views )\n" );
	CG_Printf( " : timeOffset <value> ( + or - milliseconds to be added to camera timestamp )\n" );
	CG_Printf( " : origin ( changes cam to current origin )\n" );
	CG_Printf( " : angles ( changes cam to current angles )\n" );
	CG_Printf( " : pitch <value> ( assigns pitch angle to current cam )\n" );
	CG_Printf( " : yaw <value> ( assigns yaw angle to current cam )\n" );
	CG_Printf( " : roll <value> ( assigns roll angle to current cam )\n" );
}

//===============
//CG_SaveCam_Cmd_f
//===============
void CG_SaveCam_Cmd_f( void ) {
	if( !cgs.demoPlaying )
		return;
	if( trap_Cmd_Argc() > 1 ) {
		char	*customName;
		int		custom_name_size;

		custom_name_size = sizeof(char) * (strlen("demos/") + strlen(trap_Cmd_Argv(1)) + strlen(".cam") + 1);
		customName = CG_Malloc( custom_name_size );
		Q_snprintfz( customName, custom_name_size, "demos/%s", trap_Cmd_Argv(1) );
		COM_ReplaceExtension( customName, ".cam", custom_name_size );
		CG_SaveRecamScriptFile( customName );
		CG_Free( customName );
		return;
	}

	CG_SaveRecamScriptFile( demoscriptname );
}

//===============
//CG_Democam_ImportCams_f
//===============
void CG_Democam_ImportCams_f( void ) {
	int		name_size;
	char	*customName;

	if( trap_Cmd_Argc() < 2 ) {
		CG_Printf( "Usage: importcams <filename> (relative to demos directory)\n" );
		return;
	}

	// see if there is any script for this demo, and load it
	name_size = sizeof(char) * (strlen("demos/") + strlen(trap_Cmd_Argv(1)) + strlen(".cam") + 1);
	customName = CG_Malloc( name_size );
	Q_snprintfz( customName, name_size, "demos/%s", trap_Cmd_Argv(1) );
	COM_ReplaceExtension( customName, ".cam", name_size );
	if( CG_LoadRecamScriptFile( customName ) ) {
		CG_Printf( "cam script imported\n" );
	} else {
		CG_Printf( "CG_Democam_ImportCams_f: no valid file found\n" );
	}
}

//===============
//CG_DemoEditMode_RemoveCmds
//===============
void CG_DemoEditMode_RemoveCmds( void ) {
	trap_Cmd_RemoveCommand( "addcam" );
	trap_Cmd_RemoveCommand( "deletecam" );
	trap_Cmd_RemoveCommand( "editcam" );
	trap_Cmd_RemoveCommand( "saverecam" );
	trap_Cmd_RemoveCommand( "clearcams" );
	trap_Cmd_RemoveCommand( "importcams" );
}

//===============
//CG_DemoEditMode_Cmd_f
//===============
static void CG_DemoEditMode_Cmd_f( void ) {
	if( !cgs.demoPlaying )
		return;

	if( trap_Cmd_Argc() > 1 ) {
		if( !Q_stricmp( trap_Cmd_Argv(1), "on" ) )
			democam_editing_mode = qtrue;
		else if( !Q_stricmp( trap_Cmd_Argv(1), "off" ) )
			democam_editing_mode = qfalse;
	}
	else
		democam_editing_mode = !democam_editing_mode;

	CG_Printf( "demo cam editing mode %s\n", democam_editing_mode ? "on" : "off" );
	if( democam_editing_mode ) {
		trap_Cmd_AddCommand( "addcam", CG_AddCam_Cmd_f );
		trap_Cmd_AddCommand( "deletecam", CG_DeleteCam_Cmd_f );
		trap_Cmd_AddCommand( "editcam", CG_EditCam_Cmd_f );
		trap_Cmd_AddCommand( "saverecam", CG_SaveCam_Cmd_f );
		trap_Cmd_AddCommand( "clearcams", CG_Democam_FreeCams );
		trap_Cmd_AddCommand( "importcams", CG_Democam_ImportCams_f );
	}
	else {
		CG_DemoEditMode_RemoveCmds();
	}
}

//===============
//CG_DemocamInit
//===============
void CG_DemocamInit( void ) {
	int name_size;
	democam_editing_mode = qfalse;
	demo_initial_timestamp = 0;

	if( !cgs.demoPlaying )
		return;

	demoname = trap_Cvar_Get( "demoname", "", 0 );
	if( !strlen(demoname->string) )
		CG_Error( "CG_LoadRecamScriptFile: no demo name string\n" );

	// see if there is any script for this demo, and load it
	name_size = sizeof(char) * (strlen("demos/") + strlen(demoname->string) + strlen(".cam") + 1);
	demoscriptname = CG_Malloc( name_size );
	Q_snprintfz( demoscriptname, name_size, "demos/%s", demoname->string );
	COM_ReplaceExtension( demoscriptname, ".cam", name_size );

	// add console commands
	trap_Cmd_AddCommand( "demoEditMode", CG_DemoEditMode_Cmd_f );
	trap_Cmd_AddCommand( "demoFreeFly", CG_DemoFreeFly_Cmd_f );

	if( CG_LoadRecamScriptFile( demoscriptname ) ) {
		CG_Printf( "Loaded demo cam script\n" );
	}
}

//===============
//CG_DemocamShutdown
//===============
void CG_DemocamShutdown( void ) {
	if( !cgs.demoPlaying )
		return;

	// remove console commands
	trap_Cmd_RemoveCommand( "demoEditMode" );
	trap_Cmd_RemoveCommand( "demoFreeFly" );
	if( democam_editing_mode ) {
		CG_DemoEditMode_RemoveCmds();
	}

	CG_Democam_FreeCams();
	CG_Free( demoscriptname );
	demoscriptname = NULL;
}

