/*
Copyright (C) 1997-2001 Id Software, Inc.

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  

See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

*/
// cl_demo.c  -- demo recording

#include "client.h"



/*
====================
CL_WriteDemoMessage

Dumps the current net message, prefixed by the length
====================
*/
void CL_WriteDemoMessage( msg_t *msg )
{
	int		len, swlen;

	if( cls.demofile <= 0 ) {
		cls.demorecording = qfalse;
		return;
	}

	// the first eight bytes are just packet sequencing stuff
	len = msg->cursize-8;
	swlen = LittleLong( len );

	// skip bad packets
	if( swlen ) {
		FS_Write( &swlen, 4, cls.demofile );
		FS_Write( msg->data+8, len, cls.demofile );
	}
}

/*
====================
CL_Stop_f

stop recording a demo
====================
*/
void CL_Stop_f( void )
{
	int			len, arg;
	qboolean	silent, cancel;

	// look through all the args
	silent = qfalse;
	cancel = qfalse;
	for( arg = 1; arg < Cmd_Argc(); arg++ )
	{
		if( !Q_stricmp(Cmd_Argv(arg), "silent") )
			silent = qtrue;
		else if( !Q_stricmp(Cmd_Argv(arg), "cancel") )
			cancel = qtrue;
	}

	if( !cls.demorecording )
	{
		if( !silent )
			Com_Printf( "Not recording a demo.\n" );
		return;
	}

	// finish up
	len = -1;
	FS_Write( &len, 4, cls.demofile );
	FS_FCloseFile( cls.demofile );

	// cancel the demos
	if( cancel )
	{
		// remove the file that correspond to cls.demofile
		if( !silent )
			Com_Printf( "Canceling demo: %s\n", cls.demofilename );
		if( !FS_RemoveFile(cls.demofilename) && !silent )
			Com_Printf( "Error canceling demo." );
	}

	if( !silent )
		Com_Printf( "Stopped demo: %s\n", cls.demofilename );

	cls.demofile = 0; // file id
	Mem_ZoneFree( cls.demofilename );
	cls.demofilename = NULL;
	cls.demorecording = qfalse;

}

/*
====================
CL_Record_f

record <demoname>

Begins recording a demo from the current position
====================
*/
void CL_Record_f (void)
{
	char		*name;
	int			i, name_size;
	char		buf_data[MAX_MSGLEN];
	msg_t		buf;
	int			len, length;
	qboolean	silent;
	purelist_t	*purefile;
	entity_state_t	*ent;
	entity_state_t	nullstate;

	if( cls.state != CA_ACTIVE ) {
		Com_Printf( "You must be in a level to record.\n" );
		return;
	}

	if( Cmd_Argc() < 2 ) {
		Com_Printf( "record <demoname>\n" );
		return;
	}
	
	if( Cmd_Argc() > 2 && !Q_stricmp(Cmd_Argv(2), "silent") ) {
		silent = qtrue;
	} else {
		silent = qfalse;
	}

	if( cls.demoplaying ) {
		if( !silent )
			Com_Printf( "You can't record from another demo.\n" );
		return;
	}

	if( cls.demorecording ) {
		if( !silent )
			Com_Printf( "Already recording.\n" );
		return;
	}

	//
	// open the demo file
	//
	name_size = sizeof(char) * (strlen("demos/") + strlen(Cmd_Argv(1)) + strlen(va(".wd%d", PROTOCOL_VERSION)) + 1);
	name = Mem_ZoneMalloc( name_size );

	Q_snprintfz( name, name_size, "demos/%s", Cmd_Argv(1) );
	COM_UserFilename( name );
	COM_DefaultExtension( name, va(".wd%d", PROTOCOL_VERSION), name_size );

	if( !COM_ValidateRelativeFilename(name) ) {
		if( !silent )
			Com_Printf( "Invalid filename.\n" );
		Mem_ZoneFree( name );
		return;
	}

	length = FS_FOpenFile( name, &cls.demofile, FS_WRITE );
	if( length == -1 ) {
		Com_Printf( "Error: Couldn't create the demo file.\n" );
		Mem_ZoneFree( name );
		return;
	}

	if( !silent )
		Com_Printf( "Recording demo: %s\n", name );

	// store the name in case we need it later
	cls.demofilename = name;
	cls.demorecording = qtrue;

	// don't start saving messages until a non-delta compressed message is received
	CL_AddReliableCommand( "nodelta" ); // request non delta compressed frame from server
	cls.demowaiting = qtrue;

	//
	// write out messages to hold the startup information
	//
	MSG_Init( &buf, (qbyte *)buf_data, sizeof(buf_data) );

	// send the serverdata
	MSG_WriteByte( &buf, svc_serverdata );
	MSG_WriteLong( &buf, PROTOCOL_VERSION );
	MSG_WriteLong( &buf, 0x10000 + cl.servercount );
	MSG_WriteString( &buf, FS_BaseGameDirectory() );
	MSG_WriteString( &buf, FS_GameDirectory() );
	MSG_WriteShort( &buf, cl.playernum );
	MSG_WriteString( &buf, cl.servermessage );
	if( cls.reliable ) {
		MSG_WriteByte( &buf, SV_BITFLAGS_RELIABLE );
	} else {
		MSG_WriteByte( &buf, 0 );
	}


	// pure files
	i = 0;
	purefile = cls.purelist;
	while( purefile ) {
		i++;
		purefile = purefile->next;
	}
	assert( i <= (short)0x7fff );

	MSG_WriteShort( &buf, i );
	purefile = cls.purelist;
	while( purefile ) {
		MSG_WriteString( &buf, purefile->filename );
		MSG_WriteLong( &buf, purefile->checksum );
		purefile = purefile->next;
	}

	len = LittleLong( buf.cursize );
	FS_Write( &len, 4, cls.demofile );
	FS_Write( buf.data, buf.cursize, cls.demofile );
	buf.cursize = 0;

	// configstrings
	for( i = 0; i < MAX_CONFIGSTRINGS; i++ )
	{
		if( cl.configstrings[i][0] )
		{
			MSG_WriteByte( &buf, svc_servercs );
			//MSG_WriteByte( &buf, svc_servercmd );
			MSG_WriteString( &buf, va("cs %i \"%s\"", i, cl.configstrings[i]) );

			if( buf.cursize > buf.maxsize / 2 )
			{	// write it out
				len = LittleLong( buf.cursize );
				FS_Write( &len, 4, cls.demofile );
				FS_Write( buf.data, buf.cursize, cls.demofile );
				buf.cursize = 0;
			}
		}
	}

	// baselines
	memset( &nullstate, 0, sizeof(nullstate) );
	for( i = 0; i < MAX_EDICTS; i++ )
	{
		ent = &cl_baselines[i];
		if( !ent->modelindex && !ent->sound && !ent->effects )
			continue;

		MSG_WriteByte( &buf, svc_spawnbaseline );
		MSG_WriteDeltaEntity( &nullstate, &cl_baselines[i], &buf, qtrue, qtrue, vec3_origin );
		if( buf.cursize > buf.maxsize / 2 )
		{	// write it out
			len = LittleLong( buf.cursize );
			FS_Write( &len, 4, cls.demofile );
			FS_Write( buf.data, buf.cursize, cls.demofile );
			buf.cursize = 0;
		}
	}

	MSG_WriteByte( &buf, svc_servercs );
	//MSG_WriteByte( &buf, svc_servercmd );
	MSG_WriteString( &buf, "precache" );

	// write it to the demo file
	len = LittleLong( buf.cursize );
	FS_Write( &len, 4, cls.demofile );
	FS_Write( buf.data, buf.cursize, cls.demofile );

	// the rest of the demo file will be individual frames
}


//================================================================
//
//	WARSOW : CLIENT SIDE DEMO PLAYBACK
//
//================================================================


// demo file
static int		demofilehandle;
static int		demofilelen;

/*
=================
CL_BeginDemoAviDump
=================
*/
static void CL_BeginDemoAviDump( void )
{
	cls.demoavi = qtrue;
	cls.demoavi_frame = 0;
	R_BeginAviDemo ();
}

/*
=================
CL_StopDemoAviDump
=================
*/
static void CL_StopDemoAviDump( void )
{
	cls.demoavi = qfalse;
	cls.demoavi_frame = 0;
	R_StopAviDemo ();
}

/*
=================
CL_DemoCompleted

Close the demo file and disable demo state. Called from disconnection proccess
=================
*/
void CL_DemoCompleted( void )
{
	if( cls.demoavi )
		CL_StopDemoAviDump();

	if( demofilehandle )
	{
		FS_FCloseFile( demofilehandle );
		demofilehandle = 0;
		demofilelen = 0;
	}

	cls.demoplaying = qfalse;

	Com_SetDemoPlaying( qfalse );
#ifdef DEMOCAM
	Cvar_ForceSet( "demoname", "" );
#endif
	Com_Printf( "Demo completed\n" );
}

/*
=================
CL_ReadDemoMessage

Read a packet from the demo file and send it to the messages parser
=================
*/
static void CL_ReadDemoMessage( void )
{
	static qbyte	msgbuf[MAX_MSGLEN];
	static msg_t	demomsg;
	int		msglen;

	if( !demofilehandle ) {
		CL_Disconnect( NULL );
		return;
	}

	if( demofilelen <= 0 ) {
		CL_Disconnect( NULL );
		return;
	}

	msglen = 0;

	// get the next message
	FS_Read( &msglen, 4, demofilehandle );
	demofilelen -= 4;
	
	msglen = LittleLong( msglen );
	if( msglen == -1 ) {
		CL_Disconnect( NULL );
		return;
	}

	if( msglen > MAX_MSGLEN )
		Com_Error( ERR_DROP, "Error reading demo file: msglen > MAX_MSGLEN" );
	
	if( demofilelen <= 0 )
		Com_Error( ERR_DROP, "Error reading demo file: End of file" );

	if( FS_Read(msgbuf, msglen, demofilehandle) != msglen )
		Com_Error( ERR_DROP, "Error reading demo file: End of file" );

	demofilelen -= msglen;

	demomsg.maxsize = sizeof(msgbuf);
	demomsg.data = msgbuf;
	demomsg.cursize = msglen;
	demomsg.readcount = 0;

	CL_ParseServerMessage( &demomsg );
}

/*
=================
CL_ReadDemoPackets

See if it's time to read a new demo packet
=================
*/
void CL_ReadDemoPackets( void )
{
	while( cls.demoplaying && (cl.receivedSnapNum <= 0 || !cl.frames[cl.receivedSnapNum&UPDATE_MASK].valid || cl.frames[cl.receivedSnapNum&UPDATE_MASK].serverTime < cl.serverTime) )
		CL_ReadDemoMessage();

	if( cls.demoplay_jump )
		cls.demoplay_jump = qfalse;
}

/*
====================
CL_StartDemo
====================
*/
static void CL_StartDemo( const char *demoname )
{
	char	*name, *servername;
	int		i, name_size, tempdemofilehandle, tempdemofilelen;

	if( !COM_ValidateRelativeFilename(demoname) ) {
		Com_Printf( "Invalid filename.\n" );
		return;
	}

	// have to copy the argument now, since next actions will lose it
	servername = ZoneCopyString( demoname );

	name_size = sizeof(char) * (strlen("demos/") + strlen(demoname) + strlen(va(".wd%d", PROTOCOL_VERSION)) + 1);
	name = Mem_TempMalloc( name_size );
	Q_snprintfz( name, name_size, "demos/%s", demoname );
	COM_DefaultExtension( name, va(".wd%d", PROTOCOL_VERSION), name_size );

	// open the demo file
	tempdemofilelen = FS_FOpenFile( name, &tempdemofilehandle, FS_READ );
	if( !tempdemofilehandle || tempdemofilelen < 1 ) {
		Com_Printf( "No valid demo file found\n" );
		FS_FCloseFile( tempdemofilehandle );
		Mem_TempFree( name );
		Mem_ZoneFree( servername );
		return;
	}

	// make sure a local server is killed
	Cbuf_ExecuteText( EXEC_NOW, "killserver\n" );
	CL_Disconnect( NULL );
	// wsw: Medar: fix for menu getting stuck on screen when starting demo, but maybe there is better fix out there?
	CL_UIModule_ForceMenuOff();

	demofilehandle = tempdemofilehandle;
	demofilelen = tempdemofilelen;

	cls.servername = servername;
	CL_SetClientState( CA_HANDSHAKE );
	Com_SetDemoPlaying( qtrue );
	cls.demoplaying = qtrue;

	cls.demoplay_ignore_next_frametime = qfalse;
	cls.demoplay_jump = qfalse;
#ifdef DEMOCAM
	cls.demopause = qfalse;
	Cvar_ForceSet( "demoname", servername );
#endif
	// set up for timedemo settings
	cl.timedemo_start = 0;
	cl.timedemo_frames = 0;
	for( i = 0; i < 100; i++ )
		cl.timedemo_counts[i] = 0;

	Mem_TempFree( name );
}

/*
====================
CL_PlayDemo_f

demo <demoname>
====================
*/
void CL_PlayDemo_f( void ) 
{
	if( Cmd_Argc() != 2 ) {
		Com_Printf( "demo <demoname>\n" );
		return;
	}
	CL_StartDemo( Cmd_Argv(1) );
}

#ifdef DEMOCAM
/*
====================
CL_PauseDemo_f				Can be used in democam -- PLX
====================
*/
void CL_PauseDemo_f( void )
{
	if( !cls.demoplaying ) {
		Com_Printf( "Can only demopause when playing a demo.\n" );
		return;
	}

	if( Cmd_Argc() > 1 ) {
		if( !Q_stricmp( Cmd_Argv(1), "on" ) ) 
			cls.demopause = qtrue;
		else if( !Q_stricmp( Cmd_Argv(1), "off" ) ) 
			cls.demopause = qfalse;
		return;
	}
	
	cls.demopause = !cls.demopause;
}
#endif

/*
====================
CL_DemoJump_f
====================
*/
void CL_DemoJump_f( void )
{
	qboolean relative;
	int time;
	char *p;

	if( !cls.demoplaying ) {
		Com_Printf( "Can only demojump when playing a demo\n" );
		return;
	}

	if( Cmd_Argc() != 2 ) {
		Com_Printf( "Usage: demojump <time>\n" );
		Com_Printf( "Time format is [minutes:]seconds\n" );
		Com_Printf( "Use '+' or '-' in front of the time to specify it in relation to current position\n" );
		return;
	}

	p = Cmd_Argv(1);

	if( Cmd_Argv(1)[0] == '+' || Cmd_Argv(1)[0] == '-' ) {
		relative = qtrue;
		p++;
	} else {
		relative = qfalse;
	}

	if( strchr(p, ':'))
		time = (atoi(p) * 60 + atoi(strchr(p, ':') + 1)) * 1000;
	else
		time = atoi(p) * 1000;

	if( Cmd_Argv(1)[0] == '-' )
		time = -time;

	if( relative )
		cls.gametime += time;
	else
		cls.gametime = time; // gametime always starts from 0

	if( cl.serverTime < cl.frames[cl.receivedSnapNum&UPDATE_MASK].serverTime )
		cl.pendingSnapNum = 0;

	CL_AdjustServerTime();

	if( cl.serverTime < cl.frames[cl.receivedSnapNum&UPDATE_MASK].serverTime ) {
		demofilelen += FS_Tell( demofilehandle );
		FS_Seek( demofilehandle, 0, FS_SEEK_SET );
		cl.currentSnapNum = cl.receivedSnapNum = 0;
	}

	cls.demoplay_jump = qtrue;
}

/*
====================
CL_PlayDemoToAvi_f

demoavi <demoname> (if no name suplied, toogles demoavi status)
====================
*/
void CL_PlayDemoToAvi_f( void )
{
	if( Cmd_Argc() == 1 && cls.demoplaying ) // toogle demoavi mode
	{
		if( !cls.demoavi ) {
			CL_BeginDemoAviDump();
		} else {
			CL_StopDemoAviDump();
		}
	}
	else if( Cmd_Argc() == 2 )
	{
		char *tempname = TempCopyString( Cmd_Argv(1) );

		if( cls.demoplaying )
			CL_Disconnect( NULL );

		CL_StartDemo( tempname );
		CL_BeginDemoAviDump();

		Mem_TempFree( tempname );
	}
	else
	{
		Com_Printf( "Usage: %sdemoavi <demoname>%s or %sdemoavi%s while playing a demo\n",
			S_COLOR_YELLOW, S_COLOR_WHITE, S_COLOR_YELLOW, S_COLOR_WHITE );
	}
}

//[end]
